package modules

import (
	"sync"
	"time"

	"strconv"

	Logger "pkg.deepin.io/daemon/sync/infrastructure/log"
	"pkg.deepin.io/daemon/sync/infrastructure/storage"
	"pkg.deepin.io/daemon/sync/infrastructure/utils"
	"pkg.deepin.io/daemon/sync/modules/appearance"
	"pkg.deepin.io/daemon/sync/modules/audio"
	"pkg.deepin.io/daemon/sync/modules/background"
	"pkg.deepin.io/daemon/sync/modules/dock"
	"pkg.deepin.io/daemon/sync/modules/launcher"
	. "pkg.deepin.io/daemon/sync/modules/model"
	"pkg.deepin.io/daemon/sync/modules/network"
	"pkg.deepin.io/daemon/sync/modules/peripherals"
	"pkg.deepin.io/daemon/sync/modules/power"
	"pkg.deepin.io/daemon/sync/modules/screenedge"
	"pkg.deepin.io/daemon/sync/modules/screensaver"
	"pkg.deepin.io/daemon/sync/modules/updater"
)

type Manager struct {
	hwID     string
	userID   *int64
	token    storage.TokenSource
	switcher *Switcher

	itemBusSet    map[string]*ModuleBusInfo
	itemBusLocker sync.RWMutex

	items       DataIFCList
	itemsLocker sync.RWMutex
	itemDataMap map[string]DataIFC
}

const (
	cloudDuration = time.Second * 5
)

func NewManager(userID *int64, hwID string, token storage.TokenSource) (*Manager, error) {
	var m = Manager{
		hwID:        hwID,
		userID:      userID,
		token:       token,
		items:       DataIFCList{},
		itemBusSet:  make(map[string]*ModuleBusInfo),
		itemDataMap: make(map[string]DataIFC),
	}
	m.initItemDataMap()
	m.loadSwitcher()

	return &m, nil
}

func (m *Manager) Register(module string, info *ModuleBusInfo) error {
	m.setItem(module)
	m.setItemBus(module, info)
	return nil
}

func (m *Manager) SwitcherSet(name string, enabled bool) (bool, error) {
	changed, err := m.switcher.Enable(name, enabled)
	if err != nil {
		return false, err
	}
	if !changed {
		return false, nil
	}
	return true, m.switcher.WriteFile(localSwitcherFile)
}

func (m *Manager) SwitcherGet(name string) bool {
	return m.switcher.IsEnabled(name)
}

func (m *Manager) SwitcherDump() ([]byte, error) {
	return m.switcher.Bytes()
}

func (m *Manager) GetLastSyncTime() (int64, error) {
	if *m.userID < 1 {
		return 0, nil
	}
	cloud, err := m.getCloudStorage()
	if err != nil {
		return 0, err
	}
	ret, err := cloud.GetLatest(cloudKeyIndex)
	if err != nil {
		return 0, err
	}
	if len(ret.Version) == 0 {
		return 0, nil
	}
	list := []byte(ret.Version)
	l := len(list)
	// NOTICE: version format: timestamp + 4 chars
	if l != 14 {
		return 0, nil
	}
	return strconv.ParseInt(string(list[:l-4]), 10, 64)
}

func (m *Manager) getCloudStorage() (*utils.CloudStorage, error) {
	cloud, err := utils.CloudConnect(m.token, m.hwID)
	if err != nil {
		Logger.Warning("[Modules] Failed to connect cloud:", err)
		return nil, err
	}
	return cloud, nil
}

func (m *Manager) loadSwitcher() {
	s, err := newSwitcher(ConfigSwitcherFile)
	if err != nil {
		Logger.Warning("[Modules] Failed to load system switcher:", err)
	}
	m.switcher = s

	s, err = newSwitcher(localSwitcherFile)
	if err != nil {
		return
	}
	m.switcher.Fixed(s)
}

func (m *Manager) initItemDataMap() {
	m.itemDataMap[appearance.Name] = &appearance.Data{}
	m.itemDataMap[background.Name] = &background.Data{}
	m.itemDataMap[dock.Name] = &dock.Data{}
	m.itemDataMap[launcher.Name] = &launcher.Data{}
	m.itemDataMap[network.Name] = &network.Data{}
	m.itemDataMap[peripherals.Name] = &peripherals.Data{}
	m.itemDataMap[power.Name] = &power.Data{}
	m.itemDataMap[screenedge.Name] = &screenedge.Data{}
	m.itemDataMap[screensaver.Name] = &screensaver.Data{}
	m.itemDataMap[audio.Name] = &audio.Data{}
	m.itemDataMap[updater.Name] = &updater.Data{}
}

func (m *Manager) getModuleData(name string) ([]byte, error) {
	if !m.SwitcherGet(name) {
		Logger.Debug("[Modules] the module has been disabled:",
			name)
		return nil, nil
	}
	busInfo := m.getItemBus(name)
	if busInfo == nil {
		Logger.Debug("[Modules] the module no bus info register:",
			name)
		return nil, nil
	}
	data, err := busInfo.Get()
	if err != nil {
		return nil, err
	}
	// Logger.Info("[Module] Get data:", name, string(data))
	return data, nil
}

func (m *Manager) setModuleData(name string, data []byte) error {
	busInfo := m.getItemBus(name)
	if busInfo == nil {
		Logger.Debug("[Modules] the module no bus info register:",
			name)
		return nil
	}
	// Logger.Info("[Modules] [Set] local data:", name, string(data))
	return busInfo.Set(data)
}

func (m *Manager) setItemBus(name string, info *ModuleBusInfo) {
	m.itemBusLocker.Lock()
	m.itemBusSet[name] = info
	m.itemBusLocker.Unlock()
}

func (m *Manager) getItemBus(name string) *ModuleBusInfo {
	m.itemBusLocker.RLock()
	defer m.itemBusLocker.RUnlock()
	return m.itemBusSet[name]
}

func (m *Manager) setItem(name string) {
	m.itemsLocker.Lock()
	defer m.itemsLocker.Unlock()
	info, ok := m.itemDataMap[name]
	if !ok {
		Logger.Warning("Unspported module:", name)
		return
	}
	m.items = m.items.Append(name, info)
}

func (m *Manager) getItem(name string) DataIFC {
	m.itemsLocker.RLock()
	defer m.itemsLocker.RUnlock()
	return m.items.Get(name)
}
