/* t-getmechanismlist.c - Regression test.
 * Copyright (C) 2006 g10 Code GmbH
 *
 * This file is part of Scute.
 *
 * Scute is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * Scute is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, see <https://gnu.org/licenses/>.
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#include <stdio.h>
#include <stdbool.h>

#include "t-support.h"


int
main (int argc, char *argv[])
{
  CK_RV err;
  CK_SLOT_ID_PTR slots;
  CK_ULONG slots_count;
  unsigned int i;

  (void) argc;
  (void) argv;

  init_cryptoki ();

  err = C_GetSlotList (true, NULL, &slots_count);
  fail_if_err (err);

  if (slots_count == 0)
    {
      printf ("Skipping test because no token is present.\n");
      return 77;
    }

  printf ("Number of slots with tokens: %lu\n", slots_count);
  slots = malloc (sizeof (CK_SLOT_ID) * slots_count);
  if (!slots)
    fail_if_err (CKR_HOST_MEMORY);

  err = C_GetSlotList (true, slots, &slots_count);
  fail_if_err (err);

  for (i = 0; i < slots_count; i++)
    {
      CK_MECHANISM_TYPE_PTR mechanisms;
      CK_ULONG mechanisms_count;
      unsigned int j;

      printf ("%2i. Slot ID %lu\n", i, slots[i]);

      err = C_GetMechanismList (slots[i], NULL, &mechanisms_count);
      fail_if_err (err);

      printf ("    Mechanisms: %lu\n", mechanisms_count);
      mechanisms = malloc (sizeof (CK_MECHANISM_TYPE) * mechanisms_count);
      if (!mechanisms)
	fail_if_err (CKR_HOST_MEMORY);

      err = C_GetMechanismList (slots[i], mechanisms, &mechanisms_count);
      fail_if_err (err);

      for (j = 0; j < mechanisms_count; j++)
	{
	  CK_MECHANISM_INFO info;

	  printf ("    %2i. %s\n", j, mechanism_type_str (mechanisms[j]));

	  err = C_GetMechanismInfo (slots[i], mechanisms[j], &info);
	  fail_if_err (err);

	  printf ("        Minimum key size: %lu\n", info.ulMinKeySize);
	  printf ("        Maximum key size: %lu\n", info.ulMaxKeySize);
	  printf ("        Flags: %#lx", info.flags);

	  if (info.flags)
	    {
	      bool any = false;
	      CK_FLAGS xflags = 0;

	      printf (" == ");
#define DO_FLAG(sym)					\
	  if (info.flags & sym)				\
	    {						\
	      printf ("%s" #sym, any ? " | " : "");	\
	      any = true;				\
              xflags |= sym;				\
	    }
	      DO_FLAG (CKF_HW);
	      DO_FLAG (CKF_ENCRYPT);
	      DO_FLAG (CKF_DECRYPT);
	      DO_FLAG (CKF_DIGEST);
	      DO_FLAG (CKF_SIGN);
	      DO_FLAG (CKF_SIGN_RECOVER);
	      DO_FLAG (CKF_VERIFY);
	      DO_FLAG (CKF_VERIFY_RECOVER);
	      DO_FLAG (CKF_GENERATE);
	      DO_FLAG (CKF_GENERATE_KEY_PAIR);
	      DO_FLAG (CKF_WRAP);
	      DO_FLAG (CKF_UNWRAP);
	      DO_FLAG (CKF_DERIVE);
	      DO_FLAG (CKF_EXTENSION);

	      xflags = info.flags & ~xflags;
	      if (xflags)
		printf ("%s%#lx", any ? " | " : "", xflags);
	    }
	  printf ("\n");
	}
      free (mechanisms);
    }

  return 0;
}
