% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time-point.R
\name{time-point-arithmetic}
\alias{time-point-arithmetic}
\alias{add_weeks.clock_time_point}
\alias{add_days.clock_time_point}
\alias{add_hours.clock_time_point}
\alias{add_minutes.clock_time_point}
\alias{add_seconds.clock_time_point}
\alias{add_milliseconds.clock_time_point}
\alias{add_microseconds.clock_time_point}
\alias{add_nanoseconds.clock_time_point}
\title{Arithmetic: Time points}
\usage{
\method{add_weeks}{clock_time_point}(x, n, ...)

\method{add_days}{clock_time_point}(x, n, ...)

\method{add_hours}{clock_time_point}(x, n, ...)

\method{add_minutes}{clock_time_point}(x, n, ...)

\method{add_seconds}{clock_time_point}(x, n, ...)

\method{add_milliseconds}{clock_time_point}(x, n, ...)

\method{add_microseconds}{clock_time_point}(x, n, ...)

\method{add_nanoseconds}{clock_time_point}(x, n, ...)
}
\arguments{
\item{x}{\verb{[clock_sys_time / clock_naive_time]}

A time point vector.}

\item{n}{\verb{[integer / clock_duration]}

An integer vector to be converted to a duration, or a duration
corresponding to the arithmetic function being used. This corresponds
to the number of duration units to add. \code{n} may be negative to subtract
units of duration.}

\item{...}{These dots are for future extensions and must be empty.}
}
\value{
\code{x} after performing the arithmetic.
}
\description{
These are naive-time and sys-time methods for the
\link[=clock-arithmetic]{arithmetic generics}.
\itemize{
\item \code{add_weeks()}
\item \code{add_days()}
\item \code{add_hours()}
\item \code{add_minutes()}
\item \code{add_seconds()}
\item \code{add_milliseconds()}
\item \code{add_microseconds()}
\item \code{add_nanoseconds()}
}

When working with zoned times, generally you convert to either sys-time
or naive-time, add the duration, then convert back to zoned time. Typically,
\emph{weeks and days} are added in \emph{naive-time}, and \emph{hours, minutes, seconds,
and subseconds} are added in \emph{sys-time}.

If you aren't using zoned times, arithmetic on sys-times and naive-time
is equivalent.

If you need to add larger irregular units of time, such as months, quarters,
or years, convert to a calendar type with a converter like
\code{\link[=as_year_month_day]{as_year_month_day()}}.
}
\details{
\code{x} and \code{n} are recycled against each other using
\link[vctrs:theory-faq-recycling]{tidyverse recycling rules}.
}
\examples{
library(magrittr)

# Say you started with this zoned time, and you want to add 1 day to it
x <- as_naive_time(year_month_day(1970, 04, 25, 02, 30, 00))
x <- as_zoned_time(x, "America/New_York")
x

# Note that there was a daylight saving time gap on 1970-04-26 where
# we jumped from 01:59:59 -> 03:00:00.

# You can choose to add 1 day in "system time", by first converting to
# sys-time (the equivalent UTC time), adding the day, then converting back to
# zoned time. If you sat still for exactly 86,400 seconds, this is the
# time that you would see after daylight saving time adjusted the clock
# (note that the hour field is shifted forward by the size of the gap)
as_sys_time(x)

x \%>\%
  as_sys_time() \%>\%
  add_days(1) \%>\%
  as_zoned_time(zoned_time_zone(x))

# Alternatively, you can add 1 day in "naive time". Naive time represents
# a clock time with a yet-to-be-specified time zone. It tries to maintain
# smaller units where possible, so adding 1 day would attempt to return
# "1970-04-26T02:30:00" in the America/New_York time zone, but...
as_naive_time(x)

try({
x \%>\%
  as_naive_time() \%>\%
  add_days(1) \%>\%
  as_zoned_time(zoned_time_zone(x))
})

# ...this time doesn't exist in that time zone! It is "nonexistent".
# You can resolve nonexistent times by setting the `nonexistent` argument
# when converting to zoned time. Let's roll forward to the next available
# moment in time.
x \%>\%
  as_naive_time() \%>\%
  add_days(1) \%>\%
  as_zoned_time(zoned_time_zone(x), nonexistent = "roll-forward")
}
