/*
    libzint - the open source barcode library
    Copyright (C) 2021-2022 Robin Stuart <rstuart114@gmail.com>

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:

    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the project nor the names of its contributors
       may be used to endorse or promote products derived from this software
       without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
    ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
    ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
    DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
    OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
    LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
    OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
    SUCH DAMAGE.
 */
/* SPDX-License-Identifier: BSD-3-Clause */

#include "testcommon.h"
#include "../iso3166.h"

static void test_numeric(const testCtx *const p_ctx) {

    struct item {
        int data;
        int ret;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    struct item data[] = {
        /*  0*/ { -1, 0 },
        /*  1*/ { 0, 0 },
        /*  2*/ { 1, 0 },
        /*  3*/ { 2, 0 },
        /*  4*/ { 3, 0 },
        /*  5*/ { 4, 1 },
        /*  6*/ { 5, 0 },
        /*  7*/ { 6, 0 },
        /*  8*/ { 7, 0 },
        /*  9*/ { 8, 1 },
        /* 10*/ { 9, 0 },
        /* 11*/ { 10, 1 },
        /* 12*/ { 11, 0 },
        /* 13*/ { 12, 1 },
        /* 14*/ { 13, 0 },
        /* 15*/ { 14, 0 },
        /* 16*/ { 15, 0 },
        /* 17*/ { 16, 1 },
        /* 18*/ { 17, 0 },
        /* 19*/ { 18, 0 },
        /* 20*/ { 19, 0 },
        /* 21*/ { 20, 1 },
        /* 22*/ { 21, 0 },
        /* 23*/ { 22, 0 },
        /* 24*/ { 23, 0 },
        /* 25*/ { 24, 1 },
        /* 26*/ { 25, 0 },
        /* 27*/ { 26, 0 },
        /* 28*/ { 27, 0 },
        /* 29*/ { 28, 1 },
        /* 30*/ { 29, 0 },
        /* 31*/ { 30, 0 },
        /* 32*/ { 31, 1 },
        /* 33*/ { 32, 1 },
        /* 34*/ { 33, 0 },
        /* 35*/ { 34, 0 },
        /* 36*/ { 35, 0 },
        /* 37*/ { 36, 1 },
        /* 38*/ { 37, 0 },
        /* 39*/ { 38, 0 },
        /* 40*/ { 39, 0 },
        /* 41*/ { 40, 1 },
        /* 42*/ { 41, 0 },
        /* 43*/ { 47, 0 },
        /* 44*/ { 48, 1 },
        /* 45*/ { 49, 0 },
        /* 46*/ { 50, 1 },
        /* 47*/ { 51, 1 },
        /* 48*/ { 52, 1 },
        /* 49*/ { 53, 0 },
        /* 50*/ { 67, 0 },
        /* 51*/ { 68, 1 },
        /* 52*/ { 69, 0 },
        /* 53*/ { 70, 1 },
        /* 54*/ { 71, 0 },
        /* 55*/ { 72, 1 },
        /* 56*/ { 73, 0 },
        /* 57*/ { 74, 1 },
        /* 58*/ { 75, 0 },
        /* 59*/ { 76, 1 },
        /* 60*/ { 77, 0 },
        /* 61*/ { 83, 0 },
        /* 62*/ { 84, 1 },
        /* 63*/ { 85, 0 },
        /* 64*/ { 86, 1 },
        /* 65*/ { 87, 0 },
        /* 66*/ { 99, 0 },
        /* 67*/ { 100, 1 },
        /* 68*/ { 101, 0 },
        /* 69*/ { 110, 0 },
        /* 70*/ { 119, 0 },
        /* 71*/ { 120, 1 },
        /* 72*/ { 121, 0 },
        /* 73*/ { 130, 0 },
        /* 74*/ { 131, 0 },
        /* 75*/ { 132, 1 },
        /* 76*/ { 133, 0 },
        /* 77*/ { 147, 0 },
        /* 78*/ { 148, 1 },
        /* 79*/ { 149, 0 },
        /* 80*/ { 150, 0 },
        /* 81*/ { 151, 0 },
        /* 82*/ { 152, 1 },
        /* 83*/ { 153, 0 },
        /* 84*/ { 154, 0 },
        /* 85*/ { 155, 0 },
        /* 86*/ { 156, 1 },
        /* 87*/ { 157, 0 },
        /* 88*/ { 158, 1 },
        /* 89*/ { 159, 0 },
        /* 90*/ { 160, 0 },
        /* 91*/ { 161, 0 },
        /* 92*/ { 162, 1 },
        /* 93*/ { 163, 0 },
        /* 94*/ { 169, 0 },
        /* 95*/ { 170, 1 },
        /* 96*/ { 171, 0 },
        /* 97*/ { 177, 0 },
        /* 98*/ { 178, 1 },
        /* 99*/ { 179, 0 },
        /*100*/ { 180, 1 },
        /*101*/ { 181, 0 },
        /*102*/ { 190, 0 },
        /*103*/ { 191, 1 },
        /*104*/ { 192, 1 },
        /*105*/ { 193, 0 },
        /*106*/ { 200, 0 },
        /*107*/ { 210, 0 },
        /*108*/ { 220, 0 },
        /*109*/ { 230, 0 },
        /*110*/ { 231, 1 },
        /*111*/ { 232, 1 },
        /*112*/ { 233, 1 },
        /*113*/ { 234, 1 },
        /*114*/ { 235, 0 },
        /*115*/ { 236, 0 },
        /*116*/ { 237, 0 },
        /*117*/ { 238, 1 },
        /*118*/ { 239, 1 },
        /*119*/ { 240, 0 },
        /*120*/ { 241, 0 },
        /*121*/ { 242, 1 },
        /*122*/ { 243, 0 },
        /*123*/ { 244, 0 },
        /*124*/ { 245, 0 },
        /*125*/ { 246, 1 },
        /*126*/ { 247, 0 },
        /*127*/ { 248, 1 },
        /*128*/ { 249, 0 },
        /*129*/ { 250, 1 },
        /*130*/ { 251, 0 },
        /*131*/ { 259, 0 },
        /*132*/ { 260, 1 },
        /*133*/ { 261, 0 },
        /*134*/ { 269, 0 },
        /*135*/ { 270, 1 },
        /*136*/ { 271, 0 },
        /*137*/ { 280, 0 },
        /*138*/ { 287, 0 },
        /*139*/ { 288, 1 },
        /*140*/ { 289, 0 },
        /*141*/ { 290, 0 },
        /*142*/ { 291, 0 },
        /*143*/ { 292, 1 },
        /*144*/ { 293, 0 },
        /*145*/ { 299, 0 },
        /*146*/ { 300, 1 },
        /*147*/ { 301, 0 },
        /*148*/ { 310, 0 },
        /*149*/ { 319, 0 },
        /*150*/ { 320, 1 },
        /*151*/ { 321, 0 },
        /*152*/ { 322, 0 },
        /*153*/ { 323, 0 },
        /*154*/ { 324, 1 },
        /*155*/ { 325, 0 },
        /*156*/ { 330, 0 },
        /*157*/ { 339, 0 },
        /*158*/ { 340, 1 },
        /*159*/ { 341, 0 },
        /*160*/ { 350, 0 },
        /*161*/ { 367, 0 },
        /*162*/ { 368, 1 },
        /*163*/ { 369, 0 },
        /*164*/ { 370, 0 },
        /*165*/ { 379, 0 },
        /*166*/ { 380, 1 },
        /*167*/ { 381, 0 },
        /*168*/ { 397, 0 },
        /*169*/ { 398, 1 },
        /*170*/ { 399, 0 },
        /*171*/ { 400, 1 },
        /*172*/ { 401, 0 },
        /*173*/ { 409, 0 },
        /*174*/ { 410, 1 },
        /*175*/ { 411, 0 },
        /*176*/ { 426, 1 },
        /*177*/ { 427, 0 },
        /*178*/ { 428, 1 },
        /*179*/ { 429, 0 },
        /*180*/ { 430, 1 },
        /*181*/ { 431, 0 },
        /*182*/ { 432, 0 },
        /*183*/ { 433, 0 },
        /*184*/ { 434, 1 },
        /*185*/ { 435, 0 },
        /*186*/ { 436, 0 },
        /*187*/ { 437, 0 },
        /*188*/ { 438, 1 },
        /*189*/ { 439, 0 },
        /*190*/ { 440, 1 },
        /*191*/ { 441, 0 },
        /*192*/ { 442, 1 },
        /*193*/ { 443, 0 },
        /*194*/ { 449, 0 },
        /*195*/ { 450, 1 },
        /*196*/ { 451, 0 },
        /*197*/ { 460, 0 },
        /*198*/ { 469, 0 },
        /*199*/ { 470, 1 },
        /*200*/ { 471, 0 },
        /*201*/ { 479, 0 },
        /*202*/ { 480, 1 },
        /*203*/ { 481, 0 },
        /*204*/ { 490, 0 },
        /*205*/ { 497, 0 },
        /*206*/ { 498, 1 },
        /*207*/ { 499, 1 },
        /*208*/ { 500, 1 },
        /*209*/ { 501, 0 },
        /*210*/ { 502, 0 },
        /*211*/ { 503, 0 },
        /*212*/ { 504, 1 },
        /*213*/ { 505, 0 },
        /*214*/ { 510, 0 },
        /*215*/ { 519, 0 },
        /*216*/ { 520, 1 },
        /*217*/ { 521, 0 },
        /*218*/ { 530, 0 },
        /*219*/ { 531, 1 },
        /*220*/ { 532, 0 },
        /*221*/ { 539, 0 },
        /*222*/ { 540, 1 },
        /*223*/ { 541, 0 },
        /*224*/ { 550, 0 },
        /*225*/ { 561, 0 },
        /*226*/ { 562, 1 },
        /*227*/ { 563, 0 },
        /*228*/ { 564, 0 },
        /*229*/ { 565, 0 },
        /*230*/ { 566, 1 },
        /*231*/ { 567, 0 },
        /*232*/ { 568, 0 },
        /*233*/ { 569, 0 },
        /*234*/ { 570, 1 },
        /*235*/ { 571, 0 },
        /*236*/ { 579, 0 },
        /*237*/ { 580, 1 },
        /*238*/ { 581, 1 },
        /*239*/ { 582, 0 },
        /*240*/ { 590, 0 },
        /*241*/ { 599, 0 },
        /*242*/ { 600, 1 },
        /*243*/ { 601, 0 },
        /*244*/ { 611, 0 },
        /*245*/ { 612, 1 },
        /*246*/ { 613, 0 },
        /*247*/ { 614, 0 },
        /*248*/ { 615, 0 },
        /*249*/ { 616, 1 },
        /*250*/ { 617, 0 },
        /*251*/ { 618, 0 },
        /*252*/ { 619, 0 },
        /*253*/ { 620, 1 },
        /*254*/ { 621, 0 },
        /*255*/ { 630, 1 },
        /*256*/ { 640, 0 },
        /*257*/ { 650, 0 },
        /*258*/ { 658, 0 },
        /*259*/ { 659, 1 },
        /*260*/ { 660, 1 },
        /*261*/ { 661, 0 },
        /*262*/ { 677, 0 },
        /*263*/ { 678, 1 },
        /*264*/ { 679, 0 },
        /*265*/ { 680, 0 },
        /*266*/ { 690, 1 },
        /*267*/ { 700, 0 },
        /*268*/ { 701, 0 },
        /*269*/ { 702, 1 },
        /*270*/ { 703, 1 },
        /*271*/ { 704, 1 },
        /*272*/ { 705, 1 },
        /*273*/ { 706, 1 },
        /*274*/ { 707, 0 },
        /*275*/ { 708, 0 },
        /*276*/ { 709, 0 },
        /*277*/ { 710, 1 },
        /*278*/ { 711, 0 },
        /*279*/ { 747, 0 },
        /*280*/ { 748, 1 },
        /*281*/ { 749, 0 },
        /*282*/ { 750, 0 },
        /*283*/ { 751, 0 },
        /*284*/ { 752, 1 },
        /*285*/ { 753, 0 },
        /*286*/ { 791, 0 },
        /*287*/ { 792, 1 },
        /*288*/ { 793, 0 },
        /*289*/ { 794, 0 },
        /*290*/ { 795, 1 },
        /*291*/ { 796, 1 },
        /*292*/ { 797, 0 },
        /*293*/ { 798, 1 },
        /*294*/ { 799, 0 },
        /*295*/ { 800, 1 },
        /*296*/ { 801, 0 },
        /*297*/ { 802, 0 },
        /*298*/ { 803, 0 },
        /*299*/ { 804, 1 },
        /*300*/ { 805, 0 },
        /*301*/ { 806, 0 },
        /*302*/ { 807, 1 },
        /*303*/ { 808, 0 },
        /*304*/ { 830, 0 },
        /*305*/ { 831, 1 },
        /*306*/ { 832, 1 },
        /*307*/ { 833, 1 },
        /*308*/ { 834, 1 },
        /*309*/ { 835, 0 },
        /*310*/ { 880, 0 },
        /*311*/ { 881, 0 },
        /*312*/ { 882, 1 },
        /*313*/ { 883, 0 },
        /*314*/ { 884, 0 },
        /*315*/ { 885, 0 },
        /*316*/ { 886, 0 },
        /*317*/ { 887, 1 },
        /*318*/ { 888, 0 },
        /*319*/ { 889, 0 },
        /*320*/ { 890, 0 },
        /*321*/ { 891, 0 },
        /*322*/ { 892, 0 },
        /*323*/ { 893, 0 },
        /*324*/ { 894, 1 },
        /*325*/ { 895, 0 },
        /*326*/ { 896, 0 },
        /*327*/ { 897, 0 },
        /*328*/ { 898, 0 },
        /*329*/ { 899, 0 },
        /*330*/ { 900, 0 },
        /*331*/ { 999, 0 },
        /*332*/ { 2000, 0 },
    };
    int data_size = ARRAY_SIZE(data);
    int i, ret;

    testStart("test_numeric");

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        ret = iso3166_numeric(data[i].data);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d\n", i, ret, data[i].ret);
    }

    testFinish();
}

/* Binary chop version: Whether ISO 3166-1 numeric */
static int bc_iso3166_numeric(int cc) {
    static const short codes[249] = {
        /*AFG*/   4, /*ALB*/   8, /*ATA*/  10, /*DZA*/  12, /*ASM*/  16, /*AND*/  20, /*AGO*/  24, /*ATG*/  28, /*AZE*/  31, /*ARG*/  32,
        /*AUS*/  36, /*AUT*/  40, /*BHS*/  44, /*BHR*/  48, /*BGD*/  50, /*ARM*/  51, /*BRB*/  52, /*BEL*/  56, /*BMU*/  60, /*BTN*/  64,
        /*BOL*/  68, /*BIH*/  70, /*BWA*/  72, /*BVT*/  74, /*BRA*/  76, /*BLZ*/  84, /*IOT*/  86, /*SLB*/  90, /*VGB*/  92, /*BRN*/  96,
        /*BGR*/ 100, /*MMR*/ 104, /*BDI*/ 108, /*BLR*/ 112, /*KHM*/ 116, /*CMR*/ 120, /*CAN*/ 124, /*CPV*/ 132, /*CYM*/ 136, /*CAF*/ 140,
        /*LKA*/ 144, /*TCD*/ 148, /*CHL*/ 152, /*CHN*/ 156, /*TWN*/ 158, /*CXR*/ 162, /*CCK*/ 166, /*COL*/ 170, /*COM*/ 174, /*MYT*/ 175,
        /*COG*/ 178, /*COD*/ 180, /*COK*/ 184, /*CRI*/ 188, /*HRV*/ 191, /*CUB*/ 192, /*CYP*/ 196, /*CZE*/ 203, /*BEN*/ 204, /*DNK*/ 208,
        /*DMA*/ 212, /*DOM*/ 214, /*ECU*/ 218, /*SLV*/ 222, /*GNQ*/ 226, /*ETH*/ 231, /*ERI*/ 232, /*EST*/ 233, /*FRO*/ 234, /*FLK*/ 238,
        /*SGS*/ 239, /*FJI*/ 242, /*FIN*/ 246, /*ALA*/ 248, /*FRA*/ 250, /*GUF*/ 254, /*PYF*/ 258, /*ATF*/ 260, /*DJI*/ 262, /*GAB*/ 266,
        /*GEO*/ 268, /*GMB*/ 270, /*PSE*/ 275, /*DEU*/ 276, /*GHA*/ 288, /*GIB*/ 292, /*KIR*/ 296, /*GRC*/ 300, /*GRL*/ 304, /*GRD*/ 308,
        /*GLP*/ 312, /*GUM*/ 316, /*GTM*/ 320, /*GIN*/ 324, /*GUY*/ 328, /*HTI*/ 332, /*HMD*/ 334, /*VAT*/ 336, /*HND*/ 340, /*HKG*/ 344,
        /*HUN*/ 348, /*ISL*/ 352, /*IND*/ 356, /*IDN*/ 360, /*IRN*/ 364, /*IRQ*/ 368, /*IRL*/ 372, /*ISR*/ 376, /*ITA*/ 380, /*CIV*/ 384,
        /*JAM*/ 388, /*JPN*/ 392, /*KAZ*/ 398, /*JOR*/ 400, /*KEN*/ 404, /*PRK*/ 408, /*KOR*/ 410, /*KWT*/ 414, /*KGZ*/ 417, /*LAO*/ 418,
        /*LBN*/ 422, /*LSO*/ 426, /*LVA*/ 428, /*LBR*/ 430, /*LBY*/ 434, /*LIE*/ 438, /*LTU*/ 440, /*LUX*/ 442, /*MAC*/ 446, /*MDG*/ 450,
        /*MWI*/ 454, /*MYS*/ 458, /*MDV*/ 462, /*MLI*/ 466, /*MLT*/ 470, /*MTQ*/ 474, /*MRT*/ 478, /*MUS*/ 480, /*MEX*/ 484, /*MCO*/ 492,
        /*MNG*/ 496, /*MDA*/ 498, /*MNE*/ 499, /*MSR*/ 500, /*MAR*/ 504, /*MOZ*/ 508, /*OMN*/ 512, /*NAM*/ 516, /*NRU*/ 520, /*NPL*/ 524,
        /*NLD*/ 528, /*CUW*/ 531, /*ABW*/ 533, /*SXM*/ 534, /*BES*/ 535, /*NCL*/ 540, /*VUT*/ 548, /*NZL*/ 554, /*NIC*/ 558, /*NER*/ 562,
        /*NGA*/ 566, /*NIU*/ 570, /*NFK*/ 574, /*NOR*/ 578, /*MNP*/ 580, /*UMI*/ 581, /*FSM*/ 583, /*MHL*/ 584, /*PLW*/ 585, /*PAK*/ 586,
        /*PAN*/ 591, /*PNG*/ 598, /*PRY*/ 600, /*PER*/ 604, /*PHL*/ 608, /*PCN*/ 612, /*POL*/ 616, /*PRT*/ 620, /*GNB*/ 624, /*TLS*/ 626,
        /*PRI*/ 630, /*QAT*/ 634, /*REU*/ 638, /*ROU*/ 642, /*RUS*/ 643, /*RWA*/ 646, /*BLM*/ 652, /*SHN*/ 654, /*KNA*/ 659, /*AIA*/ 660,
        /*LCA*/ 662, /*MAF*/ 663, /*SPM*/ 666, /*VCT*/ 670, /*SMR*/ 674, /*STP*/ 678, /*SAU*/ 682, /*SEN*/ 686, /*SRB*/ 688, /*SYC*/ 690,
        /*SLE*/ 694, /*SGP*/ 702, /*SVK*/ 703, /*VNM*/ 704, /*SVN*/ 705, /*SOM*/ 706, /*ZAF*/ 710, /*ZWE*/ 716, /*ESP*/ 724, /*SSD*/ 728,
        /*SDN*/ 729, /*ESH*/ 732, /*SUR*/ 740, /*SJM*/ 744, /*SWZ*/ 748, /*SWE*/ 752, /*CHE*/ 756, /*SYR*/ 760, /*TJK*/ 762, /*THA*/ 764,
        /*TGO*/ 768, /*TKL*/ 772, /*TON*/ 776, /*TTO*/ 780, /*ARE*/ 784, /*TUN*/ 788, /*TUR*/ 792, /*TKM*/ 795, /*TCA*/ 796, /*TUV*/ 798,
        /*UGA*/ 800, /*UKR*/ 804, /*MKD*/ 807, /*EGY*/ 818, /*GBR*/ 826, /*GGY*/ 831, /*JEY*/ 832, /*IMN*/ 833, /*TZA*/ 834, /*USA*/ 840,
        /*VIR*/ 850, /*BFA*/ 854, /*URY*/ 858, /*UZB*/ 860, /*VEN*/ 862, /*WLF*/ 876, /*WSM*/ 882, /*YEM*/ 887, /*ZMB*/ 894,
    };

    int s = 0, e = ARRAY_SIZE(codes) - 1;

    while (s <= e) {
        int m = (s + e) / 2;
        if (codes[m] < cc) {
            s = m + 1;
        } else if (codes[m] > cc) {
            e = m - 1;
        } else {
            return 1;
        }
    }

    return 0;
}

static void test_numeric_bc(const testCtx *const p_ctx) {

    int i, ret, bc_ret;

    testStart("test_numeric_bc");

    for (i = 0; i < 1001; i++) {
        if (testContinue(p_ctx, i)) continue;

        ret = iso3166_numeric(i);
        bc_ret = bc_iso3166_numeric(i);
        assert_equal(ret, bc_ret, "i:%d ret %d != bc_ret %d\n", i, ret, bc_ret);
    }

    testFinish();
}

static void test_alpha2(const testCtx *const p_ctx) {

    struct item {
        const char *data;
        int ret;
    };
    /* s/\/\*[ 0-9]*\*\//\=printf("\/\*%3d*\/", line(".") - line("'<")): */
    struct item data[] = {
        /*  0*/ { "", 0 },
        /*  1*/ { "A", 0 },
        /*  2*/ { "aa", 0 },
        /*  3*/ { "AA", 0 },
        /*  4*/ { "AB", 0 },
        /*  5*/ { "AC", 0 },
        /*  6*/ { "AD", 1 },
        /*  7*/ { "ad", 0 },
        /*  8*/ { "AE", 1 },
        /*  9*/ { "AF", 1 },
        /* 10*/ { "AG", 1 },
        /* 11*/ { "AH", 0 },
        /* 12*/ { "AI", 1 },
        /* 13*/ { "AJ", 0 },
        /* 14*/ { "AP", 0 },
        /* 15*/ { "AQ", 1 },
        /* 16*/ { "AR", 1 },
        /* 17*/ { "AS", 1 },
        /* 18*/ { "AT", 1 },
        /* 19*/ { "AU", 1 },
        /* 20*/ { "AV", 0 },
        /* 21*/ { "AW", 1 },
        /* 22*/ { "AX", 1 },
        /* 23*/ { "AY", 0 },
        /* 24*/ { "AZ", 1 },
        /* 25*/ { "BA", 1 },
        /* 26*/ { "BB", 1 },
        /* 27*/ { "BC", 0 },
        /* 28*/ { "BD", 1 },
        /* 29*/ { "BX", 0 },
        /* 30*/ { "BY", 1 },
        /* 31*/ { "BZ", 1 },
        /* 32*/ { "CA", 1 },
        /* 33*/ { "CB", 0 },
        /* 34*/ { "CC", 1 },
        /* 35*/ { "CD", 1 },
        /* 36*/ { "CE", 0 },
        /* 37*/ { "CT", 0 },
        /* 38*/ { "CU", 1 },
        /* 39*/ { "CV", 1 },
        /* 40*/ { "CW", 1 },
        /* 41*/ { "CX", 1 },
        /* 42*/ { "CY", 1 },
        /* 43*/ { "CZ", 1 },
        /* 44*/ { "DA", 0 },
        /* 45*/ { "DD", 0 },
        /* 46*/ { "DE", 1 },
        /* 47*/ { "DF", 0 },
        /* 48*/ { "DY", 0 },
        /* 49*/ { "DZ", 1 },
        /* 50*/ { "EA", 0 },
        /* 51*/ { "EB", 0 },
        /* 52*/ { "EC", 1 },
        /* 53*/ { "ED", 0 },
        /* 54*/ { "EZ", 0 },
        /* 55*/ { "FA", 0 },
        /* 56*/ { "FQ", 0 },
        /* 57*/ { "FR", 1 },
        /* 58*/ { "FS", 0 },
        /* 59*/ { "FZ", 0 },
        /* 60*/ { "GA", 1 },
        /* 61*/ { "GB", 1 },
        /* 62*/ { "GC", 0 },
        /* 63*/ { "GX", 0 },
        /* 64*/ { "GY", 1 },
        /* 65*/ { "GZ", 0 },
        /* 66*/ { "HA", 0 },
        /* 67*/ { "HI", 0 },
        /* 68*/ { "HJ", 0 },
        /* 69*/ { "HK", 1 },
        /* 70*/ { "HL", 0 },
        /* 71*/ { "HZ", 0 },
        /* 72*/ { "IA", 0 },
        /* 73*/ { "IC", 0 },
        /* 74*/ { "ID", 1 },
        /* 75*/ { "IE", 1 },
        /* 76*/ { "IF", 0 },
        /* 77*/ { "IZ", 0 },
        /* 78*/ { "JA", 0 },
        /* 79*/ { "JD", 0 },
        /* 80*/ { "JE", 1 },
        /* 81*/ { "JF", 0 },
        /* 82*/ { "JZ", 0 },
        /* 83*/ { "KA", 0 },
        /* 84*/ { "KD", 0 },
        /* 85*/ { "KE", 1 },
        /* 86*/ { "KF", 0 },
        /* 87*/ { "KG", 1 },
        /* 88*/ { "KH", 1 },
        /* 89*/ { "KI", 1 },
        /* 90*/ { "KJ", 0 },
        /* 91*/ { "KX", 0 },
        /* 92*/ { "KY", 1 },
        /* 93*/ { "KZ", 1 },
        /* 94*/ { "LA", 1 },
        /* 95*/ { "LB", 1 },
        /* 96*/ { "LC", 1 },
        /* 97*/ { "LD", 0 },
        /* 98*/ { "LE", 0 },
        /* 99*/ { "LX", 0 },
        /*100*/ { "LY", 1 },
        /*101*/ { "LZ", 0 },
        /*102*/ { "MA", 1 },
        /*103*/ { "MB", 0 },
        /*104*/ { "MI", 0 },
        /*105*/ { "MJ", 0 },
        /*106*/ { "MK", 1 },
        /*107*/ { "ML", 1 },
        /*108*/ { "MM", 1 },
        /*109*/ { "MN", 1 },
        /*110*/ { "MO", 1 },
        /*111*/ { "MP", 1 },
        /*112*/ { "MQ", 1 },
        /*113*/ { "MR", 1 },
        /*114*/ { "MS", 1 },
        /*115*/ { "MT", 1 },
        /*116*/ { "MU", 1 },
        /*117*/ { "MV", 1 },
        /*118*/ { "MW", 1 },
        /*119*/ { "MX", 1 },
        /*120*/ { "MY", 1 },
        /*121*/ { "MZ", 1 },
        /*122*/ { "NA", 1 },
        /*123*/ { "NB", 0 },
        /*124*/ { "NC", 1 },
        /*125*/ { "NY", 0 },
        /*126*/ { "NZ", 1 },
        /*127*/ { "OA", 0 },
        /*128*/ { "OL", 0 },
        /*129*/ { "OM", 1 },
        /*130*/ { "ON", 0 },
        /*131*/ { "OZ", 0 },
        /*132*/ { "PA", 1 },
        /*133*/ { "PB", 0 },
        /*134*/ { "PQ", 0 },
        /*135*/ { "PR", 1 },
        /*136*/ { "PS", 1 },
        /*137*/ { "PT", 1 },
        /*138*/ { "PU", 0 },
        /*139*/ { "PV", 0 },
        /*140*/ { "PW", 1 },
        /*141*/ { "PX", 0 },
        /*142*/ { "PY", 1 },
        /*143*/ { "PZ", 0 },
        /*144*/ { "QA", 1 },
        /*145*/ { "QB", 0 },
        /*146*/ { "QZ", 0 },
        /*147*/ { "RA", 0 },
        /*148*/ { "RC", 0 },
        /*149*/ { "RD", 0 },
        /*150*/ { "RE", 1 },
        /*151*/ { "RF", 0 },
        /*152*/ { "RZ", 0 },
        /*153*/ { "SA", 1 },
        /*154*/ { "SB", 1 },
        /*155*/ { "SC", 1 },
        /*156*/ { "SD", 1 },
        /*157*/ { "SE", 1 },
        /*158*/ { "SF", 0 },
        /*159*/ { "SW", 0 },
        /*160*/ { "SX", 1 },
        /*161*/ { "SY", 1 },
        /*162*/ { "SZ", 1 },
        /*163*/ { "TA", 0 },
        /*164*/ { "TB", 0 },
        /*165*/ { "TC", 1 },
        /*166*/ { "TD", 1 },
        /*167*/ { "TE", 0 },
        /*168*/ { "TU", 0 },
        /*169*/ { "TV", 1 },
        /*170*/ { "TW", 1 },
        /*171*/ { "TX", 0 },
        /*172*/ { "TY", 0 },
        /*173*/ { "TZ", 1 },
        /*174*/ { "UA", 1 },
        /*175*/ { "UB", 0 },
        /*176*/ { "UX", 0 },
        /*177*/ { "UY", 1 },
        /*178*/ { "UZ", 1 },
        /*179*/ { "VA", 1 },
        /*180*/ { "VB", 0 },
        /*181*/ { "VC", 1 },
        /*182*/ { "VD", 0 },
        /*183*/ { "VZ", 0 },
        /*184*/ { "WA", 0 },
        /*185*/ { "WE", 0 },
        /*186*/ { "WF", 1 },
        /*187*/ { "WG", 0 },
        /*188*/ { "WZ", 0 },
        /*189*/ { "XA", 0 },
        /*190*/ { "XZ", 0 },
        /*191*/ { "YA", 0 },
        /*192*/ { "YC", 0 },
        /*193*/ { "YD", 0 },
        /*194*/ { "YE", 1 },
        /*195*/ { "YF", 0 },
        /*196*/ { "YZ", 0 },
        /*197*/ { "ZA", 1 },
        /*198*/ { "ZB", 0 },
        /*199*/ { "ZL", 0 },
        /*200*/ { "ZM", 1 },
        /*201*/ { "ZN", 0 },
        /*202*/ { "ZV", 0 },
        /*203*/ { "ZW", 1 },
        /*204*/ { "ZX", 0 },
        /*205*/ { "ZY", 0 },
        /*206*/ { "ZZ", 0 },
        /*207*/ { "Z", 0 },
        /*208*/ { "zz", 0 },
        /*209*/ { "0", 0 },
        /*210*/ { "\001", 0 },
        /*211*/ { "\177", 0 },
        /*212*/ { "\200", 0 },
        /*213*/ { "\377", 0 },
    };
    int data_size = ARRAY_SIZE(data);
    int i, ret;

    testStart("test_alpha2");

    for (i = 0; i < data_size; i++) {

        if (testContinue(p_ctx, i)) continue;

        ret = iso3166_alpha2(data[i].data);
        assert_equal(ret, data[i].ret, "i:%d ret %d != %d\n", i, ret, data[i].ret);
    }

    testFinish();
}

/* Binary chop version: Whether ISO 3166-1 alpha2 */
static int bc_iso3166_alpha2(const char *cc) {
    static const char codes[249][2] = {
        {'A','D'}, {'A','E'}, {'A','F'}, {'A','G'}, {'A','I'}, {'A','L'}, {'A','M'}, {'A','O'}, {'A','Q'}, {'A','R'},
        {'A','S'}, {'A','T'}, {'A','U'}, {'A','W'}, {'A','X'}, {'A','Z'}, {'B','A'}, {'B','B'}, {'B','D'}, {'B','E'},
        {'B','F'}, {'B','G'}, {'B','H'}, {'B','I'}, {'B','J'}, {'B','L'}, {'B','M'}, {'B','N'}, {'B','O'}, {'B','Q'},
        {'B','R'}, {'B','S'}, {'B','T'}, {'B','V'}, {'B','W'}, {'B','Y'}, {'B','Z'}, {'C','A'}, {'C','C'}, {'C','D'},
        {'C','F'}, {'C','G'}, {'C','H'}, {'C','I'}, {'C','K'}, {'C','L'}, {'C','M'}, {'C','N'}, {'C','O'}, {'C','R'},
        {'C','U'}, {'C','V'}, {'C','W'}, {'C','X'}, {'C','Y'}, {'C','Z'}, {'D','E'}, {'D','J'}, {'D','K'}, {'D','M'},
        {'D','O'}, {'D','Z'}, {'E','C'}, {'E','E'}, {'E','G'}, {'E','H'}, {'E','R'}, {'E','S'}, {'E','T'}, {'F','I'},
        {'F','J'}, {'F','K'}, {'F','M'}, {'F','O'}, {'F','R'}, {'G','A'}, {'G','B'}, {'G','D'}, {'G','E'}, {'G','F'},
        {'G','G'}, {'G','H'}, {'G','I'}, {'G','L'}, {'G','M'}, {'G','N'}, {'G','P'}, {'G','Q'}, {'G','R'}, {'G','S'},
        {'G','T'}, {'G','U'}, {'G','W'}, {'G','Y'}, {'H','K'}, {'H','M'}, {'H','N'}, {'H','R'}, {'H','T'}, {'H','U'},
        {'I','D'}, {'I','E'}, {'I','L'}, {'I','M'}, {'I','N'}, {'I','O'}, {'I','Q'}, {'I','R'}, {'I','S'}, {'I','T'},
        {'J','E'}, {'J','M'}, {'J','O'}, {'J','P'}, {'K','E'}, {'K','G'}, {'K','H'}, {'K','I'}, {'K','M'}, {'K','N'},
        {'K','P'}, {'K','R'}, {'K','W'}, {'K','Y'}, {'K','Z'}, {'L','A'}, {'L','B'}, {'L','C'}, {'L','I'}, {'L','K'},
        {'L','R'}, {'L','S'}, {'L','T'}, {'L','U'}, {'L','V'}, {'L','Y'}, {'M','A'}, {'M','C'}, {'M','D'}, {'M','E'},
        {'M','F'}, {'M','G'}, {'M','H'}, {'M','K'}, {'M','L'}, {'M','M'}, {'M','N'}, {'M','O'}, {'M','P'}, {'M','Q'},
        {'M','R'}, {'M','S'}, {'M','T'}, {'M','U'}, {'M','V'}, {'M','W'}, {'M','X'}, {'M','Y'}, {'M','Z'}, {'N','A'},
        {'N','C'}, {'N','E'}, {'N','F'}, {'N','G'}, {'N','I'}, {'N','L'}, {'N','O'}, {'N','P'}, {'N','R'}, {'N','U'},
        {'N','Z'}, {'O','M'}, {'P','A'}, {'P','E'}, {'P','F'}, {'P','G'}, {'P','H'}, {'P','K'}, {'P','L'}, {'P','M'},
        {'P','N'}, {'P','R'}, {'P','S'}, {'P','T'}, {'P','W'}, {'P','Y'}, {'Q','A'}, {'R','E'}, {'R','O'}, {'R','S'},
        {'R','U'}, {'R','W'}, {'S','A'}, {'S','B'}, {'S','C'}, {'S','D'}, {'S','E'}, {'S','G'}, {'S','H'}, {'S','I'},
        {'S','J'}, {'S','K'}, {'S','L'}, {'S','M'}, {'S','N'}, {'S','O'}, {'S','R'}, {'S','S'}, {'S','T'}, {'S','V'},
        {'S','X'}, {'S','Y'}, {'S','Z'}, {'T','C'}, {'T','D'}, {'T','F'}, {'T','G'}, {'T','H'}, {'T','J'}, {'T','K'},
        {'T','L'}, {'T','M'}, {'T','N'}, {'T','O'}, {'T','R'}, {'T','T'}, {'T','V'}, {'T','W'}, {'T','Z'}, {'U','A'},
        {'U','G'}, {'U','M'}, {'U','S'}, {'U','Y'}, {'U','Z'}, {'V','A'}, {'V','C'}, {'V','E'}, {'V','G'}, {'V','I'},
        {'V','N'}, {'V','U'}, {'W','F'}, {'W','S'}, {'Y','E'}, {'Y','T'}, {'Z','A'}, {'Z','M'}, {'Z','W'},
    };

    int s = 0, e = sizeof(codes) / sizeof(codes[0]) - 1;

    while (s <= e) {
        int m = (s + e) / 2;
        int cmp = strncmp(codes[m], cc, 2);
        if (cmp < 0) {
            s = m + 1;
        } else if (cmp > 0) {
            e = m - 1;
        } else {
            return 1;
        }
    }

    return 0;
}

static void test_alpha2_bc(const testCtx *const p_ctx) {

    int i, ret, bc_ret;
    char data[2];

    (void)p_ctx;

    testStart("test_alpha2_bc");

    for (i = 0; i < 128; i++) {
        int j;
        for (j = 0; j < 128; j++) {
            data[0] = i;
            data[1] = j;
            ret = iso3166_alpha2(data);
            bc_ret = bc_iso3166_alpha2(data);
            assert_equal(ret, bc_ret, "i:%d ret %d != bc_ret %d\n", i, ret, bc_ret);
        }
    }

    testFinish();
}

int main(int argc, char *argv[]) {

    testFunction funcs[] = { /* name, func */
        { "test_numeric", test_numeric },
        { "test_alpha2", test_alpha2 },
        { "test_numeric_bc", test_numeric_bc },
        { "test_alpha2_bc", test_alpha2_bc },
    };

    testRun(argc, argv, funcs, ARRAY_SIZE(funcs));

    testReport();

    return 0;
}

/* vim: set ts=4 sw=4 et : */
