# -*- coding: utf-8 -*-
#
# Copyright (c) 2016 Wil Clouser <clouserw@micropipes.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

# Intercepts emoji characters and converts them back to their alias.  Useful if
# you're on a terminal or something that doesn't support emoji.
#
# History:
#
#   2019-07-06, Sébastien Helleu <flashcode@flashtux.org>:
#       v0.2: Make script compatible with Python 3, fix PEP8 errors.
#
#   2016-03-15, Wil Clouser <clouserw@micropipes.com>:
#       v0.1: Initial release, based on Mike Reinhardt's BSD
#             licensed emoji_aliases.py

from __future__ import print_function

import re
import sys

SCRIPT_NAME = "emoji2alias"
SCRIPT_AUTHOR = "Wil Clouser <clouserw@micropipes.com>"
SCRIPT_VERSION = "0.2"
SCRIPT_LICENSE = "MIT"
SCRIPT_DESC = "Replaces emoji characters with their aliases"

import_ok = True

try:
    import weechat as w
except ImportError:
    print("Script must be run under weechat: https://weechat.org")
    import_ok = False

EMOJI_ALIASES = {
    u'\U0001f44d': u':+1:',
    u'\U0001f44e': u':-1:',
    u'\U0001f4af': u':100:',
    u'\U0001f522': u':1234:',
    u'\U0001f3b1': u':8ball:',
    u'\U0001f170': u':a:',
    u'\U0001f18e': u':ab:',
    u'\U0001f524': u':abc:',
    u'\U0001f521': u':abcd:',
    u'\U0001f251': u':accept:',
    u'\U0001f39f': u':admission_tickets:',
    u'\U0001f6a1': u':aerial_tramway:',
    u'\U00002708': u':airplane:',
    u'\U0001f6ec': u':airplane_arriving:',
    u'\U0001f6eb': u':airplane_departure:',
    u'\U000023f0': u':alarm_clock:',
    u'\U00002697': u':alembic:',
    u'\U0001f47d': u':alien:',
    u'\U0001f47e': u':alien_monster:',
    u'\U0001f691': u':ambulance:',
    u'\U0001f3c8': u':american_football:',
    u'\U0001f3fa': u':amphora:',
    u'\U00002693': u':anchor:',
    u'\U0001f47c': u':angel:',
    u'\U0001f4a2': u':anger:',
    u'\U0001f620': u':angry:',
    u'\U0001f627': u':anguished:',
    u'\U0001f41c': u':ant:',
    u'\U0001f4f6': u':antenna_with_bars:',
    u'\U0001f34e': u':apple:',
    u'\U00002652': u':aquarius:',
    u'\U00002648': u':aries:',
    u'\U000025c0': u':arrow_backward:',
    u'\U000023ec': u':arrow_double_down:',
    u'\U000023eb': u':arrow_double_up:',
    u'\U00002b07': u':arrow_down:',
    u'\U0001f53d': u':arrow_down_small:',
    u'\U000025b6': u':arrow_forward:',
    u'\U00002935': u':arrow_heading_down:',
    u'\U00002934': u':arrow_heading_up:',
    u'\U00002b05': u':arrow_left:',
    u'\U00002199': u':arrow_lower_left:',
    u'\U00002198': u':arrow_lower_right:',
    u'\U000027a1': u':arrow_right:',
    u'\U000021aa': u':arrow_right_hook:',
    u'\U00002b06': u':arrow_up:',
    u'\U00002195': u':arrow_up_down:',
    u'\U0001f53c': u':arrow_up_small:',
    u'\U00002196': u':arrow_upper_left:',
    u'\U00002197': u':arrow_upper_right:',
    u'\U0001f503': u':arrows_clockwise:',
    u'\U0001f504': u':arrows_counterclockwise:',
    u'\U0001f3a8': u':art:',
    u'\U0001f69b': u':articulated_lorry:',
    u'\U0001f632': u':astonished:',
    u'\U0001f45f': u':athletic_shoe:',
    u'\U0001f3e7': u':atm:',
    u'\U0000269b': u':atom_symbol:',
    u'\U0001f346': u':aubergine:',
    u'\U0001f697': u':automobile:',
    u'\U0001f171': u':b:',
    u'\U0001f476': u':baby:',
    u'\U0001f37c': u':baby_bottle:',
    u'\U0001f424': u':baby_chick:',
    u'\U0001f6bc': u':baby_symbol:',
    u'\U0001f519': u':back:',
    u'\U0001f42b': u':bactrian_camel:',
    u'\U0001f3f8': u':badminton_racquet_and_shuttlecock:',
    u'\U0001f6c4': u':baggage_claim:',
    u'\U0001f388': u':balloon:',
    u'\U0001f5f3': u':ballot_box_with_ballot:',
    u'\U00002611': u':ballot_box_with_check:',
    u'\U0001f38d': u':bamboo:',
    u'\U0001f34c': u':banana:',
    u'\U0000203c': u':bangbang:',
    u'\U0001f3e6': u':bank:',
    u'\U0001f4b5': u':banknote_with_dollar_sign:',
    u'\U0001f4b6': u':banknote_with_euro_sign:',
    u'\U0001f4b7': u':banknote_with_pound_sign:',
    u'\U0001f4b4': u':banknote_with_yen_sign:',
    u'\U0001f4ca': u':bar_chart:',
    u'\U0001f488': u':barber:',
    u'\U0001f325': u':barely_sunny:',
    u'\U000026be': u':baseball:',
    u'\U0001f3c0': u':basketball:',
    u'\U0001f6c0': u':bath:',
    u'\U0001f6c1': u':bathtub:',
    u'\U0001f50b': u':battery:',
    u'\U0001f3d6': u':beach_with_umbrella:',
    u'\U0001f43b': u':bear:',
    u'\U0001f493': u':beating_heart:',
    u'\U0001f6cf': u':bed:',
    u'\U0001f41d': u':bee:',
    u'\U0001f37a': u':beer:',
    u'\U0001f37b': u':beers:',
    u'\U0001f41e': u':beetle:',
    u'\U0001f530': u':beginner:',
    u'\U0001f514': u':bell:',
    u'\U0001f515': u':bell_with_cancellation_stroke:',
    u'\U0001f6ce': u':bellhop_bell:',
    u'\U0001f371': u':bento:',
    u'\U0001f6b2': u':bicycle:',
    u'\U0001f6b4': u':bicyclist:',
    u'\U0001f459': u':bikini:',
    u'\U00002623': u':biohazard_sign:',
    u'\U0001f426': u':bird:',
    u'\U0001f382': u':birthday:',
    u'\U000026ab': u':black_circle:',
    u'\U000023fa': u':black_circle_for_record:',
    u'\U00002663': u':black_club_suit:',
    u'\U00002666': u':black_diamond_suit:',
    u'\U00002665': u':black_heart_suit:',
    u'\U0001f0cf': u':black_joker:',
    u'\U00002b1b': u':black_large_square:',
    u'\U000023ea': u':black_left-pointing_double_triangle:',
    u'\U000025fe': u':black_medium_small_square:',
    u'\U000025fc': u':black_medium_square:',
    u'\U00002712': u':black_nib:',
    u'\U00002753': u':black_question_mark_ornament:',
    u'\U000023e9': u':black_right-pointing_double_triangle:',
    u'\U00002702': u':black_scissors:',
    u'\U000025aa': u':black_small_square:',
    u'\U00002660': u':black_spade_suit:',
    u'\U0001f532': u':black_square_button:',
    u'\U000023f9': u':black_square_for_stop:',
    u'\U00002600': u':black_sun_with_rays:',
    u'\U0000260e': u':black_telephone:',
    u'\U0000267b': u':black_universal_recycling_symbol:',
    u'\U0001f33c': u':blossom:',
    u'\U0001f421': u':blowfish:',
    u'\U0001f4d8': u':blue_book:',
    u'\U0001f699': u':blue_car:',
    u'\U0001f499': u':blue_heart:',
    u'\U0001f60a': u':blush:',
    u'\U0001f417': u':boar:',
    u'\U000026f5': u':boat:',
    u'\U0001f4a3': u':bomb:',
    u'\U0001f4d6': u':book:',
    u'\U0001f516': u':bookmark:',
    u'\U0001f4d1': u':bookmark_tabs:',
    u'\U0001f4da': u':books:',
    u'\U0001f4a5': u':boom:',
    u'\U0001f462': u':boot:',
    u'\U0001f37e': u':bottle_with_popping_cork:',
    u'\U0001f490': u':bouquet:',
    u'\U0001f647': u':bow:',
    u'\U0001f3f9': u':bow_and_arrow:',
    u'\U0001f3b3': u':bowling:',
    u'\U0001f466': u':boy:',
    u'\U0001f35e': u':bread:',
    u'\U0001f470': u':bride_with_veil:',
    u'\U0001f309': u':bridge_at_night:',
    u'\U0001f4bc': u':briefcase:',
    u'\U0001f494': u':broken_heart:',
    u'\U0001f41b': u':bug:',
    u'\U0001f3d7': u':building_construction:',
    u'\U0001f4a1': u':bulb:',
    u'\U0001f685': u':bullettrain_front:',
    u'\U0001f684': u':bullettrain_side:',
    u'\U0001f32f': u':burrito:',
    u'\U0001f68c': u':bus:',
    u'\U0001f68f': u':bus_stop:',
    u'\U0001f464': u':bust_in_silhouette:',
    u'\U0001f465': u':busts_in_silhouette:',
    u'\U0001f335': u':cactus:',
    u'\U0001f370': u':cake:',
    u'\U0001f4c5': u':calendar:',
    u'\U0001f4c6': u':calendar:',
    u'\U0001f4f2': u':calling:',
    u'\U0001f4f7': u':camera:',
    u'\U0001f4f8': u':camera_with_flash:',
    u'\U0001f3d5': u':camping:',
    u'\U0000264b': u':cancer:',
    u'\U0001f56f': u':candle:',
    u'\U0001f36c': u':candy:',
    u'\U0001f520': u':capital_abcd:',
    u'\U00002651': u':capricorn:',
    u'\U0001f5c3': u':card_file_box:',
    u'\U0001f4c7': u':card_index:',
    u'\U0001f5c2': u':card_index_dividers:',
    u'\U0001f3a0': u':carousel_horse:',
    u'\U0001f38f': u':carp_streamer:',
    u'\U0001f408': u':cat2:',
    u'\U0001f431': u':cat:',
    u'\U0001f639': u':cat_face_with_tears_of_joy:',
    u'\U0001f63c': u':cat_face_with_wry_smile:',
    u'\U0001f4bf': u':cd:',
    u'\U000026d3': u':chains:',
    u'\U0001f4b9': u':chart:',
    u'\U0001f4c9': u':chart_with_downwards_trend:',
    u'\U0001f4c8': u':chart_with_upwards_trend:',
    u'\U0001f3c1': u':checkered_flag:',
    u'\U0001f4e3': u':cheering_megaphone:',
    u'\U0001f9c0': u':cheese_wedge:',
    u'\U0001f352': u':cherries:',
    u'\U0001f338': u':cherry_blossom:',
    u'\U0001f330': u':chestnut:',
    u'\U0001f414': u':chicken:',
    u'\U0001f6b8': u':children_crossing:',
    u'\U0001f43f': u':chipmunk:',
    u'\U0001f36b': u':chocolate_bar:',
    u'\U0001f384': u':christmas_tree:',
    u'\U000026ea': u':church:',
    u'\U0001f3a6': u':cinema:',
    u'\U0001f250': u':circled_ideograph_advantage:',
    u'\U00003297': u':circled_ideograph_congratulation:',
    u'\U00003299': u':circled_ideograph_secret:',
    u'\U000024c2': u':circled_latin_capital_letter_m:',
    u'\U0001f3aa': u':circus_tent:',
    u'\U0001f307': u':city_sunrise:',
    u'\U0001f306': u':city_sunset:',
    u'\U0001f3d9': u':cityscape:',
    u'\U0001f191': u':cl:',
    u'\U0001f44f': u':clap:',
    u'\U0001f3ac': u':clapper:',
    u'\U0001f3db': u':classical_building:',
    u'\U0001f4cb': u':clipboard:',
    u'\U0001f565': u':clock1030:',
    u'\U0001f559': u':clock10:',
    u'\U0001f566': u':clock1130:',
    u'\U0001f55a': u':clock11:',
    u'\U0001f567': u':clock1230:',
    u'\U0001f55b': u':clock12:',
    u'\U0001f55c': u':clock130:',
    u'\U0001f550': u':clock1:',
    u'\U0001f55d': u':clock230:',
    u'\U0001f551': u':clock2:',
    u'\U0001f55e': u':clock330:',
    u'\U0001f552': u':clock3:',
    u'\U0001f55f': u':clock430:',
    u'\U0001f553': u':clock4:',
    u'\U0001f560': u':clock530:',
    u'\U0001f554': u':clock5:',
    u'\U0001f561': u':clock630:',
    u'\U0001f555': u':clock6:',
    u'\U0001f562': u':clock730:',
    u'\U0001f556': u':clock7:',
    u'\U0001f563': u':clock830:',
    u'\U0001f557': u':clock8:',
    u'\U0001f564': u':clock930:',
    u'\U0001f558': u':clock9:',
    u'\U0001f4d5': u':closed_book:',
    u'\U0001f510': u':closed_lock_with_key:',
    u'\U0001f4ea': u':closed_mailbox_with_lowered_flag:',
    u'\U0001f4eb': u':closed_mailbox_with_raised_flag:',
    u'\U0001f302': u':closed_umbrella:',
    u'\U00002601': u':cloud:',
    u'\U0001f329': u':cloud_with_lightning:',
    u'\U0001f327': u':cloud_with_rain:',
    u'\U0001f328': u':cloud_with_snow:',
    u'\U0001f32a': u':cloud_with_tornado:',
    u'\U0001f378': u':cocktail:',
    u'\U00002615': u':coffee:',
    u'\U000026b0': u':coffin:',
    u'\U0001f630': u':cold_sweat:',
    u'\U00002604': u':comet:',
    u'\U0001f5dc': u':compression:',
    u'\U0001f4bb': u':computer:',
    u'\U0001f38a': u':confetti_ball:',
    u'\U0001f616': u':confounded:',
    u'\U0001f615': u':confused:',
    u'\U0001f6a7': u':construction:',
    u'\U0001f477': u':construction_worker:',
    u'\U0001f39b': u':control_knobs:',
    u'\U0001f3ea': u':convenience_store:',
    u'\U0001f35a': u':cooked_rice:',
    u'\U0001f36a': u':cookie:',
    u'\U0001f373': u':cooking:',
    u'\U0001f192': u':cool:',
    u'\U0001f46e': u':cop:',
    u'\U000000a9': u':copyright:',
    u'\U0001f33d': u':corn:',
    u'\U0001f6cb': u':couch_and_lamp:',
    u'\U0001f46b': u':couple:',
    u'\U0001f491': u':couple_with_heart:',
    u'\U0001f48f': u':couplekiss:',
    u'\U0001f404': u':cow2:',
    u'\U0001f42e': u':cow:',
    u'\U0001f980': u':crab:',
    u'\U0001f4b3': u':credit_card:',
    u'\U0001f319': u':crescent_moon:',
    u'\U0001f3cf': u':cricket_bat_and_ball:',
    u'\U0001f40a': u':crocodile:',
    u'\U0000274c': u':cross_mark:',
    u'\U0001f38c': u':crossed_flags:',
    u'\U00002694': u':crossed_swords:',
    u'\U0001f451': u':crown:',
    u'\U0001f622': u':cry:',
    u'\U0001f63f': u':crying_cat_face:',
    u'\U0001f52e': u':crystal_ball:',
    u'\U0001f498': u':cupid:',
    u'\U000027b0': u':curly_loop:',
    u'\U0001f4b1': u':currency_exchange:',
    u'\U0001f35b': u':curry:',
    u'\U0001f36e': u':custard:',
    u'\U0001f6c3': u':customs:',
    u'\U0001f300': u':cyclone:',
    u'\U0001f5e1': u':dagger_knife:',
    u'\U0001f483': u':dancer:',
    u'\U0001f46f': u':dancers:',
    u'\U0001f361': u':dango:',
    u'\U0001f576': u':dark_sunglasses:',
    u'\U0001f3af': u':dart:',
    u'\U0001f4a8': u':dash:',
    u'\U0001f333': u':deciduous_tree:',
    u'\U0001f69a': u':delivery_truck:',
    u'\U0001f3ec': u':department_store:',
    u'\U0001f3da': u':derelict_house_building:',
    u'\U0001f3dc': u':desert:',
    u'\U0001f3dd': u':desert_island:',
    u'\U0001f5a5': u':desktop_computer:',
    u'\U0001f4a0': u':diamond_shape_with_a_dot_inside:',
    u'\U0001f61e': u':disappointed:',
    u'\U0001f625': u':disappointed_but_relieved_face:',
    u'\U0001f4ab': u':dizzy:',
    u'\U0001f635': u':dizzy_face:',
    u'\U0001f6af': u':do_not_litter:',
    u'\U0001f415': u':dog2:',
    u'\U0001f436': u':dog:',
    u'\U0001f38e': u':dolls:',
    u'\U0001f42c': u':dolphin:',
    u'\U0001f6aa': u':door:',
    u'\U000027bf': u':double_curly_loop:',
    u'\U000023f8': u':double_vertical_bar:',
    u'\U0001f369': u':doughnut:',
    u'\U0001f54a': u':dove_of_peace:',
    u'\U0001f53b': u':down-pointing_red_triangle:',
    u'\U0001f409': u':dragon:',
    u'\U0001f432': u':dragon_face:',
    u'\U0001f457': u':dress:',
    u'\U0001f42a': u':dromedary_camel:',
    u'\U0001f4a7': u':droplet:',
    u'\U0001f4c0': u':dvd:',
    u'\U0001f4e7': u':e-mail:',
    u'\U0001f442': u':ear:',
    u'\U0001f33e': u':ear_of_rice:',
    u'\U0001f30d': u':earth_africa:',
    u'\U0001f30e': u':earth_americas:',
    u'\U0001f30f': u':earth_asia:',
    u'\U00002734': u':eight_pointed_black_star:',
    u'\U00002733': u':eight_spoked_asterisk:',
    u'\U000023cf': u':eject_symbol:',
    u'\U0001f50c': u':electric_plug:',
    u'\U0001f526': u':electric_torch:',
    u'\U0001f418': u':elephant:',
    u'\U00002709': u':email:',
    u'\U0001f3fb': u':emoji_modifier_fitzpatrick_type-1-2:',
    u'\U0001f3fc': u':emoji_modifier_fitzpatrick_type-3:',
    u'\U0001f3fd': u':emoji_modifier_fitzpatrick_type-4:',
    u'\U0001f3fe': u':emoji_modifier_fitzpatrick_type-5:',
    u'\U0001f3ff': u':emoji_modifier_fitzpatrick_type-6:',
    u'\U0001f51a': u':end:',
    u'\U0001f4e9': u':envelope_with_arrow:',
    u'\U0001f3f0': u':european_castle:',
    u'\U0001f3e4': u':european_post_office:',
    u'\U0001f332': u':evergreen_tree:',
    u'\U00002757': u':exclamation:',
    u'\U00002049': u':exclamation_question_mark:',
    u'\U0001f611': u':expressionless:',
    u'\U0001f441': u':eye:',
    u'\U0001f453': u':eyeglasses:',
    u'\U0001f440': u':eyes:',
    u'\U0001f486': u':face_massage:',
    u'\U0001f60b': u':face_savouring_delicious_food:',
    u'\U0001f631': u':face_screaming_in_fear:',
    u'\U0001f618': u':face_throwing_a_kiss:',
    u'\U0001f613': u':face_with_cold_sweat:',
    u'\U0001f915': u':face_with_head-bandage:',
    u'\U0001f624': u':face_with_look_of_triumph:',
    u'\U0001f637': u':face_with_medical_mask:',
    u'\U0001f645': u':face_with_no_good_gesture:',
    u'\U0001f646': u':face_with_ok_gesture:',
    u'\U0001f62e': u':face_with_open_mouth:',
    u'\U0001f644': u':face_with_rolling_eyes:',
    u'\U0001f61b': u':face_with_stuck-out_tongue:',
    u'\U0001f602': u':face_with_tears_of_joy:',
    u'\U0001f912': u':face_with_thermometer:',
    u'\U0001f636': u':face_without_mouth:',
    u'\U0001f44a': u':facepunch:',
    u'\U0001f3ed': u':factory:',
    u'\U0001f342': u':fallen_leaf:',
    u'\U0001f46a': u':family:',
    u'\U0001f385': u':father_christmas:',
    u'\U0001f4e0': u':fax:',
    u'\U0001f628': u':fearful:',
    u'\U0001f43e': u':feet:',
    u'\U0001f3a1': u':ferris_wheel:',
    u'\U000026f4': u':ferry:',
    u'\U0001f3d1': u':field_hockey_stick_and_ball:',
    u'\U0001f5c4': u':file_cabinet:',
    u'\U0001f4c1': u':file_folder:',
    u'\U0001f39e': u':film_frames:',
    u'\U0001f4fd': u':film_projector:',
    u'\U0001f525': u':fire:',
    u'\U0001f692': u':fire_engine:',
    u'\U0001f387': u':firework_sparkler:',
    u'\U0001f386': u':fireworks:',
    u'\U0001f313': u':first_quarter_moon:',
    u'\U0001f31b': u':first_quarter_moon_with_face:',
    u'\U0001f41f': u':fish:',
    u'\U0001f365': u':fish_cake:',
    u'\U0001f3a3': u':fishing_pole_and_fish:',
    u'\U0000270a': u':fist:',
    u'\U000026f3': u':flag_in_hole:',
    u'\U0000269c': u':fleur-de-lis:',
    u'\U0001f4aa': u':flexed_biceps:',
    u'\U0001f4be': u':floppy_disk:',
    u'\U0001f3b4': u':flower_playing_cards:',
    u'\U0001f633': u':flushed:',
    u'\U0001f32b': u':fog:',
    u'\U0001f301': u':foggy:',
    u'\U0001f463': u':footprints:',
    u'\U0001f374': u':fork_and_knife:',
    u'\U0001f37d': u':fork_and_knife_with_plate:',
    u'\U000026f2': u':fountain:',
    u'\U0001f340': u':four_leaf_clover:',
    u'\U0001f5bc': u':frame_with_picture:',
    u'\U0001f193': u':free:',
    u'\U0001f35f': u':french_fries:',
    u'\U0001f364': u':fried_shrimp:',
    u'\U0001f438': u':frog:',
    u'\U0001f425': u':front-facing_baby_chick:',
    u'\U0001f626': u':frowning:',
    u'\U000026fd': u':fuel_pump:',
    u'\U0001f315': u':full_moon:',
    u'\U0001f31d': u':full_moon_with_face:',
    u'\U000026b1': u':funeral_urn:',
    u'\U0001f3b2': u':game_die:',
    u'\U00002699': u':gear:',
    u'\U0001f48e': u':gem:',
    u'\U0000264a': u':gemini:',
    u'\U0001f47b': u':ghost:',
    u'\U0001f381': u':gift:',
    u'\U0001f49d': u':gift_heart:',
    u'\U0001f467': u':girl:',
    u'\U0001f310': u':globe_with_meridians:',
    u'\U0001f31f': u':glowing_star:',
    u'\U0001f410': u':goat:',
    u'\U0001f3cc': u':golfer:',
    u'\U0001f393': u':graduation_cap:',
    u'\U0001f347': u':grapes:',
    u'\U0001f34f': u':green_apple:',
    u'\U0001f4d7': u':green_book:',
    u'\U0001f49a': u':green_heart:',
    u'\U00002755': u':grey_exclamation:',
    u'\U00002754': u':grey_question:',
    u'\U0001f62c': u':grimacing:',
    u'\U0001f601': u':grin:',
    u'\U0001f600': u':grinning:',
    u'\U0001f638': u':grinning_cat_face_with_smiling_eyes:',
    u'\U0001f497': u':growing_heart:',
    u'\U0001f482': u':guardsman:',
    u'\U0001f3b8': u':guitar:',
    u'\U0001f52b': u':gun:',
    u'\U0001f487': u':haircut:',
    u'\U0001f354': u':hamburger:',
    u'\U0001f528': u':hammer:',
    u'\U00002692': u':hammer_and_pick:',
    u'\U0001f6e0': u':hammer_and_wrench:',
    u'\U0001f439': u':hamster:',
    u'\U0000270b': u':hand:',
    u'\U0001f45c': u':handbag:',
    u'\U0001f4a9': u':hankey:',
    u'\U0001f64b': u':happy_person_raising_one_hand:',
    u'\U0001f423': u':hatching_chick:',
    u'\U0001f3a7': u':headphone:',
    u'\U0001f649': u':hear-no-evil_monkey:',
    u'\U00002764': u':heart:',
    u'\U0001f49f': u':heart_decoration:',
    u'\U0001f60d': u':heart_eyes:',
    u'\U0001f63b': u':heart_eyes_cat:',
    u'\U00002714': u':heavy_check_mark:',
    u'\U00002797': u':heavy_division_sign:',
    u'\U0001f4b2': u':heavy_dollar_sign:',
    u'\U00002763': u':heavy_heart_exclamation_mark_ornament:',
    u'\U00002b55': u':heavy_large_circle:',
    u'\U00002796': u':heavy_minus_sign:',
    u'\U00002716': u':heavy_multiplication_x:',
    u'\U00002795': u':heavy_plus_sign:',
    u'\U0001f681': u':helicopter:',
    u'\U00002388': u':helm_symbol:',
    u'\U000026d1': u':helmet_with_white_cross:',
    u'\U0001f33f': u':herb:',
    u'\U0001f33a': u':hibiscus:',
    u'\U0001f460': u':high-heeled_shoe:',
    u'\U0001f506': u':high_brightness:',
    u'\U000026a1': u':high_voltage_sign:',
    u'\U0001f52a': u':hocho:',
    u'\U0001f573': u':hole:',
    u'\U0001f36f': u':honey_pot:',
    u'\U0001f6a5': u':horizontal_traffic_light:',
    u'\U0001f40e': u':horse:',
    u'\U0001f434': u':horse:',
    u'\U0001f3c7': u':horse_racing:',
    u'\U0001f3e5': u':hospital:',
    u'\U0001f32d': u':hot_dog:',
    u'\U0001f336': u':hot_pepper:',
    u'\U00002668': u':hot_springs:',
    u'\U0001f3e8': u':hotel:',
    u'\U0000231b': u':hourglass:',
    u'\U000023f3': u':hourglass_flowing_sand:',
    u'\U0001f3e0': u':house:',
    u'\U0001f3d8': u':house_buildings:',
    u'\U0001f3e1': u':house_with_garden:',
    u'\U0001f917': u':hugging_face:',
    u'\U0001f62f': u':hushed:',
    u'\U0001f368': u':ice_cream:',
    u'\U0001f3d2': u':ice_hockey_stick_and_puck:',
    u'\U000026f8': u':ice_skate:',
    u'\U0001f366': u':icecream:',
    u'\U0001f194': u':id:',
    u'\U0001f47f': u':imp:',
    u'\U0001f4e5': u':inbox_tray:',
    u'\U0001f4e8': u':incoming_envelope:',
    u'\U0001f481': u':information_desk_person:',
    u'\U00002139': u':information_source:',
    u'\U0001f607': u':innocent:',
    u'\U0001f523': u':input_symbol_for_symbols:',
    u'\U0001f4f1': u':iphone:',
    u'\U0001f3ee': u':izakaya_lantern:',
    u'\U0001f383': u':jack-o-lantern:',
    u'\U0001f5fe': u':japan:',
    u'\U0001f3ef': u':japanese_castle:',
    u'\U0001f47a': u':japanese_goblin:',
    u'\U0001f479': u':japanese_ogre:',
    u'\U0001f3e3': u':japanese_post_office:',
    u'\U0001f456': u':jeans:',
    u'\U0001f579': u':joystick:',
    u'\U0001f54b': u':kaaba:',
    u'\U0001f511': u':key:',
    u'\U00002328': u':keyboard:',
    u'\U0001f51f': u':keycap_ten:',
    u'\U0001f458': u':kimono:',
    u'\U0001f48b': u':kiss:',
    u'\U0001f617': u':kissing:',
    u'\U0001f63d': u':kissing_cat:',
    u'\U0001f61a': u':kissing_closed_eyes:',
    u'\U0001f619': u':kissing_face_with_smiling_eyes:',
    u'\U0001f428': u':koala:',
    u'\U0001f201': u':koko:',
    u'\U0001f3f7': u':label:',
    u'\U0001f535': u':large_blue_circle:',
    u'\U0001f537': u':large_blue_diamond:',
    u'\U0001f536': u':large_orange_diamond:',
    u'\U0001f534': u':large_red_circle:',
    u'\U0001f317': u':last_quarter_moon:',
    u'\U0001f31c': u':last_quarter_moon_with_face:',
    u'\U0000271d': u':latin_cross:',
    u'\U0001f606': u':laughing:',
    u'\U0001f343': u':leaf_fluttering_in_wind:',
    u'\U0001f4d2': u':ledger:',
    u'\U0001f50d': u':left-pointing_magnifying_glass:',
    u'\U0001f6c5': u':left_luggage:',
    u'\U00002194': u':left_right_arrow:',
    u'\U0001f4ac': u':left_speech_bubble:',
    u'\U000021a9': u':leftwards_arrow_with_hook:',
    u'\U0001f34b': u':lemon:',
    u'\U0000264c': u':leo:',
    u'\U0001f406': u':leopard:',
    u'\U0001f39a': u':level_slider:',
    u'\U0000264e': u':libra:',
    u'\U0001f688': u':light_rail:',
    u'\U0001f517': u':link:',
    u'\U0001f587': u':linked_paperclips:',
    u'\U0001f981': u':lion_face:',
    u'\U0001f444': u':lips:',
    u'\U0001f484': u':lipstick:',
    u'\U0001f512': u':lock:',
    u'\U0001f50f': u':lock_with_ink_pen:',
    u'\U0001f36d': u':lollipop:',
    u'\U0001f50a': u':loud_sound:',
    u'\U0001f62d': u':loudly_crying_face:',
    u'\U0001f4e2': u':loudspeaker:',
    u'\U0001f3e9': u':love_hotel:',
    u'\U0001f48c': u':love_letter:',
    u'\U0001f505': u':low_brightness:',
    u'\U0001f58a': u':lower_left_ballpoint_pen:',
    u'\U0001f58d': u':lower_left_crayon:',
    u'\U0001f58b': u':lower_left_fountain_pen:',
    u'\U0001f58c': u':lower_left_paintbrush:',
    u'\U0001f50e': u':mag_right:',
    u'\U0001f004': u':mahjong:',
    u'\U0001f4ec': u':mailbox_with_mail:',
    u'\U0001f4ed': u':mailbox_with_no_mail:',
    u'\U0001f468': u':man:',
    u'\U0001f574': u':man_in_business_suit_levitating:',
    u'\U0001f472': u':man_with_gua_pi_mao:',
    u'\U0001f473': u':man_with_turban:',
    u'\U0001f45e': u':mans_shoe:',
    u'\U0001f570': u':mantelpiece_clock:',
    u'\U0001f341': u':maple_leaf:',
    u'\U0001f356': u':meat_on_bone:',
    u'\U0001f3c5': u':medal:',
    u'\U000026aa': u':medium_white_circle:',
    u'\U0001f348': u':melon:',
    u'\U0001f4dd': u':memo:',
    u'\U0001f54e': u':menorah_with_nine_branches:',
    u'\U0001f6b9': u':mens:',
    u'\U0001f687': u':metro:',
    u'\U0001f3a4': u':microphone:',
    u'\U0001f52c': u':microscope:',
    u'\U0001f595': u':middle_finger:',
    u'\U0001f396': u':military_medal:',
    u'\U0001f30c': u':milky_way:',
    u'\U0001f690': u':minibus:',
    u'\U0001f4bd': u':minidisc:',
    u'\U0001f4f4': u':mobile_phone_off:',
    u'\U0001f911': u':money-mouth_face:',
    u'\U0001f4b0': u':money_bag:',
    u'\U0001f4b8': u':money_with_wings:',
    u'\U0001f412': u':monkey:',
    u'\U0001f435': u':monkey_face:',
    u'\U0001f69d': u':monorail:',
    u'\U0001f314': u':moon:',
    u'\U0001f391': u':moon_viewing_ceremony:',
    u'\U0001f54c': u':mosque:',
    u'\U0001f324': u':mostly_sunny:',
    u'\U0001f6e5': u':motor_boat:',
    u'\U0001f6e3': u':motorway:',
    u'\U0001f5fb': u':mount_fuji:',
    u'\U000026f0': u':mountain:',
    u'\U0001f6b5': u':mountain_bicyclist:',
    u'\U0001f6a0': u':mountain_cableway:',
    u'\U0001f69e': u':mountain_railway:',
    u'\U0001f401': u':mouse2:',
    u'\U0001f42d': u':mouse:',
    u'\U0001f3a5': u':movie_camera:',
    u'\U0001f5ff': u':moyai:',
    u'\U0001f3b6': u':multiple_musical_notes:',
    u'\U0001f344': u':mushroom:',
    u'\U0001f3b9': u':musical_keyboard:',
    u'\U0001f3b5': u':musical_note:',
    u'\U0001f3bc': u':musical_score:',
    u'\U0001f507': u':mute:',
    u'\U0001f485': u':nail_care:',
    u'\U0001f4db': u':name_badge:',
    u'\U0001f3de': u':national_park:',
    u'\U0001f454': u':necktie:',
    u'\U0000274e': u':negative_squared_cross_mark:',
    u'\U0001f913': u':nerd_face:',
    u'\U0001f610': u':neutral_face:',
    u'\U0001f195': u':new:',
    u'\U0001f311': u':new_moon:',
    u'\U0001f31a': u':new_moon_with_face:',
    u'\U0001f4f0': u':newspaper:',
    u'\U0001f196': u':ng:',
    u'\U0001f303': u':night_with_stars:',
    u'\U0001f6b3': u':no_bicycles:',
    u'\U000026d4': u':no_entry:',
    u'\U0001f6ab': u':no_entry_sign:',
    u'\U0001f4f5': u':no_mobile_phones:',
    u'\U0001f51e': u':no_one_under_eighteen_symbol:',
    u'\U0001f6b7': u':no_pedestrians:',
    u'\U0001f6ad': u':no_smoking:',
    u'\U0001f6b1': u':non-potable_water:',
    u'\U0001f443': u':nose:',
    u'\U0001f4d3': u':notebook:',
    u'\U0001f4d4': u':notebook_with_decorative_cover:',
    u'\U0001f529': u':nut_and_bolt:',
    u'\U0001f17e': u':o2:',
    u'\U0001f30a': u':ocean:',
    u'\U0001f419': u':octopus:',
    u'\U0001f362': u':oden:',
    u'\U0001f3e2': u':office:',
    u'\U0001f6e2': u':oil_drum:',
    u'\U0001f197': u':ok:',
    u'\U0001f44c': u':ok_hand:',
    u'\U0001f5dd': u':old_key:',
    u'\U0001f474': u':older_man:',
    u'\U0001f475': u':older_woman:',
    u'\U0001f549': u':om_symbol:',
    u'\U0001f51b': u':on:',
    u'\U0001f698': u':oncoming_automobile:',
    u'\U0001f68d': u':oncoming_bus:',
    u'\U0001f694': u':oncoming_police_car:',
    u'\U0001f696': u':oncoming_taxi:',
    u'\U0001f4c2': u':open_file_folder:',
    u'\U0001f450': u':open_hands:',
    u'\U0001f513': u':open_lock:',
    u'\U000026ce': u':ophiuchus:',
    u'\U0001f4d9': u':orange_book:',
    u'\U00002626': u':orthodox_cross:',
    u'\U0001f4e4': u':outbox_tray:',
    u'\U0001f402': u':ox:',
    u'\U0001f4e6': u':package:',
    u'\U0001f4c4': u':page_facing_up:',
    u'\U0001f4c3': u':page_with_curl:',
    u'\U0001f4df': u':pager:',
    u'\U0001f334': u':palm_tree:',
    u'\U0001f43c': u':panda_face:',
    u'\U0001f4ce': u':paperclip:',
    u'\U0001f17f': u':parking:',
    u'\U0000303d': u':part_alternation_mark:',
    u'\U000026c5': u':partly_sunny:',
    u'\U0001f326': u':partly_sunny_rain:',
    u'\U0001f389': u':party_popper:',
    u'\U0001f6f3': u':passenger_ship:',
    u'\U0001f6c2': u':passport_control:',
    u'\U0000262e': u':peace_symbol:',
    u'\U0001f351': u':peach:',
    u'\U0001f350': u':pear:',
    u'\U0001f6b6': u':pedestrian:',
    u'\U0000270f': u':pencil2:',
    u'\U0001f427': u':penguin:',
    u'\U0001f614': u':pensive:',
    u'\U0001f3ad': u':performing_arts:',
    u'\U0001f623': u':persevere:',
    u'\U0001f64d': u':person_frowning:',
    u'\U000026f9': u':person_with_ball:',
    u'\U0001f471': u':person_with_blond_hair:',
    u'\U0001f64f': u':person_with_folded_hands:',
    u'\U0001f64e': u':person_with_pouting_face:',
    u'\U000026cf': u':pick:',
    u'\U0001f416': u':pig2:',
    u'\U0001f437': u':pig:',
    u'\U0001f43d': u':pig_nose:',
    u'\U0001f48a': u':pill:',
    u'\U0001f34d': u':pineapple:',
    u'\U00002653': u':pisces:',
    u'\U0001f355': u':pizza:',
    u'\U0001f6d0': u':place_of_worship:',
    u'\U0001f447': u':point_down:',
    u'\U0001f448': u':point_left:',
    u'\U0001f449': u':point_right:',
    u'\U0000261d': u':point_up:',
    u'\U0001f446': u':point_up_2:',
    u'\U0001f693': u':police_car:',
    u'\U0001f6a8': u':police_cars_revolving_light:',
    u'\U0001f429': u':poodle:',
    u'\U0001f37f': u':popcorn:',
    u'\U0001f4ef': u':postal_horn:',
    u'\U0001f4ee': u':postbox:',
    u'\U0001f372': u':pot_of_food:',
    u'\U0001f6b0': u':potable_water:',
    u'\U0001f45d': u':pouch:',
    u'\U0001f357': u':poultry_leg:',
    u'\U0001f63e': u':pouting_cat:',
    u'\U0001f621': u':pouting_face:',
    u'\U0001f4ff': u':prayer_beads:',
    u'\U0001f478': u':princess:',
    u'\U0001f5a8': u':printer:',
    u'\U0001f49c': u':purple_heart:',
    u'\U0001f45b': u':purse:',
    u'\U0001f4cc': u':pushpin:',
    u'\U0001f6ae': u':put_litter_in_its_place:',
    u'\U0001f407': u':rabbit2:',
    u'\U0001f430': u':rabbit:',
    u'\U0001f3ce': u':racing_car:',
    u'\U0001f3cd': u':racing_motorcycle:',
    u'\U0001f4fb': u':radio:',
    u'\U0001f518': u':radio_button:',
    u'\U00002622': u':radioactive_sign:',
    u'\U0001f683': u':railway_car:',
    u'\U0001f6e4': u':railway_track:',
    u'\U0001f308': u':rainbow:',
    u'\U0001f590': u':raised_hand_with_fingers_splayed:',
    u'\U0001f64c': u':raised_hands:',
    u'\U0001f40f': u':ram:',
    u'\U0001f35c': u':ramen:',
    u'\U0001f400': u':rat:',
    u'\U000000ae': u':registered:',
    u'\U0000263a': u':relaxed:',
    u'\U0001f60c': u':relieved:',
    u'\U0001f397': u':reminder_ribbon:',
    u'\U0001f501': u':repeat:',
    u'\U0001f502': u':repeat_one:',
    u'\U0001f6bb': u':restroom:',
    u'\U0001f49e': u':revolving_hearts:',
    u'\U0001f380': u':ribbon:',
    u'\U0001f359': u':rice_ball:',
    u'\U0001f358': u':rice_cracker:',
    u'\U0001f5ef': u':right_anger_bubble:',
    u'\U0001f48d': u':ring:',
    u'\U0001f360': u':roasted_sweet_potato:',
    u'\U0001f916': u':robot_face:',
    u'\U0001f680': u':rocket:',
    u'\U0001f5de': u':rolled-up_newspaper:',
    u'\U0001f3a2': u':roller_coaster:',
    u'\U0001f413': u':rooster:',
    u'\U0001f339': u':rose:',
    u'\U0001f3f5': u':rosette:',
    u'\U0001f4cd': u':round_pushpin:',
    u'\U0001f6a3': u':rowboat:',
    u'\U0001f3c9': u':rugby_football:',
    u'\U0001f3c3': u':runner:',
    u'\U0001f3bd': u':running_shirt_with_sash:',
    u'\U0001f202': u':sa:',
    u'\U00002650': u':sagittarius:',
    u'\U0001f376': u':sake:',
    u'\U0001f461': u':sandal:',
    u'\U0001f4e1': u':satellite:',
    u'\U0001f6f0': u':satellite:',
    u'\U0001f3b7': u':saxophone:',
    u'\U00002696': u':scales:',
    u'\U0001f3eb': u':school:',
    u'\U0001f392': u':school_satchel:',
    u'\U0001f982': u':scorpion:',
    u'\U0000264f': u':scorpius:',
    u'\U0001f640': u':scream_cat:',
    u'\U0001f4dc': u':scroll:',
    u'\U0001f4ba': u':seat:',
    u'\U0001f648': u':see-no-evil_monkey:',
    u'\U0001f331': u':seedling:',
    u'\U00002618': u':shamrock:',
    u'\U0001f367': u':shaved_ice:',
    u'\U0001f411': u':sheep:',
    u'\U0001f41a': u':shell:',
    u'\U0001f6e1': u':shield:',
    u'\U000026e9': u':shinto_shrine:',
    u'\U0001f6a2': u':ship:',
    u'\U0001f455': u':shirt:',
    u'\U0001f320': u':shooting_star:',
    u'\U0001f6cd': u':shopping_bags:',
    u'\U0001f6bf': u':shower:',
    u'\U0001f918': u':sign_of_the_horns:',
    u'\U0001f642': u':simple_smile:',
    u'\U0001f52f': u':six_pointed_star:',
    u'\U0001f3bf': u':ski:',
    u'\U000026f7': u':skier:',
    u'\U0001f480': u':skull:',
    u'\U00002620': u':skull_and_crossbones:',
    u'\U0001f634': u':sleeping:',
    u'\U0001f6cc': u':sleeping_accommodation:',
    u'\U0001f4a4': u':sleeping_symbol:',
    u'\U0001f62a': u':sleepy:',
    u'\U0001f575': u':sleuth_or_spy:',
    u'\U0001f641': u':slightly_frowning_face:',
    u'\U0001f3b0': u':slot_machine:',
    u'\U0001f6e9': u':small_airplane:',
    u'\U0001f539': u':small_blue_diamond:',
    u'\U0001f538': u':small_orange_diamond:',
    u'\U0001f53a': u':small_red_triangle:',
    u'\U0001f604': u':smile:',
    u'\U0001f603': u':smiley:',
    u'\U0001f63a': u':smiley_cat:',
    u'\U0001f608': u':smiling_face_with_horns:',
    u'\U0001f605': u':smiling_face_with_open_mouth_and_cold_sweat:',
    u'\U0001f60e': u':smiling_face_with_sunglasses:',
    u'\U0001f60f': u':smirk:',
    u'\U0001f6ac': u':smoking:',
    u'\U0001f40c': u':snail:',
    u'\U0001f40d': u':snake:',
    u'\U0001f3d4': u':snow_capped_mountain:',
    u'\U0001f3c2': u':snowboarder:',
    u'\U00002744': u':snowflake:',
    u'\U00002603': u':snowman:',
    u'\U000026c4': u':snowman_without_snow:',
    u'\U000026bd': u':soccer:',
    u'\U0001f51c': u':soon:',
    u'\U0001f198': u':sos:',
    u'\U0001f509': u':sound:',
    u'\U0001f35d': u':spaghetti:',
    u'\U00002747': u':sparkle:',
    u'\U00002728': u':sparkles:',
    u'\U0001f496': u':sparkling_heart:',
    u'\U0001f64a': u':speak-no-evil_monkey:',
    u'\U0001f508': u':speaker:',
    u'\U0001f5e3': u':speaking_head_in_silhouette:',
    u'\U0001f6a4': u':speedboat:',
    u'\U0001f577': u':spider:',
    u'\U0001f578': u':spider_web:',
    u'\U0001f5d3': u':spiral_calendar_pad:',
    u'\U0001f5d2': u':spiral_note_pad:',
    u'\U0001f4a6': u':splashing_sweat_symbol:',
    u'\U0001f596': u':spock-hand:',
    u'\U0001f433': u':spouting_whale:',
    u'\U0001f199': u':squared_up_with_exclamation_mark:',
    u'\U0001f19a': u':squared_vs:',
    u'\U0001f3df': u':stadium:',
    u'\U00002b50': u':star:',
    u'\U0000262a': u':star_and_crescent:',
    u'\U00002721': u':star_of_david:',
    u'\U0001f689': u':station:',
    u'\U0001f5fd': u':statue_of_liberty:',
    u'\U0001f682': u':steam_locomotive:',
    u'\U000023f1': u':stopwatch:',
    u'\U0001f4cf': u':straight_ruler:',
    u'\U0001f353': u':strawberry:',
    u'\U0001f61d': u':stuck_out_tongue_closed_eyes:',
    u'\U0001f61c': u':stuck_out_tongue_winking_eye:',
    u'\U0001f399': u':studio_microphone:',
    u'\U0001f31e': u':sun_with_face:',
    u'\U0001f33b': u':sunflower:',
    u'\U0001f305': u':sunrise:',
    u'\U0001f304': u':sunrise_over_mountains:',
    u'\U0001f3c4': u':surfer:',
    u'\U0001f363': u':sushi:',
    u'\U0001f69f': u':suspension_railway:',
    u'\U0001f3ca': u':swimmer:',
    u'\U0001f54d': u':synagogue:',
    u'\U0001f489': u':syringe:',
    u'\U0001f3d3': u':table_tennis_paddle_and_ball:',
    u'\U0001f32e': u':taco:',
    u'\U0001f38b': u':tanabata_tree:',
    u'\U0001f34a': u':tangerine:',
    u'\U00002649': u':taurus:',
    u'\U0001f695': u':taxi:',
    u'\U0001f375': u':tea:',
    u'\U0001f4de': u':telephone_receiver:',
    u'\U0001f52d': u':telescope:',
    u'\U0001f4fa': u':television:',
    u'\U0001f3be': u':tennis:',
    u'\U000026fa': u':tent:',
    u'\U0001f321': u':thermometer:',
    u'\U0001f914': u':thinking_face:',
    u'\U0001f4ad': u':thought_balloon:',
    u'\U0001f5b1': u':three_button_mouse:',
    u'\U000026c8': u':thunder_cloud_and_rain:',
    u'\U0001f3ab': u':ticket:',
    u'\U0001f405': u':tiger2:',
    u'\U0001f42f': u':tiger:',
    u'\U000023f2': u':timer_clock:',
    u'\U0001f62b': u':tired_face:',
    u'\U00002122': u':tm:',
    u'\U0001f6bd': u':toilet:',
    u'\U0001f5fc': u':tokyo_tower:',
    u'\U0001f345': u':tomato:',
    u'\U0001f445': u':tongue:',
    u'\U0001f51d': u':top:',
    u'\U0001f3a9': u':top_hat:',
    u'\U0001f5b2': u':trackball:',
    u'\U0001f69c': u':tractor:',
    u'\U0001f686': u':train2:',
    u'\U0001f68b': u':train:',
    u'\U0001f68a': u':tram:',
    u'\U0001f6a9': u':triangular_flag_on_post:',
    u'\U0001f4d0': u':triangular_ruler:',
    u'\U0001f531': u':trident:',
    u'\U0001f68e': u':trolleybus:',
    u'\U0001f3c6': u':trophy:',
    u'\U0001f379': u':tropical_drink:',
    u'\U0001f420': u':tropical_fish:',
    u'\U0001f3ba': u':trumpet:',
    u'\U0001f337': u':tulip:',
    u'\U0001f983': u':turkey:',
    u'\U0001f422': u':turtle:',
    u'\U0001f500': u':twisted_rightwards_arrows:',
    u'\U0001f495': u':two_hearts:',
    u'\U0001f46c': u':two_men_holding_hands:',
    u'\U0001f46d': u':two_women_holding_hands:',
    u'\U00002602': u':umbrella:',
    u'\U000026f1': u':umbrella_on_ground:',
    u'\U00002614': u':umbrella_with_rain_drops:',
    u'\U0001f612': u':unamused:',
    u'\U0001f984': u':unicorn_face:',
    u'\U0001f643': u':upside-down_face:',
    u'\U0000270c': u':v:',
    u'\U0001f6a6': u':vertical_traffic_light:',
    u'\U0001f4fc': u':vhs:',
    u'\U0001f4f3': u':vibration_mode:',
    u'\U0001f4f9': u':video_camera:',
    u'\U0001f3ae': u':video_game:',
    u'\U0001f3bb': u':violin:',
    u'\U0000264d': u':virgo:',
    u'\U0001f30b': u':volcano:',
    u'\U0001f3d0': u':volleyball:',
    u'\U0001f318': u':waning_crescent_moon:',
    u'\U0001f316': u':waning_gibbous_moon:',
    u'\U000026a0': u':warning:',
    u'\U0001f5d1': u':wastebasket:',
    u'\U0000231a': u':watch:',
    u'\U0001f403': u':water_buffalo:',
    u'\U0001f6be': u':water_closet:',
    u'\U0001f349': u':watermelon:',
    u'\U0001f44b': u':wave:',
    u'\U0001f3f4': u':waving_black_flag:',
    u'\U0001f3f3': u':waving_white_flag:',
    u'\U00003030': u':wavy_dash:',
    u'\U0001f312': u':waxing_crescent_moon:',
    u'\U0001f629': u':weary:',
    u'\U0001f492': u':wedding:',
    u'\U0001f3cb': u':weight_lifter:',
    u'\U0001f40b': u':whale2:',
    u'\U00002638': u':wheel_of_dharma:',
    u'\U0000267f': u':wheelchair:',
    u'\U00002705': u':white_check_mark:',
    u'\U0001f4ae': u':white_flower:',
    u'\U00002639': u':white_frowning_face:',
    u'\U00002b1c': u':white_large_square:',
    u'\U000025fd': u':white_medium_small_square:',
    u'\U000025fb': u':white_medium_square:',
    u'\U000025ab': u':white_small_square:',
    u'\U0001f533': u':white_square_button:',
    u'\U0001f32c': u':wind_blowing_face:',
    u'\U0001f390': u':wind_chime:',
    u'\U0001f377': u':wine_glass:',
    u'\U0001f609': u':wink:',
    u'\U0001f43a': u':wolf:',
    u'\U0001f469': u':woman:',
    u'\U0001f45a': u':womans_clothes:',
    u'\U0001f452': u':womans_hat:',
    u'\U0001f6ba': u':womens:',
    u'\U0001f5fa': u':world_map:',
    u'\U0001f61f': u':worried:',
    u'\U0001f527': u':wrench:',
    u'\U0000270d': u':writing_hand:',
    u'\U0001f49b': u':yellow_heart:',
    u'\U0000262f': u':yin_yang:',
    u'\U0001f910': u':zipper-mouth_face:',
}

# Expressions taken from Martijn Pieters code at
# http://stackoverflow.com/questions/26568722/remove-unicode-emoji-using-re-in-python
try:
    # Wide UCS-4 build
    ALIAS_RE = re.compile(u'['u'\U0001F300-\U0001F64F'
                          u'\U0001F680-\U0001F6FF'
                          u'\u2600-\u26FF\u2700-\u27BF]+',
                          re.DOTALL | re.UNICODE)
except re.error:
    # Narrow UCS-2 build
    ALIAS_RE = re.compile(u'('u'\ud83c[\udf00-\udfff]|'
                          u'\ud83d[\udc00-\ude4f\ude80-\udeff]|'
                          u'[\u2600-\u26FF\u2700-\u27BF])+',
                          re.DOTALL | re.UNICODE)


def convert_emoji_to_aliases(data, modifier, modifier_data, string):
    if sys.version_info < (3, ):
        string = string.decode('utf-8')
    emoji_found = ALIAS_RE.findall(string)
    for emoji in emoji_found:
        if emoji in EMOJI_ALIASES:
            string = string.replace(emoji, EMOJI_ALIASES[emoji])
    return string


if __name__ == "__main__" and import_ok:

    if w.register(SCRIPT_NAME, SCRIPT_AUTHOR, SCRIPT_VERSION, SCRIPT_LICENSE,
                  SCRIPT_DESC, "", "utf-8"):

        w.hook_modifier("irc_in_away", "convert_emoji_to_aliases", "")
        w.hook_modifier("irc_in_cnotice", "convert_emoji_to_aliases", "")
        w.hook_modifier("irc_in_cprivmsg", "convert_emoji_to_aliases", "")
        w.hook_modifier("irc_in_kick", "convert_emoji_to_aliases", "")
        w.hook_modifier("irc_in_knock", "convert_emoji_to_aliases", "")
        w.hook_modifier("irc_in_notice", "convert_emoji_to_aliases", "")
        w.hook_modifier("irc_in_part", "convert_emoji_to_aliases", "")
        w.hook_modifier("irc_in_privmsg", "convert_emoji_to_aliases", "")
        w.hook_modifier("irc_in_quit", "convert_emoji_to_aliases", "")
        w.hook_modifier("irc_in_wallops", "convert_emoji_to_aliases", "")

        # This intercepts outgoing emoji also
        # w.hook_modifier("input_text_for_buffer",
        #                 "convert_emoji_to_aliases", "")
