#encoding: utf-8
#- Name: SiSU
#
#  - Description: documents, structuring, processing, publishing, search
#    json
#
#  - Author: Ralph Amissah
#    <ralph.amissah@gmail.com>
#
#  - Copyright: (C) 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006,
#    2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2019,
#    2020, 2021, Ralph Amissah,
#    All Rights Reserved.
#
#  - License: GPL 3 or later:
#
#    SiSU, a framework for document structuring, publishing and search
#
#    Copyright (C) Ralph Amissah
#
#    This program is free software: you can redistribute it and/or modify it
#    under the terms of the GNU General Public License as published by the Free
#    Software Foundation, either version 3 of the License, or (at your option)
#    any later version.
#
#    This program is distributed in the hope that it will be useful, but WITHOUT
#    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
#    FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
#    more details.
#
#    You should have received a copy of the GNU General Public License along with
#    this program. If not, see <http://www.gnu.org/licenses/>.
#
#    If you have Internet connection, the latest version of the GPL should be
#    available at these locations:
#    <http://www.fsf.org/licensing/licenses/gpl.html>
#    <http://www.gnu.org/licenses/gpl.html>
#
#    <http://www.sisudoc.org/sisu/en/manifest/gpl.fsf.html>
#
#  - SiSU uses:
#    - Standard SiSU markup syntax,
#    - Standard SiSU meta-markup syntax, and the
#    - Standard SiSU object citation numbering and system
#
#  - Homepages:
#    <http://www.sisudoc.org>
#
#  - Git
#    <https://git.sisudoc.org/projects/>
#    <https://git.sisudoc.org/projects/?p=software/sisu.git;a=summary>
#    <https://git.sisudoc.org/projects/?p=markup/sisu-markup-samples.git;a=summary>
module SiSU_JSON_Format
  require_relative 'dp'                                 # dp.rb
  require_relative 'json_parts'                         # json_parts.rb
  include SiSU_Param
  class ParagraphNumber
    def initialize(md,paranum)
      @md=md
      @paranum=(paranum \
      ? (/(\d+)/m.match(paranum)[1])
      : nil)
    end
    def display
      p_num_display=if @paranum
        @paranum.gsub(/(\d+)/,
        '<font size="1" color="#777777">' +
        '&nbsp;&nbsp;\1</font>')
      else ''
      end
      p_num_display
    end
    def name
      p_num_name=@paranum.gsub(/(\d+)/,'<a name="\1"></a>')
      p_num_name
    end
    def goto
      p_num_goto=@paranum.gsub(/(\d+)/,'<a href="#\1">')
      p_num_goto
    end
  end
  class HeadInformation
    include SiSU_Parts_JSON
    def initialize #dc rdf
      @full_title=@subtitle=@author=@subject=@description=@publisher=@contributor=@date=@type=@format=@identifier=@source=@language=@relation=@coverage=@rights=@copyright=@owner=@keywords=''
      @md=@@md
      # DublinCore 1 - title
      @rdfurl=%{  rdf:about="http://www.jus.uio.no/lm/toc"\n}
      if defined? @md.title.full \
      and @md.title.full                                                      # DublinCore 1 - title
        @rdf_title=%{    dc.title="#{seg_name}#{@md.title.full}"\n}
        @full_title=%{<meta name="dc.title" content="#{seg_name}#{@md.title.full}" />\n}
      end
      if defined? @md.creator.author \
      and @md.creator.author                                                  # DublinCore 2 - creator/author (author)
        @rdf_author=%{    dc.author="#{@md.creator.author}"\n}
        @author=%{<meta name="dc.author" content="#{@md.creator.author}" />\n}
      end
      if defined? @md.classify.subject \
      and @md.classify.subject=~/\S+/                                          # DublinCore 3 - subject (us library of congress, eric or udc, or schema???)
        @rdf_subject=%{    dc.subject="#{@md.classify.subject}"\n}
        @subject=%{<meta name="dc.subject" content="#{@md.classify.subject}" />\n}
      end
      if defined? @md.notes.description \
      and @md.notes.description=~/\S+/                                        # DublinCore 4 - description
        @rdf_description=%{    dc.description="#{@md.notes.description}"\n}
        @description=%{<meta name="dc.description" content="#{@md.notes.description}" />\n}
      end
      if defined? @md.publisher \
      and @md.publisher=~/\S+/                                                # DublinCore 5 - publisher (current copy published by)
        @rdf_publisher=%{    dc.publisher="#{@md.publisher}"\n}
        @publisher=%{<meta name="dc.publisher" content="#{@md.publisher}" />\n}
      end
      if defined? @md.creator.contributor \
      and @md.creator.contributor=~/\S+/                                      # DublinCore 6 - contributor
        @rdf_contributor=%{    dc.contributor="#{@md.creator.contributor}"\n}
        @contributor=%{<meta name="dc.contributor" content="#{@md.creator.contributor}" />\n}
      end
      if defined? @md.date.published \
      and @md.date.published                                                  # DublinCore 7 - date year-mm-dd
        @rdf_date=%{    dc.date="#{@md.date.published}"\n}
        @date=%{<meta name="dc.date" content="#{@md.date.published}" #{@md.date_scheme} />\n}
      end
      if defined? @md.date.created \
      and @md.date.created                                                    # DublinCore 7 - date.created year-mm-dd
        @rdf_date_created=%{    dc.date.created="#{@md.date.created}"\n}
        @date_created=%{<meta name="dc.date.created" content="#{@md.date.created}" #{@md.date_created_scheme} />\n}
      end
      if defined? @md.date.issued \
      and @md.date.issued                                                      # DublinCore 7 - date.issued year-mm-dd
        @rdf_date_issued=%{    dc.date.issued="#{@md.date.issued}"\n}
        @date_issued=%{<meta name="dc.date.issued" content="#{@md.date.issued}" #{@md.date_issued_scheme} />\n}
      end
      if defined? @md.date.available \
      and @md.date.available                                                  # DublinCore 7 - date.available year-mm-dd
        @rdf_date_available=%{    dc.date.available="#{@md.date.available}"\n}
        @date_available=%{<meta name="dc.date.available" content="#{@md.date.available}" #{@md.date_available_scheme} />\n}
      end
      if defined? @md.date.valid \
      and @md.date.valid                                                      # DublinCore 7 - date.valid year-mm-dd
        @rdf_date_valid=%{    dc.date.valid="#{@md.date.valid}"\n}
        @date_valid=%{<meta name="dc.date.valid" content="#{@md.date.valid}" #{@md.date_valid_scheme} />\n}
      end
      if defined? @md.date.modified \
      and @md.date.modified                                                   # DublinCore 7 - date.modified year-mm-dd
        @rdf_date_modified=%{    dc.date.modified="#{@md.date.modified}"\n}
        @date_modified=%{<meta name="dc.date.modified" content="#{@md.date.modified}" #{@md.date_modified_scheme} />\n}
      end
      if defined? @md.notes.coverage \
      and @md.notes.coverage=~/\S+/                                        # DublinCore 14 - coverage
        @rdf_coverage=%{    dc.coverage="#{@md.notes.coverage}"\n}
        @coverage=%{<meta name="dc.coverage" content="#{@md.notes.coverage}" />\n}
      end
      if defined? @md.notes.relation \
      and @md.notes.relation=~/\S+/                                         # DublinCore 13 - relation
        @rdf_relation=%{    dc.relation="#{@md.notes.relation}"\n}
        @relation=%{<meta name="dc.relation" content="#{@md.notes.relation}" />\n}
      end
      if defined? @md.notes.type \
      and @md.notes.type                                                            # DublinCore 8 - type (genre eg. report, convention etc)
        @rdf_type=%{    dc.type="#{@md.notes.type}"\n}
        @type=%{<meta name="dc.type" content="#{@md.notes.type}" />\n}
      end
      if defined? @md.notes.format \
      and @md.notes.format=~/\S+/                                              # DublinCore 9 - format (use your mime type)
        @rdf_format=%{    dc.format="#{@md.notes.format}"\n}
        @format=%{<meta name="dc.format" content="#{@md.notes.format}" />\n}
      end
      #if defined? @md.identifier.sisupod \
      #and @md.identifier.sisupod=~/\S+/                                       # DublinCore 10 - identifier (your identifier, could use urn which is free)
      #  @rdf_identifier=%{    dc.identifier="#{@md.identifier.sisupod}"\n}
      #  @identifier=%{<meta name="dc.identifier" content="#{@md.identifier.sisupod}" />\n}
      #end
      if defined? @md.original.source \
      and @md.original.source=~/\S+/                                           # DublinCore 11 - source (document source)
        @rdf_source=%{    dc.source="#{@md.original.source}"\n}
        @source=%{<meta name="dc.source" content="#{@md.source}" />\n}
      end
      if defined? @md.original.language \
      and @md.original.language=~/\S+/                                         # DublinCore 12 - language (English)
        @rdf_language=%{    dc.language="#{@md.original.title}"\n}
        @language=%{<meta name="dc.language" content="#{@md.language[:name]}" />\n}
      end
      if defined? @md.rights.all \
      and @md.rights.all=~/\S+/                                               # DublinCore 15 - rights
        rights=meta_content_clean(@md.rights.all)
        copyright=meta_content_clean(@md.rights.copyright.all)
        @rdf_rights=%{    dc.rights="#{rights}"\n}
        @rights=%{<meta name="dc.rights" content="#{rights}" />\n}
      end
      @copyright=%{<meta name="copyright" content="#{copyright}" />\n} \
        if @md.rights.copyright.all # possibly redundant see dc.rights
      @owner=%{<meta name="owner" content="#{@md.owner}" />\n} if @md.owner
      @keywords=%{<meta name="keywords" content="#{@md.keywords}" />\n} if @md.keywords
      @index='index'
    end
    def meta_content_clean(content='')
      content=if not content.nil?
        content=content.tr('"',"'").
           gsub(/&/,'&amp;')
        content=SiSU_XML_Munge::Trans.new(@md).char_enc.utf8(content)
      else content
      end
    end
    def table_close
      '</font> </td></tr></table>'
    end
    def toc_head
      <<WOK
<html>
<head>
<title>#{@md.html_title}</title>
<rdf:RDF xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"
         xmlns:dc="http://purl.org/dc/elements/1.1/">
 <rdf:Description
#{@rdfurl}
#{@rdf_title}
#{@rdf_subtitle}
#{@rdf_author}
#{@rdf_subject}
#{@rdf_description}
#{@rdf_publisher}
#{@rdf_contributor}
#{@rdf_date}
#{@rdf_date_created}
#{@rdf_date_issued}
#{@rdf_date_available}
#{@rdf_date_valid}
#{@rdf_date_modified}
#{@rdf_type}
#{@rdf_format}
#{@rdf_identifier}
#{@rdf_source}
#{@rdf_language}
#{@rdf_relation}
#{@rdf_coverage}
#{@rdf_rights}
  />
</rdf:RDF>
#{@full_title}
#{@author}
#{@subject}
#{@description}
#{@publisher}
#{@contributor}
#{@date}
#{@date_created}
#{@date_issued}
#{@date_available}
#{@date_valid}
#{@date_modified}
#{@type}
#{@format}
#{@identifier}
#{@source}
#{@language}
#{@relation}
#{@coverage}
#{@rights}
#{@copyright}
#{@owner}
#{@png.ico}
#{@txt.generator}
#{@js.head}
\n</head>
#{@color.body}
#{@font.css_table_file}
<a name="top"></a>
<a name="up"></a>
<a name="start"></a>
#{@js.top}
WOK
    end
  end
  class ParagraphNumber
    def initialize(md,ocn)
      @md,@ocn=md,ocn.to_s
      @ocn ||=''
    end
    def ocn_display
      @make=SiSU_Env::ProcessingSettings.new(@md)
      if @make.build.ocn?
        ocn_class='ocn'
        if @ocn.to_i==0
          @ocn.gsub(/^(\d+|)$/,
            %{<label class="#{ocn_class}"><a name="#{@ocn}">&nbsp;</a></label>})
        else
          @ocn.gsub(/^(\d+|)$/,
            %{<label class="#{ocn_class}"><a name="#{@ocn}">\\1</a></label>})
        end
      else
        ocn_class='ocn_off'
        @ocn.gsub(/^(\d+|)$/,
          %{<label class="#{ocn_class}">&nbsp;</label>})
      end
    end
    def name
      %{<a name="#{@ocn}"></a>}
    end
    def id #w3c? "tidy" complains about numbers as identifiers ! annoying
      %{id="o#{@ocn}"}
    end
    def goto
      %{<a href="##{@ocn}">}
    end
  end
  class FormatTextObject
    include SiSU_Parts_JSON
    attr_accessor :md,:dob,:txt,:ocn,:format,:table,:link,:linkname,:paranum,:p_num,:headname,:banner,:url
    def initialize(md,t_o)
      @md,@t_o=md,t_o
      if t_o.class.inspect =~/Object/
        @txt=if defined? t_o.obj; t_o.obj
        else nil
        end
        @ocn=if defined? t_o.ocn; t_o.ocn.to_s
        else nil
        end
        @headname=if t_o.is==:heading and defined? t_o.name; t_o.name
        else nil
        end
      else
        if @md.opt.act[:maintenance][:set]==:on
          p __FILE__ << ':' << __LINE__.to_s
          p t_o.class
          p caller
        end
      end
      if defined? @t_o.ocn
        ocn=((@t_o.ocn.to_s =~/\d+/) ? @t_o.ocn : nil)
        @p_num=ParagraphNumber.new(@md,ocn)
      end
      if @format and not @format.empty?
        if @format=~/^\d:(\S+)/ #need more reliable marker #if @format =~ /#{Rx[:lv]}/
          headname=$1 #format[/\d~(\S+)/m,1]
          @headname=if headname =~/^[a-zA-Z]/; %{<a name="#{headname}" id="#{headname}"></a>} #consider: h_#{headname}
          else %{<a name="h#{headname}" id="h#{headname}"></a>}
          end
        end
      end
      @dob=t_o if defined? t_o.is
    end
    def para
      para_form_css('p','norm')
    end
    def code
      para_form_css('p','code')
    end
    def center
      para_form_css('p','center')
    end
    def bold
      para_form_css('p','bold')
    end
    def bullet
      para_form_css('li','bullet')
    end
    def format(tag,attrib)
      para_form_css(tag,attrib)
    end
    def heading_normal(tag,attrib)
      %{
<div class="substance">
  #{@p_num.ocn_display}
  <#{tag} class="#{attrib}" #{@p_num.id}>#{@p_num.name}
    #{@headname}#{@txt}
  </#{tag}>
</div>
}
    end
    def heading_body
      heading_normal('p','norm')
    end
    def heading_body0
      heading_normal('h1','norm')
    end
    def heading_body1
      heading_normal('h1','norm')
    end
    def heading_body2
      heading_normal('h2','norm')
    end
    def heading_body3
      heading_normal('h3','norm')
    end
    def heading_body4
      heading_normal('h4','norm')
    end
    def heading_body5
      heading_normal('h5','norm')
    end
    def heading_body6
      heading_normal('h6','norm')
    end
    def heading_body7
      heading_normal('h7','norm')
    end
    def title_header(tag,attrib)
      %{
<div class="content">
<#{tag} class="#{attrib}">
    #{@txt}
  </#{tag}>
</div>
}
    end
    def title_header1
      title_header('h1','tiny')
    end
    def title_header2
      title_header('h2','tiny')
    end
    def title_header3
      title_header('h3','tiny')
    end
    def title_header4
      ''
    end
    def dl #check :trailer
      "<dl><b>#{@txt}</b> #{@trailer}</dl>"
    end
    def table_css_end      #<!TZ!>
      '</table>
    </p>
  </div>'
    end
    def gsub_body
#fix
      @txt=case @txt
      when /^\s*\((i+|iv|v|vi+|ix|x|xi+)\)/
        @txt.gsub(/^\((i+|iv|v|vi+|ix|x|xi+)\)/,'<b>(\1)</b>').
          gsub(/^(#{Mx[:pa_o]}i[1-9]#{Mx[:pa_c]})\s*\((i+|iv|v|vi+|ix|x|xi+)\)/,'\1<b>(\2)</b>')
      when /^\s*\(?(\d|[a-z])+\)/
        @txt.gsub(/^\((\d+|[a-z])+\)/,'<b>(\1)</b>').
          gsub(/^(#{Mx[:pa_o]}i[1-9]#{Mx[:pa_c]})\s*\((\d+|[a-z])+\)/,'\1<b>(\2)</b>')
      when /^\s*\d{1,3}\.\s/
        @txt.gsub(/^\s*(\d+\.)/,'<b>\1</b>')
      when /^\s*[A-Z]\.\s/
        @txt.gsub(/^\s*([A-Z]\.)/,'<b>\1</b>')
      else @txt
      end
    end
    def bold_para
      %{#{the_margin.txt_0}
  <p class="bold">
    #{@txt}
  </p>
#{the_margin.num_css}
  &nbsp;&nbsp;&nbsp;
#{the_table_close}}
    end
    def bold_header
      @txt=@txt.gsub(/[1-9]~(\S+)/,'<a name="\1"></a>').
        gsub(/[1-9]~/,'')
      %{<p class="bold">
    #{@txt}
  </p>
#{the_margin.num_css}
  &nbsp;&nbsp;&nbsp;
#{the_table_close}}
    end
    def toc_head_copy_at
      %{<p class="center">#{@txt}</p>\n}
    end
    def center
      %{<p class="center">#{@txt}</p>\n}
    end
    def bold
      %{<p class="bold">#{@txt}</p>\n}
    end
    def center_bold
      %{<p class="centerbold">#{@txt}</p>\n}
    end
  end
end
__END__
