/**********************************************************************
 * $read_attribute example -- PLI application using VPI routines
 *
 * C source to read specparam attribute values of Verilog objects.
 * Prints the names of all ports in the module and the value of the
 * specified attribute associated with each port.  If an object 
 * specific attribute is not found, the value of the general attribute
 * is returned.  If a general attribute is not found, then a default
 * of 99.9 is returned.
 *
 * Usage: $read_attribute(<cell_module_name>, <attribute_name>);
 *
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>    /* ANSI C standard library */
#include <stdio.h>     /* ANSI C standard input/output library */
#include <string.h>    /* ANSI C standard string library */
#include <malloc.h>    /* ANSI C standard memory allocation library */
#include "vpi_user.h"  /* IEEE 1364 PLI VPI routine library  */
#include "veriuser.h"  /* IEEE 1364 PLI TF routine library    
                          (using TF routines for simulation control) */

/* prototypes of routines in this PLI application */
int PLIbook_ReadAttribute_calltf(), PLIbook_ReadAttribute_compiletf();
double PLIbook_GetAttribute();
/**********************************************************************
 * VPI Registration Data
 *********************************************************************/
void PLIbook_ReadAttribute_register()
{
  s_vpi_systf_data tf_data;   /* allocate register data structure */
  tf_data.type      = vpiSysTask;
  tf_data.tfname    = "$read_attribute";
  tf_data.calltf    = PLIbook_ReadAttribute_calltf;
  tf_data.compiletf = PLIbook_ReadAttribute_compiletf;
  tf_data.sizetf    = NULL;
  vpi_register_systf(&tf_data);
}
/*********************************************************************/

/**********************************************************************
 * compiletf routine
 *********************************************************************/
int PLIbook_ReadAttribute_compiletf(char *user_data)
{
  vpiHandle systf_h, arg_itr, arg_h;
  int       tfarg_type;

  systf_h = vpi_handle(vpiSysTfCall, NULL);
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  if (arg_itr == NULL) {
    vpi_printf("ERROR: $read_attribute requires 2 arguments\n");
    tf_dofinish();
    return(0);
  }
  arg_h = vpi_scan(arg_itr);
  if (vpi_get(vpiType, arg_h) != vpiModule) {
    vpi_printf("ERROR: $read_attribute arg1 must be module instance\n");
    tf_dofinish();
    return(0);
  }
  arg_h = vpi_scan(arg_itr);
  if (vpi_get(vpiType, arg_h) != vpiConstant) {
    vpi_printf("$read_attribute arg2 must be quoted attribute name\n");
    tf_dofinish();
    return(0);
  }
  else if (vpi_get(vpiConstType, arg_h) != vpiStringConst) {
    vpi_printf("$read_attribute arg2 must be quoted attribute name\n");
    tf_dofinish();
    return(0);
  }

  if (vpi_scan(arg_itr) != NULL) {
    vpi_printf("ERROR: $realpow requires only 2 arguments\n");
    vpi_free_object(arg_itr);
    tf_dofinish();
    return(0);
  }
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_ReadAttribute_calltf(char *user_data)
{
  vpiHandle   systf_h, mod_h, arg_itr, arg2_h, port_itr, port_h;
  double      attribute_value;
  char       *attribute_name;
  s_vpi_value attribute_name_s;

  /* obtain handle to system task arguments;
     compiletf has already verified only 2 args with correct types */
  systf_h = vpi_handle(vpiSysTfCall, NULL);
  arg_itr = vpi_iterate(vpiArgument, systf_h);
  mod_h = vpi_scan(arg_itr);

  /* read base value of system function arg 2 */
  arg2_h = vpi_scan(arg_itr);
  vpi_free_object(arg_itr); /* free iterator--did not scan till null */
  attribute_name_s.format = vpiStringVal;
  vpi_get_value(arg2_h, &attribute_name_s);
  attribute_name = attribute_name_s.value.str;

  vpi_printf("\nModule %s:\n", vpi_get_str(vpiDefName, mod_h));    
  port_itr = vpi_iterate(vpiPort, mod_h);
  while ( (port_h = vpi_scan(port_itr)) != NULL) {
    attribute_value = PLIbook_GetAttribute(port_h, attribute_name, 9.9);
    vpi_printf("  Port name = %s, attribute %s for this port = %2.2f\n",
               vpi_get_str(vpiName, port_h),
               attribute_name, attribute_value);
  }
  return(0);  
}

/**********************************************************************
 * Function to return a specparam attribute value. Inputs are:
 *   vpiHandle obj_h:         handle for any object.
 *   char     *attribute:     pointer to string with name of attribute.
 *   double    default_value: value to return if attribute is not found
 *********************************************************************/
double PLIbook_GetAttribute(vpiHandle obj_h, char *attribute,
                       double default_value)
{
  vpiHandle module_h, param_h;
  char *object_name;
  char param_name[1024];  /* character string to hold attribute name */
  s_vpi_value param_val;  /* structure to receive attribute value */
  
  param_val.format = vpiRealVal;
  module_h = vpi_handle(vpiModule, obj_h); /* get parent module */
  
  /* build specparam name out of object name and attribute name */
  object_name = vpi_get_str(vpiName, obj_h);
  strcpy(param_name, attribute); 
  strcat(param_name, object_name);
  
  /* try to get a handle to the object specific attribute */ 
  param_h = vpi_handle_by_name(param_name, module_h);
  if (!vpi_chk_error(NULL)) {
    vpi_get_value(param_h, &param_val);
    if (!vpi_chk_error(NULL)) 
      return(param_val.value.real);  /* found specific attribute */
  }

  /* try to get a handle to a general attribute */ 
  strcpy(param_name, attribute); 
  param_h = vpi_handle_by_name(param_name, module_h);
  if (!vpi_chk_error(NULL)) {
    vpi_get_value(param_h, &param_val);
    if (!vpi_chk_error(NULL))
      return(param_val.value.real);  /* found general attribute */
  }

  /* failed to find object-specific or general attribute specparam */
  return(default_value); 
}

/*********************************************************************/

