from datetime import datetime

from sqlalchemy.orm import Mapped, mapped_column

from litestar import Litestar, post
from litestar.dto import dto_field
from litestar.plugins.sqlalchemy import SQLAlchemyDTO

from .my_lib import Base


class User(Base):
    # `Base` defines `id` field as:
    # id: Mapped[UUID] = mapped_column(default=uuid4, primary_key=True)
    name: Mapped[str]
    password: Mapped[str] = mapped_column(info=dto_field("private"))
    created_at: Mapped[datetime] = mapped_column(info=dto_field("read-only"))


UserDTO = SQLAlchemyDTO[User]


@post("/users", dto=UserDTO, sync_to_thread=False)
def create_user(data: User) -> User:
    # even though the client did not send the id field,
    # since it is a primary key it is autogenerated
    assert "id" in vars(data)
    # even though the client sent the password and created_at field, it is not in the data object
    assert "password" not in vars(data)
    assert "created_at" not in vars(data)
    # normally the database would set the created_at timestamp
    data.created_at = datetime.min
    return data  # the response includes the created_at field


app = Litestar(route_handlers=[create_user])

# run: /users -H "Content-Type: application/json" -d '{"name":"Litestar User","password":"xyz","created_at":"2023-04-24T00:00:00Z"}'
