/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcas_log.c
    \brief  Logging routines for LCAS
    \author Martijn Steenbakkers for the EU DataGrid.
*/

#define _XOPEN_SOURCE  500

/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <syslog.h>
#include <time.h>
#include <ctype.h>
#include "_lcas_log.h"

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
#ifndef DEBUG_LEVEL
#define DEBUG_LEVEL 0 /*!< default debugging level */
#endif /* DEBUG_LEVEL */

/******************************************************************************
                       Define module specific variables
******************************************************************************/
static FILE *  lcas_logfp=NULL; /*!< stream associated with logfile \internal */
static int     logging_usrlog=0; /*!< flag to do user logging \internal */
static int     logging_syslog=0; /*!< flag to use syslog \internal */
static int     debug_level=0; /*!< debugging level \internal */
static char *  extra_logstr = NULL; /*!< string to be included in every log statement \internal */
static int     should_close_lcas_logfp = 0; /*!< Flag to check if the log stream should be closed \internal */ 

/******************************************************************************
Function:       lcas_log_open()
Description:    Start logging
Parameters:
                path:    path of logfile
                fp:      file pointer to already opened file (or NULL)
                logtype: DO_USRLOG, DO_SYSLOG
Returns:        0 succes
                1 failure
******************************************************************************/
int
lcas_log_open(char * path, FILE * fp, unsigned short logtype )
{
    char * debug_env = NULL;
    char * logstr_env = NULL;

    if ((logtype & DO_SYSLOG) == DO_SYSLOG)
    {
//        fprintf(stderr,"lcas_log_open() error: attempt to do syslogging,");
//        fprintf(stderr," not supported yet\n");
        logging_syslog=1;
#if 0
        /* Not yet applicable, wait for LCAS to become daemon */
        openlog("EDG LCAS", LOG_PID, LOG_DAEMON);
#endif
    }
    if ((logtype & DO_USRLOG) == DO_USRLOG)
    {
        logging_usrlog=1;
        if (fp != NULL)
        {
            /* File already opened */
            lcas_logfp=fp;
            should_close_lcas_logfp = 0;
        }
        else if (path != NULL)
        {
            /* Try to append to file */
            if ((lcas_logfp = fopen(path, "a")) == NULL)
            {
                syslog(LOG_ERR, "lcas_log_open(): Cannot open logfile %s: %s\n",
                        path, strerror(errno));
                if (logging_syslog)
                {
                    syslog(LOG_ERR, "lcas_log_open(): Cannot open logfile %s\n", path);
                }
                return 1;
            }
        }
        else
        {
            syslog(LOG_ERR,
		"lcas_log_open(): Please specify either (open) file "
		"descriptor or name of logfile\n");
            return 1;
        }
    }
    /*
     * Set the debugging level:
     *    1. Try if DEBUG_LEVEL > 0
     *    2. Try if LCAS_DEBUG_LEVEL is set and if it is an integer
     *    3. set debug_level = 0;
     */
    if ( (int)(DEBUG_LEVEL) > 0 )
    {
        debug_level = (int)(DEBUG_LEVEL);
    }
    else if ( (debug_env = getenv("LCAS_DEBUG_LEVEL")) )
    {
        /* convert into integer */
        size_t j = 0;

        for (j = 0; j < strlen(debug_env); j++)
        {
            if (!isdigit((debug_env)[j]))
            {
                syslog(LOG_ERR,
			"lcas_log_open(): found non-digit in environment variable in \"LCAS_DEBUG_LEVEL\" = %s\n",
			debug_env);
                return 1;
            }
        }
        debug_level = atoi(debug_env);
        if (debug_level < 0)
        {
            syslog(LOG_ERR,"lcas_log_open(): environment variable in \"LCAS_DEBUG_LEVEL\" should be >= 0\n");
            return 1;
        }
    }
    else
    {
        debug_level = 0;
    }

    if (debug_level > 0)
    {
        lcas_log(0,"lcas_log_open(): setting debugging level to %d\n", debug_level);
    }

    /*
     * Check if there is an extra log string
     * These environment variables are checked: JOB_REPOSITORY_ID and GATEKEEPER_JM_ID
     */
    if ( (logstr_env = getenv("LCAS_LOG_STRING")) != NULL )
    {
        extra_logstr = strdup(logstr_env);
    }
    else if ( (logstr_env = getenv("JOB_REPOSITORY_ID")) != NULL )
    {
        extra_logstr = strdup(logstr_env);
    }
    else if ( (logstr_env = getenv("GATEKEEPER_JM_ID")) != NULL )
    {
        extra_logstr = strdup(logstr_env);
    }

    return 0;
}

/******************************************************************************
Function:       lcas_log()
Description:    Log information to file and or syslog
Parameters:
                prty:    syslog priority (if 0 don't syslog)
                fmt:     string format
Returns:        0 succes
                1 failure
******************************************************************************/
int
lcas_log(int prty, char * fmt, ...)
{
    va_list pvar;
    char    realbuf[MAX_LOG_BUFFER_SIZE];
    char    *buf;
    int     res;
    int     realprty=prty+LOG_ERR;

    if (realprty>LOG_DEBUG)
	realprty=LOG_DEBUG;

    va_start(pvar, fmt);
    res=vsnprintf(realbuf,MAX_LOG_BUFFER_SIZE,fmt,pvar);
    va_end(pvar);

    if (res<0)
	buf="lcas_log(): error logging";
    else    {
	buf=realbuf;
	if ( res >= MAX_LOG_BUFFER_SIZE )
	    snprintf(buf+MAX_LOG_BUFFER_SIZE-5,5,"...\n");
    }

    if (logging_usrlog)
    {
        if (lcas_logfp == NULL)
        {
            syslog(LOG_ERR,
		"lcas_log() error: cannot open file descriptor: %s\n",
		strerror(errno));
            return 1;
        }
        if (extra_logstr == NULL)
        {
            fprintf(lcas_logfp,"LCAS   %d: %s", prty, buf);
        }
        else
        {
            fprintf(lcas_logfp,"LCAS   %d: %s : %s", prty, extra_logstr, buf);
        }
        fflush(lcas_logfp);
    }
    /* Log to syslog starting at LOG_ERR */
    if (logging_syslog)
	syslog(realprty, "%s", buf);

    return 0;
}

/******************************************************************************
Function:       lcas_log_a_string()
Description:    Log a string according to the passed format to file and or syslog
Parameters:
                prty:       syslog priority (if 0 don't syslog)
                fmt:        string format
                the_string: the string
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcas_log_a_string(
        int prty,
        char * fmt,
        char * the_string
        )
    \brief log information
    
    This function logs information for LCAS and its plugins.
    Syslog() is called with the specified priority. No syslog() is done if the
    priority is 0.

    \param prty         syslog priority (if 0 don't syslog).
    \param fmt          string format
    \param the_string   the string

    \retval 0 succes.
    \retval 1 failure.
*/
int
lcas_log_a_string(int prty, char * fmt, char * the_string)
{
    return lcas_log(prty, fmt, the_string);
}

/******************************************************************************
Function:       lcas_log_debug()
Description:    Print debugging information
Parameters:
                debug_lvl: debugging level
                fmt:       string format
Returns:        0 succes
                1 failure
******************************************************************************/
int
lcas_log_debug(int debug_lvl, char * fmt, ...)
{
    va_list pvar;
    char    buf[MAX_LOG_BUFFER_SIZE];
    int     res;

    va_start(pvar, fmt);
    res=vsnprintf(buf,MAX_LOG_BUFFER_SIZE,fmt,pvar);
    va_end(pvar);
    if ( res<0 )    {
	lcas_log(0,"lcas_log_debug(): cannot log: %s\n",strerror(errno));
	return 1;
    }
    if ( res >= MAX_LOG_BUFFER_SIZE )
	snprintf(buf+MAX_LOG_BUFFER_SIZE-5,5,"...\n");
    if (debug_lvl <= debug_level)
        return lcas_log(debug_lvl, "%s", buf);
    return 1;
}

/******************************************************************************
Function:       lcas_log_a_string_debug()
Description:    Print debugging information
Parameters:
                debug_lvl:  debugging level
                fmt:        string format
                the_string: the string
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcas_log_a_string_debug(
        int debug_lvl,
        char * fmt,
        char * the_string
        )
    \brief Print debugging information

    This function prints debugging information
    provided debug_lvl <= DEBUG_LEVEL (default is 0).

    \param debug_lvl    debugging level
    \param fmt          string format
    \param the_string   the string

    \retval 0 succes.
    \retval 1 failure.
*/
int
lcas_log_a_string_debug(int debug_lvl, char * fmt, char * the_string)
{
    if (debug_lvl <= debug_level)
        return lcas_log_a_string(debug_lvl, fmt, the_string);
    return 1;
}

/******************************************************************************
Function:       lcas_log_close()
Description:    Stop logging
Parameters:
Returns:        0 succes
                1 failure
******************************************************************************/
int
lcas_log_close()
{
    if (extra_logstr != NULL)
    {
        free(extra_logstr);
        extra_logstr = NULL;
    }

    if (should_close_lcas_logfp)
    {
        fclose(lcas_logfp);
        lcas_logfp=NULL;
    }

    return 0;
}

/******************************************************************************
Function:       lcas_log_time()
Description:    Log information to file and or syslog with a timestamp
Parameters:
                prty:    syslog priority (if 0 don't syslog)
                fmt:     string format
Returns:        0 succes
                1 failure
******************************************************************************/
int
lcas_log_time(int prty, char * fmt, ...)
{
    va_list     pvar;
    char        buf[MAX_LOG_BUFFER_SIZE];
    char *      datetime = NULL;
    char *      tmpbuf = NULL;
    int         res;
    time_t      mclock;
    struct tm * tmp = NULL;
    int         rc;


    va_start(pvar, fmt);
    res=vsnprintf(buf,MAX_LOG_BUFFER_SIZE,fmt,pvar);
    va_end(pvar);
    if ( res<0 )    {
	lcas_log(0,"lcas_log_debug(): cannot log: %s\n",strerror(errno));
	return 1;
    }
    if ( res >= MAX_LOG_BUFFER_SIZE )
	snprintf(buf+MAX_LOG_BUFFER_SIZE-5,5,"...\n");

    if (extra_logstr == NULL)
    {
        time(&mclock);
        /* tmp = localtime(&mclock); */
        tmp = gmtime(&mclock);

        datetime = malloc(sizeof(char) * 20);

        res=snprintf(datetime, 20, "%04d-%02d-%02d.%02d:%02d:%02d",
               tmp->tm_year + 1900, tmp->tm_mon + 1, tmp->tm_mday,
               tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
        if ( (res >= 20) || (res < 0) )
        {
            lcas_log(0,"%s","lcas_log_time(): date conversion failed\n");
        }

        tmpbuf = (char *) malloc ((strlen(datetime) + strlen(buf) + strlen(" : ")) * sizeof(char) + 1);
        strcpy(tmpbuf, datetime);
        strcat(tmpbuf, " : ");
        strcat(tmpbuf, buf);
    }
    else
    {
        tmpbuf = (char *) malloc ((strlen(extra_logstr) + strlen(buf) + strlen(" : ")) * sizeof(char) + 1);
        strcpy(tmpbuf, extra_logstr);
        strcat(tmpbuf, " : ");
        strcat(tmpbuf, buf);
    }

    rc=lcas_log(prty,"%s", tmpbuf);

    if (datetime != NULL) free(datetime);
    if (tmpbuf != NULL) free(tmpbuf);

    return rc;
}


/******************************************************************************
Function:       lcas_get_debug_level()
Description:    Retrieve the debug_level
Parameters:
Returns:        the debug_level
******************************************************************************/
int
lcas_get_debug_level()
{
    return debug_level;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/src/lcas_log.c,v $
    $Date: 2010-06-23 08:30:41 $
    $Revision: 2.17 $
    $Author: msalle $
******************************************************************************/
