;;; This is an implementation of the Unlambda programming language.
;;; Version 1.92.1 of 1999/10/30
;;; $Id: unlambda.scm,v 1.10 1999/11/03 21:01:03 madore Exp $

;;; Copyright (C) 1999 by David A. Madore <david.madore@ens.fr>

;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License
;;; as published by the Free Software Foundation; either version
;;; 2 of the License, or (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty
;;; of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
;;; the GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

;; Data representation: f applied to g is represented as (f . g) (we
;; use a pair rather than a list because application *always* takes
;; *exactly* one argument).  Primitive procedures k, s, and so on are
;; represented as (k), (s) and so on.  Derived procedures (k1, s1, s2
;; and the like) are represented by lists whose first element is k1,
;; s1... and whose subsequent elements give internal data to the
;; derived procedure (data that was set there by whatever created the
;; procedure, to be remember when it is called).

;; I would be very grateful if somebody could explain to me how all
;; this works.  -- The author

;; Bail out with an error.
(define (error str)
  (display str)
  (newline)
; (quit) ; There is no standard quit operation in Scheme!!!
  (disappear-in-the-pit-of-hell) ; Force an error
  )

;; Parse the input file and return the representation of it.
(define (parse input-port)
  (define (gobble-comment)
    (if (not (let ((ch (read-char input-port)))
	       (or (eof-object? ch)
		   (eqv? ch #\newline))))
	(gobble-comment)))
  (case (let ((ch (read-char input-port)))
	  (if (eof-object? ch)
	      (error "Unexpected end of file")
	      ch))
    ((#\`) (let* ((op (parse input-port))
		  (arg (parse input-port)))
	     `(,op . ,arg)))
;   ((#\space #\ht #\cr #\newline) (parse input-port))
    ((#\space #\newline) (parse input-port)) ; #\ht and #\cr are not standard, shit!
    ((#\#) (gobble-comment) (parse input-port))
    ((#\k #\K) '(k)) ; (lambda (x) (lambda (y) x))
    ((#\s #\S) '(s)) ; (lambda (x) (lambda (y) (lambda (z) ((x z) (y z)))))
    ((#\i #\I) '(i)) ; identity (same as ``skk)
    ((#\v #\V) '(v)) ; return v
    ((#\c #\C) '(c)) ; call/cc
    ((#\d #\D) '(d)) ; delay (special form, force at next call)
    ((#\e #\E) '(e)) ; exit immediately
;; The p function has been replaced by the more general . function
;   ((#\p #\P) '(pr #\*)) ; print an asterisk (same as .*)
    ((#\r #\R) '(pr #\newline)) ; print newline
    ((#\.) `(pr ,(let ((ch (read-char input-port)))
		   (if (eof-object? ch)
		       (error "Unexpected end of file")
		       ch)))) ; print given char
    ((#\@) '(rd)) ; read next input char
    ((#\?) `(rc ,(let ((ch (read-char input-port)))
		   (if (eof-object? ch)
		       (error "Unexpected end of file")
		       ch)))) ; compare character under reading head
    ((#\|) '(pc)) ; call arg with dot function for current char
    (else (error "Character not understood"))))

;; Unparse (display) an object.
(define (unparse exp)
  (cond
   ((pair? (car exp))
    (write-char #\`) (unparse (car exp)) (unparse (cdr exp)))
   (else (case (car exp)
	   ((k) (write-char #\k))
	   ((k1) (write-char #\`) (write-char #\k) (unparse (cadr exp)))
	   ((s) (write-char #\s))
	   ((s1) (write-char #\`) (write-char #\s) (unparse (cadr exp)))
	   ((s2) (write-char #\`) (write-char #\`) (write-char #\s)
	    (unparse (cadr exp)) (unparse (caddr exp)))
	   ((i) (write-char #\i))
	   ((v) (write-char #\v))
	   ((c) (write-char #\c))
	   ((c1) (display "<continuation>"))
	   ((d) (write-char #\d))
	   ((d1) (write-char #\`) (write-char #\d) (unparse (cadr exp)))
	   ((e) (write-char #\e))
	   ((pr) (if (eqv? (cadr exp) #\newline) (write-char #\r)
		     (begin (write-char #\.) (write-char (cadr exp)))))
	   ((rd) (write-char #\@))
	   ((rc) (begin (write-char #\?) (write-char (cadr exp))))
	   ((pc) (write-char #\|))
	   (else (error "Internal error: unexpected type to unparse!"))))))

;; The eval function
(define (ev exp)
  (cond
   ((pair? (car exp))
    (let ((op (ev (car exp))))
      (if (eqv? (car op) 'd)
	  `(d1 ,(cdr exp))
	  (ap op (ev (cdr exp))))))
   (else exp)))

;; The exit continuation (makes the e function work).
(define (exit-cnt v)
  (error "Please start with entry-ev and not ev"))
;; The first eval function (begins by capturing a continuation so that
;; the exit function works).
(define (entry-ev exp)
  (call-with-current-continuation
   (lambda (cnt)
     (set! exit-cnt cnt)
     (ev exp))))

;; Character under the reading head (``current character'')
(define current-char #f)

;; The apply function
(define (ap exp arg)
;; Uncomment the following lines to enable debugging
; (display "Debug:")
; (display "Expression: ") (unparse exp)
; (display "; applied to: ") (unparse arg) (newline)
  (case (car exp)
    ((k) `(k1 ,arg))
    ((k1) (cadr exp))
    ((s) `(s1 ,arg))
    ((s1) `(s2 ,(cadr exp) ,arg))
    ((s2) (ev `((,(cadr exp) . ,arg) . (,(caddr exp) . ,arg))))
    ((i) arg)
    ((v) '(v))
    ((c) (call-with-current-continuation
	  (lambda (cont) (ev `(,arg . (c1 ,cont))))))
    ((c1) ((cadr exp) arg))
;   ((d) 'impossible)
    ((d1) (ev `(,(cadr exp) . ,arg)))
    ((e) (exit-cnt arg))
    ((pr) (display (cadr exp)) arg)
    ((rd) (set! current-char (read-char))
     (ev `(,arg . ,(if (eof-object? current-char) '(v) '(i)))))
    ((rc) (ev `(,arg . ,(if (eqv? current-char (cadr exp))
			    '(i)
			    '(v)))))
    ((pc) (ev `(,arg . ,(if (and current-char
				 (not (eof-object? current-char)))
			    `(pr ,current-char)
			    '(v)))))
    (else (error "Internal error: unexpected type to apply!"))))

;; ``Main'' function
(define (main . junk)
  (entry-ev (parse (current-input-port))))
