#GPL
#GPL  libwhisker copyright 2000,2001,2002 by rfp.labs
#GPL
#GPL  This program is free software; you can redistribute it and/or
#GPL  modify it under the terms of the GNU General Public License
#GPL  as published by the Free Software Foundation; either version 2
#GPL  of the License, or (at your option) any later version.
#GPL
#GPL  This program is distributed in the hope that it will be useful,
#GPL  but WITHOUT ANY WARRANTY; without even the implied warranty of
#GPL  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GPL  GNU General Public License for more details.
#GPL

=pod

=head1 ++ Sub package: utils

The utils subpackage contains various utility functions which serve
different purposes.

=cut

########################################################################

=pod

=head1 - Function: LW::utils_recperm
  
Params: $uri, $depth, \@dir_parts, \@valid, \&func, \%track, \%arrays, \&cfunc
Return: nothing

This is a special function which is used to recursively-permutate through
a given directory listing.  This is really only used by whisker, in order
to traverse down directories, testing them as it goes.  See whisker 2.0 for
exact usage examples.

=cut

# '/', 0, \@dir.split, \@valid, \&func, \%track, \%arrays, \&cfunc
sub utils_recperm {
 my ($p, $pp, $pn, $r, $fr, $dr, $ar, $cr)=(shift,shift,@_);
 $p=~s#/+#/#g; if($pp >= @$pn) { push @$r, $p if &$cr($$dr{$p});
 } else { my $c=$$pn[$pp];
  if($c!~/^\@/){ utils_recperm($p.$c.'/',$pp+1,@_) if(&$fr($p.$c.'/'));
  } else {	foreach $d (@{$c=~tr/\@//d&&$$ar{$c}}){
			if(&$fr($p.$d.'/')){
                  utils_recperm($p.$d.'/',$pp+1,@_);}}}}}


#################################################################

=pod

=head1 - Function: LW::utils_array_shuffle
  
Params: @array
Return: nothing

This function will randomize the order of the elements in the given array.

=cut

sub utils_array_shuffle { # fisher yates shuffle....w00p!
        my $array=shift; my $i;
        for ($i = @$array; --$i;){
                my $j = int rand ($i+1);
                next if $i==$j;
                @$array[$i,$j]=@$array[$j,$i];
}} # end array_shuffle, from Perl Cookbook (rock!)


#################################################################

=pod

=head1 - Function: LW::utils_randstr
  
Params: [ $size, $chars ]
Return: $random_string

This function generates a random string between 10 and 20 characters
long, or of $size if specified.  If $chars is specified, then the
random function picks characters from the supplied string.  For example,
to have a random string of 10 characters, composed of only the characters
'abcdef', then you would run:

LW::utils_randstr(10,'abcdef');

The default character string is alphanumeric.

=cut

sub utils_randstr {
        my $str;
        my $drift=shift||((rand() * 10) % 10)+10; 

	# 'a'..'z' doesn't seem to work on string assignment :(
	my $CHARS = shift || 'abcdefghijklmnopqrstuvwxyz' .
			'ABCDEFGHIJKLMNOPQRSTUVWXYZ' .
			'0123456789';

	my $L = length($CHARS);
        for(1..$drift){
	        $str .= substr($CHARS,((rand() * $L) % $L),1);
	}
        return $str;}

#################################################################

=pod

=head1 - Function: LW::utils_get_dir
  
Params: $uri
Return: $uri_directory

Will take a URI and return the directory base of it, i.e. /rfp/page.php 
will return /rfp/.

=cut

sub utils_get_dir {
        my ($w,$URL)=(0,shift);

	return undef if(!defined $URL);

	substr($URL,$w,length($URL)-$w)='' if( ($w=index($URL,'?')) >= 0);
	substr($URL,$w,length($URL)-$w)='' if( ($w=index($URL,'#')) >= 0);

	if( ($w=rindex($URL,'/')) >= 0){
		$URL = substr($URL,0,$w+1);
	} else {
		if(substr($URL,-1,1) ne '/'){
			$URL.='/';}
	}
        return $URL; 
}


#################################################################

=pod

=head1 - Function: LW::utils_port_open
  
Params: $host, $port
Return: $result

Quick function to attempt to make a connection to the given host and
port.  If a connection was successfully made, function will return true
(1).  Otherwise it returns false (0).

Note: this uses standard TCP connections, thus is not recommended for use
in port-scanning type applications.  Extremely slow.

=cut

sub utils_port_open {  # this should be platform-safe
        my ($target,$port)=@_;

	return 0 if(!defined $target || !defined $port);

        if(!(socket(S,PF_INET,SOCK_STREAM,0))){ return 0;}
        if(connect(S,sockaddr_in($port,inet_aton($target)))){
                close(S); return 1;
        } else { return 0;}}


#################################################################

=pod

=head1 - Function: LW::utils_split_uri
  
Params: $uri_string [, \%hin_request]
Return: @uri_parts

Return an array of the following values, in order:  uri, protocol, host,
port, params, frag, user, password.  Values not defined are given an undef
value.  If a %hin_request hash is passed in, then utils_split_uri() will
also set the appropriate values in the hash.

Note:  utils_split_uri() will only set the %hin_request if the protocol
is HTTP or HTTPS!

=cut

sub utils_split_uri {
	my ($uri,$work,$w)=(shift,'',0);
	my ($hr)=shift;
	my @res=(undef,'http',undef,80,undef,undef,undef,undef);

	return undef if(!defined $uri);

	# handle mailto's (people miswrite them as mailto:email@host)
	if(index($uri,'mailto:',0) == 0){
		$res[1]='mailto';
		($res[0]=$uri)=~s/^mailto:[\/]{0,2}//;
		return @res; }

	# handle absolute urls
	if(index($uri,'://',0) > 0 ){ # fastpath check
	 if($uri=~m#^([a-zA-Z]+)://([^/]*)(.*)$#){
		$res[1]=lc($1); 	# protocol
		$res[2]=$2;	    	# host
		$res[0]=$3;		# uri

		if($res[1] eq 'https') 	{ $res[3]=443; }

		while(($w=rindex($res[2],'@')) >=0){
			# SPEEDUP
			# $work=substr($res[2],0,$w,'');
			$work=substr($res[2],0,$w);
			substr($res[2],0,$w)='';
			$res[2]=~tr/@//d;
			if(($w=index($work,':',0)) >=0){
				$res[6]=substr($work,0,$w);
				$res[7]=substr($work,$w+1,length($work)-$2);
			} else {
				$res[6]=$work; }
		}
		
		# check for port in host
		if(($w=index($res[2],':',0)) >=0){
			# SPEEDUP
			# $res[3]=substr($res[2],$w,length($res[2])-$w,'');
			($res[2],$res[3])=split(':',$res[2],2);
			$res[3]=~tr/0-9//cd;
		}

		$res[3]||=80;
		
		$res[0]||='/'; # in case they left off URI or end slash

	 } else { $res[0]=$uri; }  # note that if the URL isn't formed
	} else { $res[0]=$uri; }   # perfectly, we make it all a URI  :/

	# remove fragments
	if(($w=index($res[0],'#',0)) >=0){
		# SPEEDUP
		# $res[5]=substr($res[0],$w+1,length($res[0])-$w,'');
		# $res[0]=~tr/#//d;
		($res[0],$res[5])=split('#',$res[0],2);
	}

	# remove parameters
	if(($w=index($res[0],'?',0)) >=0){
		# SPEEDUP
		# $res[4]=substr($res[0],$w+1,length($res[0])-$w,'');
		# $res[0]=~tr/?//d; 
		($res[0],$res[4])=split(/\?/,$res[0],2);
	}

	if(defined $hr && ref($hr) && ( $res[1] eq 'http' ||
			$res[1] eq 'https') ){
		if(defined $res[0]){
			$$hr{whisker}->{uri}=$res[0]; }
		if($res[1] eq 'https'){
			$$hr{whisker}->{ssl}=1; }
		if(defined $res[2]){
			$$hr{whisker}->{host}=$res[2]; }
		$$hr{whisker}->{port}=$res[3];
		if(defined $res[4]){
			$$hr{whisker}->{uri_param}=$res[4]; }
		if(defined $res[6]){
			$$hr{whisker}->{uri_user}=$res[6]; }
		if(defined $res[7]){
			$$hr{whisker}->{uri_password}=$res[7]; }
	}
		
	return @res;
}

#################################################################
=pod

=head1 - Function: LW::utils_lowercase_headers
  
Params: \%hash
Return: nothing

Will lowercase all the header names (but not values) of the given hash.

=cut

sub utils_lowercase_headers {
	my $href=shift;

	return if(!(defined $href && ref($href)));

	while( my ($key,$val)=each %$href ){
		delete $$href{$key};
		$$href{lc($key)}=$val;
	}
}

#################################################################
=pod

=head1 - Function: LW::utils_lowercase_hashkeys
  
Params: \%hash
Return: nothing

Alias for LW::utils_lowercase_headers.

=cut

sub utils_lowercase_hashkeys {
	goto &LW::utils_lowercase_headers;
}


#################################################################
=pod

=head1 - Function: LW::utils_find_lowercase_key
  
Params: \%hash, $key
Return: $value, undef on error or not exist

Searches the given hash for the $key (regardless of case), and
returns the value.

=cut

sub utils_find_lowercase_key {
	my ($href,$key)=(shift,lc(shift));

	return undef if(!(defined $href && ref($href)));
	return undef if(!defined $key);	

	while( my ($k,$v)=each %$href ){
		return $v if(lc($k) eq $key);
	}
	return undef;
}

#################################################################

=pod

=head1 - Function: LW::utils_join_uri
  
Params: @vals
Return: $url

Takes the @vals array output from utils_split_uri, and returns a single 
scalar/string with them joined again, in the form of:
protocol://host:port/uri?params#frag

=cut

sub utils_join_uri {
	my @V=@_;
	my $URL="$V[1]://$V[2]";
	$URL .= ":$V[3]" if defined $V[3];
	$URL.=$V[0];
	$URL .= "?$V[4]" if defined $V[4];
	$URL .= "#$V[5]" if defined $V[5];
	return $URL;
}

#################################################################

=pod

=head1 - Function: LW::utils_getline
  
Params: \$data [, $resetpos ]
Return: $line (undef if no more data)

Fetches the next \n terminated line from the given data.  Use
the optional $resetpos to reset the internal position pointer.
Does *NOT* return trialing \n.

=cut

{ $POS=0;
sub utils_getline {
	my ($dr, $rp)=@_;

	return undef if(!(defined $dr && ref($dr)));
	$POS=$rp if(defined $rp);

	my $where=index($$dr,"\n",$POS);
	return undef if($where==-1);

	my $str=substr($$dr,$POS,$where-$POS);
	$POS=$where+1;

	return $str;
}}

#################################################################

=pod

=head1 - Function: LW::utils_getline_crlf
  
Params: \$data [, $resetpos ]
Return: $line (undef if no more data)

Fetches the next \r\n terminated line from the given data.  Use
the optional $resetpos to reset the internal position pointer.
Does *NOT* return trialing \r\n.

=cut

{ $POS=0;
sub utils_getline_crlf {
	my ($dr, $rp)=@_;

	return undef if(!(defined $dr && ref($dr)));
	$POS=$rp if(defined $rp);

	my $tpos=$POS;
	while(1){
		my $where=index($$dr,"\n",$tpos);
		return undef if($where==-1);

		if(substr($$dr,$where-1,1) eq "\r"){
			my $str=substr($$dr,$POS,$where-$POS-1);
			$POS=$where+1;
			return $str;
		} else {
			$tpos=$where+1;
		}
	}
}}

#################################################################

=pod

=head1 - Function: LW::utils_absolute_uri
  
Params: $uri, $base_uri [, $normalize_flag ]
Return: $absolute_$url

Double checks that the given $uri is in absolute form (that is,
"http://host/file"), and if not (it's in the form "/file"), then
it will append the given $base_uri to make it absolute.  This
provides a compatibility similar to that found in the URI
subpackage.

If $normalize_flag is set to 1, then the output will be passed
through utils_normalize_uri before being returned.

=cut

sub utils_absolute_uri {
        my ($uri, $buri, $norm)=@_;
        return undef if(!defined $uri || !defined $buri);
	return $uri if($uri=~m#^[a-zA-Z]+://#);

	if(substr($uri,0,1) eq '/'){
		my @p=utils_split_uri($buri);
		$buri="$p[1]://$p[2]";
		$buri.=":$p[3]" if($p[3]!=80);
		$buri.='/';
	} else {
		$buri.='/' if($buri=~m#^[a-z]+://[^/]+$#i);
		$buri=~s#/[^/]*$#/#;
	}
	return utils_normalize_uri("$buri$uri") 
		if(defined $norm && $norm > 0);
        return $buri.$uri;
}

#################################################################

=pod

=head1 - Function: LW::utils_normalize_uri
  
Params: $uri [, $fix_windows_slashes ]
Return: $normalized_uri

Takes the given $uri and does any /./ and /../ dereferencing in
order to come up with the correct absolute URL.  If the $fix_
windows_slashes parameter is set to 1, all \ (back slashes) will
be converted to / (forward slashes).

=cut

sub utils_normalize_uri {
	my ($host,$uri, $win)=('',@_);

	$uri=~tr#\\#/# if(defined $win && $win>0);
	if($uri=~s#^([a-z]+://[^/]+)##i){
		$host=$1; }
	return "$host/" if($uri eq '' || $uri eq '/');

	# fast path check
	$uri=~s#/\.{0,1}/#/#g; # quickie
	return "$host$uri" if(index($uri,'/.')==-1);

	my $f='';
	$f='/' if($uri=~m#/\.{1,2}$#);

	my @final=();
	my @dirs=split('/',$uri);

	foreach (@dirs){
		next if($_ eq '.');
		next if($_ eq '');
		if($_ eq '..'){
			pop(@final);
		} else {
			push(@final,$_);
	}	} 
	$f='' if(scalar @final == 0);
	return "$host/".join('/',@final).$f;
}

#################################################################

=pod

=head1 - Function: LW::utils_save_page
  
Params: $file, \%response
Return: 0 on success, 1 on error

Saves the data portion of the given whisker %response hash to the
indicated file.  Can technically save the data portion of a
%request hash too.  A file is not written if there is no data.

Note: LW does not do any special file checking; files are opened
in overwrite mode.

=cut

sub utils_save_page {
	my ($file, $hr)=@_;
	return 1 if(!ref($hr) || ref($file));
	return 0 if(!defined $$hr{'whisker'} || 
		!defined $$hr{'whisker'}->{'data'});
	open(OUT,">$file") || return 1;
	print OUT $$hr{'whisker'}->{'data'};
	close(OUT);
	return 0;
}

#################################################################

=pod

=head1 - Function: LW::utils_getopts
  
Params: $opt_str, \%opt_results
Return: 0 on success, 1 on error

This function is a general implementation of GetOpts::Std.  It will
parse @ARGV, looking for the options specified in $opt_str, and will
put the results in %opt_results.  Behavior/parameter values are
similar to GetOpts::Std's getopts().

Note: this function does *not* support long options (--option),
option grouping (-opq), or options with immediate values (-ovalue).
If an option is indicated as having a value, it will take the next
argument regardless.

=cut

sub utils_getopts {
        my ($str,$ref)=@_;
        my (%O,$l);
        my @left;

        return 1 if($str=~tr/-:a-zA-Z0-9//c);

        while($str=~m/([a-z0-9]:{0,1})/ig){
                $l=$1;
                if($l=~tr/://d){        $O{$l}=1;
                } else {                $O{$l}=0; }
        }

        while($l=shift(@ARGV)){
                push(@left,$l)&&next if(substr($l,0,1) ne '-');
                push(@left,$l)&&next if($l eq '-');
                substr($l,0,1)='';
                if(length($l)!=1){
                        %$ref=();
                        return 1; }
                if($O{$l}==1){
                        my $x=shift(@ARGV);
                        $$ref{$l}=$x;
                } else { $$ref{$l}=1; }
        }

        @ARGV=@left;
        return 0;
}

#################################################################

=pod

=head1 - Function: LW::utils_unidecode_uri
  
Params: $unicode_string
Return: $decoded_string

This function attempts to decode a unicode (UTF-8) string by
converting it into a single-byte-character string.  Overlong 
characters are converted to their standard characters in place; 
non-overlong (aka multi-byte) characters are substituted with the 
0xff; invalid encoding characters are left as-is.

Note: this function is useful for dealing with the various unicode
exploits/vulnerabilities found in web servers; it is *not* good for
doing actual UTF-8 parsing, since characters over a single byte are
basically dropped/replaced with a placeholder.

=cut

sub utils_unidecode_uri {
        my $str = $_[0];
        return $str if($str!~tr/!-~//c); # fastpath
        my ($lead,$count,$idx);
        my $out='';
        my $len = length($str);
        my ($ptr,$no,$nu)=(0,0,0);

        while($ptr < $len){
                my $c=substr($str,$ptr,1);
                if( ord($c) >= 0xc0 && ord($c) <= 0xfd){
                        $count=0;
                        $c=ord($c)<<1;
                        while( ($c & 0x80) == 0x80){
                                $c<<=1;
                                last if($count++ ==4);
                        }
                        $c = ($c & 0xff);
                        for( $idx=1; $idx<$count; $idx++){
                                my $o=ord(substr($str,$ptr+$idx,1));
                                $no=1 if($o != 0x80);
                                $nu=1 if($o <0x80 || $o >0xbf);
                        }
                        my $o=ord(substr($str,$ptr+$idx,1));
                        $nu=1 if( $o < 0x80 || $o > 0xbf);
                        if($nu){
                                $out.=substr($str,$ptr++,1);
                        } else {
                                if($no){
                                        $out.="\xff"; # generic replacement char
                                } else {
                                        my $prior=ord(substr($str,$ptr+$count-1,1))<<6;
                                        $out.= pack("C", (ord(substr($str,$ptr+$count,1) )&0x7f)+$prior);
                                }
                                $ptr += $count+1;
                        }
                        $no=$nu=0;
                } else {
                        $out.=$c;
                        $ptr++;
                }
        }
        return $out;
}

#################################################################
