/* ***************************************************************************
 *
 * Pico Technology USB Device Driver
 *
 *//**
 * \file      PicoPortability_Linux.cpp 
 * \brief     Linux implementation of Pico Technology Cross-Platform Portability routines
/
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: PicoPortability_Linux.cpp,v 1.11 2008/05/07 10:37:53 pei Exp $
 *
 *************************************************************************** */

#include "PicoPortability.h"

/// Silently ignore this file if we're not on Linux
#ifdef PICO_OS_LINUX

#include <errno.h>
#include <pthread.h>
#include <sys/signal.h>
#include <sys/time.h>

// Define our structures
typedef enum {
	PICOHANDLE_SEMAPHORE,
	PICOHANDLE_MUTEX,
	PICOHANDLE_THREAD,
	PICOHANDLE_EVENT
} PICO_HANDLE_TYPE;

struct PICO_WINTHREAD_T
{
	LPTHREAD_START_ROUTINE lpStartAddress;
	LPTHREAD_VOID_START_ROUTINE lpVoidStartAddress;
	void * lpParameter;
};

struct PICO_HANDLE
{
	pthread_mutex_t Mutex;
	PICO_HANDLE_TYPE handleType;
	pthread_cond_t * Cond;
	long Count;
	long MaxCount;
	bool ManualReset;
	pthread_t * Thread;
	struct PICO_WINTHREAD_T winThread;
};

struct PICO_CRITICAL_SECTION
{
	pthread_mutex_t * Mutex;
};


int fopen_s(FILE ** a, const char * b, const char * c)
{
FILE * fp = fopen(b,c);
*a = fp;
return (fp>0)?0:-1;
}

long GetLastError(void)
{
	return errno;
}

void OutputDebugStr (char * str)
{
	//printf(str);
}

/// Get value of a millisecond counter (arbitrary starttime)
/// <returns>Timer value in milliseconds</returns>
DWORD GetTickCount(void)
{
	// Variables to hold the gettimeofday result
	struct timeval tv;

	// Get the time
	assert (gettimeofday(&tv, NULL) == 0);

	// Convert to ms and return
	return (DWORD)((1000 * tv.tv_sec) + (tv.tv_usec / 1000));

}

/// Get frequency in counts per second of the "Performance Counter"
/// <param name="lpFrequency">Pointer to LARGE_INTEGER to write the result in</param>
/// <returns>True if the call succeeds, false if an error occurs</returns>
BOOL QueryPerformanceFrequency(LARGE_INTEGER * lpFrequency)
{
	// The Linux gettimeofday always returns microseconds
	lpFrequency->QuadPart = 1000000;
	return true;
}

/// Get value of the "Performance Counter"
/// <param name="*pPerformanceCount">Pointer to LARGE_INTEGER to write the result in</param>
/// <returns>True if the call succeeds, false if an error occurs</returns>
BOOL QueryPerformanceCounter(LARGE_INTEGER * lpPerformanceCount)
{
	// Variables to hold the gettimeofday result
	struct timeval tv;
	int retval;

	// Get the time
	retval = gettimeofday(&tv, NULL);

	// Convert to microseconds
	lpPerformanceCount->QuadPart = 1000000 * (int64_t)tv.tv_sec + tv.tv_usec;

	return (retval == 0);
}


/// Create and init an event.
/// <param name="lpEventAttributes">For Win32 compatibility. This parameter is ignored.</param>
/// <param name="bManualReset">If false, the event is reset by the first thread woken up by it.</param>
/// <param name="bInitialState">Is the event initially signalled?</param>
/// <param name="lpName">For Win32 compatibility. This parameter is ignored and an unnamed event is always created.</param>
/// <returns>Pointer to newly-created event structure, or NULL if an error occurs</returns>
PICO_HANDLE * CreateEvent(void * lpEventAttributes, BOOL bManualReset,
		BOOL bInitialState, void * lpName)
	{
	// All possible parameter values are valid so no validation.


	// Create struct
	PICO_HANDLE * evt = new PICO_HANDLE;
	if (!evt)
		return NULL;

	evt->handleType = PICOHANDLE_EVENT;

	// Create condition
	evt->Cond = new pthread_cond_t;
	if (!evt->Cond) {
		delete evt;
		return NULL;
	}

	evt->Count = bInitialState?1:0;
	evt->ManualReset = bManualReset;

	if (pthread_mutex_init(&evt->Mutex, NULL)) {
		CloseHandle(evt);
		return NULL;
	}

	if (pthread_cond_init(evt->Cond, NULL)) {
		CloseHandle(evt);
		return NULL;
	}

	// Success!
	return evt;

		}


/// Create and init a semaphore.
/// <param name="lpSemaphoreAttributes">For Win32 compatibility. This parameter is ignored.</param>
/// <param name="lInitialCount">Initial semaphore count (0 = initially locked)</param>
/// <param name="lMaximumCount">Maximum semaphore count: incrementing beyond this will cause an error</param>
/// <param name="lpName">For Win32 compatibility. This parameter is ignored and an unnamed semaphore is always created.</param>
/// <returns>Pointer to newly-created semaphore structure, or NULL if an error occurs</returns>
PICO_HANDLE * CreateSemaphore(void * lpSemaphoreAttributes, LONG lInitialCount,
		LONG lMaximumCount, void * lpName)
	{
	// Validate params
	if (lInitialCount < 0 || lMaximumCount < 1 || lInitialCount > lMaximumCount)
		return NULL;


	// Create struct
	PICO_HANDLE * sem = new PICO_HANDLE;
	if (!sem)
		return NULL;

	sem->handleType = PICOHANDLE_SEMAPHORE;

	// Create condition
	sem->Cond = new pthread_cond_t;
	if (!sem->Cond) {
		delete sem;
		return NULL;
	}

	sem->Count = lInitialCount;
	sem->MaxCount = lMaximumCount;

	if (pthread_mutex_init(&sem->Mutex, NULL)) {
		CloseHandle(sem);
		return NULL;
	}

	if (pthread_cond_init(sem->Cond, NULL)) {
		CloseHandle(sem);
		return NULL;
	}

	// Success!
	return sem;

		}

HANDLE	CreateMutex( void * lpMutexAttributes, BOOL bInitialOwner, void * lpName)
{

	// Create struct
	PICO_HANDLE * mutex = new PICO_HANDLE;
	if (!mutex)
		return NULL;

	mutex->handleType = PICOHANDLE_MUTEX;

	if (pthread_mutex_init(&mutex->Mutex, NULL)) {
		CloseHandle(mutex);
		return NULL;
	}

	if (bInitialOwner) { 
		WaitForSingleObject(mutex, INFINITE);
	}

	// Success!
	return mutex;

}

BOOL	ReleaseMutex(HANDLE hMutex)
{
	if (!hMutex)
		return false;
	if (pthread_mutex_unlock(&hMutex->Mutex))
		return false;
	// We are done
	return true;
}

/// Wait up to a specified timeout for the thread to terminate.
/// <param name="hHandle">The thread to wait for</param>
/// <param name="dwMilliseconds">Time in ms to wait, between 0 and INFINITE</param>
/// <returns>WAIT_OBJECT_0 if the thread exits successfully,
/// WAIT_TIMEOUT if the timeout elapsed eithout the thread terminating,
/// WAIT_FAILED if an error occurs</returns>		
/// Wait up to a specified timeout for the semaphore to become available (value > 0).
/// If semaphore is/becomes available, decrement its value by 1.
/// <param name="hHandle">The PICO_SEMAPHORE object to wait for</param>
/// <param name="dwMilliseconds">Time in ms to wait, between 0 and INFINITE</param>
/// <returns>WAIT_OBJECT_0 if the semaphore has been claimed successfully,
/// WAIT_TIMEOUT if the timeout elapsed eithout the semaphore becoming available,
/// WAIT_FAILED if an error occurs</returns>		
DWORD WaitForSingleObject(PICO_HANDLE * hHandle, DWORD dwMilliseconds)
{

	int retval = 0;

	if (!hHandle)
		return WAIT_FAILED;
	if (dwMilliseconds == INFINITE) // No timeout
	{
	
		if (hHandle->handleType == PICOHANDLE_MUTEX) {
			if (pthread_mutex_lock(&hHandle->Mutex))
				return WAIT_FAILED;
			// We are done
			return WAIT_OBJECT_0;
		}
		
		if (hHandle->handleType == PICOHANDLE_SEMAPHORE || hHandle->handleType == PICOHANDLE_EVENT) {
			if (pthread_mutex_lock(&hHandle->Mutex))
				return WAIT_FAILED;
			while (hHandle->Count < 1)
			{
				if (pthread_cond_wait(hHandle->Cond, &hHandle->Mutex)) {
					pthread_mutex_unlock(&hHandle->Mutex);
					return WAIT_FAILED;
				}
			}
			if (!(hHandle->handleType == PICOHANDLE_EVENT && hHandle->ManualReset))
				hHandle->Count--;
			pthread_mutex_unlock(&hHandle->Mutex);
			return WAIT_OBJECT_0;
		}	
		
		if (hHandle->handleType == PICOHANDLE_THREAD) {
			pthread_join(*(hHandle->Thread), NULL);
			return WAIT_OBJECT_0;
		}
		// Unrecognised handle type
		return WAIT_FAILED;
		
	} else {
		
		
		struct timespec ts;
		struct timeval tv;
		gettimeofday(&tv, NULL);
		ts.tv_sec = tv.tv_sec + dwMilliseconds / 1000;
		ts.tv_nsec = tv.tv_usec*1000 + (dwMilliseconds % 1000)*1000000;
		if (ts.tv_nsec >= 1000000000) {
			ts.tv_nsec -= 1000000000;
			ts.tv_sec++;
		}
	
		if (hHandle->handleType == PICOHANDLE_MUTEX) {
			if ((retval = pthread_mutex_timedlock(&hHandle->Mutex, &ts))) {
				if (retval == ETIMEDOUT)
					return WAIT_TIMEOUT;
				else
					return WAIT_FAILED;
			}
			return WAIT_OBJECT_0;
		}
		
		if (hHandle->handleType == PICOHANDLE_SEMAPHORE || hHandle->handleType == PICOHANDLE_EVENT) {
			if (pthread_mutex_lock(&hHandle->Mutex))
				return WAIT_FAILED;
			while (hHandle->Count < 1)
			{
				if ((retval = pthread_cond_timedwait(hHandle->Cond, &hHandle->Mutex, &ts))) {
					pthread_mutex_unlock(&hHandle->Mutex);
					if (retval == ETIMEDOUT)
						return WAIT_TIMEOUT;
					else
						return WAIT_FAILED;
				}
			}
			if (!(hHandle->handleType == PICOHANDLE_EVENT && hHandle->ManualReset))
				hHandle->Count--;
			pthread_mutex_unlock(&hHandle->Mutex);
			return WAIT_OBJECT_0;
		}
		
		if (hHandle->handleType == PICOHANDLE_THREAD) {
			// TODO: This is an infinite wait
			pthread_join(*(hHandle->Thread), NULL);
			return WAIT_OBJECT_0;
		}
		
		// Unrecognised handle type
		return WAIT_FAILED;
	}
}

/// Release semaphore (increment value) and get its value before release.
/// <param name="hSemaphore">The PICO_SEMAPHORE object to release</param>
/// <param name="lReleaseCount">The count by which to increment the semaphore</param>
/// <param name="lpPreviousCount">Pointer to a location to store the count of the semaphore before the release operation</param>
/// <returns>True if the call succeeds, false if an error occurs</returns>
BOOL ReleaseSemaphore(PICO_HANDLE * hSemaphore, LONG lReleaseCount,
		long * lpPreviousCount)
{
	if (!hSemaphore  || !(hSemaphore->Cond))
		return false;

	if (lpPreviousCount)
		*lpPreviousCount = hSemaphore->Count;

	if (pthread_mutex_lock(&hSemaphore->Mutex))
		return false;

	if (hSemaphore->Count + lReleaseCount > hSemaphore->MaxCount) {
		pthread_mutex_unlock(&hSemaphore->Mutex);
		return false;
	}

	hSemaphore->Count += lReleaseCount;

	if (pthread_cond_signal(hSemaphore->Cond)) {
		pthread_mutex_unlock(&hSemaphore->Mutex);
		return false;
	}

	pthread_mutex_unlock(&hSemaphore->Mutex);
	return true;

}

/// Signal an event.
/// <param name="hEvent">The PICO_EVENT object to signal</param>
/// <returns>True if the call succeeds, false if an error occurs</returns>
BOOL SetEvent(PICO_HANDLE * hEvent)
{
	if (!hEvent  || !(hEvent->Cond))
		return false;

	if (pthread_mutex_lock(&hEvent->Mutex))
		return false;

	hEvent->Count = 1;

	if (pthread_cond_broadcast(hEvent->Cond)) {
		pthread_mutex_unlock(&hEvent->Mutex);
		return false;
	}

	pthread_mutex_unlock(&hEvent->Mutex);
	return true;

}

/// De-signal an event.
/// <param name="hEvent">The PICO_EVENT object to designal</param>
/// <returns>True if the call succeeds, false if an error occurs</returns>
BOOL ResetEvent(PICO_HANDLE * hEvent)
{
	if (!hEvent  || !(hEvent->Cond))
		return false;

	if (pthread_mutex_lock(&hEvent->Mutex))
		return false;

	hEvent->Count = 0;

	if (pthread_cond_broadcast(hEvent->Cond)) {
		pthread_mutex_unlock(&hEvent->Mutex);
		return false;
	}

	pthread_mutex_unlock(&hEvent->Mutex);
	return true;

}



/// Destroy a semaphore/thread/mutex/event and free the storage allocated.
/// <param name="hObject">The semaphore or thread object to destroy</param>
/// <returns>True if the call succeeds, false if an error occurs</returns>
BOOL CloseHandle(PICO_HANDLE * hObject)
{
	if (hObject != NULL) {
		switch (hObject->handleType) {
			case PICOHANDLE_MUTEX:
				pthread_mutex_destroy(&hObject->Mutex);
				break;
			case PICOHANDLE_THREAD:
				pthread_detach(*(hObject->Thread));
				pthread_kill(*(hObject->Thread), SIGALRM);
				delete hObject->Thread;
				break;
			case PICOHANDLE_SEMAPHORE:
			case PICOHANDLE_EVENT:
				pthread_mutex_destroy(&hObject->Mutex);
				if (hObject->Cond != NULL) {
					pthread_cond_destroy(hObject->Cond);
					delete (hObject->Cond);
				}
				break;
			default:
				break;
		}
		delete hObject;
		return true;
	} else {
		return false;
	}
}


/// Set the priority of a thread. Not implemented.
/// <param name="hThread">Thread to have its priority changed</param>
/// <param name="nPriority">New priority to set</param>	
/// <returns>True if priority changed, else false</returns>	
BOOL 	SetThreadPriority(HANDLE hThread, int nPriority)
{
	if (!hThread)
		return false;
	if (hThread->handleType != PICOHANDLE_THREAD)
		return false;
	
	// TODO: implement.
	
	return true;
}



/// Suspends a thread. Not implemented.
/// <param name="hThread">Thread to be suspended</param>
/// <returns>True if success, else false</returns>	
BOOL 	SuspendThread(HANDLE hThread)
{
	if (!hThread)
		return false;
	if (hThread->handleType != PICOHANDLE_THREAD)
		return false;
	
	// TODO: implement.
	
	return false;
}


/// Resumes a thread. Not implemented.
/// <param name="hThread">Thread to be resumed</param>
/// <returns>True if success, else false</returns>	
BOOL 	ResumeThread(HANDLE hThread)
{
	if (!hThread)
		return false;
	if (hThread->handleType != PICOHANDLE_THREAD)
		return false;
	
	// TODO: implement.
	
	return false;
}


// Function to turn integer return value of windows thread into void *
void *VoidThreadHelper(void * args)
{
	(((PICO_WINTHREAD_T *)args)->lpVoidStartAddress)(((PICO_WINTHREAD_T *)args)->lpParameter);
	return NULL;
}


/// Start a thread from the specified function.
/// <param name="start_address">Pointer to function returning void</param>
/// <param name="stack_size">Included for Win32 API compatibility, ignored</param>
/// <param name="arglist">Pointer to thread start function arguments</param>	
/// <returns>Pointer to newly-created thread, or NULL if creation fails</returns>	
PICO_HANDLE * _beginthread(void ( *start_address )( void * ), unsigned int stack_size,
		void *arglist)
{
	if (!start_address)
		return NULL;

	// Create struct
	PICO_HANDLE * thread = new PICO_HANDLE;
	if (!thread)
		return NULL;

	thread->handleType = PICOHANDLE_THREAD;

	// Create thread
	thread->Thread = new pthread_t;
	if (!thread->Thread) {
		delete thread;
		return NULL;
	}

	// Save the entry point details
	thread->winThread.lpVoidStartAddress = start_address;
	thread->winThread.lpParameter = arglist;

	if (pthread_create(thread->Thread, 0, VoidThreadHelper, (void *)&(thread->winThread))) {
		delete (thread->Thread);
		delete thread;
		return NULL;
	}

	return thread;		
}

/// Start a thread from the specified function.
/// <param name="security">Included for Win32 API compatibility, ignored</param>
/// <param name="stack_size">Included for Win32 API compatibility, ignored</param>
/// <param name="start_address">Pointer to function returning unsigned int</param>
/// <param name="arglist">Pointer to thread start function arguments</param>	
/// <param name="initflag">Included for Win32 API compatibility, ignored</param>
/// <param name="thrdaddr">Included for Win32 API compatibility, ignored</param>
/// <returns>Pointer to newly-created thread, or NULL if creation fails</returns>	
PICO_HANDLE * _beginthreadex(void *security, unsigned int stack_size,
		unsigned int ( *start_address )( void * ), void *arglist, unsigned int initflag,
		unsigned int *thrdaddr)
		{
	// Just call CreateThread with the function and arguments
	return CreateThread(NULL, 0, start_address, arglist, 0, NULL);
		}

// Function to turn integer return value of windows thread into void *
void *ThreadHelper(void * args)
{
	return (void *) ((((PICO_WINTHREAD_T *)args)->lpStartAddress)(((PICO_WINTHREAD_T *)args)->lpParameter));
}

/// Start a thread from the specified function.
/// <param name="lpThreadAttributes">Included for Win32 API compatibility, ignored</param>
/// <param name="dwStackSize">Included for Win32 API compatibility, ignored</param>
/// <param name="lpStartAddress">Pointer to function returning unsigned int</param>
/// <param name="lpParameter">Pointer to thread start function arguments</param>	
/// <param name="dwCreationFlags">Included for Win32 API compatibility, ignored</param>
/// <param name="lpThreadId">Included for Win32 API compatibility, ignored</param>
/// <returns>Pointer to newly-created thread, or NULL if creation fails</returns>	
PICO_HANDLE * CreateThread(void * lpThreadAttributes, SIZE_T dwStackSize,
		LPTHREAD_START_ROUTINE lpStartAddress, void * lpParameter,
		DWORD dwCreationFlags, DWORD * lpThreadId)
		{
	if (!lpStartAddress)
		return NULL;

	// Create struct
	PICO_HANDLE * thread = new PICO_HANDLE;
	if (!thread)
		return NULL;

	thread->handleType = PICOHANDLE_THREAD;

	// Create thread
	thread->Thread = new pthread_t;
	if (!thread->Thread) {
		delete thread;
		return NULL;
	}

	// Save the entry point details
	thread->winThread.lpStartAddress = lpStartAddress;
	thread->winThread.lpParameter = lpParameter;

	if (pthread_create(thread->Thread, 0, ThreadHelper, (void *)&(thread->winThread))) {
		delete (thread->Thread);
		delete thread;
		return NULL;
	}

	return thread;

		}

/// Get the return value of the specified thread if it has exited.
/// <param name="hThread">The thread to get the return value from</param>
/// <param name="lpExitCode">Pointer to location to write the exit code</param>
/// <returns>True if the call succeeds, false if an error occurs</returns>
BOOL GetExitCodeThread(PICO_HANDLE * hThread, DWORD * lpExitCode)
{
	*lpExitCode = STILL_ACTIVE;
	return true;
}

/// Initialize a PICO_CRITICAL_SECTION object
/// <param name="lpCriticalSection">The PICO_CRITICAL_SECTION object to initialize</param>
/// <param name="dwSpinCount">Included for Win32 compatibility, ignored</param>
/// <returns>True if the call succeeds, false if an error occurs</returns>
BOOL InitializeCriticalSectionAndSpinCount(
		CRITICAL_SECTION * lpCriticalSection, DWORD dwSpinCount)
{
	// We ignore the spincount and just init the mutex:
	return InitializeCriticalSection(lpCriticalSection);
}

/// Initialize a PICO_CRITICAL_SECTION object
/// <param name="lpCriticalSection">The PICO_CRITICAL_SECTION object to initialize</param>
/// <returns>True if the call succeeds, false if an error occurs</returns>
BOOL InitializeCriticalSection(CRITICAL_SECTION * lpCriticalSection)
{
	// Create a new object
	if (!lpCriticalSection)
		return false;
	*lpCriticalSection = new PICO_CRITICAL_SECTION;
	if (!*lpCriticalSection)
		return false;
	(*lpCriticalSection)->Mutex = new pthread_mutex_t;
	if (!(*lpCriticalSection)->Mutex)
		return false;

	// Init the mutex
	return (pthread_mutex_init((*lpCriticalSection)->Mutex, NULL) == 0);

}

/// Delete a pre-created PICO_CRITICAL_SECTION and deallocates storage.
/// <param name="lpCriticalSection">The PICO_CRITICAL_SECTION object to delete</param>
void DeleteCriticalSection(CRITICAL_SECTION * lpCriticalSection)
{
	if (lpCriticalSection != NULL) {
		if (*lpCriticalSection != NULL) {
			if ((*lpCriticalSection)->Mutex != NULL) {
				pthread_mutex_destroy((*lpCriticalSection)->Mutex);
				delete (*lpCriticalSection)->Mutex;
			}
			delete *lpCriticalSection;
		}
		lpCriticalSection = NULL;
	}
}

/// Block until the critical section is available, then lock it.
/// <param name="lpCriticalSection">The PICO_CRITICAL_SECTION object to lock</param>
void EnterCriticalSection(CRITICAL_SECTION * lpCriticalSection)
{
	// Validate params
	assert(lpCriticalSection != NULL);
	assert(*lpCriticalSection != NULL);
	assert((*lpCriticalSection)->Mutex != NULL);

	// Lock the mutex. This must succeed.
	assert (pthread_mutex_lock((*lpCriticalSection)->Mutex) == 0);

}

/// Unlock the critical section for other users.
/// <param name="lpCriticalSection">The PICO_CRITICAL_SECTION object to unlock</param>
void LeaveCriticalSection(CRITICAL_SECTION * lpCriticalSection)
{
	// Validate params
	assert(lpCriticalSection != NULL);
	assert(*lpCriticalSection != NULL);
	assert((*lpCriticalSection)->Mutex != NULL);

	// Lock the mutex. This must succeed.
	assert (pthread_mutex_unlock((*lpCriticalSection)->Mutex) == 0);

}

#endif //PICO_OS_LINUX
