/*  tiffep - TI File Format Engine and Proxy
 *  Copyright (C) 2000-2001  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "hfiles.h"
#include "tiffep.h"


/*
  Allocate a TiffepVarInfo structure and fill it with null values.
  - TiffepVarInfo [out]: the allocated structure, NULL otherwise.
*/
TIEXPORT
TiffepVarInfo* tiffep_alloc_vi_struct(void)
{ 
  TiffepVarInfo *varinfo = (TiffepVarInfo *)g_malloc0(sizeof(TiffepVarInfo));
  return varinfo;
}

/*
  Free all allocated memory
  - vi [in]: the informations stored in an allocated structure
  for being freed
  - int [out]: 0 if success, an error code otherwise.
*/
TIEXPORT
int tiffep_free_vi_struct(TiffepVarInfo **varinfo)
{
  TiffepVarInfo *vi = *varinfo;

  if(vi == NULL) 
    {
      g_warning(_("tiffep_free_vi_struct: NULL pointer.\n"));
      return -1;
    }

  // free the fields first
  g_free(VI_PATH(vi));
  g_free(VI_CALCNAME(vi));
  g_free(VI_REALNAME(vi));
  g_free(VI_CONTENT(vi));
  
  // free the structure itself
  g_free(vi);
  *varinfo = NULL;

  return 0;
}

/*
  Duplicate a TiffepVarInfo structure
  - vi [in]: the TiffepVarInfo structure to duplicate
  - TiffepVarInfo [out]: the newly allocated structure if success, 
  NULL otherwise.
*/
TIEXPORT
TiffepVarInfo* tiffep_dup_vi_struct(TiffepVarInfo *vi)
{
  TiffepVarInfo *src = vi;
  TiffepVarInfo *dst = NULL;

  dst = tiffep_alloc_vi_struct();

  VI_PATH(dst)      = g_strdup(VI_PATH(src));
  VI_CALCNAME(dst)  = g_strdup(VI_CALCNAME(src));
  VI_REALNAME(dst)  = g_strdup(VI_REALNAME(src));
  VI_TYPE(dst)      = VI_TYPE(src);
  VI_ATTRIBUTE(dst) = VI_ATTRIBUTE(src);
  VI_SIZE(dst)      = VI_SIZE(src);
  VI_CONTENT(dst)   = g_memdup(VI_CONTENT(src), VI_SIZE(src));
  VI_CALCTYPE(dst)  = VI_CALCTYPE(src);
  VI_OFFSET(dst)    = VI_OFFSET(src);

  return dst;
}

/*
  Clear a TiffepVarInfo structure
  - vi [in]: the TiffepVarInfo structure to duplicate
  - TiffepVarInfo [out]: the newly allocated structure if success, 
  NULL otherwise.
*/
TIEXPORT
TiffepVarInfo* tiffep_clear_vi_struct(TiffepVarInfo *vi)
{
  memset((void *)vi, 0, sizeof(TiffepVarInfo));
  return vi;
}

/*
  DISPLAY2 a TiffepVarInfo structure
  - vi [in]: the TiffepVarInfo structure to print
  - int [out]: none
*/
TIEXPORT
int tiffep_print_vi_struct(TiffepVarInfo *vi)
{
  if(vi == NULL)
    {
      g_warning("tiffep_print_vi_struct: NULL pointer.\n");
      return -1;
    }
  DISPLAY2("var_path   = <%s>\n", vi->var_path);
  DISPLAY2("var_name_b = <%s>\n", vi->var_name_b);
  DISPLAY2("var_name_t = <%s>\n", vi->var_name_t);
  DISPLAY2("var_type   = %i <%s>\n", vi->var_type,
	  tiffep_var_type_to_string(vi->var_type));
  DISPLAY2("var_atrb   = %i <%s>\n", vi->var_atrb,
	  tiffep_attribute_to_string(vi->var_atrb));
  DISPLAY2("var_size   = %i\n", vi->var_size);
  DISPLAY2("data       = %p\n", vi->data);
  DISPLAY2("calc_type  = %i <%s>\n", vi->calc_type,
	  tiffep_calc_type_to_string(vi->calc_type));
  DISPLAY2("offset     = 0x%04x\n", vi->offset);

  return 0;
}

/*-----------------------------------*/

/* 
   Add a variable (vi) into the file (fi). This function manages some
   special things such as updating folder list, sorting variables, ...
   - fi [in/out]: a pointer on the TiffepFileInfo structure
   - vi [in]: a pointer on the TiffepVarInfo structure
   - int [out]: 0 if success, an error code otherwise.
*/
TIEXPORT
int tiffep_add_vi_struct(TiffepFileInfo *fi, TiffepVarInfo *vi)
{
  int err = 0;
  gchar *folder_name;
  GList *list;
  gint b = 0;

  FILEINFO_NUMVARS(fi)++;
  folder_name = VI_PATH(vi);
  list = FILEINFO_DIRLIST(fi);
  printf("phase1: scanning folder list...\n");
  while(list != NULL)
    {
      TiffepFolderInfo *di = DIRLIST_DATA(list);
      if(!strcmp(DIRINFO_NAME(di), VARINFO_PATH(vi)))
	b = !0;
      
      list = g_list_next(list);
    }
  if(!b)
    {
      TiffepFolderInfo *di = tiffep_alloc_di_struct();
      DIRINFO_NAME(di) = g_strdup(VARINFO_PATH(vi));
      DIRINFO_OFFSET(di) = VARINFO_OFFSET(vi);
      printf("adding <%s> to folder list\n", DIRINFO_NAME(di));
      FILEINFO_DIRLIST(fi) = g_list_append(FILEINFO_DIRLIST(fi), di);
    }

  printf("phase2: adding <%s> to var list\n", VI_REALNAME(vi));
  FILEINFO_VARLIST(fi) = g_list_append(FILEINFO_VARLIST(fi), vi);
  
  return err;
}

/* 
   This function computes offset stored in the TiffepVarInfo & TiffepFolderInfo structures.
   This function must be called before writing a group file for instance.
   - fi [in/out]: a pointer on the TiffepFileInfo structure
   - int [out]: 0 if success, an error code otherwise.
*/
TIEXPORT
int tiffep_update_vi_offset(TiffepFileInfo *fi)
{
  GList *list1, *list2;
  gint offset;

  /* Compute offset */
  list1 = FILEINFO_VARLIST(fi);
  offset = 0x3c + 16*(g_list_length(FILEINFO_VARLIST(fi)) + 
		      g_list_length(FILEINFO_DIRLIST(fi))) + 6;
  VI_OFFSET(VARLIST_DATA(list1)) = offset;
  while(list1 != NULL)
    {
      TiffepVarInfo *vi = VARLIST_DATA(list1);
      offset += VI_SIZE(vi) + 6;
      if(g_list_next(list1) != NULL)
	VI_OFFSET(VARLIST_DATA(g_list_next(list1))) = offset;
      
      list1 = VARLIST_NEXT(list1);
    }

  /* Process folder list for counting number of vars in a given folder */
  list2 = FILEINFO_DIRLIST(fi);
  while(list2 != NULL)
    {
      TiffepFolderInfo *di = DIRLIST_DATA(list2);
      DI_SIZE(di) = 0;
      DI_TYPE(di) = TYPE_DIRECTORY;
      
      list1 = FILEINFO_VARLIST(fi);
      while(list1 != NULL)
	{
	  TiffepVarInfo *vi = VARLIST_DATA(list1);
	  if(!strcmp(VI_PATH(vi), DI_NAME(di)))
	    DI_SIZE(di)++;
	  printf("VI_TYPE(vi)=%i\n", VI_TYPE(vi));

	  list1 = VARLIST_NEXT(list1);
	}

      list2 = DIRLIST_NEXT(list2);
    }
  /* 
     Theoritically, we should also process offset of the folder list but
     the fields are not used... Maybe, I will have to fix this, isn't it ?!
  */
  return 0;
}
