/*  tiffep - Ti File Format Engine & Proxy
 *  Copyright (C) 2000  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
  This unit contains functions for managing exchange of messages
  between the TiFFEP server and the libtiffep.
  This unit is built as a overlayer of the unix/win32_sock.c unit.
*/

#include <stdio.h>

#include "hfiles.h"
#include "tiffep.h"


/*****************/
/* Misc commands */
/*****************/

/*
	Remove pending command
*/
TIEXPORT
int cmd_remove(TiffepShm s)
{
	return msg_remove(s);
}

/*
	Check if a command is available
*/
TIEXPORT
int cmd_is_available(TiffepShm s)
{
	return msg_is_available(s);
}

/*****************/
/* Send commands */
/*****************/

/*
  Send an ACK message.
  - s [in]: a shm
*/
TIEXPORT
int cmd_send_ack(TiffepShm s)
{
  TiffepMsg msg; 

  msg_clear(&msg);
  msg.cmd = CMD_ACK;
  DISPLAY2("%3i: Send ACK...\n", CMD_ACK);
  msg_send(s, msg);

  return 0;
}

/*
  Send an NAK message.
  - ns [in]: a shm
*/
TIEXPORT
int cmd_send_nak(TiffepShm s)
{
  TiffepMsg msg;

  msg_clear(&msg);
  msg.cmd = CMD_NAK;
  DISPLAY2("%3i: Send NAK...\n", CMD_NAK);
  msg_send(s, msg);

  return 0;
}

/*
  Send an OK message.
  - ns [in]: a shm
*/
TIEXPORT
int cmd_send_ok(TiffepShm s)
{
  TiffepMsg msg;

  msg_clear(&msg);
  msg.cmd = CMD_OK;
  DISPLAY2("%3i: Send OK...\n", CMD_OK);
  msg_send(s, msg);

  return 0;
}

/*
  Send an NOK message.
  - ns [in]: a shm
*/
TIEXPORT
int cmd_send_nok(TiffepShm s)
{
  TiffepMsg msg;

  msg.cmd = CMD_NOK;
  DISPLAY2("%3i: Send NOK...\n", CMD_NOK);
  msg_send(s, msg);

  return 0;
}


/*
  Send an STRING message with the associated string.
  Send several STRING messages if necessary.
  - ns [in]: a shm
  - str [in]: the string to send
*/
TIEXPORT
int cmd_send_string(TiffepShm s, gchar *str)
{
  TiffepMsg msg;

  msg_clear(&msg);
  msg.len = strlen(str)+1;
  msg.data = 0;
  msg.cmd = CMD_STRING;
  strcpy(msg.str, str);
  DISPLAY2("%3i: Send STRING...\n", CMD_STRING);
  msg_send(s, msg);
  TRY(cmd_recv_ack(s));

  return 0;
}

/*
  Send an EXEC_PLUGIN message.
  - ns [in]: a shm
*/
TIEXPORT
int cmd_send_exec_plugin(TiffepShm s, gchar *path, gchar* filename)
{
  TiffepMsg msg;

  msg_clear(&msg);
  msg.cmd = CMD_EXEC_PLUGIN; //send exec_plugin cmd
  msg.type = TP_LINKPROG;
  msg.len = 0;
  msg.data = 0;
  DISPLAY2("%3i: Send EXEC_PLUGIN...\n", CMD_EXEC_PLUGIN);
  msg_send(s, msg);
  TRY(cmd_recv_ack(s));

  TRY(cmd_send_string(s, path));
  TRY(cmd_send_string(s, filename));


  return 0;
}


/*
  Send an ECHO message.
  - s [in]: a shm
*/
TIEXPORT
int cmd_send_echo(TiffepShm s)
{
  TiffepMsg msg; 

  msg_clear(&msg);
  msg.cmd = CMD_ECHO;
  DISPLAY2("%3i: Send ECHO...\n", CMD_ECHO);
  msg_send(s, msg);

  return 0;
}

/*
  Send a TI_SEND_BACKUP message
*/
int cmd_send_sbackup     (TiffepShm s, char *backup_file)
{
  TiffepMsg msg; 

  msg_clear(&msg);
  msg.cmd = CMD_TI_SEND_BACKUP;
  DISPLAY2("%3i: Send TI_SEND_BACKUP...\n", CMD_TI_SEND_BACKUP);
  msg_send(s, msg);

  cmd_recv_ack(s);

  cmd_send_string(s, backup_file);

  return 0;
}


/*
  Send a TI_RECV_BACKUP message
*/
int cmd_send_rbackup     (TiffepShm s, char *backup_file)
{
  TiffepMsg msg; 

  msg_clear(&msg);
  msg.cmd = CMD_TI_RECV_BACKUP;
  DISPLAY2("%3i: Send TI_RECV_BACKUP...\n", CMD_TI_RECV_BACKUP);
  msg_send(s, msg);
  cmd_recv_ack(s);

  cmd_send_string(s, backup_file);

  return 0;
}


/********************/
/* Receive commands */
/********************/

/*
  Wait an ACK message.
  - ns [in]: a shm
  - int [out]: 0 if success, an error code otherwise
*/
TIEXPORT
int cmd_recv_ack(TiffepShm s)
{
  TiffepMsg msg;

  DISPLAY2("%3i: Receiving ACK...\n", CMD_ACK);  
  msg_recv(s, &msg);
  if(!is_msg_ack(msg))
    return ERR_CMD_INVALID;

  return 0;
}

/*
  Wait an NAK message.
  - ns [in]: a shm
  - int [out]: 0 if success, an error code otherwise
*/
TIEXPORT
int cmd_recv_nak(TiffepShm s)
{
  TiffepMsg msg;

  DISPLAY2("%3i: Receiving NAK...\n", CMD_NAK);  
  msg_recv(s, &msg);
  if(!is_msg_nak(msg))
    return ERR_CMD_INVALID;

  return 0;
}

/*
  Wait an OK message.
  - ns [in]: a shm
  - int [out]: 0 if success, an error code otherwise
*/
TIEXPORT
int cmd_recv_ok(TiffepShm s)
{
  TiffepMsg msg;

  DISPLAY2("%3i: Receiving OK...\n", CMD_OK);  
  msg_recv(s, &msg);
  if(!is_msg_ok(msg))
    return ERR_CMD_INVALID;

  return 0;
}

/*
  Wait an NOK message.
  - ns [in]: a shm
  - int [out]: 0 if success, an error code otherwise
*/
TIEXPORT
int cmd_recv_nok(TiffepShm s)
{
  TiffepMsg msg;

  DISPLAY2("%3i: Receiving NOK...\n", CMD_NOK);  
  msg_recv(s, &msg);
  msg.cmd = CMD_NONE;
  if(!is_msg_nok(msg))
    return ERR_CMD_INVALID;

  return 0;
}

/*
  Decode a string from socket message. Receive any subsequent messages
  until string is completed.
  - ns [in]: a shm
  - str [out]: the received string. Must be freed when no longer used !
  - int [out]: returns 0 if successful, an error code otherwise
*/
TIEXPORT
int cmd_recv_string(TiffepShm s, char **str)
{
  TiffepMsg msg;
  //int i;

  DISPLAY2("%3i: Receiving STRING...\n", CMD_STRING);
  msg_recv(s, &msg);
  *str = g_strdup(msg.str);

  TRY(cmd_send_ack(s));

  return 0;
}

