/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeUtils.h
     This file contains some utility functions
*/
#ifndef  __TERRALIB_INTERNAL_UTILS_H
#define  __TERRALIB_INTERNAL_UTILS_H

#ifdef WIN32
#pragma warning ( disable: 4786 )
#endif

#include <string>
#include <vector>
#include <cstdio>
#include <time.h>


#include "TeCoord2D.h"
#include "TeProgress.h"

using namespace std;

/** @defgroup FileFunctions Functions to manipulate file 
 *  @{
 */
	//! Get the extension part (right to the dot) of a string
	string TeGetExtension ( const char* value );

	//! Get the name part (left to the dot) of a string, path included
	string TeGetName ( const char* value );

	//! Get the base name part of a string, strip path information
	string TeGetBaseName ( const char* value );

	//! Get the path part of a file name
	string TeGetPath(const char* value);

	//! This function writes a string to a file.
	/*
		\params fileName the file name to write to, including the directory.
		\params text     the string with the text to write to.
		\params mode     the mode used to write to the file (a | w | r).
	*/
	void TeWriteToFile(const string& fileName, const string& text, const string& mode);

/** @} */ 

/** @defgroup C2StringFunctions Functions convert numbers to strings
 *  @{
 */
	//! Transforms an integer to string
	string Te2String ( const int value );

	//! Transforms an unsigned integer to string
	string Te2String ( const unsigned int value );

	//! Transforms a long to string
	string Te2String ( const long value );

	//! Transforms an unsigned long to string
	string Te2String ( const unsigned long value );

	//! Transforms a double to string
	string Te2String ( const double value, int precision=0 );
/** @} */ 

/** @defgroup StringFunctions Functions to manipulate strings
 *  @{
 */
	//! Converts a string to upper case
	void TeConvertToUpperCase ( const string& , char* );

	//! Converts a string to upper case
	string TeConvertToUpperCase (const string &name);

	//! Converts a string to lower case
	string TeConvertToLowerCase (const string &name);

	//! Removes special characteres from a string
	string TeRemoveSpecialChars ( string& str);

	//! Removes left and right blank, \t, \n  characters of a string
	void  TeTrim(string &str);

	//! Splits a string, given a separator, in a vector of parts
	int TeSplitString(const string& input, const string& delimiter, vector<string>& results);

	//! Compares two strings
	/*
		\params caseS flag indicating if it is a case sensitive comparation
	*/
	bool TeStringCompare(const string& str1, const string& str2, bool caseS=false);
/** @} */ 

/** @defgroup MathFunctions Mathematical functions
 *  @{
 */
	//! Rounds a double to int
	inline int TeRound(double val)
	{	
		if (val>=0)
			return (int)(val+.5);
		else
			return (int)(val-.5);
	}

	//! Rounds a double value to a given number of decimal digits
	double TeRoundD(double val, int precision=8);

	//! Compares two doubles
	bool TeCompareDouble(double a, double b, int precision);

	//! Adjust a number to a given precision 
	/*
		\param val the number to be adjusted
		\param precision the number of decimals places to be used
		\param reduce flag to indicate that the number should be reduced instead to increased
		\return the adjusted number
	*/
	double TeAdjustToPrecision(double val, int precision, bool reduce=false);

	//! Rounds a double raster element index to an integer
	/*
		Raster elements have area, so raster element in upper-left position has
		index from [-0.5,+0.5) in i and j dimension.
	*/
	inline int TeRoundRasterIndex(double val)
	{	
		int ind = (int) val;
		if (val < (ind-0.5))
			ind -= 1;
		else if (val >= (ind+0.5))
			ind += 1;
		return ind;
	}

	//! Check column name and change if necessary
	/*
		Raster elements have area, so raster element in upper-left position has
		index from [-0.5,+0.5) in i and j dimension.
	*/
	string TeCheckName(const string& name, bool& changed, string& invalidChar);
/** @} */ 

#endif


