/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeTheme.h
	This file contains definitions about a theme in TerraLib
*/
#ifndef  __TERRALIB_INTERNAL_THEME_H
#define  __TERRALIB_INTERNAL_THEME_H

#include "TeViewNode.h"
#include "TeLayer.h"
#include "TeLegendEntry.h"
#include "TeDataTypes.h"
#include "TeRasterTransform.h"


//! This structure is used to store the grouping parameters used in a theme
struct TeGrouping
{

	//! Returns the attribute used to group the objects of the theme
	TeAttributeRep	groupAttribute_;
	
	//! Returns the attribute used to normalize a grouping 
	string	groupNormAttribute_;
	
	//! Returns the grouping mode 
	TeGroupingMode	groupMode_;			
	
	//! Returns the number of groups
	int	groupNumSlices_;	
	
	//! Returns the numeric precision used to group objects
	int	groupPrecision_;

	//! Returns the deviation parameter used to group objects
	double	groupStdDev_;

	//! Aggregate function
	string	groupFunction_; 

	//! Show missing data
	bool	groupNullAttr_; 

	//! Minimum value used to calculate a equal step grouping
	double groupMinVal_;

	//! Maximum value used to calculate a equal step grouping 
	double groupMaxVal_;

	//! Constructor
	TeGrouping(TeAttributeRep att=TeAttributeRep(), const string& normAttr="", 
				TeGroupingMode gMode=TeEqualSteps, int numSlice=0, int gPrecision=6,  
				double	gStDev=1.0, const string& func=""): 
			groupAttribute_(att),
			groupNormAttribute_(normAttr),
			groupMode_(gMode),
			groupNumSlices_(numSlice),
			groupPrecision_(gPrecision),
			groupStdDev_(gStDev),
			groupFunction_(func),
			groupNullAttr_(false),
			groupMinVal_(TeMAXFLOAT),
			groupMaxVal_(TeMINFLOAT)
			{}

	//! Copy constructor
	TeGrouping(const TeGrouping& other)
	{	
		groupAttribute_		= other.groupAttribute_;
		groupNormAttribute_	= other.groupNormAttribute_;
		groupMode_			= other.groupMode_;
		groupNumSlices_		= other.groupNumSlices_;
		groupPrecision_		= other.groupPrecision_;
		groupStdDev_		= other.groupStdDev_;
		groupFunction_		= other.groupFunction_;
		groupNullAttr_		= other.groupNullAttr_;
		groupMinVal_		= other.groupMinVal_;
		groupMaxVal_		= other.groupMaxVal_;
	}
			
	//! Destructor
	~TeGrouping() {}
	
	TeGrouping& operator= (const TeGrouping& other)
	{	
		if ( this != &other )
		{		
			groupAttribute_		= other.groupAttribute_;
			groupNormAttribute_	= other.groupNormAttribute_;
			groupMode_			= other.groupMode_;
			groupNumSlices_		= other.groupNumSlices_;
			groupPrecision_		= other.groupPrecision_;
			groupStdDev_		= other.groupStdDev_;
			groupFunction_		= other.groupFunction_;
			groupNullAttr_		= other.groupNullAttr_;
			groupNullAttr_		= other.groupNullAttr_;
			groupMinVal_		= other.groupMinVal_;
			groupMaxVal_		= other.groupMaxVal_;
		}
		return *this;
	}
};

using namespace std;

//! TerraLib definition of a Theme
/*!
\par A Theme is a collection of objects selected from a TeLayer spatial and non spatial 
criteria. A theme points to the visual presentation parameters of its objects.

\par A Theme store the parameters about how to group its objects according com its attributes. 

\par A theme may have many associated graphics, which includes histogram, pie charts,
scatterplots, etc. These graphics are visible only when the theme is visible.

\par A theme has a threshold, that defines the minimum and maximum scales 
at which the Theme is drawn.

  \sa TeView TeTable
*/
class TeTheme: public TeViewNode
{
public:

	//! Constructor
    TeTheme( const string& name="", TeLayer* layer=0, TeViewNode* parent = 0, int view=0, int id=0)
		: TeViewNode(name, parent, view, id, TeTHEME),
		layer_(layer),
		generateAttributeRest_(""),
		generateTemporalRest_(""),
		hasSpatialRes_(false),
		geomRest_(0),
		minScale_(0.0),
		maxScale_(0.0),
		visibleRep_(0),
		enableVisibility_(1),
		grouping_(0),
		rasterVisual_(0)
		{	
			//default legends
			TeVisual visp(TePOLYGONS);
			TeVisual visl(TeLINES);
			TeVisual vispt(TePOINTS);
			TeVisual vist(TeTEXT);
			TeColor	color;
			color.init(100, 220, 220);
			visp.color(color);
			visl.color(color);
			vispt.color(color);
			color.init(100, 100, 100);
			visp.contourColor(color);
			visl.contourColor(color);
			vispt.contourColor(color);
			outOfCollectionLegend_.setVisual(visp, TePOLYGONS);		
			outOfCollectionLegend_.setVisual(visl, TeLINES);		
			outOfCollectionLegend_.setVisual(vispt, TePOINTS);		

			color.init(220, 100, 220);
			visp.color(color);
			visl.color(color);
			vispt.color(color);
			color.init(100, 100, 100);
			visp.contourColor(color);
			visl.contourColor(color);
			vispt.contourColor(color);
			withoutDataConnectionLegend_.setVisual(visp, TePOLYGONS);		
			withoutDataConnectionLegend_.setVisual(visl, TeLINES);		
			withoutDataConnectionLegend_.setVisual(vispt, TePOINTS);		

			color.init(220, 0, 0);
			visp.color(color);
			color.init(0, 220, 220);
			visl.color(color);
			color.init(220, 90, 180);
			vispt.color(color);
			color.init(100, 100, 100);
			visp.contourColor(color);
			visl.contourColor(color);
			vispt.contourColor(color);
			defaultLegend_.setVisual(visp, TePOLYGONS);		
			defaultLegend_.setVisual(visl, TeLINES);		
			defaultLegend_.setVisual(vispt, TePOINTS);		
			color.init(0, 0, 0);
			vist.color(color);
			defaultLegend_.setVisual(vist, TeTEXT);		

				
			color.init(80, 240, 100);
			visp.color(color);
			color.init(100, 100, 100);
			visp.contourColor(color);
			pointingLegend_.setVisual(visp, TePOLYGONS);		
			
			color.init(210, 210, 0);
			visp.color(color);
			color.init(100, 100, 100);
			visp.contourColor(color);		
			queryLegend_.setVisual(visp, TePOLYGONS);		
									
			color.init(255, 255, 0);
			visp.color(color);
			color.init(100, 100, 100);
			visp.contourColor(color);		
			queryAndPointingLegend_.setVisual(visp, TePOLYGONS);	

			//layer id
			if(layer)
			{
				layerId_ = layer->id();
			}
			else
				layerId_ = -1;
		}

	//! Copy constructor
	TeTheme (TeTheme& other) : TeViewNode(other.name(), other.parent(), other.view(), other.id(), TeTHEME) 
	{
		layerId_ = other.layerId();
		layer_ = other.layer();      //the same layer pointer
		generateAttributeRest_ = other.attributeRest();
		generateTemporalRest_ = other.temporalRest();
		generateSpatialRest_ = other.spatialRest(); 
		spatialRelation_ = other.spatialRelation();
		hasSpatialRes_ =  other.hasSpatialRest();
		boxRest_ = other.boxRestriction();
		geomRest_ = other.geomRestriction(); 
		geomRepRest_ = other.geomRepRestriction();	
		minScale_ = other.minScale();
		maxScale_ = other.maxScale();
		collectionTable_ = other.collectionTable();
		collectionAuxTable_ = other.collectionAuxTable();
		visibleRep_ = other.visibleRep();
		enableVisibility_ = other.visibility();

		if(grouping_)
			delete grouping_;
		grouping_ = 0;
		if(other.grouping())
		{
			grouping_ = new TeGrouping();
			(*grouping_) = (*other.grouping());
		}

		legend_ = other.legend();
		outOfCollectionLegend_ = other.outOfCollectionLegend();
		withoutDataConnectionLegend_ = other.withoutDataConnectionLegend();	
		defaultLegend_ = other.defaultLegend ();
		pointingLegend_ = other.pointingLegend();
		queryLegend_ = other.queryLegend();
		queryAndPointingLegend_ = other.queryAndPointingLegend(); 
		
		attTableVector_ = other.attrTables();
		sqlFrom_ =  other.sqlFrom();		
		sqlJoin_ = other.sqlGridJoin();
		sqlGridFrom_ = other.sqlGridFrom();
		sqlGridJoin_ = other.sqlGridJoin();
		aliasVector_ = other.aliasVector();
		sqlAttList_ = other.sqlAttList();
		sqlNumAttList_ = other.sqlNumAttList();		
	
		if(rasterVisual_)
			delete rasterVisual_;
		rasterVisual_ = 0;
		if(other.rasterVisual())
		{
			rasterVisual_ = new TeRasterTransform();
			(*rasterVisual_) = (*other.rasterVisual());
		}
	}

	//! Destructor
	~TeTheme () 
	{ 
		if (rasterVisual_)
			delete rasterVisual_;
		if (grouping_)
			delete grouping_;
	}

	//! Assignment operator
	TeTheme& operator= (TeTheme& other); 
	
	//! Returns the id of the source layer
	int	 layerId () 
	{	return layerId_; }

	//! Sets the id of the source layer
	void layerId(int i)
	{	layerId_ = i; }

	//! Set the source layer
	/*! \param layer a pointer to a layer from which theme get its objects
	*/
	void layer (TeLayer* layer)
	{
		layer_ = layer;
		if (layer)
			layerId_ = layer->id();
	}

	//! Returns a pointer to the layer from which the theme get its objects
	TeLayer* layer () { return layer_; }

	//! Returns TRUE if a theme has lower priority than another
	bool operator< (const TeTheme& r) const
	{	return priority_ < r.priority_; }

	//! Returns the minimum scale in which the theme is visible
	double minScale () { return minScale_; }
	//! Set the minimum scale in which the theme is visible
	void minScale (double s) { minScale_ = s; }

	//! Returns the maximum scale in which the theme is visible
	double maxScale () { return maxScale_; }
	//! Set the maximum scale in which the theme is visible
	void maxScale (double s) { maxScale_ = s; }

	//! Returns a vector of theme legends
	TeLegendEntryVector& legend () { return legend_; }

	//! Returns the attribute restriction (where clause) used to generate the theme  
	string attributeRest() { return generateAttributeRest_; }
	//! Set the attribute restriction (where clause) used to generate the theme
	void attributeRest(const string& s) { generateAttributeRest_ = s; }
	//! Returns if there is an attribute restriction  
	bool hasAttrRest () { return (!generateAttributeRest_.empty());}

	//! Returns the temporal restriction used to generate the theme
	string temporalRest() { return generateTemporalRest_; }
	//! Set the temporal restriction used to generate the theme
	void temporalRest(const string& t) { generateTemporalRest_ = t; };
	//! Returns if there is a temporal restriction 
	bool hasTemporalRest () { return (!generateTemporalRest_.empty());}

	//! Returns the spatial restriction used to generate the theme
	string spatialRest() { return generateSpatialRest_; }
	//! Set the spatial restriction used to generate the theme
	void spatialRest(const string& s) { generateSpatialRest_ = s; };
	
	//! Returns the spatial relation 
	TeSpatialRelation	spatialRelation() {return spatialRelation_;}
	//! Sets the spatial relation
	void spatialRelation(TeSpatialRelation s) {spatialRelation_=s;}
	
	//! Returns if there is a spatial restriction 
	bool hasSpatialRest () { return hasSpatialRes_;}
	//! Sets if there is a spatial restriction 
	void hasSpatialRest (bool a) {hasSpatialRes_ = a;}
	
	//! Returns the box associated with the spatial restriction
	TeBox boxRestriction() {return boxRest_;}
	//! Sets the box associated with the spatial restriction
	void boxRestriction (TeBox& b) {boxRest_ = b;}

	//! Returns a pointer to the geometry associated with the spatial restriction
	TeGeometry* geomRestriction() { return geomRest_;}
	//! Sets a pointer to the geometry associated with the spatial restriction
	void geomRestriction(TeGeometry* g) {geomRest_ = g;}

	//! Returns the geometry representation of the theme which will be considered in the spatial restriction  
	TeGeomRep geomRepRestriction() { return geomRepRest_; }
	//! Sets the geometry representation of the theme which will be considered in the spatial restriction  
	void geomRepRestriction (TeGeomRep& rep) { geomRepRest_ = rep; }

	//! Set the spatial restriction to be a spatial relation with a box
	void setSpatialRest(TeBox& box, TeGeomRep rep = TeGEOMETRYNONE, TeSpatialRelation relation = TeWITHIN); 
	//! Set the spatial restriction to be a spatial relation with a geometry
	void setSpatialRest(TeGeometry* geom, TeGeomRep rep = TeGEOMETRYNONE, TeSpatialRelation relation = TeWITHIN); 
	
	//! Returns the name of a table used to store the ids of the objects belonging to the theme
	string	collectionTable() { return collectionTable_; }
	//! Set the name of a table used to store the ids of the objects belonging to the theme
	void collectionTable(const string& s) { collectionTable_ = s; }

	//! Return the name of the collection auxiliary table
	string collectionAuxTable() { return collectionAuxTable_; }
	//! Set the name of the collection auxiliary table
	void collectionAuxTable(string name) { collectionAuxTable_ = name; }

	//! Set the visible representation 
	void visibleRep (int rep) { visibleRep_ = rep; }
	//! Return the visible representation 
	int visibleRep () { return visibleRep_; }

	//! Returns the status of a theme
	/*! 
		- Returns 0 if the theme is not visible and not active
		- Returns 1 if the theme is visible and not active
		- Returns 2 if theme is and active and not visible
		- Returns 3 if theme is and visible and active
	 */
	int visibility()
	{	return enableVisibility_; }
	//! Set whether the theme should be visible and/or active
	void visibility(int v)
	{	enableVisibility_ = v; }


	//! verify if the theme was generated with restrictions
	bool hasRestriction() { return (!(generateAttributeRest_.empty() || 
									  generateTemporalRest_.empty()) && hasSpatialRes_); }

	//! Returns a grouping associated the theme 
	TeGrouping* grouping() {return grouping_; }
	
	//! Sets a grouping associated the theme
	void grouping(TeGrouping* g);
	
	//! Returns the legend (visual presentation) of the layer objects that arent selected in this layer
	TeLegendEntry& outOfCollectionLegend () 
	{ return outOfCollectionLegend_; } 
	
	//! Set the legend (visual presentation) of the layer objects that arent selected in this layer
	void outOfCollectionLegend (TeLegendEntry &leg) 
	{ outOfCollectionLegend_ = leg;}
	
	//! Set the visual of the layer objects that arent selected in this layer
	void setVisualOutOfCollection (TeVisual& visual, TeGeomRep rep)
	{ outOfCollectionLegend_.setVisual(visual, rep); }

	//! Returns the legend (visual presentation) of the geometries without descriptive attributes
	TeLegendEntry& withoutDataConnectionLegend () 
	{ return withoutDataConnectionLegend_; }

	//! Set the legend (visual presentation) of the geometries without descriptive attributes
	void withoutDataConnectionLegend (TeLegendEntry &leg) 
	{ withoutDataConnectionLegend_ = leg; }

	//! Set the visualof the geometries without descriptive attributes
	void setVisualWithoutDataConnection (TeVisual& visual, TeGeomRep rep)
	{ withoutDataConnectionLegend_.setVisual(visual, rep); }

	//! Returns the default legend (visual presentation) objects of the theme
	TeLegendEntry& defaultLegend () 
	{ return defaultLegend_; }
	
	//! Set the default legend (visual presentation) objects of the theme
	void defaultLegend (TeLegendEntry& leg) 
	{ defaultLegend_ = leg; }

	//! Set the default visual of the objects of the theme
	void setVisualDefault (TeVisual& visual, TeGeomRep rep)
	{ defaultLegend_.setVisual(visual, rep); }
	
	//! Returns the legend (visual presentation) of the theme objects selected by pointing
	TeLegendEntry& pointingLegend () 
	{ return pointingLegend_; }
	
	//! Set the legend (visual presentation) of the theme objects selected by pointing
	void pointingLegend (TeLegendEntry &leg) 
	{ pointingLegend_ = leg; }

	//! Set the visual of the theme objects selected by pointing
	void setVisualPointing (TeVisual& visual, TeGeomRep rep)
	{ pointingLegend_.setVisual(visual, rep); }

	//! Returns the legend (visual presentation) of the theme objects selected by a query
	TeLegendEntry& queryLegend () 
	{ return queryLegend_; }
	
	//! Set the legend (visual presentation) of the theme objects selected by a query
	void queryLegend (TeLegendEntry &leg) 
	{ queryLegend_ = leg; }

	//! Set the visual of the theme objects selected by a query
	void setVisualQuery (TeVisual& visual, TeGeomRep rep)
	{ queryLegend_.setVisual(visual, rep); }

	//! Returns the legend (visual presentation) of the theme objects selected by query and pointing
	TeLegendEntry& queryAndPointingLegend () 
	{ return queryAndPointingLegend_; }
	
	//! Set the legend (visual presentation) of the theme objects selected by query and pointing
	void queryAndPointingLegend (TeLegendEntry &leg) 
	{ queryAndPointingLegend_ = leg; }

	//! Set the visual of the theme objects selected by query and pointing
	void setVisualQueryAndPointing (TeVisual& visual, TeGeomRep rep)
	{ queryAndPointingLegend_.setVisual(visual, rep); }

	//! Set the n-th grouping visual to a geometric representation
	bool setGroupingVisual(int n, TeVisual& visual, TeGeomRep rep);

	//! Set the n-th grouping visual of a particular group
	bool setGroupingVisual(int n, TeGeomRepVisualMap& vismap);

	//! get the grouping slices from the legends without visual
	TeSliceVector getSlices();

	//! Returns a vector of the attribute tables of the theme
	bool getAttTables(TeAttrTableVector& attrs, TeAttrTableType attType = TeAllAttrTypes); 

	//! Returns the temporal attribute table of the theme (TeEvent and TeFixedGeomDynAttr)
	bool getTemporalTable(TeTable& table);
	
	//! Set the theme tables vector
	bool setAttTables(TeAttrTableVector& attrs);  

	//! Clear the list of theme tables
	void clearAttTableVector() 
	{	attTableVector_.clear();	}

	//! Returns the vector of attribute tables 
	TeAttrTableVector& attrTables()
	{	return attTableVector_; }

	//! Return a complete SQL statement to get all the attributes of the theme objects
	string	sqlJoin() {return sqlJoin_;}

	//! Return a FROM CLAUSE of a SQL statement to get attributes of theme objects
	string	sqlFrom() {return sqlFrom_;}

	//! Returns a SQL statement to get all the attributes of the theme objects, the attributes of the 
	//! collection table, and the attributes of the extended collection table 
	string sqlGridJoin() { return sqlGridJoin_; }

	//! Returns a FROM clause of a SQL statement to get attributes of the theme objects, the attributes of the 
	//! collection table, and the attributes of the extended collection table 
	string sqlGridFrom(const string& geomTable=""); 

	//! Return the clause WHERE from restrictions (spatial, attribute and temporal)
	string sqlWhereRestrictions(TeRepresentation* rep=0);

	//! Return the list of attributes of the theme tables 
	TeAttributeList sqlAttList() {return sqlAttList_;}

	//! Clear the list of attributes associated to the theme tables 
	void clearAttList() {sqlAttList_.clear();}

	//! Return the list of numerical attributes of the theme tables 
	TeAttributeList sqlNumAttList() {return sqlNumAttList_;}

	//! Clear the list of numerical attributes associated to the theme tables 
	void clearNumAttList() {sqlNumAttList_.clear();}

	//! Return the alias vector of the names of the theme tables
	vector<string>&	aliasVector() { return aliasVector_; }

	//! Save the theme informations in database
	bool save();

	//! Build the theme collection 
	bool buildCollection();

	//! Generate the label positions (x,y) to each object of the theme
	bool generateLabelPositions();  

	//! Clear the existing grouping of objects of this theme
	void resetGrouping ();

	//! Clean legend
	void cleanLegend() { legend_.clear(); }

	//! Save the grouping parameters in memory
	bool buildGrouping(TeGrouping* g, TeSelectedObjects selectedObjects = TeAll,
		               vector<double>* dValuesVec = 0);

	//! Save grouping parameters in memory passing an arbitrary set of slices
	bool buildGrouping(TeGrouping* g, vector<TeSlice>& slices);

	//! Build the grouping and associate each object to its group in the collection table 
	bool saveGrouping(TeSelectedObjects selectedObjects = TeAll);
	
	//! Add new theme table
	virtual bool addThemeTable (TeTable& table);
	virtual void addThemeTable (string tableName);

	//! Verify if a table is a theme table, given its id
	bool isThemeTable(int tableId);

	//! Verify if a table is a theme table, given its name
	bool isThemeTable(string tableName);

	//! Return the table name of a given attribute
	string getTableName(const string& attrName);

	//! Removes an attribute table
	bool removeThemeTable(unsigned int index);

	//! Loads the theme tables in the database
	bool loadThemeTables();

	//! Delete grouping
	bool deleteGrouping(); 

	//! Create the auxiliar collection table based in the theme tables   
	bool createCollectionAuxTable();

	//! Populate the auxiliar collection table based in the theme tables
	bool populateCollectionAux();

	//! Load the new attribute list of all the theme tables.
	//! All attributes are stored into sqlAttList_ and numeric attributes are stored into sqlNumAttList_.
	void loadAttrLists();

	/** @name Locate geometries
	*  Returns the geometry(ies) of the theme that contains a given coordinate
	*/
	//@{ 	
	bool locatePolygon		(TeCoord2D &pt, TePolygon &polygon, const double& tol = 0.0);
	bool locatePolygonSet   (TeCoord2D &pt, double tol, TePolygonSet &polygons);
	bool locateLine		(TeCoord2D &pt, TeLine2D &line, const double& tol = 0.0);
	bool locatePoint	(TeCoord2D &pt, TePoint &point, const double& tol = 0.0);
	bool locateCell		(TeCoord2D &pt, TeCell &c, const double& tol = 0.0);
	//@}

	/** @name Raster Visual
	*  Methods to deal with the visual presentation of the raster
	*/
	//@{ 
	//! Returns the visual presentation of raster geometry
	TeRasterTransform* rasterVisual() 
	{ return rasterVisual_; }

	//! Sets the visual presentation of raster geometry
	void rasterVisual(TeRasterTransform* r) 
	{ rasterVisual_ = r; } 

	//! Removes the visual presentation of the raster
	void removeRasterVisual()
	{ 
		if (rasterVisual_)
		{
			delete rasterVisual_;
			rasterVisual_ = 0;
		}
	}

	//! Creates an appropriate visual presentation to the raster of the theme
	void createRasterVisual(TeRaster* rst=0);
	//@}

protected:
	// ----------------- theme information -----------------
	
	// General identification
	int			layerId_;	// layer id
		
	// Associated layer and selected objects
	TeLayer*	layer_;
	
	// Selection
	//attribute restriction
	string		generateAttributeRest_;
	
	// temporal restriction 
	string		generateTemporalRest_;
	
	//spatial restriction
	string				generateSpatialRest_; //future use

	TeSpatialRelation	spatialRelation_;
	bool				hasSpatialRes_;
	TeBox				boxRest_;		//box which defines the spatial restriction 
	TeGeometry*			geomRest_;		//geometry which defines the spatial restriction 
	TeGeomRep			geomRepRest_;	//geometry representation of the theme which will be 
										//considered in the spatial restriction  
	// Display scale
	double	minScale_;
	double	maxScale_;

	// collection table name
	string	collectionTable_;
	string	collectionAuxTable_;

	//representation visible in the theme
	int		visibleRep_;

	//if theme is visible or not
	int		enableVisibility_;

	// ----------------- grouping information -----------------
	TeGrouping*		grouping_;

	// ----------------- legend information -----------------
	TeLegendEntryVector	legend_;

	// Background Legend
	TeLegendEntry		outOfCollectionLegend_;			//group(-1) 
	TeLegendEntry		withoutDataConnectionLegend_;	//group(-2) 
	TeLegendEntry		defaultLegend_;					//group(-3) 
	TeLegendEntry		pointingLegend_;				//group(-4) 
	TeLegendEntry		queryLegend_;					//group(-5) 
	TeLegendEntry		queryAndPointingLegend_;		//group(-6) 
		
	// ----------------- theme tables information -----------------
	TeAttrTableVector	attTableVector_;
	
	//! clause FROM: join with the theme tables and collection table
	string				sqlFrom_;		
	
	//! clause SELECT and FROM: join with the theme tables and collection table
	string				sqlJoin_;
	
	//! clause FROM: join with the theme tables and auxiliar collection table
	string				sqlGridFrom_;
	
	//! clause SELECT and FROM: join with the theme tables and auxiliar collection table
	string				sqlGridJoin_;

	//! vector of alias to the attribute tables that are used more than once  
	vector<string>		aliasVector_;

	//! list containing all the attributes of the theme tables
	TeAttributeList	sqlAttList_;
	
	//! list containing only the numeric attributes of the the theme tables
	TeAttributeList	sqlNumAttList_;		

	//! visual of raster
	TeRasterTransform*		rasterVisual_;

	//! fill aliasVector_
	void loadAliasVector();

	//! Fill the sqlJoin_ and sqlFrom_ 
	void loadThemeTablesJoin();

	//! Fill the sqlGridJoin_ and sqlGridFrom_ 
	void loadTablesJoin(const string& geomTable="");

	//! Save the theme grouping legends in the collection table  
	bool saveLegendInCollection(TeSelectedObjects selectedObjects = TeAll);
	
	//! Populate the collection table based in the theme restrictions
	bool populateCollection(); 

};

//! A vector of pointers to Theme
typedef vector<TeTheme*>	TeThemeVector;

//! A map from theme identifier to pointer to theme
typedef map<int, TeTheme*>	TeThemeMap;

/*! \example ThemeExample1.cpp
	This is an example of how to create themes in TerraLib.
 */

/*! \example ThemeExample2.cpp
	This is an example of how to do a grouping on the objects of a TerraLib theme.
 */
#endif

