/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeSTElementSet.h"
#include "TeSTInstance.h"
#include "TeSTEvent.h"
#include "TeProgress.h"
#include "TeQuerier.h"
#include "TeQuerierDB.h"
#include "TeSTEFunctionsDB.h"

//------------------------------- auxiliary functions

bool insertRow (TeSTInstance* elem, TeTable& table, const string& uniqueValue, vector<int>* indexes=0)
{
	vector<string> attrs;
	table.attributeNames(attrs);
	
	string ins = " INSERT INTO "+ table.name() +" (";
	string values = " VALUES ( ";
	
	TePropertyVector prop = elem->getPropertyVector();
	int count=0;

	for(unsigned int i=0; i<prop.size(); ++i)
	{
		if(indexes && (find(indexes->begin(), indexes->end(),(int)i) == indexes->end()))
			continue;
		
		string attrName = prop[i].attr_.rep_.name_;
				
		if( (find(attrs.begin(), attrs.end(), attrName)!=attrs.end()) &&
			(attrName != table.uniqueName())      && 
			(attrName != table.linkName())		  &&
			(attrName != table.attInitialTime())  &&
			(attrName != table.attFinalTime())     )
			
		{
			if(count>0)
			{
				ins += ",";
				values += ",";
			}
			++count;
			ins += attrName;
			if(prop[i].attr_.rep_.type_==TeSTRING)
				values += "'"+ prop[i].value_ +"'";
			else if(prop[i].attr_.rep_.type_==TeDATETIME)
			{
				TeTime time(prop[i].value_, prop[i].attr_.dateChronon_, prop[i].attr_.dateTimeFormat_, prop[i].attr_.dateSeparator_, prop[i].attr_.timeSeparator_);  
				values += elem->theme()->layer()->database()->getSQLTime(time);
			}
			else
				values += prop[i].value_;
		}
	}

	// -------- object_id, unique_id and timeInterval
		
	if(count>0)
	{
		ins +=		" ,";
		values +=	" ,";
	}

	ins +=		table.linkName();
	values +=	"'"+ elem->objectId() +"'";

	if(table.linkName() != table.uniqueName())
	{
		ins +=		", "+ table.uniqueName();
		values +=	", '"+ uniqueValue +"'";
	}

	if(!table.attInitialTime().empty())
	{
		TeTime time (elem->timeInterval().getT1());
		ins +=		", "+ table.attInitialTime();
		values +=	", "+ elem->theme()->layer()->database()->getSQLTime(time); 
	}

	if((!table.attFinalTime().empty()) && (table.attInitialTime()!=table.attFinalTime()))
	{
		TeTime time (elem->timeInterval().getT2());
		ins +=		", "+ table.attFinalTime();
		values +=	", "+ elem->theme()->layer()->database()->getSQLTime(time); 
	}
	// ----------

	ins += ") "+ values +" )";
	
	if(!elem->theme()->layer()->database ()->execute (ins))
		return false;

	return true;
}


bool insertRow (TeSTEvent* elem, TeTable& table, const string& uniqueValue, vector<int>* indexes=0)
{
	vector<string> attrs;
	table.attributeNames(attrs);
	
	string ins = " INSERT INTO "+ table.name() +" (";
	string values = " VALUES ( ";
	
	TePropertyVector prop = elem->getPropertyVector();
	int count=0;

	for(unsigned int i=0; i<prop.size(); ++i)
	{
		if(indexes && (find(indexes->begin(), indexes->end(),(int)i) == indexes->end()))
			continue;
		
		string attrName = prop[i].attr_.rep_.name_;
				
		if( (find(attrs.begin(), attrs.end(), attrName)!=attrs.end()) &&
			(attrName != table.uniqueName())      && 
			(attrName != table.linkName())		  &&
			(attrName != table.attInitialTime())  &&
			(attrName != table.attFinalTime())     )
			
		{
			if(count>0)
			{
				ins += ",";
				values += ",";
			}
			++count;
			ins += attrName;
			if(prop[i].attr_.rep_.type_==TeSTRING)
				values += "'"+ prop[i].value_ +"'";
			else if(prop[i].attr_.rep_.type_==TeDATETIME)
			{
				TeTime time(prop[i].value_, prop[i].attr_.dateChronon_, prop[i].attr_.dateTimeFormat_, prop[i].attr_.dateSeparator_, prop[i].attr_.timeSeparator_);  
				values += elem->theme()->layer()->database()->getSQLTime(time);
			}
			else
				values += prop[i].value_;
		}
	}

	// -------- object_id, unique_id and timeInterval
		
	if(count>0)
	{
		ins +=		" ,";
		values +=	" ,";
	}

	ins +=		table.linkName();
	values +=	"'"+ elem->objectId() +"'";

	if(table.linkName() != table.uniqueName())
	{
		ins +=		", "+ table.uniqueName();
		values +=	", '"+ uniqueValue +"'";
	}

	if(!table.attInitialTime().empty())
	{
		TeTime time (elem->timeInterval().getT1());
		ins +=		", "+ table.attInitialTime();
		values +=	", "+ elem->theme()->layer()->database()->getSQLTime(time); 
	}

	if((!table.attFinalTime().empty()) && (table.attInitialTime()!=table.attFinalTime()))
	{
		TeTime time (elem->timeInterval().getT2());
		ins +=		", "+ table.attFinalTime();
		values +=	", "+ elem->theme()->layer()->database()->getSQLTime(time); 
	}
	// ----------

	ins += ") "+ values +" )";
	
	if(!elem->theme()->layer()->database ()->execute (ins))
		return false;

	return true;
}

bool updateRow (TeSTInstance* elem, TeTable table, const string& uniqueId, vector<int>* indexes=0)
{
	
	vector<string> attrs;
	table.attributeNames(attrs);

	string ins = " UPDATE "+ table.name() +" SET ";
	
	TePropertyVector prop = elem->getPropertyVector();
	int count = 0;
	for(unsigned int i=0; i<prop.size(); ++i)
	{
		if(indexes && (find(indexes->begin(), indexes->end(),(int)i) == indexes->end()))
			continue;

		string attrName = prop[i].attr_.rep_.name_;

		if( (find(attrs.begin(), attrs.end(), attrName)!=attrs.end()) &&
			(attrName != table.uniqueName())      && 
			(attrName != table.linkName())		  &&
			(attrName != table.attInitialTime())  &&
			(attrName != table.attFinalTime())     )
		{
		
			if(count>0)
				ins += ",";
			
			++count;
			ins += attrName +" = ";
		
			if(prop[i].attr_.rep_.type_==TeSTRING)
				ins += "'"+ prop[i].value_ +"'";
			else if(prop[i].attr_.rep_.type_==TeDATETIME)
			{
				TeTime time(prop[i].value_, prop[i].attr_.dateChronon_, prop[i].attr_.dateTimeFormat_, prop[i].attr_.dateSeparator_, prop[i].attr_.timeSeparator_);  
				ins += elem->theme()->layer()->database()->getSQLTime(time);
			}
			else
				ins += prop[i].value_;
		}
	}

	// -------- timeInterval
	if(!table.attInitialTime().empty())
	{
		TeTime time (elem->timeInterval().getT1());
		ins +=	", "+ table.attInitialTime() +" = ";
		ins +=	elem->theme()->layer()->database()->getSQLTime(time); 
	}

	if((!table.attFinalTime().empty()) && (table.attInitialTime()!=table.attFinalTime()))
	{
		TeTime time (elem->timeInterval().getT2());
		ins +=	", "+ table.attFinalTime() +" = ";
		ins +=	elem->theme()->layer()->database()->getSQLTime(time); 
	}
	// ----------

	ins += " WHERE "+ table.uniqueName() +" = '"+ uniqueId +"'";
	
	if(!elem->theme()->layer()->database()->execute (ins))
		return false;

	return true;
}

bool updateRow (TeSTEvent* elem, TeTable table, const string& uniqueId, vector<int>* indexes=0)
{
	
	vector<string> attrs;
	table.attributeNames(attrs);

	string ins = " UPDATE "+ table.name() +" SET ";
	
	TePropertyVector prop = elem->getPropertyVector();
	int count = 0;
	for(unsigned int i=0; i<prop.size(); ++i)
	{
		if(indexes && (find(indexes->begin(), indexes->end(),(int)i) == indexes->end()))
			continue;

		string attrName = prop[i].attr_.rep_.name_;

		if( (find(attrs.begin(), attrs.end(), attrName)!=attrs.end()) &&
			(attrName != table.uniqueName())      && 
			(attrName != table.linkName())		  &&
			(attrName != table.attInitialTime())  &&
			(attrName != table.attFinalTime())     )
		{
		
			if(count>0)
				ins += ",";
			
			++count;
			ins += attrName +" = ";
		
			if(prop[i].attr_.rep_.type_==TeSTRING)
				ins += "'"+ prop[i].value_ +"'";
			else if(prop[i].attr_.rep_.type_==TeDATETIME)
			{
				TeTime time(prop[i].value_, prop[i].attr_.dateChronon_, prop[i].attr_.dateTimeFormat_, prop[i].attr_.dateSeparator_, prop[i].attr_.timeSeparator_);  
				ins += elem->theme()->layer()->database()->getSQLTime(time);
			}
			else
				ins += prop[i].value_;
		}
	}

	// -------- timeInterval
	if(!table.attInitialTime().empty())
	{
		TeTime time (elem->timeInterval().getT1());
		ins +=	", "+ table.attInitialTime() +" = ";
		ins +=	elem->theme()->layer()->database()->getSQLTime(time); 
	}

	if((!table.attFinalTime().empty()) && (table.attInitialTime()!=table.attFinalTime()))
	{
		TeTime time (elem->timeInterval().getT2());
		ins +=	", "+ table.attFinalTime() +" = ";
		ins +=	elem->theme()->layer()->database()->getSQLTime(time); 
	}
	// ----------

	ins += " WHERE "+ table.uniqueName() +" = '"+ uniqueId +"'";
	
	if(!elem->theme()->layer()->database()->execute (ins))
		return false;

	return true;
}

//------------------------------- 

bool
TeSTOSetBuildDB(TeSTElementSet* stoset, bool loadGeometries, bool loadAllAttributes, vector<string> attrNames)
{
	if(!stoset->querier())
		return false;

	if(!stoset->build(loadGeometries, loadAllAttributes, attrNames))
		return false;
	return true;
}  

bool
TeSTOSetBuildDB(TeSTElementSet* stoset, TeGroupingAttr& groupAttr, bool loadGeometries)
{
	if(!stoset->querier())
		return false;

	if(!stoset->build(groupAttr, loadGeometries))
		return false;
	return true;
}

bool  
TeUpdateDBFromSet (TeSTElementSet* elemSet, const string& tableName, vector<int>* indexes)
{
	if(!elemSet->theme()->layer())
		return false;

	TeDatabase* db = elemSet->theme()->layer()->database();

	//progress bar
	int step = 0;
	int numSteps = elemSet->numSTInstance();
	if(TeProgress::instance())
		TeProgress::instance()->setTotalSteps(numSteps);
	
	try
	{
		TeAttrTableVector attrTables;
		elemSet->theme()->getAttTables(attrTables); 
		if(attrTables.empty())
			return false;	

		TeTable table;
		int	uniqueIndex = -1;
	
		//! verify if the table is in the stoset
		for(unsigned int i=0; i<attrTables.size(); i++)
		{
			if(attrTables[i].name()==tableName)
			{
				uniqueIndex = i;
				table = attrTables[i];
				break;
			}
		}
		
		if((uniqueIndex==-1) || ((table.tableType()!=TeAttrEvent) && 
								 (table.tableType()!=TeAttrStatic) && 
								 (table.tableType()!=TeFixedGeomDynAttr) &&
								 (table.tableType()!=TeAttrExternal)))
			return false;

		// get some information about the attribute table required
		string uniqueIdName = table.uniqueName();
	
		TeDatabasePortal* portal = db->getPortal();
		if(!portal)
			return false;

		vector<string> uniqueIds;
		string sql = "SELECT "+ uniqueIdName +" FROM "+ table.name();
		
		if(!portal->query (sql))
		{
			delete portal;
			return false;
		}

		while(portal->fetchRow())
			uniqueIds.push_back(portal->getData(0));
		
		delete portal;

		// Update all the objects 
		TeSTElementSet::iterator itObj = elemSet->begin();
		while (itObj != elemSet->end())
		{
			//verify if all columns exist in the table 
			if(itObj == elemSet->begin())
			{
				TePropertyVector vec = (*itObj).getPropertyVector();
				for(unsigned int i=0; i<vec.size(); ++i)
				{
					if(indexes && (find(indexes->begin(), indexes->end(),(int)i) == indexes->end()))
						continue;

					//verify if the table has this column 
					if (!elemSet->theme()->layer()->database()->columnExist(tableName, vec[i].attr_.rep_.name_ , vec[i].attr_))
					{
						if(!elemSet->theme()->layer()->database()->addColumn (tableName, vec[i].attr_.rep_))
							return false; 

						TeAttributeList attrList = table.attributeList();
						attrList.push_back (vec[i].attr_);
						table.setAttributeList (attrList);
					}
				}
			}
			
			string uniqueId =  (*itObj).uniqueId(uniqueIndex); 
			
			if(find(uniqueIds.begin(), uniqueIds.end(), uniqueId) ==  uniqueIds.end())
			{	
				if (!insertRow (&(*itObj), table, uniqueId, indexes))
					return false;

				uniqueIds.push_back(uniqueId);
			}
			else
			{	
				if (!updateRow (&(*itObj), table, uniqueId, indexes))
					return false;
			}
					
			++itObj;

			if(TeProgress::instance())
			{
				if (TeProgress::instance()->wasCancelled())
				{
					TeProgress::instance()->reset();
					return false;
				}
				else
					TeProgress::instance()->setProgress(step);
			}	
			++step;
		}
	}
	catch(...)
	{
		if (TeProgress::instance())
			TeProgress::instance()->reset();
		return false;
	}
	
	if (TeProgress::instance())
		TeProgress::instance()->reset();
	
	return true;
}


bool  
TeUpdateDBFromSet (TeSTEventSet* elemSet, const string& tableName, vector<int>* indexes)
{
	if(!elemSet->theme()->layer())
		return false;

	TeDatabase* db = elemSet->theme()->layer()->database();

	//progress bar
	int step = 0;
	int numSteps = elemSet->numSTInstance();
	if(TeProgress::instance())
		TeProgress::instance()->setTotalSteps(numSteps);
	
	try
	{
		TeAttrTableVector attrTables;
		elemSet->theme()->getAttTables(attrTables); 
		if(attrTables.empty())
			return false;	

		TeTable table;
		int	uniqueIndex = -1;
	
		//! verify if the table is in the stoset
		for(unsigned int i=0; i<attrTables.size(); i++)
		{
			if(attrTables[i].name()==tableName)
			{
				uniqueIndex = i;
				table = attrTables[i];
				break;
			}
		}
		
		if((uniqueIndex==-1) || ((table.tableType()!=TeAttrEvent) && 
								 (table.tableType()!=TeAttrStatic) && 
								 (table.tableType()!=TeFixedGeomDynAttr) &&
								 (table.tableType()!=TeAttrExternal)))
			return false;

		// get some information about the attribute table required
		string uniqueIdName = table.uniqueName();
	
		TeDatabasePortal* portal = db->getPortal();
		if(!portal)
			return false;

		vector<string> uniqueIds;
		string sql = "SELECT "+ uniqueIdName +" FROM "+ table.name();
		
		if(!portal->query (sql))
		{
			delete portal;
			return false;
		}

		while(portal->fetchRow())
			uniqueIds.push_back(portal->getData(0));
		
		delete portal;

		// Update all the objects 
		TeSTEventSet::iterator itObj = elemSet->begin();
		while (itObj != elemSet->end())
		{
			//verify if all columns exist in the table 
			if(itObj == elemSet->begin())
			{
				TePropertyVector vec = (*itObj).getPropertyVector();
				for(unsigned int i=0; i<vec.size(); ++i)
				{
					if(indexes && (find(indexes->begin(), indexes->end(),(int)i) == indexes->end()))
						continue;

					//verify if the table has this column 
					if (!elemSet->theme()->layer()->database()->columnExist(tableName, vec[i].attr_.rep_.name_ , vec[i].attr_))
					{
						if(!elemSet->theme()->layer()->database()->addColumn (tableName, vec[i].attr_.rep_))
							return false;

						TeAttributeList attrList = table.attributeList();
						attrList.push_back (vec[i].attr_);
						table.setAttributeList (attrList);
					}
				}
			}
			
			string uniqueId =  (*itObj).uniqueId(uniqueIndex); 
			
			if(find(uniqueIds.begin(), uniqueIds.end(), uniqueId) ==  uniqueIds.end())
			{	
				if (!insertRow (&(*itObj), table, uniqueId, indexes))
					return false;

				uniqueIds.push_back(uniqueId);
			}
			else
			{	
				if (!updateRow (&(*itObj), table, uniqueId, indexes))
					return false;
			}
					
			++itObj;

			if(TeProgress::instance())
			{
				if (TeProgress::instance()->wasCancelled())
				{
					TeProgress::instance()->reset();
					return false;
				}
				else
					TeProgress::instance()->setProgress(step);
			}	
			++step;
		}
	}
	catch(...)
	{
		if (TeProgress::instance())
			TeProgress::instance()->reset();
		return false;
	}
	
	if (TeProgress::instance())
		TeProgress::instance()->reset();
	
	return true;

}


