#include "TePDIUtils.hpp"

#include "TePDIAgnostic.hpp"

#include <TeDecoderMemory.h>
#include <TeDecoderTIFF.h>
#include <TeRaster.h>
#include <TeRasterParams.h>

#include <cmath>

namespace TePDIUtils{

  bool TeGenerateHistogram(
    TePDITypes::TePDIRasterPtrType& in_raster,
    unsigned int levels,
    unsigned int band,
    TePDITypes::TePDIHistogramType& out_hist,
    bool force_zero_reference )
  {
    int bands_number = in_raster->nBands();

    PDIAGN_TRUE_OR_RETURN( (int)band < bands_number,
      "Invalid band" );

    out_hist.clear();

    TeRaster::iterator raster_it = in_raster->begin();

    TeRaster::iterator raster_end_it = in_raster->end();

    /* Finding the lowest end highest level */

    double lowest_level = 0;
    bool lowest_level_enabled = 0;
    double highest_level = 0;
    bool highest_level_enabled = 0;
    double current_raster_level;

    while( raster_it != raster_end_it ) {
      current_raster_level = raster_it*( band );

      if( lowest_level_enabled ) {
        if( current_raster_level < lowest_level) {
          lowest_level = current_raster_level;
        }
      } else {
        lowest_level = current_raster_level;
        lowest_level_enabled = true;
      }

      if( highest_level_enabled ) {
        if( current_raster_level > highest_level) {
          highest_level = current_raster_level;
        }
      } else {
        highest_level = current_raster_level;
        highest_level_enabled = true;
      }

      ++raster_it;
    }

    if( force_zero_reference ) {
      if( highest_level > 0 ) {
        lowest_level = 0;
      } else {
        highest_level = 0;
      }
    }

    /* Calculating the histogram step */

    double step = 1;
    unsigned int hist_computed_levels = 1;

    if( levels == 0 ) {
      /* Auto step feature */
      PDIAGN_TRUE_OR_RETURN( ( highest_level - lowest_level ) > 1,
        "Unable to use the Histogram Auto-step feature" );

      hist_computed_levels = 1 +
        ( unsigned int ) std::floor( highest_level - lowest_level );
    } else {
      hist_computed_levels = levels;
    }

    PDIAGN_TRUE_OR_RETURN( hist_computed_levels > 1,
      "Invalid levels number" );

    step = ( highest_level - lowest_level ) / (double)hist_computed_levels;

    /* Ordered Histogram pre-allocation */

    for( unsigned int level = 0 ; level < ( hist_computed_levels - 1 ) ;
         ++level ) {

      out_hist[ ( level * step ) + lowest_level ] = 0;
    }
    out_hist[ highest_level ] = 0;

    /* Histogram generation */

    raster_it = in_raster->begin();
    TePDITypes::TePDIHistogramType::iterator hist_it;
    TePDITypes::TePDIHistogramType::iterator hist_next_it;
    TePDITypes::TePDIHistogramType::iterator hist_end_it = out_hist.end();

    while( raster_it != raster_end_it ) {
      current_raster_level = raster_it*( band );

      /* locating the raster level position relative to histogram levels */

      hist_it = out_hist.begin();
      hist_next_it = hist_it;
      ++hist_next_it;

      while( hist_next_it != hist_end_it ) {
        if( current_raster_level <= hist_next_it->first ) {
          break;
        }

        ++hist_it;
        ++hist_next_it;
      }

      /* Interpolating level */

      if( hist_next_it == hist_end_it ) {
        (*hist_it).second = (*hist_it).second + 1;
      } else {
        if( std::abs( hist_it->first - current_raster_level ) <
            std::abs( hist_next_it->first - current_raster_level ) ) {

          (*hist_it).second = (*hist_it).second + 1;
        } else {
          (*hist_next_it).second = (*hist_next_it).second + 1;
        }
      }


      ++raster_it;
    }

    return true;
  }


  bool TeAllocRAMRaster(
    TePDITypes::TePDIRasterPtrType& template_raster,
    TePDITypes::TePDIRasterPtrType& RAMRaster )
  {
    PDIAGN_TRUE_OR_RETURN( template_raster.isActive(),
      "Inactive Input Raster" );
    PDIAGN_TRUE_OR_RETURN( template_raster->status() != TeNOTREADY,
      "Input not Ready" );

    if( ! RAMRaster.isActive() ) {
      RAMRaster.reset( new TeRaster() );
    }

    TeRasterParams temp_params = template_raster->params();
    temp_params.mode_ = 'c';

    /* Assigning a RAM decoder */
    RAMRaster->setDecoder( new TeDecoderMemory( temp_params ) );

    return RAMRaster->init();
  }


  bool TeRaster2Geotiff(
    TePDITypes::TePDIRasterPtrType& in_raster,
    const std::string& file_name )
  {
    PDIAGN_TRUE_OR_RETURN( in_raster.isActive(),
      "Inactive Input Raster" );
    PDIAGN_TRUE_OR_RETURN( in_raster->status() != TeNOTREADY,
      "Input not Ready" );
    PDIAGN_TRUE_OR_RETURN( file_name.size() != 0,
      "Invalid file name" );

    TeRasterParams temp_params = in_raster->params();

    temp_params.mode_ = 'c';
    temp_params.fileName_ = file_name;

    TePDITypes::TePDIRasterPtrType outRaster( new TeRaster( temp_params ) );

    outRaster->setDecoder( new TeDecoderTIFF( outRaster->params() ) );

    PDIAGN_TRUE_OR_RETURN( outRaster->init(),
      "Unable to init GeoTIFF Raster" );

    return TeCopyRasterPixels( in_raster, outRaster );
  }


  bool TeCopyRasterPixels(
    TePDITypes::TePDIRasterPtrType& source_raster,
    TePDITypes::TePDIRasterPtrType& target_raster )
  {
    PDIAGN_TRUE_OR_RETURN( source_raster.isActive(),
      "Inactive Input Raster" );
    PDIAGN_TRUE_OR_RETURN( target_raster.isActive(),
      "Inactive Output Raster" );
    PDIAGN_TRUE_OR_RETURN( source_raster->status() != TeNOTREADY,
      "Input not Ready" );
    PDIAGN_TRUE_OR_RETURN( target_raster->status() == TeREADYTOWRITE,
      "OutPut not Ready" );

    TeRasterParams& source_params = source_raster->params();
    TeRasterParams& target_params = target_raster->params();

    PDIAGN_TRUE_OR_RETURN(
      ( ( source_params.nBands() == target_params.nBands() ) &&
      ( source_params.nlines_ == target_params.nlines_ ) &&
      ( source_params.ncols_ == target_params.ncols_ ) ),
      "Input and Output Raster's have different dimentions" );

    double pixel_value;

    for( int band = 0 ; band < source_params.nBands() ; ++band )
      for( int line = 0 ; line < source_params.nlines_ ; ++line )
        for( int column = 0 ; column < source_params.ncols_ ; ++column ) {
          if( source_raster->getElement( column, line, pixel_value, band ) ) {
            PDIAGN_TRUE_OR_THROW( target_raster->setElement( column, line,
              pixel_value, band ), "Pixel copy error" );
          }

        }

    return true;
  }


  bool TeResetRaster(TePDITypes::TePDIRasterPtrType& raster,
      unsigned int bands, unsigned int lines, unsigned int columns,
      TePDIRgbPalette* palette  )
  {
    PDIAGN_TRUE_OR_RETURN( raster.isActive(),
      "Inactive Raster" );
    PDIAGN_TRUE_OR_RETURN( raster->status() == TeREADYTOWRITE,
      "Raster not ready" );

    TeRasterParams params = raster->params();

    params.nBands( (int) bands );

    params.nlines_ = (int) lines;
    params.ncols_ = (int) columns;
    params.mode_ = 'c';
    params.useDummy_ = false;

    if( palette != 0 ) {
      params.setPhotometric( TeRASTERPALETTE );
      params.setNumberPalleteEntries( palette->size() );

      TePDIRgbPalette::iterator pal_it = palette->begin();
      TePDIRgbPalette::iterator pal_it_end = palette->end();

      for( unsigned int lut_index = 0 ; lut_index < palette->size() ;
           ++lut_index ) {

        params.lutr_[ lut_index ] = (unsigned short)pal_it.red();
        params.lutg_[ lut_index ] = (unsigned short)pal_it.green();
        params.lutb_[ lut_index ] = (unsigned short)pal_it.blue();

        ++pal_it;
      }

      params.lutr_[ palette->size() ] = 255;
      params.lutg_[ palette->size() ] = 255;
      params.lutb_[ palette->size() ] = 255;
    }

    PDIAGN_TRUE_OR_RETURN( raster->init( params ),
      "Unable to init raster" );

    return true;
  }

}

