/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeProxMatrixWeightsStrategy.h
    This file contains structures and definitions about weight strategies of proximity matrix 
*/


#ifndef TeProxMatrixWeightsStrategy_H
#define TeProxMatrixWeightsStrategy_H

#include "TeProxMatrixImplementation.h"
#include <vector>


//! An abstract class to representate weight strategies of proximity matrix    
class TeProxMatrixWeightsStrategy  
{
protected:
	bool norm_;

	//! Constructor
	TeProxMatrixWeightsStrategy(bool norm = false) { norm_ = norm; }

public:
	//! Compute weigths
	virtual bool ComputeWeigths (TeProxMatrixImplementation* /* imp */) {return true;}

	//! Destructor
	virtual ~TeProxMatrixWeightsStrategy() {}

	//! Verify if is equal
	virtual bool IsEqual (const TeProxMatrixWeightsStrategy& /* rhs */) const {return true;}

	//! Equal operator
	virtual bool operator== (const TeProxMatrixWeightsStrategy& /* rhs */) const {return true;}

};

//! A class to implement the no weight strategy of proximity matrix (i.e., all weights are 1, only indicating that a connection exists).
class TeProxMatrixNoWeightsStrategy : public TeProxMatrixWeightsStrategy
{

public:

	//! Constructor
	TeProxMatrixNoWeightsStrategy (bool norm = true) {norm_ = norm;}

	//! Compute weigths
	virtual bool ComputeWeigths (TeProxMatrixImplementation* imp);

	//! Destructor
	~TeProxMatrixNoWeightsStrategy() {}
};


//! A class to implement the inverse distance weight strategy of proximity matrix; if network distances were computed, they can be also considered.
//! Formula: w = (a*1/dist_centroids + b*1/dist_to_net + c*1/dist_net_connection)*factor
//! These values can be normalized or not.
class TeProxMatrixInverseDistanceStrategy : public TeProxMatrixWeightsStrategy
{
private:
	double a_;
	double b_;
	double c_;
	double factor_;

public:

	//! Constructor
	TeProxMatrixInverseDistanceStrategy  (double a = 1.0, double b = 1.0, double c = 1.0, double factor = 1.0, bool norm = true) : TeProxMatrixWeightsStrategy (norm)
	{ a_ = a; b_ = b; c_ = c; factor_ = factor;}

	//! Compute weights
	virtual bool ComputeWeigths (TeProxMatrixImplementation* imp); 

	//! Destructor	
	~TeProxMatrixInverseDistanceStrategy() {}
};


//! A class to implement the inverse distance weight strategy of proximity matrix; if network distances were computed, they can be also considered.
//! Formula: w = (a*1/(dist_centroids)2 + b*1/(dist_to_net)2 + c*1/(dist_net_connection)2)*factor
//! These values can be normalized or not.
class TeProxMatrixSquaredInverseDistanceStrategy : public TeProxMatrixWeightsStrategy
{
private:
	double a_;
	double b_;
	double c_;
	double factor_;

public:

	//! Constructor
	TeProxMatrixSquaredInverseDistanceStrategy  (double a = 1.0, double b = 1.0, double c = 1.0, double factor = 1.0, bool norm = true) : TeProxMatrixWeightsStrategy (norm)
	{ a_ = a; b_ = b; c_ = c; factor_ = factor;}

	//! Compute weights
	virtual bool ComputeWeigths (TeProxMatrixImplementation* imp); 

	//! Destructor	
	~TeProxMatrixSquaredInverseDistanceStrategy() {}
};


//! A class to implement the connection strenght weight strategy of proximity matrix
//! If centroids distance is smaller them max_local_distance, w will be only the local distance inverse (multiplied by factor).
//! otherwise, it will be w = 1/(dist_ratio*dist_to_net + dist_net_connection)*factor.
class TeProxMatrixConnectionStrenghtStrategy : public TeProxMatrixWeightsStrategy
{
private:
	double dist_ratio_;
	double max_local_distance_;
	double factor_;
public:

	//! Constructor
	TeProxMatrixConnectionStrenghtStrategy  (double dist_ratio = 1.0, double max_local_distance = 0.0, double factor = 1.0, bool norm = true) 
		: TeProxMatrixWeightsStrategy (norm)
	{ 
		dist_ratio_ = dist_ratio;
		max_local_distance_ = max_local_distance;
		factor_ = factor;
	}

	//! Compute weights
	virtual bool ComputeWeigths (TeProxMatrixImplementation* imp); 

	//! Destructor	
	~TeProxMatrixConnectionStrenghtStrategy() {}
};

#endif 
