/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeGeneralizedProxMatrix.h
    This file contains structures and definitions about 
	generalized proximity matrices support in TerraLib
*/


#ifndef TeProxMatrix_H
#define TeProxMatrix_H

#include "TeNeighbours.h"
#include "TeProxMatrixImplementation.h"
#include "TeProxMatrixConstructionStrategy.h"
#include "TeProxMatrixSlicingStrategy.h"
#include "TeProxMatrixWeightsStrategy.h"


//! A class to represent a generalized proximity matrix 
class TeGeneralizedProxMatrix  
{
private:

	TeProxMatrixImplementation*				imp_;  
	TeProxMatrixConstructionStrategy*		sc_;   
	TeProxMatrixSlicingStrategy*			ss_;   
	TeProxMatrixWeightsStrategy*			sw_;   

	int										total_slices_;

protected:

	//! Get the implementation from a concrete factory defined by impl_type
	TeProxMatrixImplementation*				getImplementation(const string implementation_type = "Breymann");

	//! Verify if a matrix was created correctly by the constructors.	
	bool IsValid() const;
	
public:

	//! Empty constructor
	TeGeneralizedProxMatrix ()
	{imp_ = 0; sc_ = 0; sw_ = 0; ss_ = 0; total_slices_=1;}

	//! Constructor parametrized with specific strategies. Each stragegy must be previously created and correctly parametrized.
	TeGeneralizedProxMatrix (TeProxMatrixConstructionStrategy* sc,  TeProxMatrixWeightsStrategy* sw = 0, TeProxMatrixSlicingStrategy* ss = 0, const string type = "Breymann");

	//! Constructor based on default strategies: (a) Local adjacency of first order; (b) No siling;  and (c)  No weighs (all equal to 1).
	TeGeneralizedProxMatrix(TeSTElementSet* objects, TeGeomRep geom_type, const string type = "Breymann");

	//! Copy constructor
	TeGeneralizedProxMatrix(const TeGeneralizedProxMatrix& p);

	//! Attribution Operator 
	TeGeneralizedProxMatrix& operator=(const TeGeneralizedProxMatrix& p);

	bool ClearImplementation ();

	//! Comparison Operator
	bool operator==(const TeGeneralizedProxMatrix& p) const;
	
	
	/** @name Methods for return the neighbours
	*  All methods return the  neighbours of a given object in a given slice. The default is the first slice.
	*  Slices are defined according to the Slicing Strategy in use (e.g., according to distance zones, corresponding to neighbourhood orders, weights intervals, etc.). 
	*  If the parameter slice is not provided, the first slice is returned.
	*  The operator[] method should preferably be used 
	*  For each object, all the connection attributes are stored as properties (weight is the first).
	*/
	//@{ 
	//! Return the neighbours of an object in a slice, packed in a TeNeighbours 
	TeNeighbours getNeighbours (const string& object_id, int slice = 1);

	//! Return the neighbours of an object in a slice, packed in a TeNeighboursMap
	TeNeighboursMap	getMapNeighbours (const string& object_id, int slice = 1);
	
	//! Operator [], return the neighbours packed in a TeNeighbours
	TeNeighbours operator[] (const string& object_id); 

	//! Return the neighbours of an object and their attributes in a spatial temporal element set (TeSTElementSet)
	TeSTElementSet getSTENeighbours(const string& object_id); 

	//@}

	
	/** @name Methods for changing current strategies 
	*  change current strategies 
	*/
	//@{ 
	//! Set current construction strategy 
	bool setCurrentConstructionStrategy (TeProxMatrixConstructionStrategy* sc);  
	
	//! Set current weights strategy 
	bool setCurrentWeightsStrategy (TeProxMatrixWeightsStrategy* sw); 
	
	//! Set current slicing strategy 
	bool setCurrentSlicingStrategy (TeProxMatrixSlicingStrategy* ss); 
	//@}
	

	//! Reconstruct matrix and recompute weights and slicing, accornding to current strategies.
	bool ConstructMatrix ();

	//! Recomputes the weigths, given a new strategy. The matrix is not reconstructed.
	bool RecomputeWeights ();

	//! Sets the slicing strategy for neighbours selection.
	bool RecomputeSlicing ();

	//! Verify if two objects are connected
	bool IsConnected (const string& object_id1, const string& object_id2); 

	//! Connect two objects
	bool ConnectObjects (const string& object_id1, const string& object_id2, const TeProxMatrixAttributes& attr); 
	
	//! Connect two objects
	bool ConnectObjects (const string& object_id1, const string& object_id2);

	//! Disconnect two objects
	bool DisconnectObjects (const string& object_id1, const string& object_id2); 

	//! Remove object
	bool RemoveObject (const string& object_id);  

	//! Get connection attributes
	bool GetConnectionAttributes (const string& object_id1, string& object_id2, TeProxMatrixAttributes& attr); 

	//! Return the number of objects
	int NumberOfObjects (); 

	//! Return the number of slices
	int NumerOfSlices () {return total_slices_;}

	//! Save the matrix in a text file
	bool SaveTextFile (string name); 

		//! Save the matrix in a text file
	bool SaveGALFile (string name); 	
	
	//! Save the matrix in a text file
	bool SaveGWTFile (string name); 

	//! Destructor
	virtual ~TeGeneralizedProxMatrix();

};

#endif
