/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/
/*! \file TeDecoderTIFF.h
    This file contais functions to deal with raster images in Tiff/GeoTIFF format.
*/

#ifndef  __TERRALIB_INTERNAL_DECODERTIFF_H
#define  __TERRALIB_INTERNAL_DECODERTIFF_H

// LibTIFF includes
#include "geotiff.h"
#include "xtiffio.h"
#include "geo_normalize.h"
#include "geovalues.h"
#include "tiffiop.h"
#include "geo_tiffp.h" /* external TIFF interface */
#include "geo_keyp.h"  /* private interface       */

#include <iostream>
#include <stdio.h>
#include <sys/types.h>

#include "TeDecoderVirtualMemory.h"

#include <map>
using namespace std;

#define MONOIMAGE	0	/* Image type monochrome */
#define RGBIMAGE	1	/* Image type color image */
#define PALLETEIMAGE 2  /* Image type pallete image */


//! Implements a decoder to a raster in TIFF (Tagged Image File Format) format
/*!
	Decoder TIFF is based on geotiff libray (http://remotesensing.org/geotiff/geotiff.html).
	This imposes some restrictions to this decoder. Using libtiff TIFF image files may not 
	be opened for both reading and writing; there is no support for altering the contents of 
	a TIFF file. So this decoder is limited to create new tiff files or to read existing ones.
*/
class TeDecoderTIFF : public TeDecoder
{
	TIFF 	*tif;       // TIFF directory structure
	GTIF	*gtif;		// GEOTIFF directory structure

	unsigned char  *TBuffer_;
	unsigned char  *TBufferTile_;

	bool			isGeoTiff_;	// flag that indicates georeference
	bool			isTiled_;	// flag indicating if image is tiled
	unsigned short	photom_;    // photometric interpretation
	unsigned short	planar_;	// storage organization : PLANARCONFIG_CONTIG or PLANARCONFIG_SEPARATE
	unsigned short	compress_;	// compression flag: COMPRESSION_NONE, COMPRESSION_CCITTRLE, COMPRESSION_CCITTFAX3 and others
	short 			TImage;	    // MONOIMAGE, RGBIMAGE, PALLETE 
	int				nBands_;	// number of bands
	int				nbitsperpixel_;
	unsigned long	bytesperline_;
	unsigned long	tilew_,tileh_;
	unsigned long	rowtilesize_;
	unsigned long	bytespertile_;
	unsigned long	rowsperstrip_;
	int				TCurLine_;		// current line in memory
	int				TCurTile_;		// current tile in memory


	bool first;				// Flag to test the first access to To8Bits() function
    unsigned char Lut[256];	// Lut to convert nbits/pixel to 8bits/pixel
	unsigned short mask;	// mask bits 0000...1 for 1 bit
							// 000.1111 for 2 bits and on

	double *transMatrix_;
	double *pixelScale_;
	double *tiePoints_;

 
	// if PHOTOMETRIC_YCBCR:	
	float 	*YCbCrCoeffs,	*refBlackWhite;	// Chrominance coefficients
	unsigned short YCbCrHorizSampling,		// Horizontal Smapling
		           YCbCrVertSampling;		// Vertical and
													
													


	//!	Reads TIFF Directory and fills tif structure
	void ReadTiffDirectory();

	//! Reads the georeferencing keys of a geotiff data
	bool GetGeoTIFF ();

	//! Reads and inteprets a .tfw file associated to a tiff
	bool DecodeTFW(const string& filename);

	//! Sets the georeferencing keys of a geotiff data
	void SetGeoKeys();

	//! Write a dummy image (when creating a tiff file)
	bool WriteDummyImage();

	//!	Reads a line of monochrome image
	/*!		\param buf image
			\param lin line number to read
			\return true if line is secessfully read and false otherwise
	*/
	bool ReadMonoImage(unsigned long lin);

	//!	Reads a line of color image
	/*!		\param lin line number to read
			\return true if line is secessfully read and false otherwise
	*/
	bool ReadRGBImage(unsigned long lin);

	//!	Reads color image if planar = PLANARCONFIG_CONTIG
	/*!		\param lin line number to read
			\return non 0 if line is secessfully written and 0 otherwise
	*/
	bool ReadRGBContig(unsigned long lin);

	//!	Reads color image if planar = PHOTOMETRIC_PALETTE
	/*!		\param lin line number to read
			\return non 0 if line is secessfully written and 0 otherwise
	*/
	bool ReadPaletteContig(unsigned long lin);

	bool ReadTileImageContig(unsigned long lin, unsigned char *line);

	//!	Allocates the internal buffers
	bool AllocateBuffer();

	//!	Allocates the internal buffers
	void DeallocateBuffer();

	//!	Reads Lut associated to a raster pallete
	bool ReadLut();

	//!	Saves the associated LUT for pallete raster 
	void SaveLut();

	//! Converts buf to 8 bits per pixel into ret buffer
	//		Input:
	//			buf - a line of image stored in TIFF
	//		Output:
	//			ret - image line converted to 8 bits per pixel
	//		Preconditions:
	//			image line buffer (ret) must be allocated
	//			previously before call this function
	void To8Bits(u_char *buf,u_char *ret,short vmax=0, short vmin = 0,short nx=0);

	//!	Prints to standard output contents of TIFF directory
	void Print();
	
	//	Returns true if tiff data is compressed
	bool IsCompressed() { return compress_ != COMPRESSION_NONE; }

public:

	//! Construtor from parameters 
	TeDecoderTIFF( const TeRasterParams& );

	//!	Normal destructor
	~TeDecoderTIFF();

	//! Initializes raster structure
	void init();

	//! Clear internal parameters
	bool clear();

	TeCoord2D index2Coord (TeCoord2D& pt);
	TeCoord2D coord2Index (TeCoord2D& pt);

	bool setElement (int col,int lin, double val, int band=0);
	bool getElement (int col,int lin, double &val,int band=0);
};

//! Implements a tiff decoder factory
class TeDecoderTIFFFactory : public TeDecoderFactory
{
public:

	TeDecoderTIFFFactory(const string& name) : TeDecoderFactory(name) {}

	virtual TeDecoder* build (const TeRasterParams& arg)
	{  return new TeDecoderTIFF(arg); }
};

#endif
