/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TePostgreSQL.h"
#include "TePGUtils.h"
#include <sys/types.h>
#include <sys/stat.h>







#include "TeSpatialOperations.h"
#include "TeGeometryAlgorithms.h"
#include <string>
using namespace std;


//----- TePostgreSQL methods ---
TePostgreSQL::TePostgreSQL()
{
	errorMessage_ = "";
	dbmsName_ = "PostgreSQL";
}

TePostgreSQL::~TePostgreSQL()
{
	if(isConnected_)
		close();
}

string TePostgreSQL::errorMessage()
{
	if(errorMessage_.empty())
		return tepg_connection_.err_msg();
	else
		return errorMessage_;
	
}

bool TePostgreSQL::newDatabase(const string& database, const string& user, const string& password, const string& host, const int& port, bool terralibModel)
{
	errorMessage_ = "";

	string createDB  = "CREATE DATABASE ";
	       createDB += database;

	//if(!realConnect(host, user, password, "template1", port))
	//	return false;	

	if(!connect(host, user, password, "template1", port))
		return false;

	
	bool connected = false;
	
	if(execute(createDB))
		connected = connect(host, user, password, database, port);
	else
		return false;

	if(terralibModel)
	{
		//create conceptual model
		if(connected)
		{
			if(!createConceptualModel())
				return false;
		}
		else
			return false;
	}

	return true;
}


bool TePostgreSQL::connect(const string& host, const string& user, const string& password, const string& database, int port)
{
	if(TePostgreSQL::realConnect(host, user, password, database, port))
	{
		// See if PostGIS is present
		TePGRecordset rec;

		string sql = "SELECT postgis_version()";

		if(rec.open(sql.c_str(), &tepg_connection_) && rec.recordCount() > 0)
		{
			rec.close();

			this->close();

			errorMessage_ = "PostGIS extension found! Please use a PostGIS driver!";

			return false;	
		}

		return true;
	}
	
	return false;
}

bool TePostgreSQL::realConnect(const string& host, const string& user, const string& password, const string& database, int port)
{
	errorMessage_ = "";

	host_             = (host == "")     ? (string("")) : (" host = " + host + " ");
	user_             = (user == "")     ? (string("")) : (" user = " + user + " ");
	password_         = (password == "") ? (string("")) : (" password = " + password + " ");
	database_         = (database == "") ? (string("")) : (" dbname = " + database + " ");
	string portNumber = (port == 0)      ? (string("")) : (" port = " + Te2String(port)  + " ");

	portNumber_ = port;

	string connInfo = host_ + user_ + password_ + database_ + portNumber;

	this->close();

	if(!tepg_connection_.open(connInfo))
	{
		errorMessage_ = "Could not connect to the database server!";
		return false;
	}

	isConnected_ = true;
	
	host_     = host;
	user_     = user;
	password_ = password;
	database_ = database;	

	return true;
}

bool TePostgreSQL::showDatabases(const string& host, const string& user, const string& password, vector<string>& dbNames, int port)
{
	errorMessage_ = "";

	if(realConnect(host, user, password, "template1", port))
	{
		string sql = "SELECT datname FROM pg_database WHERE datname NOT IN ('template0', 'template1') ORDER BY datname";

		TePostgreSQLPortal p(this);

		if(p.query(sql) && p.fetchRow())
		{
			do
			{
				dbNames.push_back(p.getData("datname"));
			}while(p.fetchRow());

			return true;
		}
		else
			errorMessage_ = "Didn't find any database!";
	}
	
	return false;
}

void TePostgreSQL::close(void)
{
	clear();

	errorMessage_ = "";

	tepg_connection_.close();
	isConnected_ = false;	

	return;
}

bool TePostgreSQL::listTables(vector<string>& tableList)
{
	errorMessage_ = "";

	TePGRecordset rec;

	string sql = "SELECT tablename FROM pg_table WHERE schemaname = 'public' ORDER BY tablename";

	if(!rec.open(sql, &tepg_connection_) || rec.recordCount() <= 0)
		return false;

	while(!rec.eof())
	{
		tableList.push_back(rec.getData(0));
		rec.moveNext();
	}
	
	return true;

}

bool TePostgreSQL::tableExist(const string& table)
{
	errorMessage_ = "";

	TePGRecordset rec;

	string sql = "SELECT relname FROM pg_class WHERE lower(relname) = lower('" + table + "')";

	if(!rec.open(sql, &tepg_connection_))
		return false;

	return (rec.recordCount() > 0);
}

bool TePostgreSQL::columnExist(const string& table, const string& column, TeAttribute& attr)
{
	errorMessage_ = "";

	TePGRecordset rec;

	string sql  = "SELECT atttypid, attname, atttypmod, attlen  FROM pg_class, pg_attribute ";
	       sql += "WHERE lower(pg_class.relname) = lower('" + table + "') AND ";
		   sql += "      pg_class.oid = pg_attribute.attrelid AND ";
		   sql += "      pg_attribute.attname = lower('" + column + "')";

	if(!rec.open(sql, &tepg_connection_))
		return false;

	if(rec.recordCount() > 0)
	{
		attr.rep_.name_ = rec.getData("attname");

		Oid nType = static_cast<Oid>(rec.getInt("atttypid"));
		switch(nType)
		{
			case 16:    //BOOL
			case 20:    //INT8
			case 21:    //INT2
			case 23:    //INT4
			case 26:    //OID			
						attr.rep_.type_ = TeINT;
						attr.rep_.numChar_ = 15;
						break;

			case 700 :  //float4  -> float(p)
			case 701 :  //float8  -> float(p)
			case 790 :  //money   -> decimal(9, 2)  
			case 1700:  //numeric -> numeric(p, s)  
						attr.rep_.type_ = TeREAL;
						attr.rep_.numChar_ = 15;
						break;

			case 1082:	//date -> date
			case 1083:  //time -> time
			case 1114:  //timestamp
			case 1186:  //interval
			case 1266:	//
						attr.rep_.type_ = TeDATETIME;
						attr.rep_.numChar_ = rec.getInt("attlen");
						break;

			//case 26:    //OID
			case 17:	//bytea
						attr.rep_.type_ = TeBLOB;
						attr.rep_.numChar_ = 0;
						break;
			case 1042:
						attr.rep_.type_ = TeCHARACTER;
						attr.rep_.numChar_ = rec.getInt("atttypmod") - 4;

			case 1043:  //varchar(n)
			case 25:    //text
						attr.rep_.type_ = TeSTRING;
						attr.rep_.numChar_ = rec.getInt("atttypmod") - 4;
						break;

			default:
						attr.rep_.type_ = TeUNKNOWN;
						attr.rep_.numChar_ = 0;
						break;
		}

		return true;
	}

	return false;
}

bool TePostgreSQL::createTable(const string& table, TeAttributeList &attr)
{
	errorMessage_ = "";

	bool first = true;

	TeAttributeList::iterator it = attr.begin();
	
	string createTable ="CREATE TABLE " + table +" (";
	
	string type;
	char	size[8];

	string pkeys;

	while(it != attr.end())
	{
		switch ((*it).rep_.type_)
		{
			case TeSTRING:		type = "VARCHAR ";
								sprintf (size, "(%d)", (*it).rep_.numChar_);
								type += string(size);
								break;

			case TeREAL:		type = "FLOAT8";
								break;

			case TeINT:			type = ((*it).rep_.isAutoNumber_) ? "SERIAL" : "INT";
								break;

			case TeBLOB:		//type = "OID";
				                type = "BYTEA";
								break;

			case TeDATETIME:	type = "TIMESTAMP(0)";
								break;

			case TeCHARACTER:	type = "CHAR ";
								sprintf (size, "(%d)", (*it).rep_.numChar_);
								type += string (size);
								break;

			default:			type = "VARCHAR ";
								sprintf (size, "(%d)", (*it).rep_.numChar_);
								type += string (size);
								break;
		}
		
		if(!first)
			createTable += ",  ";
		else
			first = false;

		createTable += (*it).rep_.name_ + " ";
		createTable += type;

		// check if column is part of primary key
		if((*it).rep_.isPrimaryKey_ && (*it).rep_.type_ != TeBLOB )
		{
			if(!pkeys.empty())
				pkeys += ", ";
			
			pkeys += (*it).rep_.name_;
		}
		else
			createTable += " NULL ";

		++it;
	}

	if(!pkeys.empty())
	{	string pk = ", PRIMARY KEY(";
	           pk += pkeys;
			   pk += ")";

		createTable += pk;
	}


	createTable += ");";

	return execute(createTable);
}

bool TePostgreSQL::addColumn(const string& table, TeAttributeRep &rep)
{
	errorMessage_ = "";
	
	string field = TeGetExtension(rep.name_.c_str());
	if(field.empty())
		field = rep.name_;

	string new_column  = "ALTER TABLE " + table + " ADD COLUMN " + field + " ";

	switch(rep.type_)
	{
		case TeSTRING:	new_column += "VARCHAR(" + Te2String(rep.numChar_) + ")";
						break;
		case TeREAL:	new_column += "FLOAT8";
						break;
		case TeINT:		new_column += "INTEGER";
						break;
		case TeBLOB:	//new_column += "OID";
						new_column += "BYTEA";
						break;
		default:		new_column += "VARCHAR";
						break;
	}

	return this->execute(new_column);
}

bool TePostgreSQL::createRelation(const string& relName, const string& table, const string& fieldName, const string& relatedTable, const string& relatedField, bool cascadeDeletion)
{
	errorMessage_ = "";

	string alter  = "ALTER TABLE " +  table + " ADD CONSTRAINT " + relName + " ";
	       alter += "FOREIGN KEY (" + fieldName + ") "; 
	       alter += "REFERENCES " + relatedTable + "(" + relatedField + ") ";

	if(cascadeDeletion)
		alter += " ON DELETE CASCADE";

	return execute(alter);
}

TeDBRelationType TePostgreSQL::existRelation(const string& /*tableName*/, const string& relName)
{
	errorMessage_ = "";

	string sql  = "SELECT proname FROM pg_trigger, pg_proc WHERE tgconstrname = '" + relName + "' AND pg_trigger.tgfoid = pg_proc.oid AND pg_proc.proname LIKE 'RI_FKey_cascade_del%'";
	TeDBRelationType resp = TeNoRelation;

	TePGRecordset r(sql, &tepg_connection_);
	if(r.recordCount() > 0)
		resp = TeRICascadeDeletion; 
	else
	{
		sql  = "SELECT proname FROM pg_trigger, pg_proc WHERE tgconstrname = '" + relName + "' AND pg_trigger.tgfoid = pg_proc.oid AND pg_proc.proname LIKE 'RI_FKey_check_ins%'";
		r.open(sql, &tepg_connection_);
		if(r.recordCount() > 0)
			resp = TeRINoCascadeDeletion; 
	}

	r.close();
	
	return resp;
}

bool TePostgreSQL::execute(const string &sql)
{
	try
	{
		//TeWriteToFile("EXECS.SQL", sql + "\n", "aw");

		this->tepg_connection_.exec_cmd(sql);
		
		
	}

	catch(...)
	{
		return false;
	}

	return true;
}

TeDatabasePortal* TePostgreSQL::getPortal()
{
	errorMessage_ = "";

	TeDatabasePortal *portal = new TePostgreSQLPortal(this);

	return portal;
}

bool TePostgreSQL::createDatabaseTable()
{
	string create  = "CREATE TABLE te_database ";
           create += "(";
           create += "  db_version		VARCHAR(50) NOT NULL,";
           create += "  db_creation		TIMESTAMP   NULL,";
           create += "  PRIMARY KEY (db_version)";
           create += ")";

	if(!this->execute(create))
	{
		errorMessage_ = "Erro na criao da tabela database! Message: " + string(this->tepg_connection_.err_msg());
		return false;
	}

	return true;
}

bool TePostgreSQL::createProjectionTable()
{
	errorMessage_ = "";

	string create  = "CREATE TABLE te_projection ";
	       create += "(";
		   create += " projection_id    SERIAL,";
		   create += " name				VARCHAR(255) NOT NULL,";
		   create += " long0			FLOAT8       NOT NULL,";
		   create += " lat0				FLOAT8       NOT NULL,";
		   create += " offx				FLOAT8       NOT NULL,";
		   create += " offy				FLOAT8       NOT NULL,";
		   create += " stlat1			FLOAT8       NOT NULL,";
		   create += " stlat2			FLOAT8       NOT NULL,";
		   create += " unit				VARCHAR(50)  NOT NULL,";
		   create += " scale			FLOAT8       NOT NULL,";
		   create += " hemis			INTEGER      NOT NULL,";
		   create += " datum			VARCHAR(255) NOT NULL,";
		   create += " radius			FLOAT8       NOT NULL,";
		   create += " flattening		FLOAT8       NOT NULL,";
		   create += " dx				FLOAT8       NOT NULL,";
		   create += " dy				FLOAT8       NOT NULL,";
		   create += " dz				FLOAT8       NOT NULL,";
		   create += " PRIMARY KEY (projection_id)";
		   create += ")";

	return this->execute(create);
}

bool TePostgreSQL::createLayerTable()
{
	errorMessage_ = "";

	string create  = "CREATE TABLE te_layer ";
	       create += "(";
		   create += " layer_id       SERIAL,";
		   create += " projection_id  INTEGER       NOT NULL,";
		   create += " name           VARCHAR(255)  NOT NULL,";
		   create += " lower_x        FLOAT8        NULL,";
		   create += " lower_y        FLOAT8        NULL,";
		   create += " upper_x        FLOAT8		NULL,";
		   create += " upper_y        FLOAT8		NULL,";
		   create += " initial_time   TIMESTAMP(0)  NULL,";
		   create += " final_time     TIMESTAMP(0)  NULL,";
		   create += " PRIMARY KEY (layer_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	return createIndex("te_layer", "layer_projection_idx", "projection_id");
}

bool TePostgreSQL::createRepresentationTable()
{
	errorMessage_ = "";

	string create  = "CREATE TABLE te_representation ";
		   create += "(";
		   create += " repres_id         SERIAL,";
	       create += " layer_id          INTEGER      NOT NULL,";
		   create += " geom_type         INTEGER      NOT NULL,";
		   create += " geom_table        VARCHAR(255) NOT NULL,";
		   create += " description       VARCHAR(255) NULL,";
		   create += " lower_x           FLOAT8       NOT NULL,";
		   create += " lower_y           FLOAT8       NOT NULL,";
		   create += " upper_x           FLOAT8       NOT NULL,";
   		   create += " upper_y           FLOAT8       NOT NULL,";
		   create += " res_x             FLOAT8       NULL,";
		   create += " res_y             FLOAT8       NULL,";
		   create += " num_cols          INTEGER      NULL ,";
		   create += " num_rows          INTEGER      NULL,";
		   create += " initial_time      TIMESTAMP(0) NULL,";
		   create += " final_time        TIMESTAMP(0) NULL,";		   
		   create += " PRIMARY KEY (repres_id)";
		   create += ")";

	if(!execute(create))
		return false;

	return createIndex("te_representation", "repres_layer_geom_type_idx", "layer_id, geom_type");
}

bool TePostgreSQL::createViewTable()
{
	errorMessage_ = "";

	string create  = "CREATE TABLE te_view ";
	       create += "(";
	       create += " view_id			SERIAL,";
	       create += " projection_id    INTEGER      NULL,";
		   create += " name				VARCHAR(255) NOT NULL,";
		   create += " user_name        VARCHAR(255) NULL,"; 
		   create += " visibility       INTEGER      NULL,";
		   create += " PRIMARY KEY (view_id)";
		   create += ")";

    if(!this->execute(create))
		return false;

	return createIndex("te_view", "view_index", "name, user_name");
}

bool TePostgreSQL::createThemeTable()
{
	errorMessage_ = "";

	string create  = "CREATE TABLE te_theme ";
	       create += "(";
		   create += " theme_id                 SERIAL,";
		   create += " layer_id                 INTEGER       NULL,";         
		   create += " view_id                  INTEGER       NOT NULL,";
		   create += " name                     VARCHAR(255)  NOT NULL,";
		   create += " parent_id                INTEGER       NULL,";
		   create += " priority                 INTEGER       NULL,";
		   create += " node_type                INTEGER       NULL,";
		   create += " min_scale                FLOAT8        NULL,";
		   create += " max_scale                FLOAT8        NULL,";
		   create += " generate_attribute_where VARCHAR(500)  NULL,";
		   create += " generate_spatial_where   VARCHAR(500)  NULL,";
		   create += " generate_temporal_where  VARCHAR(255)  NULL,";
           create += " collection_table         VARCHAR(255)  NULL,";
		   create += " visible_rep				INTEGER       NULL,";
		   create += " enable_visibility		INTEGER       NULL,";
		   create += " PRIMARY KEY (theme_id)";
		   create += ")";

	if(!execute(create))
		return false;

	if(!createIndex("te_theme", "theme_layer_idx", "layer_id"))
		return false;

	return createIndex("te_theme", "theme_view_idx", "view_id");
}

bool TePostgreSQL::createGroupingTable()
{
    errorMessage_ = "";

	string create  = "CREATE TABLE te_grouping ";
	       create += "(";
		   create += " theme_id                 INTEGER       NOT NULL,";
		   create += " grouping_number          INTEGER       NULL,";         
		   create += " grouping_attr            VARCHAR(255)  NULL,";
		   create += " grouping_attr_type       INTEGER       NULL,";
		   create += " grouping_mode            INTEGER       NULL,";
		   create += " grouping_norm_attr       VARCHAR(255)  NULL,";
		   create += " grouping_std_dev         FLOAT8        DEFAULT 0.0,";
		   create += " grouping_precision       INTEGER       NULL,";
		   create += " grouping_function		VARCHAR(50)   NULL,";
		   create += " PRIMARY KEY (theme_id)";
		   create += ")";

	return execute(create);
}

bool TePostgreSQL::createThemeTablesTable()
{
	errorMessage_ = "";

	string create  = "CREATE TABLE te_theme_table ";
	       create += "(";
	       create += " theme_table_id	SERIAL,";
	       create += " theme_id			INTEGER	 NOT NULL,";
		   create += " table_id			INTEGER	 NOT NULL,";
		   create += " relation_id      INTEGER  NULL,"; 
		   create += " table_order      INTEGER  NOT NULL,";
		   create += " PRIMARY KEY (theme_table_id)";
		   create += ")";

    if(!this->execute(create))
		return false;

	//creates indexes
	if(!createIndex("te_theme_table", "te_theme_table_theme_idx", "theme_id"))
		return false;

	if(!createIndex("te_theme_table", "te_theme_table_table_idx", "table_id"))
		return false;

	return createIndex("te_theme_table", "te_theme_table_relation_idx", "relation_id");
}

bool TePostgreSQL::createLegendTable()
{
	errorMessage_ = "";

	string create  = "CREATE TABLE te_legend ";
	       create += "(";
		   create += " legend_id		SERIAL,";
		   create += " theme_id         INT          NOT NULL,";
		   create += " group_id         INT          NULL,";
		   create += " num_objs         INT          NULL,";
		   create += " lower_value      VARCHAR(255) NULL,";
		   create += " upper_value      VARCHAR(255) NULL,";
		   create += " label			VARCHAR(255) NULL,";
		   create += " PRIMARY KEY (legend_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	return createIndex("te_legend", "te_legend_theme_idx", "theme_id");
}

bool TePostgreSQL::createVisualTable()
{
	errorMessage_ = "";

	string create  = "CREATE TABLE te_visual";
	       create += "(";
		   create += " legend_id		INT          NOT NULL,";
		   create += " geom_type		INT          NOT NULL,";
		   create += " lib_name         VARCHAR(255) NULL,";
		   create += " symb_id          INTEGER      NULL,";
		   create += " red              INT          NULL,";
		   create += " green            INT          NULL,";
		   create += " blue             INT          NULL,";
		   create += " transparency     INT          NULL,";
		   create += " width            INT          NULL,";
		   create += " contour_lib_name VARCHAR(255) NULL,";
		   create += " contour_symb_id  INTEGER      NULL,";
		   create += " contour_red      INT          NULL,";
		   create += " contour_green    INT          NULL,";
		   create += " contour_blue     INT          NULL,";
		   create += " contour_transp   INT          NULL,";
		   create += " contour_width    INT          NULL,";
		   create += " size_value       INT          NULL,";
		   create += " pt_angle         INT          NULL,";
		   create += " family           VARCHAR(255) NULL,";
		   create += " bold             INT          NULL,";
		   create += " italic           INT          NULL,";
		   create += " alignment_vert   FLOAT8       NULL,";
		   create += " alignment_horiz  FLOAT8       NULL,";
		   create += " tab_size         INT          NULL,";
		   create += " line_space       INT          NULL,";
		   create += " fixed_size       INT          NULL,";		   
		   create += " PRIMARY KEY (legend_id, geom_type)";
		   create += ")";

	return this->execute(create);
}

bool TePostgreSQL::createVisualRasterTable()
{
	string create  = "CREATE TABLE te_visual_raster (";
	       create += " theme_id		INT NOT NULL,";
	       create += " band_in		INT NOT NULL,";
	       create += " band_out		INT,";
	       create += " transf_type	INT,";
	       create += " param1		FLOAT8,";
	       create += " param2		FLOAT8,";
	       create += " lut_table	VARCHAR(255),";
	       create += " PRIMARY KEY (theme_id, band_in))";

	return execute(create);
}

bool TePostgreSQL::createLayerTableTable()
{
	errorMessage_ = "";

	string create  = "CREATE TABLE te_layer_table ";
	       create += "(";
		   create += " table_id          SERIAL,";
		   create += " layer_id			 INT		   NULL,";
		   create += " attr_table		 VARCHAR(255)  NOT NULL,";
		   create += " unique_id         VARCHAR(255)  NULL,";
		   create += " attr_link		 VARCHAR(255)  NULL,";
		   create += " attr_initial_time VARCHAR(255)  NULL,";
		   create += " attr_final_time   VARCHAR(255)  NULL,";
		   create += " attr_time_unit    INT           NULL,";
		   create += " attr_table_type   INT		   NULL,";
		   create += " user_name         VARCHAR(255)  NULL,";
		   create += " initial_time      TIMESTAMP(0)  NULL,";
		   create += " final_time        TIMESTAMP(0)  NULL,";
		   create += " PRIMARY KEY (table_id)";
		   create += ")";

	if(!execute(create))
		return false;

	return createIndex("te_layer_table", "te_layer_table_layer_idx", "layer_id");
}

bool TePostgreSQL::createTablesRelationTable()
{
	errorMessage_ = "";

	string create  = "CREATE TABLE te_tables_relation ";
	       create += "(";
		   create += " relation_id		    SERIAL,";
		   create += " related_table_id     INTEGER       NOT NULL,";
		   create += " related_attr		    VARCHAR(255)  NOT NULL,";
		   create += " external_table_name  VARCHAR(255)  NOT NULL,";
		   create += " external_attr        VARCHAR(255)  NOT NULL,";
		   create += " PRIMARY KEY (relation_id)";
		   create += ")";

	return this->execute(create);
}

bool TePostgreSQL::createPolygonGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id		SERIAL,";
		   create += " object_id    VARCHAR(255) NULL,";
		   create += " num_coords   INTEGER      NOT NULL,";
		   create += " num_holes    INTEGER      NOT NULL,";
		   create += " parent_id    INTEGER      NOT NULL,";
		   create += " lower_x      FLOAT8       NOT NULL,";
		   create += " lower_y      FLOAT8       NOT NULL,";
		   create += " upper_x      FLOAT8       NOT NULL,";
		   create += " upper_y      FLOAT8       NOT NULL,";
		   create += " ext_max      FLOAT8       NOT NULL,";
		   create += " spatial_data POLYGON,";
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	//creates indexes
	if(!createIndex(tableName, tableName + "_idx_obj_id", "object_id"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_ext_max", "ext_max"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_box", "lower_x, lower_y, upper_x, upper_y"))
		return false;

	return true;
}

bool TePostgreSQL::createLineGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id		SERIAL,";
		   create += " object_id    VARCHAR(255) NULL,";
		   create += " num_coords   INTEGER      NOT NULL,";
		   create += " lower_x      FLOAT8       NOT NULL,";
		   create += " lower_y      FLOAT8       NOT NULL,";
		   create += " upper_x      FLOAT8       NOT NULL,";
		   create += " upper_y      FLOAT8       NOT NULL,";
		   create += " ext_max      FLOAT8       NOT NULL,";
		   create += " spatial_data POLYGON,";
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	if(!createIndex(tableName, tableName + "_idx_obj_id", "object_id"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_ext_max", "ext_max"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_box", "lower_x, lower_y, upper_x, upper_y"))
		return false;

	return true;
}

bool TePostgreSQL::createPointGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id		SERIAL,";
		   create += " object_id	VARCHAR(255)  NULL,";
		   create += " x		    FLOAT8         DEFAULT 0.0,";
		   create += " y            FLOAT8        DEFAULT 0.0,";		   
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";


	if(!this->execute(create))
		return false;
	
	if(!createIndex(tableName, tableName + "_idx_obj_id", "object_id"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_xy", "x, y"))
		return false;

	return true;
}

bool TePostgreSQL::createCellGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id		SERIAL,";
		   create += " object_id    VARCHAR(255) NULL,";
		   create += " lower_x      FLOAT8       NOT NULL,";
		   create += " lower_y      FLOAT8       NOT NULL,";
		   create += " upper_x      FLOAT8       NOT NULL,";
		   create += " upper_y      FLOAT8       NOT NULL,";
	       create += " col_number   INTEGER      NOT NULL,";
	       create += " row_number	INTEGER      NOT NULL,";
	       create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	if(!createIndex(tableName, tableName + "_idx_obj_id", "object_id"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_box", "lower_x, lower_y, upper_x, upper_y"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_rc", "row_number, col_number"))
		return false;

	return true;
}

bool TePostgreSQL::createTextGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id		   SERIAL,";
		   create += " object_id       VARCHAR(255)  NULL,";
		   create += " x			   FLOAT8        DEFAULT 0.0,";
		   create += " y			   FLOAT8        DEFAULT 0.0,";
		   create += " text_value      VARCHAR(255)  NULL,";
		   create += " height		   FLOAT8        DEFAULT 0.0,";
		   create += " angle		   FLOAT8        DEFAULT 0.0,";
		   create += " alignment_vert  FLOAT8		 NULL,";
		   create += " alignment_horiz FLOAT8		 NULL,";		   
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	//creates indexes
	if(!createIndex(tableName, tableName + "_idx_obj", "object_id"))
		return false;

	return createIndex(tableName, tableName + "_idx_xy", "x, y");
}

bool TePostgreSQL::createArcGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id    SERIAL,";
		   create += " object_id  VARCHAR(255) NULL,";
		   create += " from_node  INT          NOT NULL,";
		   create += " to_node    INT          NOT NULL,";
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	//creates indexes
	if(!createIndex(tableName, tableName + "obj_idx", "object_id"))
		return false;

	if(!createIndex(tableName, tableName + "fn_idx", "from_node"))
		return false;

	return createIndex(tableName, tableName + "tn_idx", "to_node");
}

bool TePostgreSQL::createNodeGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id    SERIAL,";
		   create += " object_id  VARCHAR(255) NULL,";
		   create += " x		  FLOAT8       DEFAULT 0.0,";
		   create += " y		  FLOAT8       DEFAULT 0.0,";
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	if(!createIndex(tableName, tableName + "_idx_obj_id", "object_id"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_xy", "x, y"))
		return false;

	return true;	
}

bool TePostgreSQL::createRasterGeometry(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " geom_id           SERIAL,";
		   create += " object_id         VARCHAR(255) NULL,";
		   create += " raster_table      VARCHAR(255) NULL,";
		   create += " lut_table		 VARCHAR(255) NULL,";		   
		   create += " res_x             FLOAT8       DEFAULT 0.0,";
		   create += " res_y		     FLOAT8       DEFAULT 0.0,";		   
		   create += " num_bands         INTEGER      NULL,";
		   create += " num_cols          INTEGER      NULL,";
		   create += " num_rows          INTEGER      NULL,";
		   create += " block_height      INTEGER      NULL,";
		   create += " block_width       INTEGER      NULL,";
		   create += " lower_x           FLOAT8       NOT NULL,";
		   create += " lower_y           FLOAT8       NOT NULL,";
		   create += " upper_x           FLOAT8       NOT NULL,";
		   create += " upper_y           FLOAT8       NOT NULL,";
		   create += " tiling_type       INTEGER, ";
		   create += " PRIMARY KEY (geom_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	//creates indexes
	return createIndex(tableName, tableName + "_idx_obj", "object_id");
}

bool TePostgreSQL::createRasterMetadataTable(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName;
	       create += "(";
		   create += " geom_id          INTEGER       NOT NULL,";
		   create += " band_id          INTEGER       NOT NULL,";
		   create += " min_value        FLOAT8        DEFAULT 0.0,";
		   create += " max_value        FLOAT8        DEFAULT 0.0,";
		   create += " num_bits         INTEGER       NULL,";
		   create += " data_type        INTEGER       NULL,";
		   create += " photometric_type INTEGER       NULL,";
		   create += " compression_type INTEGER       NULL,";
		   create += " dummy            FLOAT8        DEFAULT 0.0,";
		   create += " PRIMARY KEY (geom_id, band_id)";
		   create += ")";

	return this->execute(create);
}

bool TePostgreSQL::createRasterTable(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " block_id          VARCHAR(50) NOT NULL,";
		   create += " lower_x           FLOAT8      NOT NULL,";
	       create += " lower_y           FLOAT8      NOT NULL,";
	       create += " upper_x           FLOAT8      NOT NULL,";
	       create += " upper_y           FLOAT8      NOT NULL,";
		   create += " band_id		     INTEGER     NOT NULL,";		   
		   create += " resolution_factor INTEGER     NOT NULL,";
		   create += " subband		     INTEGER     NOT NULL,";		   
		   //create += " spatial_data      OID,";
		   create += " spatial_data      BYTEA,";
		   create += " block_size        INTEGER  NOT NULL,";
		   create += " PRIMARY KEY (block_id)";
		   create += ")";


	if(!execute(create))
		return false;

	//creates indexes
	if(!createIndex(tableName, tableName + "_idx_band", "band_id"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_subband", "subband"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_resfactor", "resolution_factor"))
		return false;

	if(!createIndex(tableName, tableName + "_idx_box", "lower_x, lower_y, upper_x, upper_y"))
		return false;

	return true;
}

bool TePostgreSQL::createCollectionTable(const string& tableName)
{
	errorMessage_ = "";

	string create  = "CREATE TABLE " + tableName + " ";
	       create += "(";
		   create += " c_object_id		VARCHAR(255) NOT NULL,";
		   create += " c_legend_id		INT          NULL,";
		   create += " label_x			FLOAT8       NULL,";
		   create += " label_y			FLOAT8       NULL,";
           create += " c_legend_own     INT          NULL,";
		   create += " c_object_status  INT          NULL,";
		   create += " PRIMARY KEY (c_object_id)";
		   create += ")";

	if(!this->execute(create))
		return false;

	//creates indexes

	return createIndex(tableName, tableName + "_idx_legend", "c_legend_id");
}

bool TePostgreSQL::insertRelationInfo(const int tableId, const string& tField, const string& rTable, const string& rField, int& relId)
{
	// check if relation already exists
	TePostgreSQLPortal* portal = (TePostgreSQLPortal*)this->getPortal();

	if(!portal)
		return false;

	relId = -1;
	
	string sel = "SELECT relation_id FROM te_tables_relation WHERE";
	sel += " related_table_id = " + Te2String(tableId);
	sel += " AND related_attr = '" + tField + "'";
	sel += " AND external_table_name = '" + rTable + "'";
	sel += " AND external_attr = '" + rField + "'";

	if(!portal->query(sel))
	{
		delete portal;
		return false;
	}

	if(portal->fetchRow())
	{
		relId = atoi(portal->getData("relation_id"));
		delete portal;
		return true;
	}

	delete portal;

	string sql =  "INSERT INTO te_tables_relation (related_table_id, related_attr, external_table_name, external_attr) VALUES(";
	       sql += Te2String(tableId);
		   sql += ", '";
		   sql += tField;
		   sql += "', '";
		   sql += rTable;
		   sql += "', '";
		   sql += rField;
		   sql += "')";

	if(this->execute(sql))
	{
		sql = "SELECT currval('te_tables_relation_relation_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);

		if(rec.recordCount() > 0)
			relId = rec.getInt(0);

		rec.close();
	}
	else
		return false;

	return true;
}

bool TePostgreSQL::insertTableInfo(int layerId, TeTable& table, const string& user)
{
	errorMessage_ = "";

	if(table.name().empty())
		return false;

	if(table.id() <= 0)
	{
		string sql  = "INSERT INTO te_layer_table (";
		
		if(layerId > 0)
		{
			sql += "layer_id, ";
		}

		sql += "attr_table, unique_id, attr_link, attr_initial_time, attr_final_time, ";
		sql += "attr_time_unit, attr_table_type, user_name) VALUES(";
		
		if(layerId > 0)
		{
			sql += Te2String(layerId) + ", ";
		}	
		
		sql += "'" + table.name();
		sql += "', '" + table.uniqueName();		
		sql += "', '";
		sql += table.linkName();
		sql += "', '";
		sql += table.attInitialTime();
		sql += "', '";
		sql += table.attFinalTime();
		sql += "', ";
		sql += Te2String(table.attTimeUnit());
		sql += ", ";
		sql += Te2String(table.tableType());
		sql += ", '";
		sql += user;
		sql += "')";

		if(this->execute(sql))
		{
			sql = "SELECT currval('te_layer_table_table_id_seq')";
			TePGRecordset rec;
			rec.open(sql, &tepg_connection_);

			if(rec.recordCount() > 0)
				table.setId(rec.getInt(0));

			rec.close();
		}
		else
			return false;
	}
	
	return true;	
}

bool TePostgreSQL::insertTable(TeTable &table)
{
	string tableName = table.name();
	TeAttributeList att = table.attributeList();
	TeAttributeList::iterator it = att.begin();

	TeTableRow row;
	unsigned int i;
	unsigned int j;
	for ( i = 0; i < table.size(); i++  )
	{
		row = table[i];
		it = att.begin();
		string q = 	"INSERT INTO "+tableName+" values(";
		j = 1;
		int jj = 0;
		string dt = "";
		while ( it != att.end() )
		{
			if(((*it).rep_.type_==TeDATETIME) && (!row[jj].empty()))
			{
				const string temp_dt = string(row[jj].c_str());
				TeTime t(temp_dt, (*it).dateChronon_, (*it).dateTimeFormat_, (*it).dateSeparator_, (*it).timeSeparator_, (*it).indicatorPM_);
				dt=t.getDateTime("YYYYsMMsDDsHHsmmsSS", "-");
			}

  			switch ((*it).rep_.type_)
  			{
  				case TeSTRING:
					q += "'"+this->escapeSequence(row[jj])+"'";
  				break;
  				case TeREAL:
					q += "'"+row[jj]+"'";
  				break;
  				case TeINT:
					q += "'"+row[jj]+"'";
  				break;
				case TeDATETIME:
					q += "'"+dt+"'";   
  				break;
				default:
					q += "'"+this->escapeSequence(row[jj])+"'";
  				break;
  			}
			if (j<att.size())
				q+= ",";
			++it;
			j++;
			jj++;
		}
		q += ")";
		if (!this->execute(q))
//			return false;
			continue;
	}
	return true;
}

bool TePostgreSQL::alterTable(const string& tableName, TeAttributeRep& rep, const string& oldColName)
{
	if(!tableExist(tableName))
		return false;

	string tab;

	if(!oldColName.empty())
	{
		tab  = " ALTER TABLE " + tableName + " RENAME COLUMN ";
		tab += oldColName +  " TO " + rep.name_;
	}
	else
	{
		tab  = " ALTER TABLE " + tableName + " ALTER COLUMN ";
		tab += rep.name_ + " TYPE ";

		switch (rep.type_)
		{
			case TeSTRING:
				tab += "VARCHAR(" + Te2String(rep.numChar_) + ") ";
				break;
				
			case TeREAL:
				tab += "FLOAT8";	
				break;
				
			case TeINT:
				tab += "INTEGER";
				break;

			case TeDATETIME:
				tab += "TIMESTAMP(0)";
				break;

			case TeCHARACTER:
				tab += "CHAR";
				break;

			case TeBLOB:
				tab += "BYTEA";
				break; 
			
			default:
				tab += "VARCHAR(" + Te2String(rep.numChar_) + ") ";
				break;
		}	
	}

	

	if(!execute(tab))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error alter table " + tableName + " !";

		return false;
	}

	string tableId;
	TeDatabasePortal* portal = getPortal();
	string sql = "SELECT table_id FROM te_layer_table WHERE attr_table = '" + tableName + "'";
	if(portal->query(sql) && portal->fetchRow())
		tableId = portal->getData(0);

	delete portal;

	if(tableId.empty() == false)
	{
		if(oldColName.empty() == false) // column name changed
		{
			 // update relation
			sql = "UPDATE te_tables_relation SET related_attr = '" + rep.name_ + "'";
			sql += " WHERE related_table_id = " + tableId;
			sql += " AND related_attr = '" + oldColName + "'";
			if(execute(sql) == false)
				return false;

			sql = "UPDATE te_tables_relation SET external_attr = '" + rep.name_ + "'";
			sql += " WHERE external_table_name = '" + tableName + "'";
			sql += " AND external_attr = '" + oldColName + "'";
			if(execute(sql) == false)
				return false;

			 // update grouping
			sql = "UPDATE te_grouping SET grouping_attr = '" + tableName + "." + rep.name_ + "'";
			sql += " WHERE grouping_attr = '" + tableName + "." + oldColName + "'";
			if(execute(sql) == false)
				return false;
		}
		else // column type changed
		{
			// delete relation
			sql = "DELETE FROM te_tables_relation WHERE (related_table_id = " + tableId;
			sql += " AND related_attr = '" + rep.name_ + "')";
			sql += " OR (external_table_name = '" + tableName + "'";
			sql += " AND external_attr = '" + rep.name_ + "')";
			if(execute(sql) == false)
				return false;

			// delete grouping
			TeDatabasePortal* portal = getPortal();
			sql = "SELECT theme_id FROM te_grouping WHERE grouping_attr = '" + tableName + "." + oldColName + "'";
			if(portal->query(sql) && portal->fetchRow())
			{
				string themeId = portal->getData(0);

				sql = "DELETE FROM te_legend WHERE theme_id = " + themeId + " AND group_id >= 0";
				if(execute(sql) == false)
				{
					delete portal;
					return false;
				}
			}
			delete portal;

			sql = "DELETE FROM te_grouping";
			sql += " WHERE grouping_attr = '" + tableName + "." + oldColName + "'";
			if(execute(sql) == false)
				return false;
		}
	}
	return true;
}



bool TePostgreSQL::insertBlob (const string& tableName, const string& columnBlob, TeAttributeRep& columnId, const string& valueId, unsigned char* data, int size)
{
	errorMessage_ = "";

	TePostgreSQLPortal* portal = static_cast<TePostgreSQLPortal*>(this->getPortal());
	
	if(!portal)
		return false;

	string q = "SELECT * FROM "+ tableName +" WHERE "+ columnId.name_ +" = ";
	
	switch(columnId.type_ )
	{
		case TeSTRING:
			q += "'"+ valueId + "'";
			break;
		default:
			q += valueId;
			break;
	}

	if((!portal->query(q)) || (!portal->fetchRow()))
	{
		delete portal;
		return false;
	}

//	int exist = portal->getInt(columnBlob);

	delete portal;

	size_t newLen = 0;

	unsigned char* newbuf = TePGConnection::escapeBytea(data, size, &newLen);

	string sql  = "UPDATE ";
	       sql += tableName;
	       sql += " SET ";
	       sql += columnBlob;
	       sql += " = '";
	       sql += (char*)newbuf;
		   sql += "' WHERE ";
		   sql += columnId.name_;
		   sql += " = ";

	switch (columnId.type_ )
	{	
		case TeSTRING:  sql += "'"+ valueId + "'";
						break;

		default:		sql += valueId;
	}

	if(!execute(sql))
	{
		TePGConnection::freeMem(newbuf);

		errorMessage_ = "Couldn't update blob!";

		return false;
	}

	TePGConnection::freeMem(newbuf);

	return true;
}

bool TePostgreSQL::insertBlob (const string& tableName, const string& columnBlob, TeAttributeRep& columnId, const string& valueId, const string& fileName)
{
	unsigned char	*cdata = 0;
	int		size;
	FILE	*fp = 0;
	
	struct	stat buf;
	int		result;
	
	result = stat(fileName.c_str(), &buf);
	
	if(result != 0)
		return false;
	
	size = buf.st_size;

	cdata = new unsigned char[size];
	fp = fopen(fileName.c_str(), "rb");
	fread(cdata, sizeof(unsigned char), size, fp); 

	bool status = insertBlob(tableName, columnBlob, columnId,  valueId, cdata, size);

	if(fp)
		fclose(fp);

	if(cdata)
		delete [] cdata;

	return status;
}

bool TePostgreSQL::insertProjection(TeProjection *proj)
{
	errorMessage_ = "";

	TeProjectionParams par = proj->params();

	string sql  = "INSERT INTO te_projection (name, long0, lat0, offx, ";
		   sql += "offy, stlat1, stlat2, unit, scale, ";
		   sql += "hemis, datum, radius, flattening, ";
		   sql += "dx, dy, dz) VALUES('";
		   sql += proj->name();
		   sql += "', ";
		   sql += Te2String(par.lon0*TeCRD);
		   sql += ", ";
		   sql += Te2String(par.lat0*TeCRD);
		   sql += ", ";
		   sql += Te2String(par.offx);
		   sql += ", ";
		   sql += Te2String(par.offy);
		   sql += ", ";
		   sql += Te2String(par.stlat1*TeCRD);
		   sql += ", ";
		   sql += Te2String(par.stlat2*TeCRD);
		   sql += ", '";
		   sql += par.units;
		   sql += "', ";
		   sql += Te2String(par.scale);
		   sql += ", ";
		   sql += Te2String(par.hemisphere);
		   sql += ", '";
		   sql += proj->datum().name();
		   sql += "', ";
		   sql += Te2String(proj->datum().radius());
		   sql += ", ";
		   sql += Te2String(proj->datum().flattening());
		   sql += ", ";
		   sql += Te2String(proj->datum().xShift());
		   sql += ", ";
		   sql += Te2String(proj->datum().yShift());
		   sql += ", ";
		   sql += Te2String(proj->datum().zShift());
		   sql += ")";

	if(this->execute(sql))
	{
		sql = "SELECT currval('te_projection_projection_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
			proj->id(rec.getInt(0));

		rec.close();
	}
	else
		return false;
		
	return true;	
}

bool TePostgreSQL::insertLayer(TeLayer *layer)
{
	errorMessage_ = "";

	TeProjection* proj = layer->projection();
	if (!proj || !insertProjection(proj))
	{
		errorMessage_ = "No  possvel inserir layer sem projeo";
		return false;
	}

	string sql  = "INSERT INTO te_layer (projection_id, name, lower_x, lower_y, upper_x, ";
		   sql += "upper_y) VALUES(";
		   sql += Te2String(proj->id());
		   sql += ", '";
		   sql += layer->name();
	       sql += "', ";
	       sql += Te2String(layer->box().x1());
	       sql += ", ";
	       sql += Te2String(layer->box().y1());
	       sql += ", ";
	       sql += Te2String(layer->box().x2());
	       sql += ", ";
	       sql += Te2String(layer->box().y2());
	       sql += ")";

	if(this->execute(sql))
	{
		sql = "SELECT currval('te_layer_layer_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
		{
			layer->id(rec.getInt(0));
		}
		rec.close();
	}
	else
		return false;

	layerMap_[layer->id()] = layer;

	return true;	
}

bool TePostgreSQL::insertRepresentation(int layerId, TeRepresentation& rep)
{	
	errorMessage_ = "";

	if(layerId <= 0)
		return false;
	

	string sql  = "INSERT INTO te_representation (layer_id, geom_type, geom_table, description, lower_x, ";
		   sql += "lower_y, upper_x, upper_y, res_x, res_y, num_cols, num_rows) VALUES(";
		   sql += Te2String(layerId);
		   sql += ", ";
		   sql += Te2String(static_cast<long>(rep.geomRep_));
		   sql += ", '";
		   sql += rep.tableName_;
		   sql += "', '";
		   sql += rep.description_;		   
		   sql += "', ";
           sql += Te2String(rep.box_.x1());
		   sql += ", ";
		   sql += Te2String(rep.box_.y1());
		   sql += ", ";
		   sql += Te2String(rep.box_.x2());
		   sql += ", ";
		   sql += Te2String(rep.box_.y2());
		   sql += ", ";
		   sql += Te2String(rep.resX_);
		   sql += ", ";
		   sql += Te2String(rep.resY_);
		   sql += ", ";
		   sql += Te2String(static_cast<long>(rep.nCols_));
		   sql += ", ";
		   sql += Te2String(static_cast<long>(rep.nLins_));
		   sql += ")";

	if(this->execute(sql))
	{
		sql = "SELECT currval('te_representation_repres_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
			rep.id_ = rec.getInt(0);

		rec.close();
	}
	else
		return false;

	return true;		
}

bool TePostgreSQL::insertView(TeView *view)
{
	errorMessage_ = "";

	// save its projection
	TeProjection* proj = view->projection();
	if ( !proj || !insertProjection(proj))
	{
		errorMessage_ = "No  possvel inserir vista sem projeo";
		return false;
	}

	string sql  = "INSERT INTO te_view (projection_id, name, user_name, visibility) VALUES(";
	       sql += Te2String(proj->id());
		   sql += ", '";
		   sql += view->name();
		   sql += "', '";
		   sql += view->user();
		   sql += "', ";
		   sql += Te2String(view->isVisible());
		   sql += ")";

	if(this->execute(sql))
	{
		sql = "SELECT currval('te_view_view_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
		{
			view->id(rec.getInt(0));
		}
		rec.close();
	}
	else
		return false;

	for(unsigned int th = 0; th < view->size(); ++th)
	{
		TeViewNode* node = view->get (th);
		if (node->type() == TeTHEME)
		{
			TeTheme *theme = (TeTheme*) node;
			insertTheme (theme);
		}
		else
		{
			TeViewTree* tree = (TeViewTree*)node;
			insertViewTree (tree);
		}
	}

	viewMap_[view->id()] = view;

	return true;
}

bool TePostgreSQL::insertViewTree(TeViewTree *tree)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO te_theme (view_id, name, parent_id, priority, node_type) VALUES(";

		   sql += Te2String(tree->view());
		   sql += ", '";
	       sql += tree->name();
	       sql += "', ";
	       sql += Te2String(tree->parentId());
	       sql += ", ";
		   sql += Te2String(tree->priority());
	       sql += ", ";
	       sql += Te2String(tree->type());
	       sql += ", ";
	       
	       sql += ")";

	if(this->execute(sql))
	{
		sql = "SELECT currval('te_theme_theme_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
			tree->id(rec.getInt(0));

		rec.close();
	}
	else
		return false;
		
	return true;	
}

bool TePostgreSQL::insertThemeGroup(TeViewTree* tree)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO te_theme (layer_id, view_id, name, parent_id, priority, node_type) VALUES(";
	       sql += "1";
		   sql += ", ";
		   sql += Te2String(tree->view());
		   sql += ", '";
	       sql += tree->name();
	       sql += "', ";
	       sql += Te2String(tree->parentId());
	       sql += ", ";
		   sql += Te2String(tree->priority());
	       sql += ", ";
	       sql += "1";
	       sql += ", ";
	       
	       sql += ")";

	if(this->execute(sql))
	{
		sql = "SELECT currval('te_theme_theme_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
			tree->id(rec.getInt(0));

		rec.close();
	}
	else
		return false;
		
	return true;	

}

bool TePostgreSQL::insertTheme(TeTheme *theme)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO te_theme (layer_id, view_id, name, parent_id, priority, node_type, ";
	       sql += "min_scale, max_scale, generate_attribute_where, generate_spatial_where, ";
		   sql += "generate_temporal_where, collection_table, visible_rep, enable_visibility) VALUES(";
		   sql += Te2String(theme->layerId());
		   sql += ", ";
		   sql += Te2String(theme->view());
		   sql += ", '";
		   sql += theme->name();
		   sql += "', ";
		   sql += Te2String(theme->parentId());
		   sql += ", ";
		   sql += Te2String(theme->priority());
		   sql += ", ";
		   sql += Te2String(theme->type());
		   sql += ", ";
		   sql += Te2String(theme->minScale());
		   sql += ", ";
		   sql += Te2String(theme->maxScale());
		   sql += ", '";
		   sql += escapeSequence(theme->attributeRest());
		   sql += "', '";
		   sql += escapeSequence(theme->spatialRest());
		   sql += "', '";
		   sql += escapeSequence(theme->temporalRest());
		   sql += "', '";
		   sql += theme->collectionTable();
		   sql += "', ";
		   sql += Te2String(theme->visibleRep());
		   sql += ", ";
		   sql += Te2String(theme->visibility());
		   sql += " )";

	if(this->execute(sql))
	{
		sql = "SELECT currval('te_theme_theme_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
		{
			theme->id(rec.getInt(0));
		}
		rec.close();
	}
	else
		return false;


	if(theme->collectionTable().empty())
	{
		sql  = "UPDATE te_theme SET collection_table = '";
		sql += "te_collection_";
		sql += Te2String(theme->id());
		sql += "' WHERE theme_id = ";
		sql += Te2String(theme->id());

		theme->collectionTable(string("te_collection_") + Te2String(theme->id()));

		if(!this->execute(sql))
			return false;
	}


	bool status;

	// insert grouping
	int numSlices = 0;
	if(theme->grouping())
	{
		if(!insertGrouping (theme->id(), theme->grouping()))
			return false;
		numSlices = theme->grouping()->groupNumSlices_;
	}
		
	// insert legend
	theme->outOfCollectionLegend().group(-1); 
	theme->outOfCollectionLegend().theme(theme->id()); 
	status = insertLegend (&(theme->outOfCollectionLegend())); 
	if (!status)
		return status;

	theme->withoutDataConnectionLegend().group(-2); 
	theme->withoutDataConnectionLegend().theme(theme->id()); 
	status = insertLegend (&(theme->withoutDataConnectionLegend())); 
	if (!status)
		return status;

	theme->defaultLegend().group(-3); 
	theme->defaultLegend().theme(theme->id()); 
	status = insertLegend (&(theme->defaultLegend())); 
	if (!status)
		return status;

	theme->pointingLegend().group(-4); 
	theme->pointingLegend().theme(theme->id()); 
	status = insertLegend (&(theme->pointingLegend())); 
	if (!status)
		return status;

	theme->queryLegend().group(-5); 
	theme->queryLegend().theme(theme->id()); 
	status = insertLegend (&(theme->queryLegend())); 
	if (!status)
		return status;

	theme->queryAndPointingLegend().group(-6); 
	theme->queryAndPointingLegend().theme(theme->id()); 
	status = insertLegend (&(theme->queryAndPointingLegend())); 
	if (!status)
		return status;

	for (int i = 0; i < numSlices; i++)
	{
		theme->legend()[i].group(i);
		theme->legend()[i].theme(theme->id());
		status = insertLegend (&(theme->legend()[i]));
		if (!status)
			return status;
	}
	if (!status)
		return status;

	themeMap_[theme->id()] = theme;
	return updateThemeTable(theme);
}

bool TePostgreSQL::insertThemeTable(int themeId, int tableId, int relationId, int tableOrder)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO te_theme_table (theme_id, table_id, ";

	if(relationId > 0)
		sql += "relation_id, ";

	sql += "table_order) VALUES(";
	sql += Te2String(themeId);
	sql += ", ";
	sql += Te2String(tableId);
	sql += ", ";

	if(relationId > 0)
	{
		sql += Te2String(relationId);
		sql += ", ";
	}

	sql += Te2String(tableOrder);
	sql += ")";

	if(this->execute(sql))
	{
		//sql = "SELECT currval('te_theme_table_theme_table_id_seq')";
		//TePGRecordset rec;
		//rec.open(sql, &tepg_connection_);
		//if(rec.recordCount() > 0)
		//{
		//	tree->id(atoi(rec.value(0)));
		//}
		//rec.close();
	}
	else
		return false;
		
	return true;	
}

bool TePostgreSQL::generateLabelPositions(TeTheme *theme)
{
	string	geomTable, upd;
	string	collTable = theme->collectionTable();
	
	if((collTable.empty()) || (!tableExist(collTable)))
		return false;

	if(theme->layer()->hasGeometry(TeCELLS)    || 
	   theme->layer()->hasGeometry(TePOLYGONS) ||
	   theme->layer()->hasGeometry(TeLINES))
	{
		geomTable = theme->layer()->tableName(TeCELLS);
		
		if(geomTable.empty())
		{
			geomTable = theme->layer()->tableName(TePOLYGONS);
			if(geomTable.empty())
			{
				geomTable = theme->layer()->tableName(TeLINES);

				if(geomTable.empty())
					geomTable = theme->layer()->tableName(TePOINTS);
			}
		}
		
		upd  = " UPDATE " + collTable + " SET ";
		upd += " label_x = (SELECT MAX(lower_x + (upper_x";
		upd += " -  lower_x) / 2.0) ";
		upd += "FROM " + geomTable + " WHERE object_id = c_object_id), ";
		
		upd += " label_y = (SELECT MAX(lower_y + (upper_y";
		upd += " - lower_y) / 2.0) ";
		upd += "FROM " + geomTable + " WHERE object_id = c_object_id) ";

		upd += " WHERE (label_x IS NULL) OR (label_y IS NULL)";
	}	
	else
		if(theme->layer()->hasGeometry(TePOINTS))
		{
			geomTable = theme->layer()->tableName(TePOINTS);
		
			upd  = " UPDATE " + collTable + " SET ";
			upd += " label_x = (SELECT MAX(x)";
			upd += " FROM " + geomTable + " WHERE object_id = c_object_id), ";

			upd += " label_y = (SELECT MAX(y)";
			upd += " FROM " + geomTable + " WHERE object_id = c_object_id) ";

			upd += " WHERE (label_x IS NULL) OR (label_y IS NULL)";
		}

	return execute(upd);
}

bool TePostgreSQL::insertLegend(TeLegendEntry *legend)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO te_legend (theme_id, group_id, num_objs, lower_value, upper_value, ";
		   sql += "label) VALUES(";
		   sql += Te2String(legend->theme());
		   sql += ", ";
		   sql += Te2String(legend->group());
		   sql += ", ";
	   	   sql += Te2String(legend->count());
		   sql += ", '";
		   sql += legend->from();
		   sql += "', '";
		   sql += legend->to();
		   sql += "', '";
		   sql += legend->label();
		   sql += "')";

	if(this->execute(sql))
	{
		sql = "SELECT currval('te_legend_legend_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
		{
			legend->id(rec.getInt(0));
		}
		rec.close();
	}
	else
		return false;	

	legendMap_[legend->id()] = legend;

	return insertVisual(legend);
}

bool TePostgreSQL::insertPolygon(const string& table, TePolygon &p)
{
	errorMessage_ = "";

	//TeRemoveDuplicatedCoordinates(p);

	int parentId = 0;

	for(unsigned int k = 0u; k < p.size(); ++k)
	{
		TeLinearRing ring = p[k];

		TeBox b = ring.box();

		unsigned int ringSize = ring.size();

		double extMax = MAX(b.width(), b.height());

		unsigned int numberOfHoles = 0u;
		
		if(k == 0u)
			numberOfHoles = p.size() - 1u;

		string strPol = "";

		Te2PgGeomPolygon(ring, strPol);

		string sql  = "INSERT INTO ";
		       sql += table;
			   sql += " (object_id, num_coords, num_holes, parent_id, lower_x, lower_y, upper_x, upper_y, ext_max, spatial_data) VALUES ('";
			   sql += p.objectId();
			   sql += "', ";
			   sql += Te2String(ringSize);
			   sql += ", ";
			   sql += Te2String(numberOfHoles);
			   sql += ", ";
			   sql += Te2String(parentId);
			   sql += ", ";
			   sql += Te2String(b.lowerLeft().x());
			   sql += ", ";
			   sql += Te2String(b.lowerLeft().y());
			   sql += ", ";
			   sql += Te2String(b.upperRight().x());
			   sql += ", ";
			   sql += Te2String(b.upperRight().y());
			   sql += ", ";
			   sql += Te2String(extMax);
			   sql += ", '";
			   sql += strPol;
			   sql += "')";

		if(this->execute(sql))
		{
			sql = "SELECT currval('" + table + "_geom_id_seq')";
		
			TePGRecordset rec;
			rec.open(sql, &tepg_connection_);

			if(rec.recordCount() > 0)
				ring.geomId(rec.getInt(0));

			if(k == 0u)
			{
				parentId = ring.geomId();

				string newSQL  = "UPDATE " + table + " SET parent_id = ";
				       newSQL += Te2String(parentId);
					   newSQL += " WHERE geom_id = ";
					   newSQL += Te2String(parentId);

				if(!this->execute(newSQL))
					return false;
			}

			rec.close();
		}
		else
			return false;
	}

	return true;
}

bool TePostgreSQL::updatePolygon(const string& table, TePolygon &p)
{
	errorMessage_ = "";

	//TeRemoveDuplicatedCoordinates(p);

	int parentId = 0;

	for(unsigned int k = 0u; k < p.size(); ++k)
	{
		TeLinearRing ring = p[k];

		TeBox b = ring.box();

		unsigned int ringSize = ring.size();

		double extMax = MAX(b.width(), b.height());

		unsigned int numberOfHoles = 0u;
		
		if(k == 0u)
			numberOfHoles = p.size() - 1u;

		string strPol = "";

		Te2PgGeomPolygon(ring, strPol);

		string sql  = "UPDATE " + table + " SET object_id = '";
		       sql += p.objectId();
			   sql += "', num_coords = ";
			   sql += Te2String(ringSize);
			   sql += ", num_holes = ";
			   sql += Te2String(numberOfHoles);
			   sql += ", parent_id = ";
			   sql += Te2String(parentId);
			   sql += ", lower_x = ";
			   sql += Te2String(b.x1_);
			   sql += ", lower_y = ";
			   sql += Te2String(b.y1_);
			   sql += ", upper_x = ";
			   sql += Te2String(b.x2_);
			   sql += ", upper_y = ";
			   sql += Te2String(b.y2_);
			   sql += ", ext_max = ";
			   sql += Te2String(extMax);
			   sql += ", spatial_data = '";
			   sql += strPol;
			   sql += "' WHERE geom_id = ";
			   sql += Te2String(ring.geomId());

		if(!this->execute(sql))
			return false;
	}

	return true;
}

bool TePostgreSQL::insertLine(const string& table, TeLine2D& l)
{
	errorMessage_ = "";

	//TeRemoveDuplicatedCoordinates(l);

	string strLine = "";
		
	Te2PgGeomPolygon(l, strLine);

	double extMax = MAX(l.box().width(), l.box().height());

	string sql  = "INSERT INTO " + table + " (object_id, num_coords, lower_x, lower_y, upper_x, upper_y, ext_max, spatial_data) ";
		   sql += "VALUES('";
		   sql += l.objectId();
		   sql += "', ";
		   sql += Te2String(l.size());
		   sql += ", ";
		   sql += Te2String(l.box().x1_);
		   sql += ", ";
		   sql += Te2String(l.box().y1_);
		   sql += ", ";
		   sql += Te2String(l.box().x2_);
		   sql += ", ";
		   sql += Te2String(l.box().y2_);
		   sql += ", ";
		   sql += Te2String(extMax);
		   sql += ", '";
		   sql += strLine;
		   sql += "')";

	if(this->execute(sql))
	{
		sql = "SELECT currval('" + table + "_geom_id_seq')";

		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);

		if(rec.recordCount() > 0)
			l.geomId(rec.getInt(0));

		rec.close();
	}
	else
		return false;

	return true;
}

bool TePostgreSQL::updateLine(const string& table, TeLine2D& l)
{
	errorMessage_ = "";

	//TeRemoveDuplicatedCoordinates(l);

	string strLine = "";
	
	Te2PgGeomPolygon(l, strLine);

	double extMax = MAX(l.box().width(), l.box().height());

	string sql  = "UPDATE " + table + " SET  object_id = '";
		   sql += l.objectId();
		   sql += "', num_coords = ";
		   sql += Te2String(l.size());
		   sql += ", lower_x = ";
		   sql += Te2String(l.box().x1_);
		   sql += ", lower_y = ";
		   sql += Te2String(l.box().y1_);
		   sql += ", upper_x = ";
		   sql += Te2String(l.box().x2_);
		   sql += ", upper_y = ";
		   sql += Te2String(l.box().y2_);
		   sql += ", ext_max = ";
		   sql += Te2String(extMax);
		   sql += ", '";
		   sql += strLine;
		   sql += "' WHERE geom_id = ";
		   sql += Te2String(l.geomId());			

	return this->execute(sql);
}

bool TePostgreSQL::insertPoint(const string& table, TePoint &p)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO " + table + " (object_id, x, y) VALUES('";
 		   sql +=  p.objectId();
		   sql += "', ";
		   sql += Te2String(p.location().x_);
		   sql += ", ";
		   sql += Te2String(p.location().y_);
		   sql += ")";

	if(this->execute(sql))
	{
		sql = "SELECT currval('" + table + "_geom_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
			p.geomId(rec.getInt(0));

		rec.close();
	}
	else
		return false;
	
	return true;
}

bool TePostgreSQL::insertText(const string& table, TeText& t)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO " + table + " (object_id, x, y, text_value, angle, height, alignment_vert, alignment_horiz) VALUES('";
		   sql += t.objectId();
		   sql += "', ";
		   sql += Te2String(t.location().x());
		   sql += ", ";
		   sql += Te2String(t.location().y());
		   sql += ", '";
		   sql += escapeSequence(t.textValue());
		   sql += "', ";
		   sql += Te2String(t.angle());
		   sql += ", ";
		   sql += Te2String(t.height());
		   sql += ", ";
		   sql += Te2String(t.alignmentVert());
		   sql += ", ";
		   sql += Te2String(t.alignmentHoriz());
		   sql += ")";

	if(this->execute(sql))
	{
		sql = "SELECT currval('" + table + "_geom_id_seq')";

		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
			t.geomId(rec.getInt(0));

		rec.close();
	}
	else
		return false;

	return true;
}

bool TePostgreSQL::insertArc(const string& table, TeArc& arc)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO " + table + " (object_id, from_node, to_node) VALUES('";
		   sql += arc.objectId();
		   sql += "', ";
		   sql += Te2String(arc.fromNode().geomId());
	       sql += ", ";
		   sql += Te2String(arc.toNode().geomId());
		   sql += ")";

	if(this->execute(sql))
	{
		sql = "SELECT currval('" + table + "_geom_id_seq')";
		TePGRecordset rec;
		rec.open(sql, &tepg_connection_);
		if(rec.recordCount() > 0)
			arc.geomId(rec.getInt(0));

		rec.close();
	}
	else
		return false;
	

	return true;
}

bool TePostgreSQL::insertNode(const string& table, TeNode& node)
{
	errorMessage_ = "";

	string  sql = "INSERT INTO " + table + " (object_id, x, y) VALUES('";
			sql += node.objectId();
			sql += "', ";
			sql += Te2String(node.location().x_);
			sql += ", ";
			sql += Te2String(node.location().y_);
			sql += ")";

	if(!this->execute(sql))
		return false;

	sql = "SELECT currval('" + table + "_geom_id_seq')";
	TePGRecordset rec;
	rec.open(sql, &tepg_connection_);
	if(rec.recordCount() > 0)
		node.geomId(rec.getInt(0));

	rec.close();

	return true;
}

bool TePostgreSQL::insertCell(const string& table, TeCell &c)
{
	errorMessage_ = "";

	string sql  = "INSERT INTO " + table + " (object_id, lower_x, lower_y, upper_x, upper_y, col_number, row_number) VALUES('";
		   sql += c.objectId();
		   sql += "', ";
		   sql += Te2String(c.box().x1_);
		   sql += ", ";
		   sql += Te2String(c.box().y1_);
		   sql += ", ";
		   sql += Te2String(c.box().x2_);
		   sql += ", ";
		   sql += Te2String(c.box().y2_);
		   sql += ", ";
		   sql += Te2String(c.column());
		   sql += ", ";
		   sql += Te2String(c.line());
		   sql += ")";

	if(!this->execute(sql))
		return false;

	sql = "SELECT currval('" + table + "_geom_id_seq')";
	TePGRecordset rec;
	rec.open(sql, &tepg_connection_);
	if(rec.recordCount() > 0)
		c.geomId(rec.getInt(0));
	
	rec.close();	

	return true;
}

bool TePostgreSQL::insertRasterBlock(const string& table, const string& blockId, const TeCoord2D& ll, const TeCoord2D& ur, unsigned char* buf, unsigned long size, int band, unsigned int res, unsigned int subband)
{
	errorMessage_ = "";

	if (blockId.empty()) 
	{
		errorMessage_ = "No block identifier provided!";
		return false;
	}

	TePostgreSQLPortal* portal = static_cast<TePostgreSQLPortal*>(this->getPortal());
	if(!portal)
		return false;

	bool update = false;
	string q ="SELECT * FROM " + table; 
	q += " WHERE block_id='" + blockId + "'";

	if (!portal->query(q))
	{
		delete portal;
		return false;
	}

	// check if this block is alread in the database
	if(portal->fetchRow())
		update = true;

	delete portal;


	string sql = "";
		
	size_t newLen = 0;

	unsigned char* newbuf = TePGConnection::escapeBytea(buf, size, &newLen);

	if(!update)
		getInsertRasterBlock(table, blockId, ll, ur, band, res, subband, (char*)newbuf, size, sql);
	else
		getUpdateRasterBlock(table, blockId, ll, ur, band, res, subband, (char*)newbuf, size, sql);


	if(!execute(sql))
	{
		TePGConnection::freeMem(newbuf);
		errorMessage_ = "Couldn't insert/update a rasterblock!";
		return false;
	}

	TePGConnection::freeMem(newbuf);

	return true;
}

bool TePostgreSQL::createLUTTable(const string& name)
{
	errorMessage_ = "";

	string create = "CREATE TABLE " + name + "(";
	create += "index_id		SERIAL,";
	create += "r_val		INTEGER NOT NULL,";
	create += "g_val		INTEGER NOT NULL,";
	create += "b_val		INTEGER NOT NULL,";
	create += "PRIMARY KEY (index_id))";
	
	return execute(create);
}

string TePostgreSQL::getSQLStatistics(TeGroupingAttr& attrs)
{
	string sql = "";
	string virg = "";

	TeGroupingAttr::iterator it = attrs.begin();
	int count = 0;
	while(it != attrs.end())
	{
		if(count > 0)
			virg = ",";

		string strAux = (*it).first.name_;
		size_t pos = strAux.find(".");
		if(pos != string::npos)
			strAux[pos] = '_';

		switch((*it).second)
		{
			case TeSUM:
				sql += virg +" SUM( "+ (*it).first.name_ +") AS SUM_"+ strAux;
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeMAXVALUE:
				sql += virg +" MAX( "+ (*it).first.name_ +") AS MAX_"+ strAux;
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeMINVALUE:
				sql += virg +" MIN( "+ (*it).first.name_ +") AS MIN_"+ strAux;
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeCOUNT:
				sql += virg +" COUNT( "+ (*it).first.name_ +") AS COUNT_"+ strAux;
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeMEAN:
				sql += virg +" AVG( "+ (*it).first.name_ +") AS AVG_"+ strAux;
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeSTANDARDDEVIATION:
				sql += virg +" STDDEV( "+ (*it).first.name_ +") AS STDDEV_"+ strAux;
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeVARIANCE:
				sql += virg +" VARIANCE( "+ (*it).first.name_ +") AS VAR_"+ strAux;
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			default:
				break;
		}
		++it;

	}

	return sql;
}

string TePostgreSQL::getSQLTemporalWhere(TeTimeInterval& timeInterval, TeTemporalRelation timeOperator, const string& initialTime, const string& finalTime)
{
	string sql;
	
	string t1 = timeInterval.getInitialDateTime("YYYYsMMsDDsHHsmmsSS");
	string t2 = timeInterval.getFinalDateTime ("YYYYsMMsDDsHHsmmsSS");
	
	switch(timeOperator)
	{
		case TeTIMEBEFORE:		sql = finalTime +" < '"+ t1 +"'";
								break;

		case TeTIMEAFTER:		sql = initialTime +" > '"+ t2 +"'";
								break;

		case TeTIMEEQUAL:       sql = "("+ initialTime +" >= '"+ t1 +"'";
								sql += " AND "+ initialTime +" <= '"+ t2 +"')";
								sql += " AND ";
								sql += "("+ finalTime +" >= '"+ t1 +"'";
								sql += " AND "+ finalTime +" <= '"+ t2 +"')";
								break;

		case TeTIMEMEETS:       sql = finalTime +" = '"+ t1 +"'";
								sql += " OR "+ initialTime +" = '"+ t2 +"'";
								break;

		case TeTIMEDURING:		sql = initialTime +" > '"+ t1 +"'";
								sql += " AND "+ initialTime +" < '"+ t2 +"'";
								sql += " AND "+ finalTime +" > '"+ t1 +"'";
								sql += " AND "+ finalTime +" < '"+ t2 +"'";
								break;

		case TeTIMEOVERLAPS:    sql = "("+ initialTime +" < '"+ t1 +"'";
								sql += " AND "+ finalTime +" > '"+ t1 +"'";
								sql += " AND "+ finalTime +" < '"+ t2 +"')";
								sql += " OR ";
								sql += "("+ initialTime + " > '"+ t1 +"'";
								sql += " AND "+ initialTime +" < '"+ t2 +"'";
								sql += " AND "+ finalTime +" > '"+ t2 +"')";
								break;

		case TeTIMEENDS:		sql = finalTime +" = '"+ t2 +"'";
								break;

		case TeTIMESTARTS:		sql = initialTime +" = '"+ t1 +"'";
								break;
		case TeTIMEUNDEFINED:	
				return sql;				
	}

	return sql; 
}

string TePostgreSQL::getSQLTemporalWhere(int time1, int time2, TeChronon chr, TeTemporalRelation rel, const string& initialTime, const string& finalTime)
{
	string func = " EXTRACT(";
	string sql;

	switch(chr)
	{
		case TeSECOND:		func = "second FROM "; 
							break;
		
		case TeMINUTE:		func = "minute FROM ";
							break;
		
		case TeHOUR:        func = "hour FROM ";
							break;
		
		case TeDAY:         func = "day FROM ";
							break;

		case TeMONTH:		func = "month FROM ";
							break;

		case TeYEAR:        func = "year FROM ";	
							break;
		default:			return "";
	}

	switch(rel)
	{
		case TeTIMEBEFORE:		sql = func + finalTime + ") < " + Te2String(time1);
								break;
		
		case TeTIMEAFTER:		sql = func + initialTime + ") > " + Te2String(time2);
								break;
		
		case TeTIMEEQUAL:		sql  = func + initialTime + ") = " + Te2String(time1);
								sql += " AND "+ func + finalTime +") = " + Te2String(time2);
								break;

		case TeTIMEMEETS:		sql  = func + finalTime + ") = " + Te2String(time1);
								sql += " OR "+ func + initialTime +") = " + Te2String(time2);
								break;

		case TeTIMEDURING:		sql  = func + initialTime + ") >= " + Te2String(time1);
								sql += " AND " + func + initialTime + ") <= " + Te2String(time2);
								sql += " AND " + func + finalTime + ") >= " + Te2String(time1);
								sql += " AND " + func + finalTime + ") <= " + Te2String(time2);
								break;

		case TeTIMEOVERLAPS:    sql  =  "("+ func + initialTime + ") <= " + Te2String(time1);
								sql += " AND " + func + finalTime + ") >= " + Te2String(time1);
								sql += " AND " + func + finalTime + ") <= " + Te2String(time2) + ")";
								sql += " OR ";
								sql += "(" + func + initialTime + ") >= " + Te2String(time1);
								sql += " AND " + func + initialTime + ") <= " + Te2String(time2);
								sql += " AND " + func + finalTime + ") >= " + Te2String(time2) +")";
								break;

		case TeTIMEENDS:		sql = func + finalTime + ") = " + Te2String(time2);
								break;

		case TeTIMESTARTS:		sql = func + initialTime + ") = " + Te2String(time1);
								break;
		case TeTIMEUNDEFINED:	
				return sql;				
	}
	
	return sql;
}

string TePostgreSQL::getAutoNumberSQL(const string& table)
{
	errorMessage_ = "";

	TePGRecordset rec;

	string sql = "SELECT adsrc FROM pg_class, pg_attrdef  WHERE lower(pg_class.relname) = lower('" + table + "') AND pg_attrdef.adnum = 1 AND pg_class.oid = pg_attrdef.adrelid";

	if(!rec.open(sql, &tepg_connection_))
		return false;

	if(rec.recordCount() > 0)
	{
		return rec.getData(0);
	}
	else
		return "";
}

string TePostgreSQL::concatValues(vector<string>& values, const string& unionString)
{
	string concat = "";
	
	for(unsigned int i = 0; i < values.size(); ++i)
	{
		if(i != 0)
		{
			concat += " || ";

			if(!unionString.empty())
			{
				concat += "'";
				concat += unionString;
				concat += "'";
				concat += " || ";
			}
		}

		concat += values[i];
	}

	return concat;
}

string TePostgreSQL::toUpper(const string& value)
{
	string result  = "upper(";
	       result += value;
		   result += ")";

	return result;
}

void TePostgreSQL::getInsertRasterBlock(const string& table, const string& blockId, const TeCoord2D& ll, const TeCoord2D& ur, const int& band, const unsigned int& res, const unsigned int& subband, char* buf, const unsigned long& size, string& sql)
{
	sql += "INSERT INTO " + table + " (block_id, lower_x, lower_y, upper_x, upper_y, band_id, resolution_factor, subband, spatial_data, block_size) VALUES('";
	sql += blockId;
	sql += "', ";
	sql += Te2String(ll.x_);
	sql += ", ";
	sql += Te2String(ll.y_);
	sql += ", ";
	sql += Te2String(ur.x_);
	sql += ", ";
	sql += Te2String(ur.y_);
	sql += ", ";
	sql += Te2String(band);
	sql += ", ";
	sql += Te2String(res);
	sql += ", ";
	sql += Te2String(subband);
	sql += ", '";
	//sql += Te2String(obj_oid);
	sql += buf;
	sql += "', ";
	sql += Te2String(size);
	sql += ")";

	return;
}

void TePostgreSQL::getUpdateRasterBlock(const string& table, const string& blockId, const TeCoord2D& ll, const TeCoord2D& ur, const int& band, const unsigned int& res, const unsigned int& subband, char* buf, const unsigned long& size, string& sql)
{
	sql += "UPDATE ";
	sql += table;
	sql += " SET lower_x = ";
	sql += Te2String(ll.x_);
	sql += ", lower_y = ";
	sql += Te2String(ll.y_);
	sql += ", upper_x = ";
	sql += Te2String(ur.x_);
	sql += ", upper_y = ";
	sql += Te2String(ur.y_);
	sql += ", band_id = ";
	sql += Te2String(band);
	sql += ", resolution_factor = ";
	sql += Te2String(res);
	sql += ", subband = ";
	sql += Te2String(subband);
	sql += ", spatial_data = '";
	//sql += Te2String(static_cast<long>(obj_oid));;
	sql += buf;
	sql += "', block_size = ";
	sql += Te2String(size);
	sql += " WHERE block_id = '";
	sql += blockId;
	sql += "'";

	return;
}


string TePostgreSQL::escapeSequence(const string& from)
{
	size_t newLen = 0;


	char* aux = TePGConnection::escapeString(from.c_str(), from.length(), newLen);

	string str = aux;

	delete [] aux;

	return str;
}


//----- TePostgreSQLPortal methods ---
TePostgreSQLPortal::TePostgreSQLPortal() : con_(0), curRow_(-1)
{
}

TePostgreSQLPortal::TePostgreSQLPortal(TeDatabase *pDatabase) : curRow_(-1)
{
	db_ = pDatabase;
	con_ = &((static_cast<TePostgreSQL*>(pDatabase))->tepg_connection_);
}

TePostgreSQLPortal::~TePostgreSQLPortal()
{
	this->freeResult();
	con_ = 0;
}

bool TePostgreSQLPortal::query(const string &qry, TeCursorLocation l, TeCursorType t, TeCursorEditType /*e*/, TeCursorDataType dt)
{
	errorMessage_ = "";

	freeResult();	
		
	
	if(!tepg_recordset_.open(qry, con_, t, l, dt))
	//{
	//	TeWriteToFile("QUERYS.SQL", qry + "\n", "a");

		return false;
	//}


	numRows_ = tepg_recordset_.recordCount();

	numFields_ = tepg_recordset_.getFieldCount(); 

	attList_.clear ();

	for(int i = 0; i < numFields_; ++i)
	{
		TeAttribute attribute;

		Oid nType = tepg_recordset_.fieldType(i);

		switch(nType)
		{
			case 16:    //BOOL			
			case 20:    //INT8
			case 21:    //INT2
			case 23:    //INT4
			case 26:    //OID			
						attribute.rep_.type_ = TeINT;
						break;
			case 700 :  //float4  -> float(p)
			case 701 :  //float8  -> float(p)
			case 790 :  //money   -> decimal(9, 2)
			case 1700:  //numeric -> numeric(p, s)
						attribute.rep_.type_ = TeREAL;
						break;

			case 1082:	//date -> date
			case 1083:  //time -> time
			case 1114:  //timestamp
			case 1186:  //interval
			case 1266:	//
						attribute.rep_.type_ = TeDATETIME;
						break;

			//case 26:    //OID
			case 17:	//bytea
						attribute.rep_.type_ = TeBLOB;
						break;
			case 1042:
						attribute.rep_.type_ = TeCHARACTER;

			case 1043:  //varchar(n)
			case 25:    //text
						attribute.rep_.type_ = TeSTRING;
						break;

			default:
						attribute.rep_.type_ = TeUNKNOWN;
						break;
		}

		attribute.rep_.name_ = tepg_recordset_.fieldName(i);
		pair<int, int> len(tepg_recordset_.fieldSize(i), tepg_recordset_.fieldSizeFractionaryPart(i));
		attribute.rep_.numChar_ = len.first + len.second;
		attList_.push_back(attribute);
	}

	curRow_ = 0;	

	return true;
}

bool TePostgreSQLPortal::fetchRow()
{
	errorMessage_ = "";

	if(numFields_ <= 0)
	{
		errorMessage_ = "The PostgreSQL portal is empty!";
		return false;
	}

	if(curRow_ > 0)
		tepg_recordset_.moveNext();

	if(tepg_recordset_.eof())
		return false;

	++curRow_;	

	return true;
}

bool TePostgreSQLPortal::fetchRow(int i)
{
	errorMessage_ = "";

	//if(tepg_recordset_.recordCount() == 0)
	//{
	//	errorMessage_ = "The PostgreSQL portal is empty!";
	//	return false;
	//}

	curRow_ = i;
	return tepg_recordset_.moveTo(i + 1);
	//return true;
}

void TePostgreSQLPortal::freeResult()
{
	tepg_recordset_.close();

	return;
}

string TePostgreSQLPortal::errorMessage()
{
	if(errorMessage_.empty())
		return con_->err_msg();	
	else
		return errorMessage_;
}

char* TePostgreSQLPortal::getData(int i)
{
	errorMessage_ = "";
	
	try
	{
		return tepg_recordset_.getData(i);
	}

	catch(...)
	{
		return "";
	}

	return "";

}

char* TePostgreSQLPortal::getData(const string& s)
{
	errorMessage_ = "";

	try
	{
		return tepg_recordset_.getData(s);
	}

	catch(...)
	{
		return "";
	}

	return "";
}

double TePostgreSQLPortal::getDouble(int i)
{
	errorMessage_ = "";

	try
	{
		return tepg_recordset_.getDouble(i);
	}

	catch(...)
	{
		return 0.0;
	}

	return 0.0;
}

double TePostgreSQLPortal::getDouble(const string& s)
{
	errorMessage_ = "";

	try
	{
		return tepg_recordset_.getDouble(s);
	}

	catch(...)
	{
		return 0.0;
	}

	return 0.0;
}

int TePostgreSQLPortal::getInt(int i)
{
	errorMessage_ = "";

	try
	{
		return tepg_recordset_.getInt(i);
	}

	catch(...)
	{
		return 0;
	}

	return 0;
}

int TePostgreSQLPortal::getInt(const string& s)
{
	errorMessage_ = "";

	try
	{
		return tepg_recordset_.getInt(s);
	}

	catch(...)
	{
		return 0;
	}

	return 0;
}

bool TePostgreSQLPortal::getBool(const string& s)
{
	errorMessage_ = "";

	try
	{
		if(tepg_recordset_.getInt(s))
			return true;
		else
			return false;
	}

	catch(...)
	{
		return false;
	}
	
	return false;
}

bool TePostgreSQLPortal::getBool(int i)
{
	errorMessage_ = "";

	try
	{
		if(tepg_recordset_.getInt(i))
			return true;
		else
			return false;
	}

	catch(...)
	{
		return false;
	}

	return false;
}

TeTime TePostgreSQLPortal::getDate(int i)
{
	string s = getData(i);

	TeTime t(s, TeSECOND, "YYYYsMMsDDsHHsmmsSS"); 

	return t;
}

TeTime TePostgreSQLPortal::getDate(const string& s)
{
	string ss = getData(s);

	TeTime t(ss, TeSECOND, "YYYYsMMsDDsHHsmmsSS"); 

	return t;
}

string TePostgreSQLPortal::getDateAsString(int i)
{
	return getData(i);
}

string TePostgreSQLPortal::getDateAsString(const string& s)
{
	return getData(s);
}

bool TePostgreSQLPortal::getBlob(const string& s, unsigned char*& data, long& size)
{
	errorMessage_ = "";

	data = 0;

	char* ptDataAux = (char*)data;

	unsigned long newLen = tepg_recordset_.getBytea(s, ptDataAux);

	if(newLen > 0)
	{
		size = newLen;
		data = (unsigned char*)ptDataAux;
		return true;
	}

	errorMessage_ = "Couldn't read blob! Blob size error!";

	if(data)
	{
		delete [] data;
		data = 0;
	}

	return false;
}

bool TePostgreSQLPortal::getRasterBlock(unsigned long& size, unsigned char* ptData)
{
	errorMessage_ = "";

	// get the actual length of the compressed data
	size = tepg_recordset_.getInt("block_size");
	
	
	if(size > 0)
	{
		char* ptDataAux = (char*)ptData;

		unsigned long newLen = tepg_recordset_.getBytea("spatial_data", ptDataAux);

		if(newLen == size)
		{
			ptData = (unsigned char*)ptDataAux;
			return true;
		}
	}

	return false;
}


bool TePostgreSQLPortal::fetchGeometry(TePolygon& pol)
{
	errorMessage_ = "";

	int numberOfHoles;

	TeLinearRing ring = this->getLinearRing(numberOfHoles);

	pol.objectId(ring.objectId());
	pol.geomId(ring.geomId());
	pol.add(ring);
	int parentId = pol.geomId();

	while(fetchRow())
	{
		if(atoi(this->getData("parent_id")) == parentId)
		{
			int dummy;

			TeLinearRing ring = getLinearRing(dummy);
			pol.add(ring);
		}
		else
			return true;
	}

	return false;
}

bool TePostgreSQLPortal::fetchGeometry(TeLine2D& line)
{
	errorMessage_ = "";

	tepg_recordset_.getPGLine2D("spatial_data", line);

	int geomId = tepg_recordset_.getInt("geom_id");

	string objectId = tepg_recordset_.value("object_id");

	line.objectId(objectId);
	
	line.geomId(geomId);

	return fetchRow();
}

bool TePostgreSQLPortal::fetchGeometry(TeNode& n)
{
	errorMessage_ = "";
	
	TeCoord2D pt(getDouble("x"), getDouble("y"));

	n.add(pt);
	n.geomId(atoi(getData("geom_id")));
	n.objectId(string(getData("object_id")));

	return fetchRow();
}

bool TePostgreSQLPortal::fetchGeometry(TePoint& p)
{
	errorMessage_ = "";

	p.location().setXY(getDouble("x"), getDouble("y"));
	p.geomId(atoi(getData("geom_id")));
	p.objectId(string(getData("object_id")));
	
	return fetchRow();
}

TeLinearRing TePostgreSQLPortal::getLinearRing(int& numberOfHoles)
{
	int geomId = tepg_recordset_.getInt("geom_id");
	string objectId = tepg_recordset_.value("object_id");

	numberOfHoles = tepg_recordset_.getInt("num_holes");	

	TeLine2D line;

	tepg_recordset_.getPGLine2D("spatial_data", line);

	TeLinearRing ring = line;

	ring.objectId(objectId);
	
	ring.geomId(geomId);

	return ring;
}

string TePostgreSQLPortal::escapeSequence(const string& from)
{
	size_t newLen = 0;

	char* aux = TePGConnection::escapeString(from.c_str(), from.length(), newLen);

	string str = aux;

	delete [] aux;

	return str;
}
