/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include <libsyncml/syncml.h>
#include <libsyncml/syncml_internals.h>

#ifdef ENABLE_OBEX
#include <libsyncml/sml_transport_internals.h>

#include "obex_server.h"
#include "obex_server_internals.h"

#include <fcntl.h>
#include <sys/poll.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <termios.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

static gboolean _dispatch_obex(gpointer data)
{
	SmlLinkObexServerEnv *linkenv = data;
	
	if (OBEX_HandleInput(linkenv->handle, 0) < 0) {
		SmlError *error = NULL;
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to handle input");
		smlTransportReceiveEvent(linkenv->env->tsp, linkenv->link, SML_TRANSPORT_EVENT_ERROR, NULL, error);
		smlTrace(TRACE_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
		smlErrorDeref(&error);
		return FALSE;
	}
	
	if (linkenv->disconnect) {
		smlTrace(TRACE_INTERNAL, "disconnecting link");
		OBEX_TransportDisconnect(linkenv->handle);
		close(linkenv->fd);
		linkenv->destroy = TRUE;
	}
	
	
	if (linkenv->destroy) {
		smlTrace(TRACE_INTERNAL, "Destroying link %p %p", linkenv, linkenv->link);
		smlTransportReceiveEvent(linkenv->env->tsp, linkenv->link, SML_TRANSPORT_EVENT_DISCONNECT_DONE, NULL, NULL);
		smlLinkDeref(linkenv->link);
		g_source_unref(linkenv->source);
		OBEX_Cleanup(linkenv->handle);
		g_free(linkenv);
		return FALSE;
	}
	return TRUE;
}

static void _smlObexEvent(obex_t *handle, obex_object_t *object, int mode, int event, int obex_cmd, int obex_rsp)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %i, %i, %i, %i)", __func__, handle, object, mode, event, obex_cmd, obex_rsp);
	SmlLinkObexServerEnv *linkenv = OBEX_GetUserData(handle);
	SmlError *error = NULL;
	obex_headerdata_t header;
	uint8_t headertype = 0;
	uint32_t len = 0;
	uint32_t conid = 0;		
	SmlMimeType mimetype = SML_MIMETYPE_UNKNOWN;
	
	if (linkenv->disconnect) {
		OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
		smlTrace(TRACE_EXIT, "%s: This link was disconnected", __func__);
		return;
	}
	
	switch (event)  {
		case OBEX_EV_PROGRESS:
			smlTrace(TRACE_INTERNAL, "Progress");
			break;
		case OBEX_EV_REQDONE:
			smlTrace(TRACE_INTERNAL, "Done");
			if (obex_cmd == OBEX_CMD_DISCONNECT) {
				linkenv->destroy = TRUE;
			}
			break;
        case OBEX_EV_REQHINT:
			smlTrace(TRACE_INTERNAL, "Hint");
			/* Comes BEFORE the lib parses anything. */
			switch (obex_cmd) {
				case OBEX_CMD_GET:
					/* At this point we received a get request. If we dont have any data
					 * to send yet, we iterate the main context until the data arrived. */
					while (!linkenv->send_data && !linkenv->disconnect && !linkenv->error) {
						if (!g_main_context_pending(linkenv->env->tsp->context)) {
							usleep(100);
							continue;
						}
						smlTrace(TRACE_INTERNAL, "Dispatching command queue since we dont have data to send");
						g_main_context_iteration(linkenv->env->tsp->context, TRUE);
					}
					
					/* We received a disconnect event so we reply with an error */
					if (linkenv->disconnect) {
						OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
						smlTrace(TRACE_INTERNAL, "Get was aborted");
						break;
					}
				case OBEX_CMD_PUT:
				case OBEX_CMD_CONNECT:
				case OBEX_CMD_DISCONNECT:
					OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
					break;
				default:
					OBEX_ObjectSetRsp(object, OBEX_RSP_NOT_IMPLEMENTED, OBEX_RSP_NOT_IMPLEMENTED);
					break;
			}
			break;
        case OBEX_EV_REQ:
			switch (obex_cmd) {
				case OBEX_CMD_SETPATH:
					smlTrace(TRACE_INTERNAL, "Got SETPATH command");
					OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
					break;
				case OBEX_CMD_PUT:;
					smlTrace(TRACE_INTERNAL, "Got PUT command");
					uint32_t length = 0;
					char *body = NULL;
					
					while (OBEX_ObjectGetNextHeader(handle, object, &headertype, &header, &len)) {
						smlTrace(TRACE_INTERNAL, "Next header %i, %d, %p", headertype, header.bq4, header.bs);
						switch (headertype) {
							case OBEX_HDR_CONNECTION:
								smlTrace(TRACE_INTERNAL, "Found connection number: %d", header.bq4);
								conid = header.bq4;
								break;
							case OBEX_HDR_TYPE:;
								char *mimetypestr = g_strndup((char *)header.bs, len);
								/*char *mimetypestr = g_malloc0(len);// / 2 + 1);
								if (OBEX_UnicodeToChar((unsigned char *)mimetypestr, header.bs, len / 2 + 1) == -1) {
									g_free(mimetypestr);
									smlErrorSet(&error, SML_ERROR_GENERIC, "unable to convert from unicode");
									goto error;
								}*/
								
								smlTrace(TRACE_INTERNAL, "Found type: %s", mimetypestr);
								
								if (!strcmp(mimetypestr, SML_ELEMENT_WBXML)) {
									mimetype = SML_MIMETYPE_WBXML;
								} else if (!strcmp(mimetypestr, SML_ELEMENT_XML)) {
									mimetype = SML_MIMETYPE_XML;
								} else if (!strcmp(mimetypestr, SML_ELEMENT_SAN)) {
									mimetype = SML_MIMETYPE_SAN;
								} else {
									g_free(mimetypestr);
									smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown mime type");
									goto error;
								}
								
								g_free(mimetypestr);
								break;
							case OBEX_HDR_LENGTH:
								smlTrace(TRACE_INTERNAL, "Found length: %d", header.bq4);
								length = header.bq4;
								break;
							case OBEX_HDR_BODY:
								if (!length) {
									smlErrorSet(&error, SML_ERROR_GENERIC, "Length must come before the body");
									goto error;
								}
								
								body = smlTryMalloc0(length, &error);
								if (!body)
									goto error;
								
								memcpy(body, header.bs, length);
								break;
							default:
								smlTrace(TRACE_INTERNAL, "Unknown header");
						}
					}
					
					if (!conid) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing connection id");
						goto error;
					}
					
					if (conid != linkenv->conid) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Wrong connection id");
						goto error;
					}
					
					if (mimetype == SML_MIMETYPE_UNKNOWN) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing mime type");
						goto error;
					}
					
					if (!length) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing length");
						goto error;
					}
					
					if (!body) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing body");
						goto error;
					}
					
					SmlTransportData *tspdata = smlTransportDataNew(body, length, mimetype, TRUE, &error);
					if (!tspdata)
						goto error;
					
					if (smlTransportReceiveEvent(linkenv->env->tsp, linkenv->link, SML_TRANSPORT_EVENT_DATA, tspdata, NULL))
						OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
					else
						OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
					
					smlTransportDataDeref(tspdata);
					break;
				case OBEX_CMD_GET:;
					smlTrace(TRACE_INTERNAL, "Got GET command");
					
					while (OBEX_ObjectGetNextHeader(handle, object, &headertype, &header, &len)) {
						smlTrace(TRACE_INTERNAL, "Next header %i, %d, %p", headertype, header.bq4, header.bs);
						switch (headertype) {
							case OBEX_HDR_CONNECTION:
								smlTrace(TRACE_INTERNAL, "Found connection number: %d", header.bq4);
								conid = header.bq4;
								break;
							case OBEX_HDR_TYPE:;
								char *mimetypestr = g_strndup((char *)header.bs, len);
								/*char *mimetypestr = g_malloc0(len / 2 + 1);
								if (OBEX_UnicodeToChar((unsigned char *)mimetypestr, header.bs, len / 2 + 1) == -1) {
									g_free(mimetypestr);
									smlErrorSet(&error, SML_ERROR_GENERIC, "unable to convert from unicode");
									goto error;
								}*/
								
								smlTrace(TRACE_INTERNAL, "Found type: %s", mimetypestr);
								
								if (!strcmp(mimetypestr, SML_ELEMENT_WBXML)) {
									mimetype = SML_MIMETYPE_WBXML;
								} else if (!strcmp(mimetypestr, SML_ELEMENT_XML)) {
									mimetype = SML_MIMETYPE_XML;
								} else if (!strcmp(mimetypestr, SML_ELEMENT_SAN)) {
									mimetype = SML_MIMETYPE_SAN;
								} else {
									g_free(mimetypestr);
									smlErrorSet(&error, SML_ERROR_GENERIC, "Unknown mime type");
									goto error;
								}
								
								g_free(mimetypestr);
								break;
							default:
								smlTrace(TRACE_INTERNAL, "Unknown header");
						}
					}
					
					if (!conid) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing connection id");
						goto error;
					}
					
					if (conid != linkenv->conid) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Wrong connection id");
						goto error;
					}
					
					if (mimetype == SML_MIMETYPE_UNKNOWN) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing mime type");
						goto error;
					}
					
					if (linkenv->error) {
						if (smlErrorGetClass(&(linkenv->error)) <= SML_ERRORCLASS_RETRY)
							OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
						else
							OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
						
						smlErrorDeref(&(linkenv->error));
						linkenv->error = NULL;
						smlTrace(TRACE_INTERNAL, "Sent error in response to get");
						break;
					}
					
					if (!linkenv->send_data) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "No data to send");
						goto error;
					}
					
					if (mimetype != linkenv->send_data->type) {
						smlErrorSet(&error, SML_ERROR_GENERIC, "Wrong mimetype requested");
						goto error;
					}
					
					/* Now the data and size */
					header.bq4 = (uint32_t)linkenv->send_data->size;
					OBEX_ObjectAddHeader(handle, object, OBEX_HDR_LENGTH, header, sizeof(uint32_t), 0);
					
					header.bs = (unsigned char *)linkenv->send_data->data;
					OBEX_ObjectAddHeader(handle, object, OBEX_HDR_BODY, header, linkenv->send_data->size, 0);
					
					smlTransportDataDeref(linkenv->send_data);
					linkenv->send_data = NULL;
					
					OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
					break;
				case OBEX_CMD_CONNECT:;
					smlTrace(TRACE_INTERNAL, "Got CONNECT command");
					char *target = NULL;
					
					while (OBEX_ObjectGetNextHeader(handle, object, &headertype, &header, &len)) {
						smlTrace(TRACE_INTERNAL, "Next header %i, %d, %p", headertype, header.bq4, header.bs);
						switch (headertype) {
							case OBEX_HDR_TARGET:
								target = g_strndup((char *)header.bs, len);
								smlTrace(TRACE_INTERNAL, "Found target: %s", target);
								break;
							default:
								smlTrace(TRACE_INTERNAL, "Unknown header");
						}
					}
					
					if (!target || strcmp(target, "SYNCML-SYNC")) {
						g_free(target);
						smlErrorSet(&error, SML_ERROR_GENERIC, "Missing target");
						goto error;
					}
					g_free(target);
					
					header.bq4 = linkenv->conid;
					OBEX_ObjectAddHeader(handle, object, OBEX_HDR_CONNECTION, header, sizeof(linkenv->conid), OBEX_FL_FIT_ONE_PACKET);
					
					header.bs = (unsigned char *)"SYNCML-SYNC";
					OBEX_ObjectAddHeader(handle, object, OBEX_HDR_WHO, header, strlen((char *)header.bs), OBEX_FL_FIT_ONE_PACKET);
 
					
					linkenv->link = smlLinkNew(linkenv->env->tsp, linkenv, &error);
					if (!linkenv->link)
						goto error;
					
					if (smlTransportReceiveEvent(linkenv->env->tsp, linkenv->link, SML_TRANSPORT_EVENT_CONNECT_DONE, NULL, NULL))
						OBEX_ObjectSetRsp(object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
					else {
						OBEX_ObjectSetRsp(object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
						linkenv->disconnect = TRUE;
					}
					break;
				case OBEX_CMD_DISCONNECT:
					smlTrace(TRACE_INTERNAL, "Got DISCONNECT command");
					OBEX_ObjectSetRsp(object, OBEX_RSP_SUCCESS, OBEX_RSP_SUCCESS);
					break;
				default:
					smlTrace(TRACE_INTERNAL, "Denied %02x request", obex_cmd);
					OBEX_ObjectSetRsp(object, OBEX_RSP_NOT_IMPLEMENTED, OBEX_RSP_NOT_IMPLEMENTED);
					break;
			}

			break;
		case OBEX_EV_LINKERR:
			smlTrace(TRACE_INTERNAL, "Link broken (this does not have to be an error)!");
			break;
		default:
			smlTrace(TRACE_INTERNAL, "Unknown event!");
			break;
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error:
	OBEX_ObjectSetRsp(object, OBEX_RSP_BAD_REQUEST, OBEX_RSP_BAD_REQUEST);
	smlTransportReceiveEvent(linkenv->env->tsp, linkenv->link, SML_TRANSPORT_EVENT_ERROR, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

/* These function supervise the file descriptor where we are listening for incoming
 * connections */
static gboolean _fd_prepare(GSource *source, gint *timeout_)
{
	/* There is not need to dispatch this too fast since
	 * it does not influence the transmission of data, only
	 * the connection */
	*timeout_ = 50;
	return FALSE;
}

static gboolean _fd_check(GSource *source)
{
	SmlTransportObexServerEnv *env = *((SmlTransportObexServerEnv **)(source + 1));
	fd_set rfds;
	FD_ZERO(&rfds);
	FD_SET(env->fd, &rfds);
           
	struct timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = 1;

	int retval = select(env->fd + 1, &rfds, NULL, NULL, &tv);
	
	if (retval == -1)
		smlTrace(TRACE_ERROR, "%s: Unable to select()", __func__);
	else if (retval)
        return TRUE;
	
	return FALSE;
}

static gboolean _fd_dispatch(GSource *source, GSourceFunc callback, gpointer user_data)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, source, callback, user_data);
	SmlTransportObexServerEnv *env = user_data;
	SmlError *error = NULL;
	
	struct sockaddr_in addr;
	memset(&addr, 0, sizeof(addr));
	socklen_t addrlen = sizeof(addr);
	
	smlTrace(TRACE_INTERNAL, "dispatch %i", env->fd);
	
	/* Create the env that handles this link */
	SmlLinkObexServerEnv *linkenv = smlTryMalloc0(sizeof(SmlLinkObexServerEnv), &error);
	if (!linkenv)
		goto error;
	linkenv->env = env;
	
	linkenv->fd = accept(env->fd, (struct sockaddr *)&addr, &addrlen);
	if (linkenv->fd == -1) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "error: %s, %i", strerror(errno), errno);
		goto error_free_env;
	}
	smlTrace(TRACE_INTERNAL, "New fs %i", linkenv->fd);
	
	
	linkenv->handle = OBEX_Init(OBEX_TRANS_FD, _smlObexEvent, 0);
	if (!linkenv->handle) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to create new handle");
		goto error_close_fd;
	}
		
	/* Start the obex transport */
	if (FdOBEX_TransportSetup(linkenv->handle, linkenv->fd, linkenv->fd, 4096) < 0) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "Unable to setup");
		goto error_clean_obex;
	}
	
	/* Now we create a source that watches the new obex connection */
	linkenv->source = g_idle_source_new();
	g_source_set_callback(linkenv->source, _dispatch_obex, linkenv, NULL);
	g_source_attach(linkenv->source, env->tsp->context);
	
	/* We first have to add the connection id which we generate for this transport */
	env->lastConId++;
	linkenv->conid = env->lastConId;

	OBEX_SetUserData(linkenv->handle, linkenv);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_clean_obex:
	OBEX_Cleanup(linkenv->handle);
error_close_fd:
	close(linkenv->fd);
error_free_env:
	g_free(linkenv);
error:
	smlTransportReceiveEvent(env->tsp, NULL, SML_TRANSPORT_EVENT_ERROR, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	return TRUE;
}

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/

static void *smlTransportObexServerInit(SmlTransport *tsp, const void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, tsp, data, error);
	smlAssert(tsp);
	smlAssert(data);
	
	SmlTransportObexServerEnv *env = smlTryMalloc0(sizeof(SmlTransportObexServerEnv), error);
	if (!env)
		goto error;
	
	const SmlTransportObexServerConfig *conf = data;
	env->tsp = tsp;
	env->path = g_strdup(conf->path);
	env->lastConId = 0;
	
	switch (conf->type) {
		case SML_OBEX_SERVER_TYPE_NET:;
			struct sockaddr_in addr;
			memset(&addr, 0, sizeof(addr));
			
			addr.sin_family = AF_INET;
			addr.sin_port = htons(conf->port);
			addr.sin_addr.s_addr = INADDR_ANY;

			env->fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
			if (env->fd < 0) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Cannot create socket: %s", strerror(errno));
				goto error_free_env;
			}
			
			smlTrace(TRACE_INTERNAL, "fd is %i", env->fd);
			
			if (fcntl(env->fd, F_SETFL, O_NONBLOCK) < 0) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Cannot control socket: %s", strerror(errno));
				goto error_close_handle;
			}
			
			if (bind(env->fd, (struct sockaddr*) &addr, sizeof(struct sockaddr_in))) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Cannot bind socket: %s", strerror(errno));
				goto error_close_handle;
			}
			smlTrace(TRACE_INTERNAL, "bind successfull");

			if (listen(env->fd, 256) != 0) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Cannot listen on socket: %s", strerror(errno));
				goto error_close_handle;
			}
			
			smlTrace(TRACE_INTERNAL, "listen successfull on %i", env->fd);
			break;
		case SML_OBEX_SERVER_TYPE_SERIAL:
			/*env->obexhandle = OBEX_Init(OBEX_TRANS_FD, _smlObexEvent, 0);
			if (!env->obexhandle) {
				smlErrorSet(error, SML_ERROR_GENERIC, "Unable to open connection");
				goto error_free_env;
			}*/
						
			break;
		default:
			smlErrorSet(error, SML_ERROR_GENERIC, "Unknown obex type");
			goto error_free_env;
	}
	
	/* Now we create a source that watches for incoming connection requests */
	env->functions = smlTryMalloc0(sizeof(GSourceFuncs), error);
	if (!env->functions)
		goto error_close_handle;
	env->functions->prepare = _fd_prepare;
	env->functions->check = _fd_check;
	env->functions->dispatch = _fd_dispatch;
	env->functions->finalize = NULL;

	env->source = g_source_new(env->functions, sizeof(GSource) + sizeof(SmlTransportObexServerEnv *));
	SmlTransportObexServerEnv **envptr = (SmlTransportObexServerEnv **)(env->source + 1);
	*envptr = env;
	g_source_set_callback(env->source, NULL, env, NULL);
	g_source_attach(env->source, tsp->context);
	if (tsp->context)
		g_main_context_ref(tsp->context);
    
	smlTrace(TRACE_EXIT, "%s: %p", __func__, env);
	return env;

error_close_handle:
	close(env->fd);
error_free_env:
	g_free(env->path);
	g_free(env);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}

static SmlBool smlTransportObexServerFinalize(void *data, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, data, error);
	smlAssert(data);
	SmlTransportObexServerEnv *env = data;
	
	smlAssert(env->tsp);
	
	close(env->fd);
	g_free(env->functions);
	//g_source_destroy(env->source);
	g_source_unref(env->source);
	g_free(env->path);
	
	g_free(env);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
}

static void smlTransportObexServerSend(void *userdata, void *link, SmlTransportData *data, SmlError *error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, userdata, link, data, error);
	smlAssert(data || error);
	smlAssert(userdata);
	SmlLinkObexServerEnv *linkenv = link;
	
	if (error) {
		smlAssert(!data);
		linkenv->error = error;
		
		smlTrace(TRACE_EXIT, "%s: Error set", __func__);
		return;
	}
	
	if (linkenv->send_data) {
		smlErrorSet(&error, SML_ERROR_GENERIC, "We already have waiting data");
		goto error;
	}
	
	linkenv->send_data = data;
	smlTransportDataRef(linkenv->send_data);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error:
	smlErrorDeref(&error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	return;
}

static void smlTransportObexServerDisconnect(void *data, void *linkdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, data, linkdata);
	smlAssert(data);
	smlAssert(linkdata);
	SmlLinkObexServerEnv *linkenv = linkdata;
	
	linkenv->disconnect = TRUE;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
}

/** @brief Function description
 * 
 * @param parameter This parameter does great things
 * @returns What you always wanted to know
 * 
 */
SmlBool smlTransportObexServerNew(SmlTransport *tsp, SmlError **error)
{
	tsp->functions.initialize = smlTransportObexServerInit;
	tsp->functions.finalize = smlTransportObexServerFinalize;
	tsp->functions.send = smlTransportObexServerSend;
	tsp->functions.disconnect = smlTransportObexServerDisconnect;
	return TRUE;
}

#endif //ENABLE_OBEX

/*@}*/
