/*
    LIBSNSP - A C Library for the Simple Network Scanning Protocol
    Copyright (C) 2001 Michael R. Kllejan 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* $Id: reply.c,v 1.2 2001/09/26 17:37:25 michael Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include "reply.h"
#include "util.h"

/* snsp_reply_line, snsp_reply */

snsp_reply_line_t *
snsp_reply_line_alloc ()
{
  snsp_reply_line_t *reply_line;

  reply_line = malloc (sizeof (snsp_reply_line_t));
  reply_line->code = LINKED_LIST_ALLOC;
  reply_line->parameters = LINKED_LIST_ALLOC;
  return reply_line;
}

void
snsp_reply_line_free (snsp_reply_line_t ** reply_line)
{
  if ((*reply_line)->code)
    LINKED_LIST_FREE (&((*reply_line)->code));
  LINKED_LIST_FREE (&((*reply_line)->parameters));
  free (*reply_line);
  *reply_line = NULL;
}

char *
snsp_reply_line_render (snsp_reply_line_t * reply_line,
			snsp_reply_line_linetype_t mline)
{
  char *str, *reply_line_str = malloc (SNSP_REPLYLINE_LEN_MAX);
  reply_line_str[0] = '\0';
  
  strcat (reply_line_str, reply_line->type ? "+" : "-");
  strcat (reply_line_str, mline ? "_" : " ");

  reply_line->code = LINKED_LIST_FIRST (reply_line->code);
  reply_line->parameters = LINKED_LIST_FIRST (reply_line->parameters);

  do {

      if ((!reply_line->code) || (!reply_line->code->data)) break;

      str = reply_line->code->data;
      strcat (reply_line_str, str);
      strcat (reply_line_str, "-");
      
  } while (LINKED_LIST_NEXT (&(reply_line->code)));
  
  if ( reply_line_str[ strlen (reply_line_str) -1] == '-' )
    reply_line_str[ strlen (reply_line_str) -1] = '\0';
  
  if ( (LINKED_LIST_LEN (reply_line->code)) == 0 ) 
    strcat( reply_line_str, "*" );
 
  do {

    if ((!reply_line->parameters) || (!reply_line->parameters->data)) break;
  
    str = reply_line->parameters->data;
    if ((strchr (str, ' ')) != (char) NULL) {
  	  if (LINKED_LIST_LEN(reply_line->parameters) == 1) {
        strcat (reply_line_str, "\" ");
        strcat (reply_line_str, str);
      } else {
        if ((strchr (str, '\"')) != (char) NULL )
          continue;
        strcat (reply_line_str, " ");
        strcat (reply_line_str, "\"");
    	  strcat (reply_line_str, str);
  	    strcat (reply_line_str, "\"");
      };
	  } else {
      strcat (reply_line_str, " ");
	    strcat (reply_line_str, str);
    };

  } while (LINKED_LIST_NEXT (&(reply_line->parameters)));

  strcat (reply_line_str, "\r\n");

  return reply_line_str;
};

snsp_reply_line_t *
snsp_reply_line_parse (char *reply_line_str,
		       snsp_reply_line_linetype_t * mline)
{
  snsp_reply_line_t *reply_line = snsp_reply_line_alloc ();
  int i = 0, j = 0, k = 0, qs = 0;
  char *str = malloc (SNSP_REPLYLINE_LEN_MAX);
  int reply_line_str_len;

  if ((reply_line_str == NULL) || (mline == NULL))
    return NULL;

  reply_line_str_len = strlen (reply_line_str);

  if (reply_line_str_len > SNSP_REPLYLINE_LEN_MAX)
    return NULL;

  switch (reply_line_str[i++]) {
    case '+':
      reply_line->type = snsp_reply_line_type_positive;
      break;
    case '-':
      reply_line->type = snsp_reply_line_type_negative;
      break;
    default:
      return NULL;
  }

  switch (reply_line_str[i++]) {
    case ' ':
      *mline = snsp_reply_line_linetype_singleline;
      break;
    case '_':
      *mline = snsp_reply_line_linetype_multiline;
      break;
    default:
      return NULL;
  }

  while (1) {
    if (reply_line_str[i] == '*') {
      LINKED_LIST_PUSH(reply_line->code, "*", 2);
      break;
    };
    
    while (isalpha (reply_line_str[i])) {
	    str[j++] = reply_line_str[i++];
	    if ((j > SNSP_REPLYLINE_CODE_LEN) || (i >= reply_line_str_len))
	      goto error;
	  };

    if (strlen (str) != SNSP_REPLYLINE_CODE_LEN)
	    goto error;
    if (++k > SNSP_REPLYLINE_CODE_MAX)
	    goto error;
    
    str[j] = '\0'; j = 0;
    LINKED_LIST_PUSH (reply_line->code, (void *) str, strlen (str) + 1);

    if (snsp_isspace (reply_line_str[i]) || reply_line_str[i] == '\r')
	    break;

    if (reply_line_str[i++] != '-')
	    goto error;
    if (i >= reply_line_str_len)
	    goto error;
  }

  if (k < SNSP_REPLYLINE_CODE_MIN)
    goto error;

  while (1) {
    k = 0; j = 0;

    while (snsp_isspace (reply_line_str[i++])) {
	    if ((++k >= SNSP_WHITESPACE_MAX) || (i >= reply_line_str_len))
	      goto error;
	  }; i--;

    qs = (reply_line_str[i] == '\"') ? (i++, 1) : 0;
    
    if (i >= reply_line_str_len)
	    goto error;

    while ((qs) ? (reply_line_str[i] != '\"') :
           (snsp_isspace (reply_line_str[i]) == 0)) {

  	  if (snsp_isprint (reply_line_str[i]) ||
	        snsp_isspace (reply_line_str[i]))
	      str[j++] = reply_line_str[i++];
	    else {
	      if (reply_line_str[i] == '\r')
		      break;
	      else
		      goto error;
	    }

	    if (j > SNSP_REPLYLINE_PARA_MAX)
	      goto error;
    }; 
    
    i++; str[j] = '\0';
    LINKED_LIST_PUSH (reply_line->parameters, (void *) str,
    strlen (str) + 1);

    if (reply_line_str[i] == '\r')
	    break;
  }

  if (i++ >= reply_line_str_len)
    goto error;
  if (reply_line_str[i] == '\n')
    goto success;

error:
  snsp_reply_line_free (&reply_line);
  reply_line = NULL;
success:
  free (str);
  return reply_line;
};

snsp_reply_t *
snsp_reply_alloc ()
{
  snsp_reply_t *reply;

  reply = malloc (sizeof (snsp_reply_t));
  reply->lines = LINKED_LIST_ALLOC;

  return reply;
}

void
snsp_reply_free (snsp_reply_t ** reply)
{

  (*reply)->lines = LINKED_LIST_FIRST ((*reply)->lines);

  if ((*reply)->lines == NULL)
    return;

  do {

    snsp_reply_line_free (((snsp_reply_line_t**) (&(*reply)->lines->data)));
    
  } while (LINKED_LIST_NEXT (&(*reply)->lines));

  LINKED_LIST_FREE (&(*reply)->lines);

  free (*reply);
  *reply = NULL;
}

char *
snsp_reply_render (snsp_reply_t * reply)
{
  snsp_reply_line_linetype_t line_type;
  char *reply_str = malloc (SNSP_REPLYLINE_LEN_MAX * SNSP_REPLYLINE_MAX);
  LINKED_LIST_T(snsp_reply_line_t*) *lines = (void*)reply->lines;

  reply_str[0] = '\0';
  
  if (!lines) return NULL;
  
  do {
    if (lines->next == NULL)
      line_type = snsp_reply_line_linetype_singleline;
    else
      line_type = snsp_reply_line_linetype_multiline;
   
    strcat (reply_str, snsp_reply_line_render (lines->data, line_type));

  } while (LINKED_LIST_NEXT (&lines));

  return reply_str;
};

snsp_reply_t *
snsp_reply_parse (char *reply_str)
{
  char *next_str, *this_str;
  snsp_reply_t *reply = snsp_reply_alloc ();
  snsp_reply_line_t *reply_line = NULL;
  snsp_reply_line_linetype_t *mline =
    malloc (sizeof (snsp_reply_line_linetype_t));

  next_str = reply_str;

  while (*(this_str = next_str) != '\0') {
    reply_line = snsp_reply_line_alloc ();
    reply_line = snsp_reply_line_parse (this_str, mline);
    LINKED_LIST_PUSH (reply->lines, reply_line, sizeof (snsp_reply_line_t));
    next_str = (char *) (strstr (this_str, "\r\n") + 2);
  }

  free (mline);

  return reply;
};

snsp_reply_line_t *
snsp_reply_line_create (snsp_reply_line_type_t reply_line_type, ...)
{
  snsp_reply_line_t *reply_line = snsp_reply_line_alloc ();
  va_list ap;
  char *str;

  reply_line->type = reply_line_type;

  va_start (ap, reply_line_type);

  while ((str = va_arg (ap, char *)) != NULL)
      LINKED_LIST_PUSH (reply_line->code, str, strlen (str) + 1);

  while ((str = va_arg (ap, char *)) != NULL)
      LINKED_LIST_PUSH (reply_line->parameters, str, strlen (str) + 1);

  va_end (ap);

  return reply_line;
};

snsp_reply_t *
snsp_reply_create (char dummy, ...)
{

  snsp_reply_line_t *reply_line = NULL;
  snsp_reply_line_linetype_t reply_line_linetype =
    snsp_reply_line_linetype_multiline;
  snsp_reply_t *reply = snsp_reply_alloc ();
  va_list ap;

  va_start (ap, dummy);

  while ((reply_line = va_arg (ap, snsp_reply_line_t *)) != NULL)
    LINKED_LIST_PUSH (reply->lines, reply_line, sizeof (snsp_reply_line_t));

  va_end (ap);

  return reply;
}
