#ifndef PACKAGE_NAMESPACE_PATHFINDER_H
#define PACKAGE_NAMESPACE_PATHFINDER_H
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain



#include <string>
#include <list>
#include <map>
#include <iostream>

// #include <PACKAGE_NAMESPACE/StringList.h>	// todo: get rid of this in the public interface.

namespace PACKAGE_NAMESPACE
{
        /**
           path_finder searches for keys using a set of prefixes
           (paths) and suffixes (file extensions).
           
           Example:

           <pre>
           path_finder p;
           p.path( "/lib/lib:/usr/lib/lib" );
           p.extensions( ".a:.so" );
           cout << p.find( "z" );
           </pre>

           That would print an empty string if it finds nothing, or a
           string if it finds any of the following:

           - z
           - /lib/libz
           - /lib/libz.a
           - /lib/libz.so
           - /usr/lib/libz
           - /usr/lib/libz.a
           - /usr/lib/libz.so



           Please excuse the large number of unnecessary i/o operators
           in this interface: this is old code from when i started
           learning STL and wanted to try a lot of stuff out. :/
         */
	class path_finder
	{
	      public:

                /**
                   A list type returned by some functions.
                 */
                typedef std::list<std::string> string_list;


		/**
                   Creates object with the given path/extension list.
                */
                path_finder( const std::string & path = std::string(), const std::string & ext = std::string(), const std::string & pathsep = ":" );

                virtual ~path_finder();
                
		/**
                   Returns a ":"-separated string of all paths added via add/path().
                */
		  std::string path_string() const;

                /**
                   Sets the string used as a separator for the
                   string-based variants of path(), extentions(), etc.
                */
		void path_separator( const std::string & sep );

                /**
                   Returns the path separator string. Default is ":";
                 */
		const std::string & path_separator() const;

		/**
                   Sets the path to p, which should be a path_separtor()-delimited string.
                   Returns the number of path elements parsed from p.
                */
		virtual unsigned int path( const std::string & p );

                /**
                   Sets the path to the given list of directories.
                 */
		virtual unsigned int path( const string_list & p );

		/**
                   Adds p to the path. May be path_separtor()-delimited.
                */
		virtual void add_path( const std::string & p );

		/**
                   Adds a "search extension." Sample:
                   finder.extension( ".txt:README" ); Will now try
                   all path combinations with the rightmost characters
                   matching ext. Be sure to include a period if you
                   want that searched (that is so this class can be
                   used to find non-files and those with
                   non-traditional extensions).
                */
		virtual void add_extension( const std::string & ext = std::string() );
		/**
                   like extension(), but overwrites extension list.
                */
		virtual unsigned int extensions( const std::string & ext );
                /**
                   Sets the extensions list to the given list.
                 */
		virtual unsigned int extensions( const string_list & p );

		/**
                   Returns the path_separator()-delimited listed of file
                   suffixes to use when searching for a path.
                */
		  std::string extensions_string() const;
                /**
                   Returns this object's extensions list.
                 */
		const string_list & extensions() const;

                /**
                   Helper function to collapse a list into a string.
                 */
                std::string join_list( const string_list & list, const std::string & separator ) const;

                /**
                   Returns true if path is readable.
                 */
		static bool is_accessible( const std::string & path );

		/**
                   Returns the "base name" of the given string: any part
                   following the final directory separator character.
                */
		static std::string basename( const std::string & );

                /**
                   Returns a platform-dependent path separator. This
                   is set when the class is compiled.
                 */
		static std::string dir_separator();

		/**
                   Returns the full path of the given resource, provided it could be found
                   in all paths or with all extensions added and provided it is readable.
                   Note that this might return a relative path, especially if the
                   resourcename passed to it immediately resolves to an existing resource.
                   It returns an empty string if the resourcename cannot be found in
                   the filesystem tree (or is otherwise unaccessible).

                   If check_cache is false then this function ignores
                   it's lookup cache and searches again, otherwise it
                   uses a cache. When caching it will always return
                   the same result for any given resourcename.
                */
		std::string find( const std::string & resourcename, bool check_cache = true ) const;

		/**
                   Returns a list of all items added via add_path() and path().
                */
		const string_list & path() const;


	      private:
		string_list paths;
		string_list exts;
		std::string pathseparator;
		typedef std::map < std::string, std::string > StringStringMap;
		typedef StringStringMap::iterator StringStringIterator;
		mutable StringStringMap hitcache;
	};


        /**
          operator << adds this object's path to os.
        */
        std::ostream & operator <<( std::ostream & os, const path_finder & );

        /**
           Similar to the ostream version, it exports this
           object's path to the given string.
        */
        std::string & operator <<( std::string & os, const path_finder & );

        /**
          Adds p to this object's path().
        */
        path_finder & operator +=( path_finder & list, const std::string & p );

};				// namespace PACKAGE_NAMESPACE
#endif // PACKAGE_NAMESPACE_PATHFINDER_H
