////////////////////////////////////////////////////////////////////////////////
// main_dn.cpp
// Implmentation for s11nconvert utility, to convert data between the various
// Serializers.
// Author: stephan@s11n.net
// License: Public Domain
////////////////////////////////////////////////////////////////////////////////


#include <iostream>
#include <string>
#include <memory> // auto_ptr

#include <unistd.h> // isatty()

#include "config.h" // S11N_NS_S11N_LIBRARY_VERSION, PACKAGE_{VERSION,NAME}

#include <S11N_NS/argv_parser.h> // argv_parser class
#include <S11N_NS/debuggering_macros.h> // CERR
#include <S11N_NS/file_util.h> // get_i/ostream()

#include <S11N_NS/s11n.h> // s11n core framework

#include <S11N_NS/s11nlite.h> // s11nlite framework

#include <S11N_NS/pointer_cleaner.h> // pointer_cleaner class
#include <S11N_NS/algo.h> // free_list_entries<>()

#define APP_NAME "s11nconvert"
#define APP_VERSION "$Revision: 1.2 $"
#define APP_LICENSE "Public Domain"

bool verbose = false;
#define VERBOSE if( verbose ) CERR

typedef S11N_NS::argv_parser ARGV;


void show_version_info( std::ostream & os ) {
        os << APP_NAME << " version " << APP_VERSION << "\n";
        os << "Built with libs11n ["  << S11N_NS_S11N_LIBRARY_VERSION << "]\n";
        os << "License: " << APP_LICENSE << "\n";
}
int
main( int argc, char **argv )
{
        ARGV & args = ARGV::args( argc, argv );

        if( args.get_bool( "v", false ) || args.get_bool( "verbose", false ) )
        {
                verbose = true;
        }

        if( args.get_bool( "cldebug", false ) )
        {
                S11N_NS::class_loader_debug_level( 1 );
        }

        enum {
        ErrorNoSerializer = 1,
        ErrorFileNotFound,
        ErrorNoInputFile,
        ErrorInputReadFailed,
        ErrorOutputOpenFailed,
        ErrorLoadNodeFailed,
        ErrorSaving
        };


        static std::string const known_ser = "compact funtxt funxml parens simplexml";

        args.set_help( "?, help",
                       "print help text" );
        args.set_help( "f filename, file=filename",
                       "specify input filename. Same as --file. Filename of - means stdin." );
        args.set_help( "z",
                       "Enable gzip compression (works only for files). Overrides -bz." );
        args.set_help( "bz",
                       "Enable bz2ip compression (works only for files)." );
	args.set_help( "o filename, output=filenmae",
                       "specify output filename. Filename of - means stdout." );
        args.set_help( "version, V",
                       "print app and libs11n version info and quit." );

        args.set_help( "cldebug",
                       "toggles on classloader debug output." );


        args.set_help( "s SERIALIZER, format SERIALIZER",
                       std::string("output format (i.e., a Serializer's name). ") +
                       "Built-in serializers:\n\t\t" + known_ser + 
                       "\n\tAny name which is valid for s11nlite::create_serializer() is accepted. "
                       ); // "s" is for serializer

        if( ! args.is_set( "format" ) && ! args.is_set( "s" ) )
        {
                args.set( "?", 1 ); // trigger help later
        }

	if ( args.is_help_set() )
	{
                std::ostream & os = std::cout; 
                show_version_info( os );
                os << "Command-line arguments:\n"
                   << args.dump_help() << std::endl
                   << "\nThe -gz and -bz options only work if your "
                   << "libs11n was compiled with corresponding compression support.\n"
		   << "\n";
                   ;
                return 0;
	}

        if( args.is_set( "version" ) || args.is_set( "V" ) )
        {
                show_version_info( std::cout );
                return 0;
        }




#define ERR std::cerr << "Error: "        


        std::string fmt = args.get( "s", args.get( "format", "" ) );
        if( fmt.empty() )
        {
                ERR << "No output format specified. "
                    << "Try using [-s or --format] with one of:\n"
                    << "\t" + known_ser << "\n";
                return ErrorNoSerializer;
        }
        

        typedef std::auto_ptr<s11nlite::serializer_base_type> APSER;
        APSER ser = APSER( s11nlite::create_serializer( fmt ) );
        if( ! ser.get() )
        {
                ERR << "No Serializer found for name '"<<fmt<<"'. Try one of:"
                    << "\n\t" << known_ser << "\n";
                return ErrorNoSerializer;
        }

 
        S11N_NS::CompressionPolicy comp = S11N_NS::compression_policy();
        if( args.is_set( "z" ) ) comp = S11N_NS::GZipCompression;
        else if( args.is_set( "bz" ) ) comp = S11N_NS::BZipCompression;
        S11N_NS::compression_policy( comp );



        std::string ifname = args.get( "f", args.get( "file", "" ) );
        typedef S11N_NS::pointer_cleaner<std::istream> IStreamClean;
        IStreamClean icleaner;
        std::istream * is = 0;
        if( ! isatty(STDIN_FILENO) || ( "-" == ifname ) )
        {
                is = &std::cin;
        }
        else if( ifname.empty() )
        {
                ERR << "No input file specified.\n";
                return ErrorNoInputFile;
        }
        else
        {
                is = S11N_NS::get_istream( ifname );
                if( ! is )
                {
                        ERR << "Error reading input file! (Not found? Read access failed?)\n";
                        return ErrorInputReadFailed;
                }
                icleaner.add( is );
        }


        typedef S11N_NS::pointer_cleaner<std::ostream> OStreamClean;
        OStreamClean ocleaner;
        std::ostream * os = 0;

        std::string ofname = args.get( "o", args.get( "output","" ) );
        if( ofname.empty() || ("-" == ofname) )
        {
                os = &std::cout;
        }
        else
        {
                os = S11N_NS::get_ostream( ofname );
                if( ! os )
                {
                        ERR << "Error opening output file '"<<ofname<<"'!\n";
                        return ErrorOutputOpenFailed;
                }
                ocleaner.add( os );
        }

        // And FINALLY... we get down to the real work...
        // these couple lines almost now don't seem worth all the hassle ;)

        typedef std::auto_ptr<s11nlite::node_type> NAP;
        NAP n = NAP( s11nlite::load_node( *is ) );
        if( ! n.get() )
        {
                ERR << "Error loading node tree from input stream!\n";
                return ErrorLoadNodeFailed;
        }
        
        if( ! ser->serialize( *n, *os ) )
        {
                ERR << "Error saving output!\n";
                return ErrorSaving;
        }

	return 0;
#undef ERR
};



