/*
 * Copyright (C) The MX4J Contributors.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.adaptor.interceptor;

import java.lang.reflect.Method;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.loading.ClassLoaderRepository;

import mx4j.ImplementationException;
import mx4j.loading.RepositoryClassLoader;

/**
 * The interceptor that sets the context classloader for incoming calls
 *
 * @version $Revision: 1.3 $
 */
public class ContextClassLoaderAdaptorInterceptor extends AdaptorInterceptor
{
   private boolean m_isMX4J;
   private Method m_getClassLoader;
   private Method m_getClassLoaderFor;
   private Method m_getClassLoaderRepository;
   private ThreadLocal m_classloader = new ThreadLocal();

   public void setMBeanServer(MBeanServer server)
   {
      super.setMBeanServer(server);
      m_isMX4J = isMX4JImplementation(server);

      if (m_isMX4J)
      {
         try
         {
            m_getClassLoader = server.getClass().getMethod("getClassLoader", new Class[]{ObjectName.class});
            m_getClassLoaderFor = server.getClass().getMethod("getClassLoaderFor", new Class[]{ObjectName.class});
            m_getClassLoaderRepository = server.getClass().getMethod("getClassLoaderRepository", new Class[0]);
         }
         catch (Exception ignored)
         {
         }
      }
   }

   public String getType()
   {
      return "contextclassloader";
   }

   /**
    * Cannot change the status of this attribute, this interceptor is always enabled
    */
   public void setEnabled(boolean value)
   {
   }

   /**
    * This interceptor is always enabled
    */
   public boolean isEnabled()
   {
      return true;
   }

   protected void preInvoke(Invocation invocation) throws Exception
   {
      // The line below imposes a dependency between the RMIAdaptor and the MBeanServer implementation
      // However, this dependency should be removed with JMX 1.2.
      // Use reflection to minimize the dependecy

      if (m_isMX4J)
      {
         ClassLoader old = Thread.currentThread().getContextClassLoader();
         m_classloader.set(old);

         ClassLoader cl = null;
         String key = invocation.getClassLoaderKey();
         if (key != null)
         {
            if (key.equals(Invocation.CLASSLOADER_MBEANSERVER))
            {
               cl = getMBeanServer().getClass().getClassLoader();
            }
            else if (key.equals(Invocation.CLASSLOADER_REPOSITORY))
            {
               cl = getRepositoryClassLoader();
            }
            else if (key.equals(Invocation.CLASSLOADER_LOADER))
            {
               ObjectName name = invocation.getObjectNameForClassLoader();
               cl = getClassLoader(name);
            }
            else if (key.equals(Invocation.CLASSLOADER_MBEAN))
            {
               ObjectName name = invocation.getObjectNameForClassLoader();
               cl = getClassLoaderFor(name);
            }
            else
            {
               // Unknown key, a bug
               throw new ImplementationException();
            }
         }

         if (cl != null && cl != old) Thread.currentThread().setContextClassLoader(cl);
      }
   }

   protected void postInvoke(Invocation invocation) throws Exception
   {
      if (m_isMX4J)
      {
         ClassLoader cl = (ClassLoader)m_classloader.get();
         Thread.currentThread().setContextClassLoader(cl);
      }
   }

   private boolean isMX4JImplementation(MBeanServer server)
   {
      try
      {
         ClassLoader loader = server.getClass().getClassLoader();
         if (loader == null) loader = Thread.currentThread().getContextClassLoader();
         if (loader == null) loader = ClassLoader.getSystemClassLoader();
         Class cls = loader.loadClass("mx4j.server.MX4JMBeanServer");
         return cls.isInstance(server);
      }
      catch (ClassNotFoundException ignored)
      {
         return false;
      }
   }

   private ClassLoader getClassLoader(ObjectName name)
   {
      try
      {
         return (ClassLoader)m_getClassLoader.invoke(getMBeanServer(), new Object[]{name});
      }
      catch (Exception x)
      {
         return null;
      }
   }

   private ClassLoader getClassLoaderFor(ObjectName name)
   {
      try
      {
         return (ClassLoader)m_getClassLoaderFor.invoke(getMBeanServer(), new Object[]{name});
      }
      catch (Exception x)
      {
         return null;
      }
   }

   private ClassLoader getRepositoryClassLoader()
   {
      try
      {
         ClassLoaderRepository clr = (ClassLoaderRepository)m_getClassLoaderRepository.invoke(getMBeanServer(), new Object[0]);
         return new RepositoryClassLoader(clr);
      }
      catch (Exception x)
      {
         return null;
      }
   }
}
