/*!
 * \file mei_evaluate.c
 * 
 * \brief Build an interpreter for a mathematical expression
 */

/*
  This file is part of the "Mathematical Expression Interpreter" library.

  Copyright (C) 2008-2009  EDF

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*----------------------------------------------------------------------------
 * Standard C library headers
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

/*----------------------------------------------------------------------------
 * BFT library headers
 *----------------------------------------------------------------------------*/

#include <bft_mem.h>
#include <bft_printf.h>
#include <bft_error.h>

/*----------------------------------------------------------------------------
 * Local headers
 *----------------------------------------------------------------------------*/

#include "mei_node.h"
#include "mei_parser_glob.h"
#include "mei_parser.h"

/*----------------------------------------------------------------------------
 * Header for the current file
 *----------------------------------------------------------------------------*/

#include "mei_evaluate.h"

/*----------------------------------------------------------------------------*/

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*----------------------------------------------------------------------------
 * Local macro definitions
 *----------------------------------------------------------------------------*/

/*!
 * \brief Default modulo for hash table.
 */

#define HASHSIZE 701

/*============================================================================
 * Private function definitions
 *============================================================================*/

/*!
 * \brief Copy the symbol table pointer into the each node of the interpreter.
 *
 * \param [in] n node of the interpreter
 * \param [in] h table of symbols
 */

static void
_init_symbol_table(mei_node_t *n, hash_table_t *h)
{
    int i;

    if (!n) return;

    /* Copy the symbol table pointer into the current node */

    n->ht = h;

    /* Search the next node in the interpreter */

    if (n->flag == FUNC1)
    {
        _init_symbol_table(n->type->func.op, h);
    }
    else if (n->flag == FUNC2 || n->flag == FUNC3 || n->flag == FUNC4)
    {
        for (i = 0; i < n->type->funcx.nops; i++)
            _init_symbol_table(n->type->funcx.op[i], h);
    }
    else if (n->flag == OPR)
    {
        for (i = 0; i < n->type->opr.nops; i++)
            _init_symbol_table(n->type->opr.op[i], h);
    }
}

/*!
 * \brief Return the number of errors related to the use of undefined symbol
 * in the expression.
 *
 * \param [in] p node of an interpreter
 * \return number of undefined symbol
 */

static int
_check_symbol(mei_node_t *p)
{
    int iok = 0;
    int l;

    if (!p) return 0;

    switch(p->flag)
    {
        case CONSTANT:
            return 0;

        case ID:
            if (mei_hash_table_lookup(p->ht, p->type->id.i) == NULL)
            {
                /* bft_printf("Warning: identifiant %s is unknown\n", p->type->id.i); */
                /* bft_printf("- line:   %i \n", p->type->id.l); */
                /* bft_printf("- column: %i \n", p->type->id.c); */

                BFT_REALLOC(label_list,  ierr_list+1, char*);
                BFT_REALLOC(line_list,   ierr_list+1, int);
                BFT_REALLOC(column_list, ierr_list+1, int);

                l = strlen("Warning, identifiant ")+1;
                BFT_MALLOC(label_list[ierr_list], l, char);
                strncpy(label_list[ierr_list], "Warning: identifiant ", l);

                l = l + strlen(p->type->id.i);
                BFT_REALLOC(label_list[ierr_list], l, char);
                strncat(label_list[ierr_list], p->type->id.i, l);

                l = l + strlen(" is unknown.\n");
                BFT_REALLOC(label_list[ierr_list], l, char);
                strncat(label_list[ierr_list], " is unknown.\n", l);

                line_list[ierr_list]   = p->type->id.l;
                column_list[ierr_list] = p->type->id.c;

                ierr_list++;

                return 1;
            }
            return 0;

        case FUNC1:
            if (mei_hash_table_lookup(p->ht, p->type->func.name) == NULL)
            {
                /* in a normal way it is impossible to arrive here     */
                /* because the parser has already check functions name */

                bft_error(__FILE__, __LINE__, 0, "Error: _check_symbol\n");

                return 1;
             }
             return _check_symbol(p->type->func.op);

        case FUNC2:
            if (mei_hash_table_lookup(p->ht, p->type->funcx.name) == NULL)
            {
                /* in a normal way it is impossible to arrive here     */
                /* because the parser has already check functions name */

                bft_error(__FILE__, __LINE__, 0, "Error: _check_symbol\n");

                return 1;
            }
            iok  = _check_symbol(p->type->funcx.op[0]);
            iok += _check_symbol(p->type->funcx.op[1]);
            return iok;

        case FUNC3:
                bft_error(__FILE__, __LINE__, 0, "not implemented yet \n");
		break;

        case FUNC4:
                bft_error(__FILE__, __LINE__, 0, "not implemented yet \n");
		break;

        case OPR:

            switch (p->type->opr.oper)
            {

            case IF:
                iok  = _check_symbol(p->type->opr.op[0]);
                iok += _check_symbol(p->type->opr.op[1]);
                if (p->type->opr.nops > 2)
                    iok += _check_symbol(p->type->opr.op[2]);
                return iok;

            case PRINT:
                return _check_symbol(p->type->opr.op[0]);

            case '=':
                mei_hash_table_insert(p->ht,
                                      p->type->opr.op[0]->type->id.i,
                                      CONSTANT,
                                      0,
                                      NULL,
                                      NULL,
                                      NULL,
                                      NULL);
                return _check_symbol(p->type->opr.op[1]);

            case UPLUS:
                return _check_symbol(p->type->opr.op[0]);

            case UMINUS:
                return _check_symbol(p->type->opr.op[0]);

            case '!':
                return _check_symbol(p->type->opr.op[0]);

            default:
                iok  = _check_symbol(p->type->opr.op[0]);
                iok += _check_symbol(p->type->opr.op[1]);
                return iok;
        }
    }

    bft_error(__FILE__, __LINE__, 0, "Error: _check_symbol\n");

    return iok;
}

/*!
 * \brief Return the evaluation of the expression.
 *
 * \param [in] p node of an interpreter
 * \return value of evaluated expression
 */

static double
_evaluate(mei_node_t *p)
{
    double t1, t2;
    func1_t f1;
    func2_t f2;

    if (!p) return 0;

    switch(p->flag)
    {

        case CONSTANT:
            return p->type->con.value;

        case ID:
            return (mei_hash_table_lookup(p->ht, p->type->id.i))->data->value;

        case FUNC1:
            f1 = (mei_hash_table_lookup(p->ht, p->type->func.name))->data->func;
            return f1(_evaluate(p->type->func.op));

        case FUNC2:
            f2 = (mei_hash_table_lookup(p->ht, p->type->funcx.name))->data->f2;
            return f2(_evaluate(p->type->funcx.op[0]), _evaluate(p->type->funcx.op[1]));

        case FUNC3:
                bft_error(__FILE__, __LINE__, 0, "not implemented\n");
		break;

        case FUNC4:
                bft_error(__FILE__, __LINE__, 0, "not implemented\n");
		break;

        case OPR:

            switch(p->type->opr.oper)
            {

                case WHILE:
                    while(_evaluate(p->type->opr.op[0])) _evaluate(p->type->opr.op[1]);
                    return 0;

                case IF:
                    if (_evaluate(p->type->opr.op[0]))
                        _evaluate(p->type->opr.op[1]);
                    else if (p->type->opr.nops > 2)
                        _evaluate(p->type->opr.op[2]);
                    return 0;

                case PRINT:
                    bft_printf("PRINT %f\n", _evaluate(p->type->opr.op[0]));
                    return 0;

                case ';':
                    _evaluate(p->type->opr.op[0]);
                    return _evaluate(p->type->opr.op[1]);

                case '=':
                    t2 = _evaluate(p->type->opr.op[1]);
                    mei_hash_table_insert(p->ht,
                                          p->type->opr.op[0]->type->id.i,
                                          CONSTANT,
                                          t2,
                                          NULL,
                                          NULL,
                                          NULL,
                                          NULL);
                    return 0;

                case UPLUS:
                    return _evaluate(p->type->opr.op[0]);

                case UMINUS:
                    return -_evaluate(p->type->opr.op[0]);

                case '+':
                    return _evaluate(p->type->opr.op[0]) + _evaluate(p->type->opr.op[1]);

                case '-':
                    return _evaluate(p->type->opr.op[0]) - _evaluate(p->type->opr.op[1]);

                case '*':
                    return _evaluate(p->type->opr.op[0]) * _evaluate(p->type->opr.op[1]);

                case '/':
                    t1 = _evaluate(p->type->opr.op[0]);
                    t2 = _evaluate(p->type->opr.op[1]);
                    if (t2)
                        return t1 / t2;
                    else
                        bft_error(__FILE__, __LINE__, 0, "Error: floating point exception\n");

                case '^':
                    return pow(_evaluate(p->type->opr.op[0]), _evaluate(p->type->opr.op[1]));

                case '<':
                    return _evaluate(p->type->opr.op[0]) < _evaluate(p->type->opr.op[1]);

                case '>':
                    return _evaluate(p->type->opr.op[0]) > _evaluate(p->type->opr.op[1]);

                case '!':
                    return ! _evaluate(p->type->opr.op[0]);

                case GE:
                    return _evaluate(p->type->opr.op[0]) >= _evaluate(p->type->opr.op[1]);

                case LE:
                    return _evaluate(p->type->opr.op[0]) <= _evaluate(p->type->opr.op[1]);

                case NE:
                    return _evaluate(p->type->opr.op[0]) != _evaluate(p->type->opr.op[1]);

                case EQ:
                    return _evaluate(p->type->opr.op[0]) == _evaluate(p->type->opr.op[1]);

                case AND:
                    return _evaluate(p->type->opr.op[0]) && _evaluate(p->type->opr.op[1]);

                case OR:
                    return _evaluate(p->type->opr.op[0]) || _evaluate(p->type->opr.op[1]);
            }
    }

    return 0;
}

/*!
 * \brief Store error message.
 *
 * \param [in] ev interpreter
 */

static void
_manage_error(mei_tree_t *ev)
{
    int i;
    size_t l;

    assert(ev != NULL);

    ev->errors = ierr_list;

    BFT_MALLOC(ev->labels,  ierr_list, char*);
    BFT_MALLOC(ev->lines,   ierr_list, int);
    BFT_MALLOC(ev->columns, ierr_list, int);

    for (i=0; i<ev->errors; i++)
    {
        ev->lines[i]   = line_list[i];
        ev->columns[i] = column_list[i];

        l = strlen(label_list[i]) + 1;
        BFT_MALLOC(ev->labels[i], l, char);
        strncpy(ev->labels[i], label_list[i], l);
    }
}

/*!
 * \brief Check if the symbol \em str exists in the expression stored
 *  in \em ev. Return 0 if the symbol exists in the symbol table, 1 if not.
 *
 * \param [in] ev  interpreter in which we want to know if \em str exists
 * \param [in] str symbol to find
 * \return 0 if the symbol exists in the symbol table, 1 if not
 */

static int
_find_symbol(mei_tree_t *const ev, const char *const str)
{
    int i, iok = 0;
    size_t l;

    assert(ev != NULL);
    assert(str != NULL);

    if (!mei_hash_table_lookup(ev->symbol, str))
    {
        /* the symbol is not found: this information is stored */

        iok = 1;

        i = ev->errors;
        ev->errors++;

        if (!ev->labels)
            BFT_MALLOC(ev->labels, ev->errors, char*);
        else
            BFT_REALLOC(ev->labels, ev->errors, char*);

        l = strlen(str) + 1;
        BFT_MALLOC(ev->labels[i], l, char);
        strncpy(ev->labels[i], str, l);
    }

    return iok;
}

/*============================================================================
 * Public function definitions
 *============================================================================*/

/*!
 * \brief Returns a new interpreter. The node member is empty,
 * the string member contains the mathematical expression and
 * the symbol table is initialize with the standard symbols.
 *
 * \param [in] expr string characters of the mathematical expression
 * \return new empty interpreter
 */

mei_tree_t*
mei_tree_new(const char *const expr)
{
    mei_tree_t *ev = NULL;
    size_t length;

    assert(expr != NULL);

    BFT_MALLOC(ev, 1, mei_tree_t);

    BFT_MALLOC(ev->symbol, 1, hash_table_t);

    length = strlen(expr)+1;
    BFT_MALLOC(ev->string, length, char);

    strncpy(ev->string, expr, length);

    mei_hash_table_create(ev->symbol, HASHSIZE);
    ev->symbol->n_inter = 1;
    mei_hash_table_init(ev->symbol);

    /* Default initializations */

    ev->errors  = 0;
    ev->columns = NULL;
    ev->lines   = NULL;
    ev->labels  = NULL;
    ev->node    = NULL;

    return ev;
}

/*!
 * \brief Returns a new interpreter. The node member is empty,
 * the string member contains the mathematical expression and
 * the symbol table is initialize with a existing table, which is shared by
 * multiple interpreter.
 *
 * \param [in] expr         string characters of the mathematical expression
 * \param [in] symbol_table shared table of symbols
 * \return new empty interpreter
 */

mei_tree_t*
mei_tree_new_with_shared_symbols(const char   *const expr,
                                 hash_table_t *const symbol_table)
{
    mei_tree_t *ev = NULL;
    size_t length;

    assert(expr != NULL);
    assert(symbol_table != NULL);

    BFT_MALLOC(ev, 1, mei_tree_t);

    length = strlen(expr)+1;
    BFT_MALLOC(ev->string, length, char);

    strncpy(ev->string, expr, length);

    /* copy the adress of the shared table of symbols and
      incremente the number of interpreters that share this table */

    ev->symbol = symbol_table;
    ev->symbol->n_inter++;

    /* Default initializations */

    ev->errors  = 0;
    ev->columns = NULL;
    ev->lines   = NULL;
    ev->labels  = NULL;
    ev->node    = NULL;

    return ev;
}

/*!
 * \brief Returns a new table of symbols.
 * The table contains standard mathematical symbols.
 *
 * \return table of symbols
 */

hash_table_t*
mei_table_symbols_new(void)
{
    hash_table_t *ht = NULL;

    BFT_MALLOC(ht, 1, hash_table_t);

    mei_hash_table_create(ht, HASHSIZE);
    ht->n_inter = 0;
    mei_hash_table_init(ht);

    return ht;
}

/*!
 * \brief Call the yacc parser.
 * Return 0 if the parsing is ok, or the number of errors, if errors occurs.
 *
 * \param [in] ev interpreter
 * \return number of parsing errors
 */

int
mei_tree_builder(mei_tree_t *ev)
{
    int i;

    assert(ev != NULL);

    /* Initialize the global variables of the parser */
    /*-----------------------------------------------*/

    /* Initialize the pointer on the node returned by the parser */

    root = NULL;

    /* Begin/end of the expression is set in these global variables */

    string_begin = ev->string;
    string_end   = ev->string + strlen(ev->string);

    /* Initialize line and column counters,
       incrementation is done in scanner.l */

    line   = 1;
    column = 1;

    /* Initialize error counter
       incrementation is done in parser.y (yyerror function) */

    ierr_list = 0;

    /* Parse the expression string and construct the interpreter */
    /*-----------------------------------------------------------*/

    yyparse();

    if (ierr_list)
    {

        /* Parsing error: copy informations for display */

        _manage_error(ev);

       /* Free memory. Warning: if there is more than one error,
          not all pointers are cleaning */

        mei_free_node(root);
    }
    else
    {
        /* If the parsing is ok, copy the data in the current interpreter */

        ev->node = root;

        /* For all nodes of the interpreter, copy the symbols table pointer */

        _init_symbol_table(ev->node, ev->symbol);

        /* Check if all new symbols are defined in the expression string */

        ierr_list = _check_symbol(ev->node);

        if (ierr_list)
        {

            /* Check symbol error: copy informations for display */

            /* if (ierr_list == 1) */
            /*   bft_printf("Warning: there is %i error.\n",  ierr_list); */
            /* else if (ierr_list > 1) */
            /*   bft_printf("Warning: there are %i errors.\n", ierr_list); */

          _manage_error(ev);
        }

    }

    /* Free memory of the parser global variables if necessary */

    for (i=0; i<ierr_list; i++)
        BFT_FREE(label_list[i]);

    BFT_FREE(label_list);
    BFT_FREE(line_list);
    BFT_FREE(column_list);

    return ierr_list;
}

/*----------------------------------------------------------------------------
 * WARNING: obsolete function use insteed mei_tree_find_symbol
 *                                        --------------------
 * Return 0 if the expression is able to evaluate the symbol 'str',
 * return the number of error otherwise.
 *
 * parameters:
 *   ev         <-- interpreter
 *   str        <-- symbol to check
 *----------------------------------------------------------------------------*/

//int mei_tree_check_required_symbol(mei_tree_t *const ev, char *const str)
//{
//  int i, iok;
//  char *buf;
//  size_t length;
//  mei_tree_t *e;
//  struct item *item, *next;
//
//  /* create a new string with the expression and the symbol to find */
//
//  length = strlen(ev->string) + strlen(str) + 3;
//  BFT_MALLOC(buf, length, char);
//
//  strncpy(buf, ev->string, strlen(ev->string)+1);
//  strncat(buf, ";", 1);
//  strncat(buf, str, strlen(str));
//  strncat(buf, ";", 1);
//
//  /* create a new interpreter */
//
//  e = mei_tree_new(buf);
//
//  BFT_FREE(buf);
//
//  /* ride trought out the symbol table */
//
//  for (i = 0; i < ev->symbol->length; i++) {
//
//    item = ev->symbol->table[i];
//
//    while (item) {
//
//      next = item->next;
//
//      if (item->type != FUNC1 &&
//          item->type != FUNC2 &&
//          item->type != FUNC3 &&
//          item->type != FUNC4) {
//
//        /* update the symbol table */
//
//        mei_hash_table_insert(e->symbol,
//                              item->key,
//                              item->type,
//                              item->data->value,
//                              NULL,
//                              NULL,
//                              NULL,
//                              NULL);
//      }
//
//      item = next;
//
//    }
//
//  }
//
//  /* try to construct the new interpreter */
//
//  iok = mei_tree_builder(e);
//
//  mei_tree_destroy(e);
//
//  return iok;
//}

/*----------------------------------------------------------------------------
 * WARNING: obsolete function use insteed mei_tree_find_symbols
 *                                        ---------------------
 * Return 0 if the expression is able to evaluate the list of symbols,
 * return the number of error otherwise.
 *
 * parameters:
 *   ev         <-- interpreter
 *   size       <-- number of symbols to check
 *   symbol     <-- list of symbols
 *----------------------------------------------------------------------------*/

//int mei_tree_check_required_symbols(mei_tree_t *const ev,
//                                    const int         size,
//                                    char            **symbol)
//{
//  int i, iok = 0;
//
//  for(i=0; i<size; i++) {
//    iok = iok +mei_tree_check_required_symbol(ev, symbol[i]);
//  }
//
//  return iok;
//}

/*!
 * \brief Inserts a constant (label and value) in the table of symbols
 * associated to an interpreter.
 *
 * \param [in] ev    interpreter
 * \param [in] str   label of the constant
 * \param [in] value value associated to the constant
 */

void
mei_tree_insert(      mei_tree_t *const ev,
                const char       *const str,
                const double            value)
{
    assert(ev != NULL);
    assert(str != NULL);

    mei_hash_table_insert(ev->symbol,
                          str,
                          CONSTANT,
                          value,
                          NULL,
                          NULL,
                          NULL,
                          NULL);
}

/*!
 * \brief Inserts a constant (label and value) in a table of symbols.
 *
 * \param [in]  symbol_table table of symbols
 * \param [in]  str          label of the constant
 * \param [in]  value        value associated to the constant
 */

void
mei_symbol_table_insert(      hash_table_t *const symbol_table,
                        const char         *const str,
                        const double              value)
{
    assert(symbol_table!= NULL);
    assert(str != NULL);

    mei_hash_table_insert(symbol_table,
                          str,
                          CONSTANT,
                          value,
                          NULL,
                          NULL,
                          NULL,
                          NULL);
}

/*!
 * \brief Check if the symbol \em str exists in the expression stored
 * in \em ev.
 * Return 0 if the symbol exists in the symbol table, 1 if not.
 *
 * \param [in] ev  interpreter in which we want to know if str exists
 * \param [in] str symbol to find
 * \return 0 if the symbol exists in the symbol table, 1 if not
 */

int
mei_tree_find_symbol(mei_tree_t *const ev, const char *const str)
{
    int i;

    assert(ev != NULL);
    assert(str != NULL);

    /* restart error's counter */

    for (i=0; i < ev->errors; i++)
        BFT_FREE(ev->labels[i]);

    BFT_FREE(ev->labels);
    BFT_FREE(ev->lines);
    BFT_FREE(ev->columns);
    ev->errors = 0;

    return _find_symbol(ev, str);
}

/*!
 * \brief Check if the symbol \em str from a list exists in the expression.
 * Return 0 if all symbols exist, otherwise return the number of errors.
 * The list of missing symbols is stored in \em ev->labels.
 *
 * \param [in]  ev     interpreter
 * \param [in]  size   number of symbols to check
 * \param [in]  symbol list of symbols
 * \return 0 if all symbols exist, otherwise return the number of errors
 */

int
mei_tree_find_symbols(mei_tree_t *const ev,
                      const int         size,
                      const char      **symbol)
{
    int i, iok = 0;

    assert(ev != NULL);

    /* restart error's counter */

    for (i=0; i < ev->errors; i++)
        BFT_FREE(ev->labels[i]);

    BFT_FREE(ev->labels);
    BFT_FREE(ev->lines);
    BFT_FREE(ev->columns);
    ev->errors = 0;

    /* check if each symbol from the list exists */

    for(i=0; i<size; i++)
        iok += _find_symbol(ev, symbol[i]);

    return iok;
}

/*!
 * \brief Returns a value of the \em str symbol (variable or constant)
 *  from table of symbols of \em ev interpreter.
 *
 * \param [in]   ev  interpreter
 * \param [in]   str name of the symbol
 * \return value of a symbol
 */

double
mei_tree_lookup(mei_tree_t *const ev, const char *const str)
{
    struct item *item = NULL;

    assert(ev != NULL);
    assert(str != NULL);

    item = mei_hash_table_lookup(ev->symbol, str);

    if (!item)
    {
        bft_error(__FILE__, __LINE__, 0,
                  "Error in mei_tree_lookup function: "
                  "%s does not exist in the symbol table\n", str);
    }
    return item->data->value;
}

/*!
 * \brief Evaluates the expression \em ev :
 *   1) computes all values for variables inside the expression
 *   2) returns a value from the intrepreted expression.
 *
 * \param [in] ev interpreter
 * \return value from the intrepreted expression
 */

double
mei_evaluate(mei_tree_t *const ev)
{
    assert(ev != NULL);
    return _evaluate(ev->node);
}

/*!
 * \brief Free memory and return NULL.
 *
 * \param [in] ev interpreter
 */

void
mei_tree_destroy(mei_tree_t *ev)
{
    int i;

    if (ev != NULL)
    {
        if (ev->symbol->n_inter == 1) /* tables of symbol for a single interpreter */
        {
            mei_hash_table_free(ev->symbol);
            BFT_FREE(ev->symbol);
        }
        else
        {
            ev->symbol->n_inter--; /* shared tables of symbol */
        }
        BFT_FREE(ev->string);
        mei_free_node(ev->node);

        for (i=0; i < ev->errors; i++)
            BFT_FREE(ev->labels[i]);

        BFT_FREE(ev->labels);
        BFT_FREE(ev->lines);
        BFT_FREE(ev->columns);
        BFT_FREE(ev);
    }
}

/*----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif /* __cplusplus */
