/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: AbstractFileOpenCommand.java,v 1.5 2007/01/11 08:28:39 pietschy Exp $
 */
package org.pietschy.command.file;

import org.pietschy.command.CommandManager;

import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import java.awt.*;
import java.io.File;

/**
 * The command provides generic file open behaviour.  One execution, the command will display
 * a {@link JFileChooser} and if the selection is successful {@link #performOpen} will be
 * invoked with the selected files.  Subclasses must implement {@link #performOpen} to implement
 * the required behaviour.
 * <p>
 * Subclasses can change the default settings by overriding {@link #getFileChooser} and configuring
 * appropriately.
 *
 * @version $Revision: 1.5 $
 * @author andrewp
 */
public abstract class
AbstractFileOpenCommand
extends AbstractFileCommand
{
   private static final String _ID_ = "$Id: AbstractFileOpenCommand.java,v 1.5 2007/01/11 08:28:39 pietschy Exp $";

   private boolean multiselectionEnabled = true;

   /**
    * Creates a new command with the specified id and {@link FileFilter}.  This command is bound
    * to the default command manager.
    * @param id the id of the command.
    * @param filter the {@link FileFilter} to use.
    */
   public AbstractFileOpenCommand(String id, FileFilter filter)
   {
      this(CommandManager.defaultInstance(), id, filter);
   }

   /**
    * Creates a new command with the specified id and list of {@link FileFilter}.  This command is
    * bound to the default command manager.
    * @param id the id of the command.
    * @param filters a list of {@link FileFilter} instances to use.
    */
   public AbstractFileOpenCommand(String id, FileFilter[] filters)
   {
      this(CommandManager.defaultInstance(), id, filters);
   }

   /**
    * Creates a new command with the specified id and {@link FileFilter}
    * @param id the id of the command.
    * @param filter the {@link FileFilter} to use.
    */
   public AbstractFileOpenCommand(CommandManager manager, String id, FileFilter filter)
   {
      super(manager, id, new FileFilter[]{filter});
   }

   /**
    * Creates a new command with the specified id and list of {@link FileFilter}
    * @param id the id of the command.
    * @param filters a list of {@link FileFilter} instances to use.
    */
   public AbstractFileOpenCommand(CommandManager manager, String id, FileFilter[] filters)
   {
      super(manager, id, filters);
   }


   protected int
   showChooserDialog(JFileChooser chooser, Window invoker)
   {
      chooser.setMultiSelectionEnabled(isMultiselectionEnabled());
      return chooser.showOpenDialog(invoker);
   }

   protected void
   performFileAction(File[] files, JFileChooser chooser, Window invoker)
   {
      performOpen(files);
   }

   /**
    * Checks if multiple selection is enabled.
    * @return <tt>true</tt> if multiple selection is enabled.
    */
   public boolean
   isMultiselectionEnabled()
   {
      return multiselectionEnabled;
   }

   /**
    * Configures if the {@link JFileChooser} allows multiple selection.
    * @param multiselectionEnabled <tt>true</tt> to allow multiple file selection, <tt>false</tt>
    * otherwise.
    */
   public void
   setMultiselectionEnabled(boolean multiselectionEnabled)
   {
      this.multiselectionEnabled = multiselectionEnabled;
   }

   /**
    * This method is called when the user selects one or more files to open.  Subclasses must
    * implement this method to perform the required behaviour.
    * <p>
    * If {@link #isMultiselectionEnabled multiple selection} is disabled, the file list will
    * contain only one file.
    * @param files the files that the user has selected.
    */
   protected abstract void
   performOpen(File[] files);

}

