/*******************************************************************************************************************************************
 ccolorbutton.c
*******************************************************************************************************************************************/

#include "ccolorbutton.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CColorButtonListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CColorButtonListener::CColorButtonListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CColorButtonListener::~CColorButtonListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnSetColor
//-----------------------------------------------------------------------------------------------------------------------------------------
void CColorButton::OnSetColor (GtkColorButton *inGtkColorButton, gpointer inData)
{
	// retreive our gtkol instance
	CColorButton *inColorButton = reinterpret_cast <CColorButton *> (inData);

	// pointer check
	if (inColorButton != NULL && inColorButton -> GetListener() != NULL)
	{
		// get the selected color
		GdkColor ioGdkColor; ::gtk_color_button_get_color (inGtkColorButton, &ioGdkColor);

		// set the notification
		static_cast <CColorButtonListener *> (inColorButton -> GetListener()) -> 
			OnSetColor (inColorButton, ::gtk_color_button_get_alpha (inGtkColorButton), ioGdkColor);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CColorButton);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CColorButton::CColorButton (CContainer *inOwner, const CColorButtonListener *inListener)
	     :CButton	   (NULL, inListener)
{
	// set the color button component's owner and launch the gtk instanciation process
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CColorButton::~CColorButton ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CColorButton::PerformWidgetInstanciate ()
{
	// ok
	return ::gtk_color_button_new ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CColorButton::PerformWidgetInitialize ()
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// specific signal connection
	::g_signal_connect (G_OBJECT(inGtkWidget), "color-set", G_CALLBACK(CColorButton::OnSetColor), this);

	// generic call
	CButton::PerformWidgetInitialize ();

	// caption / dialog title
	SetCaption (CString("Color Chooser"));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected listener type
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CColorButton::ListenerMustBe () const
{
	// ok
	return __metaclass(CColorButtonListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// color button selected color
//-----------------------------------------------------------------------------------------------------------------------------------------
void CColorButton::SetColor (const GdkColor &inGdkColor)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// ok
	::gtk_color_button_set_color (GTK_COLOR_BUTTON(inGtkWidget), &inGdkColor);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// color button selected color
//-----------------------------------------------------------------------------------------------------------------------------------------
GdkColor CColorButton::GetColor () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// the output selected color
	GdkColor outGdkColor; ::bzero (&outGdkColor, sizeof(GdkColor));

	// get the color
	if (inGtkWidget != NULL) ::gtk_color_button_get_color (GTK_COLOR_BUTTON(inGtkWidget), &outGdkColor);

	// ok
	return outGdkColor;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// alpha value of the selected color
//-----------------------------------------------------------------------------------------------------------------------------------------
void CColorButton::SetAlpha (const UInt16 inAlpha)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// ok
	::gtk_color_button_set_alpha (GTK_COLOR_BUTTON(inGtkWidget), inAlpha);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// alpha value of the selected color
//-----------------------------------------------------------------------------------------------------------------------------------------
UInt16 CColorButton::GetAlpha () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// ok ?
	return inGtkWidget != NULL ? ::gtk_color_button_get_alpha (GTK_COLOR_BUTTON(inGtkWidget)) : 0L;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// use alpha attribute
//-----------------------------------------------------------------------------------------------------------------------------------------
void CColorButton::SetUseAlpha (const bool inUseAlpha)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// ok
	::gtk_color_button_set_use_alpha (GTK_COLOR_BUTTON(inGtkWidget), inUseAlpha);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// use alpha attribute
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CColorButton::GetUseAlpha () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// ok ?
	return inGtkWidget != NULL ? ::gtk_color_button_get_use_alpha (GTK_COLOR_BUTTON(inGtkWidget)) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// associated dialog box title
//-----------------------------------------------------------------------------------------------------------------------------------------
void CColorButton::SetCaption (const CString &inTitle)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL) return;

	// ok ?
	::gtk_color_button_set_title (GTK_COLOR_BUTTON(inGtkWidget), inTitle.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// associated dialog box title
//-----------------------------------------------------------------------------------------------------------------------------------------
CString CColorButton::GetCaption () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// ok ?
	return inGtkWidget != NULL ? CString(::gtk_color_button_get_title (GTK_COLOR_BUTTON(inGtkWidget))) : CString();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// serialization handling
//-----------------------------------------------------------------------------------------------------------------------------------------
void CColorButton::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CButton::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element under the current one
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_COLOR_BUTTON_ELEMENT);

			// add the attributes
			newXMLElement -> AddAttribute (XML_COLOR_BUTTON_ATTR_ARGB, CString((UInt32)GetAlpha()) + "," +
										   CString((UInt32)GetColor().red) + "," +
										   CString((UInt32)GetColor().green) + "," +
										   CString((UInt32)GetColor().blue));
			newXMLElement -> AddAttribute (XML_COLOR_BUTTON_ATTR_USE_ALPHA, CString(GetUseAlpha()?"true":"false"));

			// modify the input output pointer so that the overwritten serialization process continues under this element
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the child node this serialization process is interested in
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_COLOR_BUTTON_ELEMENT);

			// check we got the expected node
			if (inXMLNode == NULL)
				throw new CException (CString("CColorButton::Serialize, specified xml node is not a \"") + 
							      XML_COLOR_BUTTON_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the in/out element node pointer so that the serialization process will continue under the cbutton node
			ioXMLElementNode = inXMLNode;

			// set the button caption from the expected attribute
			SetUseAlpha 	(::xml_node_get_attribute (inXMLNode, XML_COLOR_BUTTON_ATTR_USE_ALPHA).GetValue().ToBool());
			CString inARGB 	(::xml_node_get_attribute (inXMLNode, XML_COLOR_BUTTON_ATTR_ARGB).GetValue());
			if (inARGB.Cut(CString(",")).GetLength() == 4)
			{
				GdkColor outGdkColor = { 0L, inARGB.Cut(CString(","))[1]->ToULong(), 
							 inARGB.Cut(CString(","))[2]->ToULong(), inARGB.Cut(CString(","))[3]->ToULong() };
				SetColor (outGdkColor);
				SetAlpha (inARGB.Cut(CString(","))[0]->ToULong());
			}
		}
		break;
	}
}


