/*******************************************************************************************************************************************
 cgraphics.h
 
 CClass		CClass			CClass
 |--CGraphics	|--CClothPainter	|--CPixbuf

 Definition of a set of tools to paint on drawables.
*******************************************************************************************************************************************/

#ifndef __CGRAPHICS_H__
#define __CGRAPHICS_H__

#include "cclass.h"

#include <gdk/gdk.h>
#include <gtk/gtk.h>

// at this point, opaque definition of a gtkol widget...
class CWidget;

// class defined here
class CGraphics;
class CClothPainter;
class CPixbuf;

//-----------------------------------------------------------------------------------------------------------------------------------------
// let's define some basics : point, size and bounds...
//-----------------------------------------------------------------------------------------------------------------------------------------
struct TPoint
{
        SInt32  x;
        SInt32  y;
                                                                                                                                                
        TPoint  (const SInt32 =0L, const SInt32 =0L);
        TPoint  (const TPoint &);
};
                                                                                                                                                
struct TSize
{
        SInt32  w;
        SInt32  h;
                                                                                                                                                
        TSize   (const SInt32 =0, const SInt32 =0);
        TSize   (const TSize &);
};
                                                                                                                                                
struct TBounds : public TPoint, public TSize
{
        TBounds (const SInt32 =0L, const SInt32 =0L, const SInt32 =0L, const SInt32 =0L);
        TBounds (const TPoint &, const TSize  & =TSize ());
        TBounds (const TSize  &, const TPoint & =TPoint());
        TBounds (const TBounds &);
                                                                                                                                                
        TBounds & operator = (const TBounds &);
	TBounds & operator = (const TSize &);
	TBounds & operator = (const TPoint &);
};

//-----------------------------------------------------------------------------------------------------------------------------------------
// CGraphics class, gdk_draw_xxx set implementation
//-----------------------------------------------------------------------------------------------------------------------------------------
class CGraphics : public CClass
{
	// instanciation section
	public :

		CGraphics			(CWidget *inGtkolWidget, const TBounds &inInvalidated=TBounds());
		CGraphics			(GtkWidget *inGtkWidget, const TBounds &inInvalidated=TBounds());
		CGraphics			(GdkDrawable *inGdkDrawable, GtkStyle *inGtkStyle, PangoLayout *inPangoLayout,
						 const TBounds &inInvalidated=TBounds());
		virtual ~CGraphics		();

	// graphics services
	public :

		// returns if graphics tools have been used or not on the associated drawable (used to stop expose events handling)
		Bool				GetDrawn		() const;

		// what bounds have been painted with graphics tools on the associated drawable ?
		TBounds				GetPainted		() const;
		
		// get invalidated bounds of the associated drawable to be painted
		TBounds				GetInvalidated		() const;

	// specific graphics drawing functions
	public :

		// get specified string size (in devices units), uses pango layout of the associated drawable 
		TSize				GetStringSize		(const CString &) const;

		// draws specified string on associayed drawable 
		void				DrawString		(const CString &, const TPoint & =TPoint());

		// draws a line from point to point on associated drawable
		void				DrawLine		(const TPoint &, const TPoint &);

		// draw a part of a pixbuf at the specified point on the graphics associated drawable
		void				DrawPixbuf		(CPixbuf *, const TBounds &, const TPoint &);

		// paint a focus indicator around the given rectangle on the graphics associated drawable using the given style
		void				PaintFocus		(const TBounds &, const GtkStateType =GTK_STATE_SELECTED);

		// draws a rectangular outline or filled rectangle, using the foreground color and other attributes of the gc
		void				DrawRectangle		(const TBounds &, const bool inFill=true);

	// protected attributes
	protected :

		// drawn with graphics ?
		Bool				m_Drawn;

		// invalidated entry bounds
		TBounds				m_Invalidated;

		// the targeted drawable, the used GC, the used pango layout
		GdkDrawable *			m_Drawable;
		PangoLayout *			m_Layout;
		GdkGC	    *			m_GC;
		GtkStyle    *			m_Style;

		// painted bounds if any
		TBounds				m_Painted;

		// metaclass specifications
		SECTION_CAPSULE_METACLASS;
};

// metaclass and class tag declaration
DECLARE_CAPSULE_METACLASS ('gphc', CGraphics, CClass);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CClothPainter class, used to paint in an expose events independant way
//-----------------------------------------------------------------------------------------------------------------------------------------
class CClothPainter : public CClass
{
	// instanciation section
	public :

		CClothPainter			(CWidget *inGtkolWidget);
		CClothPainter			(GtkWidget *inGtkWidget);
		CClothPainter			(GdkDrawable *inGdkDrawable, GtkStyle *inGtkStyle, PangoLayout *inPangoLayout, 
						 const TSize &inClothSize);
		virtual ~CClothPainter		();

	// specific functions
	public :

		// get a graphics tool to paint on the drawable cloth 
		CGraphics *			CreateGraphics		() const;

		// release the graphics tool used from CreateGraphics and update the targeted drawable from the worked on cloth
		void				DeleteGraphics 		(CGraphics *inGraphics) const;

	// protected attributes
	protected :

		// targeted drawable, used GC, cloth painted on, original cloth, pango layout, cloth size
		GdkDrawable *			m_Drawable;
		GdkGC *				m_GC;
		GdkPixmap *			m_Pixmap;
		GdkPixmap *			m_Cloth;
		PangoLayout *			m_Layout;
		TSize				m_Size;
		GtkStyle *			m_Style;

		// metaclass code section
		SECTION_CAPSULE_METACLASS;
};

// metaclass and class tag declaration
DECLARE_CAPSULE_METACLASS ('chpt', CClothPainter, CClass);

//-----------------------------------------------------------------------------------------------------------------------------------------
// pixel format definition
//-----------------------------------------------------------------------------------------------------------------------------------------
typedef enum
{
	PIXELFORMAT_8	=8,	// 8  bits per pixel
	PIXELFORMAT_16	=16,	// 16 bits per pixel (565, no alpha !)
	PIXELFORMAT_24	=24,	// 24 bits per pixel
	PIXELFORMAT_32	=32	// 32 bits per pixel, alpha blending
} TPixelFormat;

//-----------------------------------------------------------------------------------------------------------------------------------------
// CPixbuf, basic accessible memory representation of an image (maps GdkPixbuf)
//-----------------------------------------------------------------------------------------------------------------------------------------
class CPixbuf : public CClass
{
	// instanciation section
	public :

		// new empty pixbuf of specified size and pixel format
		CPixbuf				(const TSize &, const TPixelFormat);

		// new pixbuf representation of part of specified gtkol widget (8 bits pixel format mapping)
		CPixbuf				(CWidget *, const TBounds &);

		// new pixbuf representation of part of specified gtkol widget to specified pixel format
		CPixbuf				(CWidget *, const TPixelFormat, const TBounds &);

		// new pixbuf representation of part of specified drawable (8 bits pixel format mapping)
		CPixbuf				(GdkDrawable *, const TBounds &);

		// new pixbuf representation of part of specified drawable to specified pixel format
		CPixbuf				(GdkDrawable *, const TPixelFormat, const TBounds &);

		// new pixbuf from specified buffer, width, height, row bytes number and pixel format; the buffer is handled by CPixbuf
		CPixbuf				(const UInt8 *, const TSize &, const UInt16, const TPixelFormat);

		// new pixbuf from specified file name
		CPixbuf				(const CString &);

		// new pixbuf from stock id
		CPixbuf				(const char *);
	
		// pixbuf copy
		CPixbuf				(const CPixbuf &);

		// destructor
		virtual ~CPixbuf		();

	// specific functions
	public :

		// pixel format of the pixbuf
		TPixelFormat			GetPixelFormat		() const;

		// pixbuf size
		TSize				GetSize			() const;

		// pixbuf bytes per line number
		UInt16				GetRowBytes		() const;
	
		// image buffer direct access
		UInt8 *				GetBaseAddress		() const;

		// get the gdk pixbuf pointer
		GdkPixbuf *			GetPixbuf		() const;

		// get file name or stock id (id:xxx if so)
		CString				GetName			() const;

	// services functions
	public :

		// draws a line from point to point on a specified buffer (Brensenham algorithm application), constant alpha parameter 
		// not handled yet 
		static void			DrawLine		(const TPoint &inP1, const TPoint &inP2, UInt8 *inData, 
									 const TPixelFormat inPixelFormat, const UInt16 inRowBytes, 
									 const UInt32 inColor=0xFF000000, const Float32 inAlpha=1.0f);

		// draws an empty rectangle on the specified bounds on the buffer, constant alpha parameter
		static void			DrawRectangle		(const TBounds &inRectangle, UInt8 *inData, 
									 const TPixelFormat inPixelFormat, const UInt16 inRowBytes,
									 const UInt32 inColor=0xFF000000, const Float32 inAlpha=1.0f);

	// protected attributes
	protected :

		// associated pixbuf of the instance
		GdkPixbuf *			m_Pixbuf;

		// filename or stockid if any
		CString				m_Name;

		// metaclass section
		SECTION_CAPSULE_METACLASS;
};

// metaclass and class tag declaration
DECLARE_CAPSULE_METACLASS ('pxbf', CPixbuf, CClass);

#endif
