/*******************************************************************************************************************************************
 ctogglebutton.c
*******************************************************************************************************************************************/

#include "ctogglebutton.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CToggleButton);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CToggleButton::CToggleButton (CContainer *inOwner, const CButtonListener *inListener)
	      :CButton 	     (NULL, inListener)
{
	// launch the gtkol instanciation process if the owner is specified
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CToggleButton::~CToggleButton ()
{
	// widget deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CToggleButton::PerformWidgetInstanciate ()
{
	return ::gtk_toggle_button_new ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CToggleButton::PerformWidgetInitialize ()
{
	// generic call first
	CButton::PerformWidgetInitialize ();

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// pointer check
	if (inGtkWidget == NULL) return;

	// disconnect the default click function
	::g_signal_handlers_disconnect_by_func (G_OBJECT(inGtkWidget), (void*)G_CALLBACK(CButton::OnClick), this);

	// connect the toggle signal
	::g_signal_connect (G_OBJECT(inGtkWidget), "toggled", G_CALLBACK(CToggleButton::OnClick), this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// set state
//-----------------------------------------------------------------------------------------------------------------------------------------
void CToggleButton::Check (const bool inCheck)
{
	GtkWidget *inGtkWidget = GetGtkWidget();
	if (inGtkWidget != NULL) ::gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(inGtkWidget), inCheck);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get state
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CToggleButton::IsChecked () const
{
	GtkWidget *inGtkWidget = GetGtkWidget();
	return inGtkWidget != NULL ? ::gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(inGtkWidget)) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CToggleButton::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic serialization first
	CButton::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_TOGGLEBUTTON_ELEMENT);

			// add specific attributes
			newXMLElement -> AddAttribute (XML_TOGGLEBUTTON_ATTR_PRESS, CString(IsChecked()?"true":"false"));

			// modify the current xml node so that the derived definition will continue the serialization process under
			// the current xml node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the expected xml child node
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_TOGGLEBUTTON_ELEMENT);

			// check we got it
			if (inXMLNode == NULL)
				throw new CException (CString("CToggleButton::Serialize : the current node is not a ") + 
						      XML_TOGGLEBUTTON_ELEMENT + " one.", __exception(XMLPARSE));

			// modify the current xml node so that the derived definition will continue the serialization process under
			// the current xml node
			ioXMLElementNode = inXMLNode;

			// set the toggle state
			Check (::xml_node_get_attribute (inXMLNode, XML_TOGGLEBUTTON_ATTR_PRESS).GetValue().ToBool());
		}
		break;
	}
}
