/*
 *  @(#)XMLFileServer.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.pmti.v1.autodoc.v1.xml;


import java.io.Writer;
import java.io.StringWriter;
import java.io.PrintWriter;
import java.io.IOException;

import net.sourceforge.groboutils.autodoc.v1.testserver.TestData;
import net.sourceforge.groboutils.autodoc.v1.testserver.TestInfo;

import net.sourceforge.groboutils.pmti.v1.autodoc.v1.server.AbstractNewFileServer;

import net.sourceforge.groboutils.pmti.v1.autodoc.v1.ITFTestData;

import junit.framework.AssertionFailedError;

import net.sourceforge.groboutils.util.xml.v1.XMLUtil;


/**
 * An interface which corresponds to a part of the framework that knows how
 * to deal with the framework's <tt>TestData</tt>.  It may directly deal with
 * the data, or pass it off to a remote server.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since      March 17, 2002
 * @version    $Date: 2003/02/10 22:51:57 $
 */
public class XMLFileServer extends AbstractNewFileServer
{
    public XMLFileServer( String prefix )
    {
        this( prefix, ".xml" );
    }
    
    
    public XMLFileServer( String prefix, String postfix )
    {
        super( prefix, postfix, false );
    }
    
    
    /**
     * Write the BugTestResult with all the result's encountered BugTestLog
     * events.
     */
    protected void writeTestData( TestData td, Writer w )
            throws IOException
    {
        if (td == null || !(td instanceof ITFTestData))
        {
            return;
        }
        ITFTestData itd = (ITFTestData)td;
        TestInfo ti = td.getTestInfo();
        String newline = System.getProperty( "line.separator" );
        if (newline == null)
        {
            newline = "\n";
        }

        AssertionFailedError failures[] = itd.getFailures();
        Throwable errors[] = itd.getErrors();
        String logs[] = itd.getIssues();
        
        StringBuffer sb = new StringBuffer(
            "<?xml version=\"1.0\"?>" );
        sb.append( newline ).
            append( "<testresult suite=\"" );
        escapeString( sb, ti.getSuite() );
        sb.append( "\" test=\"" );
        escapeString( sb, ti.getMethod() );
        sb.append( "\" tests=\"" ).
            append( itd.getTestCount() ).
            append( "\" time=\"" ).
            // These are just numbers, so we shouldn't need to escape them.
            // Heh... famous last words...
            append( ( (double)itd.getRunTime() / 1000.0 ) ).
            append( "\" failures=\"" ).
            append( (failures == null ? 0 : failures.length) ).
            append( "\" errors=\"" ).
            append( (errors == null ? 0 : errors.length ) ).
            append( "\" >" ).
            append( newline );
        if (failures != null)
        {
            for (int i = 0; i < failures.length; ++i)
            {
                sb.append( "    <failure message=\"" );
                escapeString( sb, failures[i].getMessage() );
                sb.append( "\" >" ).append( newline );
                escapeString( sb, toString( failures[i] ) );
                sb.append( "    </failure>" ).append( newline );
            }
        }
        if (errors != null)
        {
            for (int i = 0; i < errors.length; ++i)
            {
                sb.append( "    <error message=\"" );
                escapeString( sb, errors[i].getMessage() );
                sb.append( "\" >" ).append( newline );
                escapeString( sb, toString( errors[i] ) );
                sb.append( "    </error>" ).append( newline );
            }
        }
        for (int i = 0; i < logs.length; ++i)
        {
            sb.append( "    <issue issueid=\"" );
            escapeString( sb, logs[i] );
            sb.append( "\" />" ).
                append( newline );
        }
        sb.append( "</testresult>" ).
            append( newline );
        
        w.write( sb.toString() );
        w.flush();
    }
    
    
    protected void escapeString( StringBuffer sb, String raw )
    {
        XMLUtil.getInstance().utf2xml( raw, sb );
    }
    
    
    protected String toString( Throwable t )
    {
        StringWriter sw = new StringWriter();
        t.printStackTrace( new PrintWriter( sw ) );
        return sw.toString();
    }
}

