/*
 * @(#)AnalysisModuleSetIOUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;


/**
 * Tests the AnalysisModuleSetIO class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     January 7, 2003
 */
public class AnalysisModuleSetIOUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = AnalysisModuleSetIOUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public AnalysisModuleSetIOUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    public void testWriteAnalysisModuleSet1() throws Exception
    {
        AnalysisModuleSetIO amsio = new AnalysisModuleSetIO();
        StringWriter sw = new StringWriter();
        AnalysisModuleSet ams = new AnalysisModuleSet();
        
        amsio.writeAnalysisModuleSet( ams, sw );
        
        String res = sw.toString();
        DOC.getLog().info( "Wrote module set ["+res+"]" );
        assertEquals(
            "Incorrect result format.",
            "0{}",
            res );
    }
    
    public void testWriteAnalysisModuleSet2() throws Exception
    {
        AnalysisModuleSetIO amsio = new AnalysisModuleSetIO();
        StringWriter sw = new StringWriter();
        AnalysisModuleSet ams = createAnalysisModuleSet(
            new IAnalysisModule[] {
                createIAnalysisModule( "n", "u", "m" ),
            } );
        
        amsio.writeAnalysisModuleSet( ams, sw );
        
        String res = sw.toString();
        DOC.getLog().info( "Wrote module set ["+res+"]" );
        assertEquals(
            "Incorrect result format.",
            "1{[0;11;1:n,1:u,1:m]}",
            res );
    }
    
    public void testWriteAnalysisModuleSet3() throws Exception
    {
        AnalysisModuleSetIO amsio = new AnalysisModuleSetIO();
        StringWriter sw = new StringWriter();
        AnalysisModuleSet ams = createAnalysisModuleSet(
            new IAnalysisModule[] {
                createIAnalysisModule( "n", "u", "m" ),
                createIAnalysisModule( "", "", "" ),
                createIAnalysisModule( "aaaa", "bbbb", "1234" ),
            } );
        
        amsio.writeAnalysisModuleSet( ams, sw );
        
        String res = sw.toString();
        DOC.getLog().info( "Wrote module set ["+res+"]" );
        assertEquals(
            "Incorrect result format.",
            "3{[0;11;1:n,1:u,1:m][1;8;0:,0:,0:][2;20;4:aaaa,4:bbbb,4:1234]}",
            res );
    }
    
    
    public void testReadAnalysisModuleSet1() throws Exception
    {
        AnalysisModuleSetIO amsio = new AnalysisModuleSetIO();
        StringReader sr = new StringReader( "0{}" );
        AnalysisModuleSet ams = amsio.readAnalysisModuleSet( sr );
        IAnalysisModule[] amL = ams.getAnalysisModules();
        assertNotNull(
            "Returned null data list.",
            amL );
        assertEquals(
            "Did not return an empty data set.",
            0,
            amL.length );
    }
    
    
    public void testReadAnalysisModuleSet2() throws Exception
    {
        AnalysisModuleSetIO amsio = new AnalysisModuleSetIO();
        StringReader sr = new StringReader( "1{[0;11;1:n,1:u,1:m]}" );
        AnalysisModuleSet ams = amsio.readAnalysisModuleSet( sr );
        IAnalysisModule[] amL = ams.getAnalysisModules();
        assertNotNull(
            "Returned null data list.",
            amL );
        assertEquals(
            "Did not return an empty data set.",
            1,
            amL.length );
        IAnalysisModule am = amL[0];
        assertNotNull(
            "Returned null module 0.",
            am );
        assertEquals(
            "name incorrect.",
            "n",
            am.getMeasureName() );
        assertEquals(
            "unit incorrect.",
            "u",
            am.getMeasureUnit() );
        assertEquals(
            "encoding incorrect.",
            "m",
            am.getMimeEncoding() );
    }
    
    
    public void testReadAnalysisModuleSet3() throws Exception
    {
        AnalysisModuleSetIO amsio = new AnalysisModuleSetIO();
        StringReader sr = new StringReader(
            "2{[0;11;1:n,1:u,1:m][1;8;0:,0:,0:]}" );
        AnalysisModuleSet ams = amsio.readAnalysisModuleSet( sr );
        IAnalysisModule[] amL = ams.getAnalysisModules();
        assertNotNull(
            "Returned null data list.",
            amL );
        assertEquals(
            "Did not return an empty data set.",
            2,
            amL.length );
        
        IAnalysisModule am = amL[0];
        assertNotNull(
            "Returned null module 0.",
            am );
        assertEquals(
            "0: name incorrect.",
            "n",
            am.getMeasureName() );
        assertEquals(
            "0: unit incorrect.",
            "u",
            am.getMeasureUnit() );
        assertEquals(
            "0: encoding incorrect.",
            "m",
            am.getMimeEncoding() );
        
        am = amL[1];
        assertNotNull(
            "Returned null module 1.",
            am );
        assertEquals(
            "1: name incorrect.",
            "",
            am.getMeasureName() );
        assertEquals(
            "1: unit incorrect.",
            "",
            am.getMeasureUnit() );
        assertEquals(
            "1: encoding incorrect.",
            "",
            am.getMimeEncoding() );
    }
    
    
    public void testReadAnalysisModuleSet4() throws Exception
    {
        AnalysisModuleSetIO amsio = new AnalysisModuleSetIO();
        StringReader sr = new StringReader(
            "{}" );
        try
        {
            amsio.readAnalysisModuleSet( sr );
            fail( "Did not throw IOException." );
        }
        catch (IOException ioe)
        {
            // test exception
        }
    }
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    protected AnalysisModuleSet createAnalysisModuleSet( IAnalysisModule amL[] )
    {
        return CCCreatorUtil.createAnalysisModuleSet( amL );
    }
    
    protected IAnalysisModule createIAnalysisModule( String name, String unit,
            String mime )
    {
        return CCCreatorUtil.createIAnalysisModule( name, unit, mime );
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();

       
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

