/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  




-----------------------------------------------------------------------
FILENAME:                eval_speed.h
LIBRARY:                 libgpiv

SOURCE:                  piv_eval_speed.c
EXTERNAL FUNCTIONS:      gpiv_piv_shift_grid,
                         gpiv_piv_dxdy_at_new_grid,
                         gpiv_piv_gridgen
                         gpiv_piv_gridadapt
                         gpiv_piv_speedup_cmpr
			 gpiv_piv_compress
			 gpiv_piv_decompress

LAST MODIFICATION DATE: $Id: eval_speed.h,v 1.11 2006/01/31 13:27:22 gerber Exp $
--------------------------------------------------------------------- */
/* NOTE: cmpr is variable, not input parameter (anymore) */

#ifndef __LIBGPIV_EVAL_SPEED_H__
#define __LIBGPIV_EVAL_SPEED_H__

#define CMPR_FACT 2            /* Image compression factor for speeding up the
                                  evaluation */
#define GPIV_SHIFT_FACTOR 2    /* For initial grid, apply 
                                  int_shift = int_size_2 / GPIV_SHIFT_FACTOR */


/*--------------------------------------------------------------------
  Function prototypes */


char *
gpiv_piv_shift_grid(GpivPivData gpd,
                    GpivPivData *gpd_out
                    );
/*-----------------------------------------------------------------------------
 * shifts the knots of a 2-dimensional grid containing PIV data for improved 
 * (bi-linear) interpolation
 *
 * See: T. Blu, P. Thevenaz, "Linear Interpolation Revitalized",
 * IEEE Trans. in Image Processing, vol13, no 5, May 2004
 *
-----------------------------------------------------------------------------*/



char *
gpiv_piv_dxdy_at_new_grid(GpivPivData piv_data_src,
                          GpivPivData *piv_data_dest
                          );
/*-----------------------------------------------------------------------------
 * calculates dx, dy of piv_data_dest from piv_data_src
 * by bi-linear interpolation of inner points with shifted knots
 * or extrapolation of outer lying points

    dist_:      distance
    _n:         NORTH
    _s:         SOUTH
    _e:         EAST
    _w:         WEST
    _nn:        at NORTH of NORTH, etc.
-----------------------------------------------------------------------------*/



char *
gpiv_piv_gridgen(GpivPivData * piv_data,
                 GpivImagePar  image_par,
                 GpivEvalPar piv_eval_par
                 );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Generates grid by Calculating the positions of interrogation areas
 *
 * INPUTS:
 *     image_par:      structure of image parameters
 *     piv_eval_par:   structure of piv evaluation parameters
 *
 * OUTPUTS:
 *     out_data:       output piv data from image analysis
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/



void 
gpiv_piv_gridadapt(GpivImagePar * image_par, 
                   GpivEvalPar piv_eval_par_in,
                   GpivEvalPar * piv_eval_par_out,
                   GpivPivData * piv_data,
                   int sweep, 
                   gboolean * grid_last
                   );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Adjust grid nodes if zero_off or adaptive interrogation 
 *     area has been used. This is performed by modifying int_shift equal 
 *     to int_shift / GPIV_SHIFT_FACTOR , until it reaches (src)
 *     int_shift. Then, grid_last is set TRUE, which will avoid
 *     changing the interrogation shift in next calls and signal the
 *     (while loop in) the calling function.
 *     
 * INPUTS:
 *     @image_par:              image parameters
 *     @piv_eval_par_src:       piv evaluation parameters
 *     @piv_data:               input PIV data
 *     @sweep:                  interrogation sweep step
 *
 * OUTPUTS:
 *     @image_par:              image parameters
 *     @piv_eval_par_dest:      modified piv evaluation parameters
 *     @grid_last:              flag if final grid refinement has been 
 *                              reached
 *     @piv_data:               modified PIV data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/



/*--------------------------------------------------------------------
  Speeds up the evaluation process if zero_off or adaptive interrogation 
  has been used by compressing the image */
void 
gpiv_piv_speedup_cmpr(guint16 **img1, 
                      guint16 * **img1_ACT, 
                      guint16 **img2, 
                      guint16 * **img2_ACT, 
                      GpivImagePar image_par, 
                      GpivImagePar * image_par_ACT, 
                      GpivEvalPar piv_eval_par, 
                      GpivEvalPar * piv_eval_par_ACT, 
                      GpivPivData * piv_data,
                      int *cmpr_fact,
                      int *int_size_0, 
                      int sweep, 
                      int sweep_last);

/* -------------------------------------------------------------------
 Compresses image(s) and adapts relevant variables and parameters  */
void 
gpiv_piv_compress(GpivPivData * old_data,	/* piv data from a previous analysis */
                  guint16 **img_1,	/* raw (binary) image data of first piv image  */
                  guint16 **img_2,	/* raw (binary) image data of second piv image  */
                  guint16 **img_cmpr_1,	/* raw (binary) compressed image data of first piv image  */
                  guint16 **img_cmpr_2,	/* raw (binary) compressed image data of second piv image  */
                  GpivImagePar * image_par,
                  GpivEvalPar * piv_eval_par
                  );
                       
/* --------------------------------------------------------------------
   Adapts ONLY relevant variables and parameters to cancel out 
   gpiv_piv_compress */
void 
gpiv_piv_decompress(GpivPivData * piv_data,
                    /*                          guint16 **img_1, */
                    /*                          guint16 **img_2, */
                    GpivImagePar * image_par,
                    GpivEvalPar * piv_eval_par
                    );


#endif /* __LIBGPIV_EVAL_PEED_H__ */
