/**
 * @file description.c Description functions
 *
 * $Id: description.c,v 1.3 2001/07/29 03:35:34 chipx86 Exp $
 *
 * @Copyright (C) 1999-2001 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <gnurdf.h>
#include <libgnurdf/internal.h>

RdfDescription *
rdfFirstDescription(RdfSchema *schema)
{
	xmlNode *tree;
	RdfNamespace *rdfNs;

	if (schema == NULL)
		return NULL;
	
	rdfNs = rdfGetRdfNamespace(schema);
	tree  = schema->xmlRootNode;

	if (tree == NULL)
		return NULL;
	
	tree = tree->xmlChildrenNode;

	while (tree != NULL)
	{
		if ((tree->ns == rdfNs) && (!strcmp((char *)tree->name, "Description")))
			return tree;

		tree = tree->next;
	}

	return NULL;
}

RdfDescription *
rdfNextDescription(RdfDescription *desc)
{
	RdfNamespace *rdfNs;

	if (desc == NULL)
		return NULL;
	
	rdfNs = desc->ns;
	desc  = desc->next;

	while (desc != NULL)
	{
		if ((desc->ns == rdfNs) && (!strcmp((char *)desc->name, "Description")))
			return desc;

		desc = desc->next;
	}

	return NULL;
}

RdfDescription *
rdfAddDescription(RdfSchema *schema, const char *id, const char *about)
{
	RdfNamespace *rdfNs;
	xmlNode *tree;

	if (schema == NULL)
		return NULL;
	
	rdfNs  = rdfGetRdfNamespace(schema);

	tree = xmlNewChild(schema->xmlRootNode, rdfNs, "Description", NULL);

	if (id != NULL)
		xmlNewNsProp(tree, rdfNs, "ID", (xmlChar *)id);

	if (about != NULL)
		xmlNewNsProp(tree, rdfNs, "about", about);

	return (RdfDescription *)tree;
}

RdfDescription *
rdfGetDescriptionFromAbout(RdfSchema *schema, const char *about)
{
	RdfDescription *desc;
	char *found_about;

	if (schema == NULL || about == NULL)
		return NULL;
	
	desc = rdfFirstDescription(schema);

	while (desc != NULL)
	{
		found_about = (char *)rdfGetDescriptionAbout(desc);
		
		if (found_about != NULL && !strcmp(found_about, about))
		{
			free(found_about);
			
			return desc;
		}

		free(found_about);

		desc = rdfNextDescription(desc);
	}

	return NULL;
}

void
rdfSetDescriptionId(RdfDescription *desc, const char *id)
{
	RdfNamespace *rdfNs;
	
	if (desc == NULL || id == NULL)
		return;

	rdfNs = rdfGetRdfNamespace(rdfGetElementSchema(desc));

	if (rdfGetAttribute(desc, "ID") != NULL)
	{
		rdfSetAttribute(desc, "ID", id);
	}
	else
	{
		rdfSetNsAttribute(desc, "ID", id, rdfNs);
	}
}

char *
rdfGetDescriptionId(RdfDescription *desc)
{
	char *res;
	RdfNamespace *rdfNs;
	
	if (desc == NULL)
		return NULL;
	
	rdfNs = rdfGetRdfNamespace(rdfGetElementSchema(desc));

	if ((res = rdfGetNsAttribute(desc, "ID", rdfNs)) != NULL)
	{
		return res;
	}
	else if ((res = rdfGetAttribute(desc, "ID")) != NULL)
	{
		return res;
	}
	
	return NULL;
}

void
rdfSetDescriptionAbout(RdfDescription *desc, const char *about)
{
	RdfNamespace *rdfNs;
	
	if (desc == NULL || about == NULL)
		return;

	rdfNs = rdfGetRdfNamespace(rdfGetElementSchema(desc));
	
	if (rdfGetNsAttribute(desc, "About", rdfNs) != NULL)
	{
		rdfSetNsAttribute(desc, "About", about, rdfNs);
	}
	else if (rdfGetNsAttribute(desc, "about", rdfNs) != NULL)
	{
		rdfSetNsAttribute(desc, "about", about, rdfNs);
	}
	else if (rdfGetAttribute(desc, "About") != NULL)
	{
		rdfSetAttribute(desc, "About", about);
	}
	else if (rdfGetAttribute(desc, "about") != NULL)
	{
		rdfSetAttribute(desc, "about", about);
	}
	else
	{
		rdfSetNsAttribute(desc, "about", about, rdfNs);
	}
}

char *
rdfGetDescriptionAbout(RdfDescription *desc)
{
	char *res;
	RdfNamespace *rdfNs;

	if (desc == NULL)
		return NULL;
	
	rdfNs = rdfGetRdfNamespace(rdfGetElementSchema(desc));
	
	/* Try looking for "about" with a namespace prefix */
	if ((res = rdfGetNsAttribute(desc, "about", rdfNs)) != NULL)
	{
		return res;
	}
	/* Try looking for "about" without a namespace prefix */
	else if ((res = rdfGetAttribute(desc, "about")) != NULL)
	{
		return res;
	}
	/* Try "About" with a namespace prefix */
	else if ((res = rdfGetNsAttribute(desc, "About", rdfNs)) != NULL)
	{
		return res;
	}
	/* Try "About" without a namespace prefix */
	else if ((res = rdfGetAttribute(desc, "About")) != NULL)
	{
		return res;
	}

	return NULL;
}

RdfDescription *
rdfGetDescriptionFromId(RdfSchema *schema, const char *id)
{
	RdfDescription *desc;
	char *found_id;

	if (schema == NULL || id == NULL)
		return NULL;
	
	desc = rdfFirstDescription(schema);

	while (desc != NULL)
	{
		found_id = (char *)rdfGetDescriptionId(desc);
		
		if (found_id != NULL && !strcmp(found_id, id))
		{
			free(found_id);

			return desc;
		}

		free(found_id);

		desc = rdfNextDescription(desc);
	}

	return NULL;
}

