/*
 * Galago Presence API
 *
 * Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_PRESENCE_H_
#define _GALAGO_PRESENCE_H_

#include <sys/time.h>
#include <time.h>

typedef struct _GalagoPresence        GalagoPresence;
typedef struct _GalagoPresenceClass   GalagoPresenceClass;
typedef struct _GalagoPresencePrivate GalagoPresencePrivate;

#include <libgalago/galago-account.h>
#include <libgalago/galago-object.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-status.h>

/**
 * A presence.
 *
 * A presence is associated with an account and contains a list of the
 * account's statuses, as well as idle times and other information.
 */
struct _GalagoPresence
{
	GalagoObject parent_object;

	GalagoPresencePrivate *priv;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

/**
 * GalagoPresence's class.
 */
struct _GalagoPresenceClass
{
	GalagoObjectClass parent_class;

	/* Signals */
	void (*idle_changed)(GalagoPresence *presence, gboolean idle,
						 time_t idle_start_time);
	void (*status_added)(GalagoPresence *presence, GalagoStatus *status);
	void (*status_removed)(GalagoPresence *presence, GalagoStatus *status);
	void (*status_updated)(GalagoPresence *presence, GalagoStatus *status,
						   const char *attr_id);
	void (*changed)(GalagoPresence *presence);

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

#define GALAGO_TYPE_PRESENCE (galago_presence_get_type())
#define GALAGO_PRESENCE(obj) \
		(G_TYPE_CHECK_INSTANCE_CAST((obj), GALAGO_TYPE_PRESENCE, \
									GalagoPresence))
#define GALAGO_PRESENCE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_CAST((klass), GALAGO_TYPE_PRESENCE, \
								 GalagoPresenceClass))
#define GALAGO_IS_PRESENCE(obj) \
		(G_TYPE_CHECK_INSTANCE_TYPE((obj), GALAGO_TYPE_PRESENCE))
#define GALAGO_IS_PRESENCE_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_TYPE((klass), GALAGO_TYPE_PRESENCE))
#define GALAGO_PRESENCE_GET_CLASS(obj) \
		(G_TYPE_INSTANCE_GET_CLASS((obj), GALAGO_TYPE_PRESENCE, \
								   GalagoPresenceClass))

#define GALAGO_DBUS_PRESENCE_INTERFACE "org.freedesktop.Galago.Presence"

G_BEGIN_DECLS

GType galago_presence_get_type(void);

void galago_presence_set_idle(GalagoPresence *presence, gboolean idle,
							  time_t idle_start_time);
gboolean galago_presence_is_idle(const GalagoPresence *presence);
time_t galago_presence_get_idle_time(const GalagoPresence *presence);
time_t galago_presence_get_idle_start_time(const GalagoPresence *presence);

GalagoAccount *galago_presence_get_account(const GalagoPresence *presence);

gboolean galago_presence_is_discarded(const GalagoPresence *presence);
gboolean galago_presence_is_available(const GalagoPresence *presence);


void galago_presence_set_statuses(GalagoPresence *presence, GList *statuses);
void galago_presence_add_status(GalagoPresence *presence,
								GalagoStatus *status);
void galago_presence_remove_status(GalagoPresence *presence,
								   const char *status_id);
void galago_presence_clear_statuses(GalagoPresence *presence);
GList *galago_presence_get_statuses(const GalagoPresence *presence);
GalagoStatus *galago_presence_get_active_status(const GalagoPresence *presence);

gboolean galago_presence_is_status_exclusive(const GalagoPresence *presence,
											 const char *status_id);
GalagoStatus *galago_presence_get_status(const GalagoPresence *presence,
										 const char *status_id);
gboolean galago_presence_has_status(const GalagoPresence *presence,
									const char *status_id);
gboolean galago_presence_has_status_type(const GalagoPresence *presence,
										 GalagoStatusType type);

int galago_presence_compare(const GalagoPresence *presence1,
							const GalagoPresence *presence2);

G_END_DECLS

#endif /* _GALAGO_PRESENCE_H_ */
