/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: IPAddress.hh,v 1.3 2001/12/19 12:46:45 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#ifndef __IPADDRESS_HH_FLAG__
#define __IPADDRESS_HH_FLAG__

#include <string>

#include <sys/types.h>
#ifndef __MINGW32__
# include <sys/socket.h>
# include <netinet/in.h>
# include <arpa/inet.h>
#else
# include <winsock2.h>
typedef unsigned long ulong;
#endif

#include <fwbuilder/FWException.hh>

namespace libfwbuilder
{

class IPAddress
{
    protected:

    friend class IPNetwork;

    unsigned int octets[4];
    virtual void validate() throw(FWException);
    
    public:
    
    explicit IPAddress();

    IPAddress(const unsigned char *data, size_t len) throw(FWException);
    IPAddress(const struct in_addr *) throw(FWException);
    explicit IPAddress(const std::string &) throw(FWException, FWNotSupportedException);
    IPAddress(const IPAddress &);

    IPAddress& operator=(const IPAddress &);

    IPAddress& operator=(const std::string &s) throw(FWException, FWNotSupportedException);

    friend bool operator<(const IPAddress &a, const IPAddress &b);
    friend bool operator==(const IPAddress &a, const IPAddress &b);
    unsigned const int operator[](size_t pos) const { return octets[pos]; }
    
    operator std::string() const { return toString(); }
    operator ulong() const;
    
    std::string toString() const;
   
};

class Netmask: public IPAddress
{
    protected:
    
    virtual void validate() throw(FWException);

    public:

    explicit Netmask();
    explicit Netmask(const std::string &) throw(FWException);
    Netmask(const unsigned char *data, size_t len) throw(FWException);

    /**
     * This constructor creates natural classful netmask for given IP address
     *
     * Cheatsheet:
     *                                   Net     Host    Total
     * Net      Addr                      Addr    Addr    Number
     * Class   Range      NetMask         Bits    Bits   of hosts
     * ----------------------------------------------------------
     * A        0-127    255.0.0.0         8      24     16777216
     * B      128-191    255.255.0.0      16      16        65536
     * C      192-254    255.255.255.0    24       8          256
     * 
     * (ref: RFC1375 & www.isi.edu/in-notes/iana/assignments/ipv4-address-space)
     *
     */
    explicit Netmask(const IPAddress &);

    Netmask& operator=(const std::string &s) throw(FWException);

    friend bool operator==(const Netmask &a, const Netmask &b);

};

class IPNetwork
{
    private:
    
    IPAddress address    ;
    Netmask   netmask    ;
    int       bcast_bits ;

    public:

    IPNetwork(const IPAddress &, const Netmask&, int bcast_bits=1);

    const IPAddress &getAddress         () const;
    const Netmask   &getNetmask         () const;
    IPAddress getBroadcastAddress () const;

    /**
     * Broadcast :  255.255.255.255  and  0.0.0.0,
     */
    bool isBroadcast() const;

    /**
     * Multicast :  224.0.0.0 - 239.0.0.0
     */
    bool isMulticast() const;

    IPNetwork& operator=(const IPNetwork &o);
    friend bool operator==(const IPNetwork &a, const IPNetwork &b);
    friend bool operator<(const IPNetwork &a, const IPNetwork &b);
    operator std::string() const { return address.toString()+"/"+netmask.toString(); }

    bool belongs(const IPAddress &) const;
    
};

class Interface;
class IPRoute
{
    public:

    IPRoute(const IPRoute &);
    IPRoute(const IPAddress &_dst, const Netmask &_nm, const IPAddress &_gw, const Interface *_intf,  bool _direct);
    virtual ~IPRoute();

    bool isDirect() const { return direct;}
    const Netmask   &getNetmask    () const { return nm;   }
    const IPAddress &getDestination() const { return dst;  }
    const IPAddress &getGateway    () const { return gw;   }

    /**
     * @return interface associated with this route, or
     * NULL if none.
     */
    const Interface *getInterface  () const { return intf; }
    
    private:
    
    Netmask   nm     ;
    IPAddress dst    ;
    IPAddress gw     ;
    const Interface  *intf;
    bool      direct ;
};

}

#endif

