/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "test.h"
#include "flux.h"
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>


int test_tt_size()
{
  int tts;
  
  tts = sizeof(TT);
  if (tts == 32 || tts == 56) return(tts);
  return(-tts);
}


int test_tt_allocation()
{
  TT *tt0, *tt1, *tt2;
  int i;

  for (i = 0; i < 32; i++)  /* Repeat to promote heap corruption */
  {
    if (!(tt0 = tt_new())) return(-1);
    tt_del(tt0);
  
    if (!(tt0 = tt_new_with_data("01234567890123456789012345678901", 32))) return(-1);
    if (!(tt1 = tt_dup(tt0))) return(-1);
    if (!(tt2 = tt_dup(tt1))) return(-1);
    tt_del(tt0);
    tt_del(tt1);
    tt_del(tt2);
  }

  return(0);
}


int test_tt_navigation()
{
  TT *tt0, *tt1, *tt2;
  int i, j;

  for (j = 0; j < 32; j++)  /* Repeat to promote heap corruption */
  {
    tt0 = tt_new();  /* root */
  
    for (i = 0; i < 32; i++)
    {
      tt1 = tt_new();
      tt_add_as_last_child(tt0, tt1);
    }
  
    for (i = 0; i < 32; i++)
    {
      tt2 = tt_new();
      tt_add_as_first_sibling(tt1, tt2);
    }

    /* Force creation of new root node */
  
    for (i = 0; i < 31; i++)
    {
      tt1 = tt_new();
      tt_add_as_last_sibling(tt0, tt1);
    }

    if (!(tt0 = tt_get_parent(tt0))) return(-1);

    /* Check the resulting structure */

    if (tt_count_children(tt0) != 32) return(-1);
    if (tt_count_siblings(tt_get_first_child(tt0)) != 32) return(-1);
    if (tt_count_children(tt_get_first_child(tt0)) != 64) return(-1);
    if (tt_count_children_all(tt0) != 96) return(-1);
    if (tt_count_siblings(tt0) != 1) return(-1);
    if (tt_count_children(tt_get_next(tt_get_first_child(tt0)))) return(-1);

    /* Free all */
  
    tt_del(tt0);
  }
  
  return(0);
}


int test_tt_matching()
{
  TT *tt0, *tt1;
  int i;
  
  for (i = 0; i < 32; i++)
  {
    tt0 = tt_new_with_data("root", 4);
    tt_new_with_parent_and_data(tt0, "tabernakel", 10);
    tt_new_with_parent_and_data(tt0, "nisselue", 8);
    tt_new_with_parent_and_data(tt0, "snurpenot", 9);
    tt_new_with_parent_and_data(tt0, "tabernakel", 10);
    tt_new_with_parent_and_data(tt0, "tabernakel", 10);

    if (!(tt1 = tt_find_first_child(tt0, "tabernakel", 10))) return(-1);
    if (!(tt1 = tt_find_next(tt1, "tabernakel", 10))) return(-1);
    if (!(tt1 = tt_find_next(tt1, "tabernakel", 10))) return(-1);
    if (tt_find_next(tt1, "tabernakel", 10)) return(-1);
    if (!(tt1 = tt_find_prev(tt1, "tabernakel", 10))) return(-1);
    if (!(tt_find_first_sibling(tt1, "nisselue", 8))) return(-1);

    tt_del(tt0);
  }
  
  return(0);
}


int test_tt_dataops()
{
  TT *tt0;
  int i;
  
  for (i = 0; i < 32; i++)
  {
    tt0 = tt_new_with_data("89A", 3);
    tt_data_prepend_bytes(tt0, "01234567", 8);
    tt_data_append_bytes(tt0, "BCDEFGHIJKLM", 12);
    if (tt_strcmp(tt0, "0123456789ABCDEFGHIJKLM")) return(-1);
    tt_del(tt0);
  }

  return(0);
}


int test_tt_external()
{
  TT *tt0;
  FILE *f;
  int i;
  
  for (i = 0; i < 32; i++)
  {
    f = fopen("temporary_tt_test", "wt");
    fwrite("0123456789", 10, 1, f);
    fclose(f);
  
    tt0 = tt_new();
    if (!tt_data_set_file(tt0, "temporary_tt_test", TRUE)) return(-1);
    if (tt_strcmp(tt0, "0123456789")) return(-1);

    tt_data_prepend_bytes(tt0, "ABCD", 4);
    tt_data_append_bytes(tt0, "EFGH", 4);
    if (tt_strcmp(tt0, "ABCD0123456789EFGH")) return(-1);

    tt_del(tt0);
  }

  return(0);
}


int test_tt_hashing()
{
  TT *tt0;
  int i;
  u32 hash, hash_prev = 0;  /* Keeps compiler happy */
  
  for (i = 0; i < 32; i++)
  {
    tt0 = tt_new_with_data("root", 4);
    tt_new_with_parent_and_data(tt0, "tabernakel", 10);
    tt_new_with_parent_and_data(tt0, "nisselue", 8);
    tt_new_with_parent_and_data(tt0, "snurpenot", 9);
    
    hash = tt_hash_all(tt0);
    tt_del(tt0);

    if (i && hash != hash_prev) return(-1);
    hash_prev = hash;
  }
  
  return(0);
}


int test_tt_printability()
{
  TT *tt0, *tt1;
  FILE *f;
  int i;
  
  for (i = 0; i < 32; i++)
  {
    f = fopen("temporary_tt_test", "w+t");
    if (!f) return(-1);

    tt0 = tt_new_with_data("root", 4);
    tt_new_with_parent_and_data(tt0, "taber{nakel}", 12);
    tt_new_with_parent_and_data(tt0, "\"nisse\"lue", 10);
    tt_new_with_parent_and_data(tt0, "snurpe (not)", 12);
    
    tt_print_to_file(tt0, f, TT_PRINT_ALLMAN, FALSE);
    rewind(f);

    tt_del(tt0);
    
    if (!(tt1 = tt_scan_from_file(f))) return(-1);
    fclose(f);
    remove("temporary_tt_test");

    if (!tt_find_first_child(tt1, "taber{nakel}", 12) ||
        !tt_find_first_child(tt1, "\"nisse\"lue", 10) ||
        !tt_find_first_child(tt1, "snurpe (not)", 12))
    {
      tt_del(tt1);
      return(-1);
    }
    
    tt_del(tt1);
  }

  return(0);
}


int test_tt_speed()
{
  TT *tt0, *tt1;
  unsigned int i;
  double r1;

  bm_start();

  for (i = 0; i < 500000; i++)
  {
    tt0 = tt_new_with_data("01234567890123456789012345678901", 32);
    tt1 = tt_new_with_parent_and_data(tt0, "sub", 3);

    tt1 = tt_new_with_parent_and_data(tt0, "sub2", 4);

    tt_del(tt0);
  }
  
  bm_end();
  
  r1 = (double) (5.0 * 500000.0 * 1000000.0) / (double) bm_diff_get_usec();
  return((int) r1);
}
