/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/types.h>
#include <fcntl.h>
#include "fifobuf.h"
#include "sock.h"
#include "fstring.h"


#define TRIES 1

/* --- New functions --- */


/* ---- *
 * Read *
 * ---- */

#ifdef __GNUC__
__inline
#endif
int _sock_feed(SOCK *s)
{
  int t;
  int fd;
  int i0, i1;
  register FIFOBUF *fib;

  fd = SOCK_FDREAD(s);
  fib = s->fib_in;
  i0 = fifobuf_free_cur(fib);
  i1 = fib->bufsize - fib->byte_in;

  if (!i1)
  {
    fib->byte_in = 0;
    fib->node_in = fib->node_in->next;
    i1 = fib->bufsize;
  }

  fcntl(fd, F_SETFL, O_NONBLOCK);
  t = read(fd, fib->node_in->data + fib->byte_in, i0 < i1 ? i0 : i1);

  if (t < 0)
  {
    if (errno != EWOULDBLOCK) _sock_chkerrno(s);
  }
  else
  {
    fib->byte_in += t;
    fib->enqueued += t;
  }

  fcntl(fd, F_SETFL, 0);
  return(t);  /* Used to detect disconnects */
}


int sock_read(SOCK *s, void *dest, unsigned int size)
{
  int fd, t;

  t = fifobuf_dequeue(s->fib_in, dest, size);
  dest += t;
  size -= t;

  for (fd = SOCK_FDREAD(s); size; )
  {
    fcntl(fd, F_SETFL, 0);
    t = read(fd, dest, size);
    if (t < 1 && errno != EINTR) break;
    dest += t;
    size -= t;
  }

  if (size) { _sock_chkerrno(s); return(-1); }
  return(0);
}


void *sock_dread(SOCK *s, unsigned int size)
{
  void *dest;
  
  dest = malloc(size);
  if (!sock_read(s, dest, size)) return(dest);

  free(dest);
  return(0);
}


int sock_getc(SOCK *s)
{
  unsigned char c;

#ifdef TRIES
  if (!s->fib_in->enqueued) _sock_feed(s);
#endif
  if (!sock_read(s, &c, sizeof(unsigned char))) return((int) c);
  return(-1);
}


int sock_gets(SOCK *s, char *dest, unsigned int size_max)
{
  int c;
  unsigned int size;

  size = 0;
  while (size < size_max && (c = sock_getc(s)) != '\n')
  {
    if (c < 0) return(-1);
    if (c == '\r') continue;
    dest[size++] = c;
  }

  dest[size] = 0;
  return(size);
}


int _sock_fifobuf_lf(void *data, unsigned long len, void *extra)
{
  unsigned char *p;

  p = memchr(data, '\n', len);

  if (p)
  {
    (*((unsigned int *) extra)) += (p - (unsigned char *) data);
    return(-1);
  }

  (*((unsigned int *) extra)) += len;
  return(0);
}


char *sock_dgets(SOCK *s)
{
  char *r = 0, *p0;
  unsigned int lf;
  int fd, t;
  fd_set sock_read;
  char buf[256];

  lf = 0;
  if (fifobuf_do(s->fib_in, s->fib_in->enqueued, _sock_fifobuf_lf, (void *) &lf))
  {
    r = fifobuf_dequeue_dyn(s->fib_in, (unsigned long) lf + 1);
    *(r + lf) = 0;
    if (lf && *(r + lf - 1) == '\r') *(r + lf - 1) = 0;
    if (fifobuf_peek(s->fib_in, buf, 1) && buf[0] == '\r')
      fifobuf_drop(s->fib_in, 1);
    return(r);
  }

  fd = SOCK_FDREAD(s);
  fcntl(fd, F_SETFL, O_NONBLOCK);

  for (;;)
  {
#if 0    
    printf("[sock_dgets blocking]"); fflush(stdout);
#endif
    FD_ZERO(&sock_read);
    FD_SET(fd, &sock_read);
    t = select(fd + 1, &sock_read, 0, 0, 0);
    if (t < 0 && !_sock_chkerrno(s)) break;

    t = read(fd, buf, 255);
    if (t < 1)
    {
#if 0      
      printf("[sock_dgets got empty select]"); fflush(stdout);
#endif
      if (!_sock_chkerrno(s)) break;
      continue;
    }

    p0 = memchr(buf, '\n', t);
    if (p0)
    {
      int len;
      
      len = p0 - buf;
      
      if (len && *(p0 - 1) == '\r') len--;
      r = malloc(s->fib_in->enqueued + len + 1);
      memcpy(r + s->fib_in->enqueued, buf, len);
      *((r + s->fib_in->enqueued) + len) = 0;
      fifobuf_dequeue(s->fib_in, r, s->fib_in->enqueued);
      p0++; /* Skip newline */

      if (p0 < (buf + t))
      {
        if (*p0 == '\r') p0++;
        if (p0 < (buf + t))
        {
#if 0
          printf("[sock_dgets leftover: %d bytes]\n\n", t - (p0 - buf));
          fwrite(p0 + 1, t - (p0 - buf), 1, stdout);
#endif
          fifobuf_enqueue(s->fib_in, p0 + 1, t - (p0 + 1 - buf));
        }
      }
      
      break;
    }

    fifobuf_enqueue(s->fib_in, buf, t);
#if 0
    printf("[sock_dgets got %d bytes]", t); fflush(stdout);
#endif
  }

  fcntl(fd, F_SETFL, 0);
  return(r);
}


/* FIXME: Some systems wholly lack v*scanf. Compensating for this
 *        involves a bit of work. */

#ifdef HAVE_VSSCANF

int sock_scanf(SOCK *s, char *format, ...)
{
  va_list args;
  char *str;
  int ret;

  if (!(str = sock_dgets(s))) return (0);

  va_start(args, format);
  ret = vsscanf(str, format, args);
  va_end(args);
  free(str);
  
  return(ret);
}

#endif


/* ----- *
 * Write *
 * ----- */

#ifdef __GNUC__
__inline
#endif
int _sock_spill(SOCK *s)
{
  int i0;
  int fd;
  int t;
  register FIFOBUF *fib;

  fd = SOCK_FDWRITE(s);
  fib = s->fib_out;
  i0 = fib->bufsize - fib->byte_out;

  if (!i0)
  {
    fib->byte_out = 0;
    fib->node_out = fib->node_out->next;
    i0 = fib->bufsize;
  }

  fcntl(fd, F_SETFL, O_NONBLOCK);

  t = write(fd, fib->node_out->data + fib->byte_out,
            fib->enqueued < i0 ? fib->enqueued : i0);
  
  if (t < 0) _sock_chkerrno(s);
  else
  {
    fib->byte_out += t;
    fib->enqueued -= t;
  }

  fcntl(fd, F_SETFL, 0);
  return(t);  /* For _sock_feed() symmetry. */
}


int _sock_write(void *data, unsigned long len, void *s)
{
  int t;
  int fd;

  fd = SOCK_FDWRITE(((SOCK *) s));
  while (len)
  {
    t = write(fd, data, len);
    if (t < 0 && errno != EINTR) return(-1);
    data += t;
    len -= t;
  }

  return(0);
}


int sock_flush(SOCK *s)
{
  int n;

  if (!(s->flags & SOCK_CONNECTED)) return(-1);
  
  n = fifobuf_do(s->fib_out, s->fib_out->enqueued, _sock_write, s);
  if (n > 0)
  {
    fifobuf_drop(s->fib_out, n);
    _sock_chkerrno(s);
    return(-1);
  }

  fifobuf_drop(s->fib_out, s->fib_out->enqueued);
  return(0);
}


int sock_write(SOCK *s, void *data, unsigned int size)
{
  if (size > fifobuf_free_cur(s->fib_out))
    if (sock_flush(s) < 0) return(-1);

  if (size < fifobuf_free_max(s->fib_out))
  {
    fifobuf_enqueue(s->fib_out, data, size);
#ifdef TRIES
    _sock_spill(s);
#endif
    return(0);
  }

  return(_sock_write(data, size, s));
}


int sock_putc(SOCK *s, char c)
{
  if (!fifobuf_free_cur(s->fib_out))
    if (sock_flush(s) < 0) return(-1);

  fifobuf_enqueue(s->fib_out, &c, sizeof(char));
#ifdef TRIES
  if (s->fib_out->enqueued > 16) _sock_spill(s);
#endif
  return(0);
}


int sock_puts(SOCK *s, char *str)
{
  int len;
  
  len = strlen(str);
  if (sock_write(s, str, len) < 0) return(-1);

  return(len);
}


int sock_printf(SOCK *s, char *format, ...)
{
  int len;
  va_list args;
  char *str = 0;
  
  va_start(args, format);
  len = vasprintf(&str, format, args);
  va_end(args);
  
  if (len < 1)
  {
    if (str) free(str);
    return(-1);
  }
  
  if (sock_write(s, str, len) < 0) 
  {
    free(str);
    return(-1);
  }

  free(str);
  return(len);
}


int sock_vprintf(SOCK *s, char *format, va_list args)
{
  int len;
  char *str = 0;
  
  len = vasprintf(&str, format, args);
  
  if (len < 1)
  {
    if (str) free(str);
    return(-1);
  }
  
  if (sock_write(s, str, len) < 0) 
  {
    free(str);
    return(-1);
  }

  free(str);
  return(len);
}
