/***************************************************************************
  dbmusic.h  -  description
  -------------------
begin                : Wed Dec 19 2001
copyright            : (C) 2001 by root
email                : badhack@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef DBMUSIC_H
#define DBMUSIC_H

#include "dbmusicglobals.h"
#include <libpq++.h>
#include <qlistview.h>
#include <qtextstream.h>
#include <qfile.h>
#include <kapp.h>
#include <kiconloader.h>




/**This is the database interface class. This class provides all the music database functionality
 * through functions. The data type to be modified will be passed const x & and these functions
 * return an integer error code value.
 *
 * There are some standard int values that are used for genres/labels/etc. They are so similar the queries
 * end up nearly identical, so it's easier to pass an int and combine everything into one function.
 * Those values are:
 *@li @p 1-	Boxset
 *@li @p 2-	Format
 *@li @p 3-	Genre
 *@li @p 4-	Label
 *@li @p 5-	Method
 *@li @p 6-	Type
 *@li @p 7-	Bands To Explore (if implemented)
 *
 * Instead of using integers values, I have included some #define's to make it easier:
 *@li @p 0-	KBOXSET
 *@li @p 1-	KFORMAT
 *@li @p 2-	KGENRE
 *@li @p 3-	KLABEL
 *@li @p 4-	KMETHOD
 *@li @p 5-	KTYPE
 *@li @p 6-	KBTE
 *@author Will DeRousse
 *@short Music database query,add,delete,update and reports functionality.
 */

class dbMusic : public PgDatabase  {
	public:
		/** Constructor.
		 * The connection to the backend is made, and all the QMap caches are filled.
		 * <pre>
		 * dbMusic *music=new dbMusic("dbname");
		 * music->addArtist();
		 * delete music;
		 * </pre>
		 *
		 * Initializing a dbMusic object makes the connect. Deleting the object breaks the connect.
		 *@param string PostgreSQL style connection string
		 */
		dbMusic(const QString &string);
		/**
		 * Destructor
		 */
		~dbMusic();
		/** Adds new artist to the db.
		 *
		 * Returns error code.
		 *@param ArtistInfo Pointer to the artist info struct.
		 */
		int addArtist( const ArtistInfo *const);
		/** Adds a category to the database.
		 *
		 * Returns error code.
		 *@param int Category type.
		 *@param QString Name
		 */
		int addCategory(const int &, const QString &);
		/** Adds the title to the db.
		 *
		 * Returns error code.
		 *@param QString Artist Name
		 *@param TitleInfo Pointer to the title information.
		 */
		int addTitle(const QString &, const TitleInfo *const);
		/** Adds a track to the dataabse.
		 *
		 * Returns error code.
		 *@param QString CD ID
		 *@param TrackInfo Pointer to TrackInfo struct containing track description.
		 */
		int addTrack(const QString &, TrackInfo *const);
		/** Creates a pg_dump compatible file
		 *
		 * Returns error code.
		 *@param string name of file
		 */
		int backup(const QString &string);
		/** Checks the connection with the database to insure that it is ok.
		 *
		 * Returns error code.
		 */
		bool checkConnect( );
		/** Retrieves the currently opened database version and sets the QString passed.
		 *
		 *Returns the error code.
		 *@param QString version
		 */
		int dbVersionLive(QString &);
		/** Retrieves the database version this library was compiled for and sets the QString passed.
		 *
		 *Returns the error code.
		 *@param QString version
		 */
		int dbVersionCompat(QString &);
		/** Deletes artist from the database.
		 *
		 * Returns error code.
		 * @param QString artist name
		 */
		int deleteArtist(const QString &);
		/** Deletes a specific category entry from the database.
		 *
		 * Returns error code.
		 *@param int Category type
		 *@param QString Name
		 */
		int deleteCategory(const int &, const QString &);
		/** Deletes title (and subsequent tracks) from the db.
		 *
		 * Returns error code.
		 *@param QString CDID Index Number
		 */
		int deleteTitle(const QString &);
		/** Deletes a track from the databse.
		 *
		 * Returns error code.
		 *@param QString Track ID
		 *@param QString CD ID
		 */
		int deleteTrack(const QString &, const QString &);
		/** Fills QStringList with artist names (for a combobox) from cache.
		 *
		 * Returns error code.
		 *@param QStringList Pointer to the list
		 */
		void populateArtistQList(QStringList *const) const;
		/**
		 * "formats" the database out for use with dbMusic. It lays out all the basic tables, indexes, sequences,  and triggers for referential integrity.
		 * Also adds a few basic entries to the categories. For more details, see dbmusic.dump or kmusicdb manual.
		 *
		 * Returns error code.
		 */
		int formatDb();
		/** Fills struct with a particular artist.
		 *
		 * Returns error code.
		 *@param QString artist name
		 *@param ArtistInfo struct to be filled
		 */
		int getArtist(const QString &, ArtistInfo *const);
		/** Fills strcut with the info on a title.
		 *
		 * Returns error code.
		 *@param QString CDID
		 *@param TitleInfo struct to be filled
		 */
		int getTitle(const QString &, TitleInfo *const);
		/** Runs VACUUM (optimizer/defrag) on the database.
		 *
		 * Returns error code.
		 */
		int opti();
		/** Fills the qmap with category information.
		 *@param int Which value to populate (genre, label, etc.)
		 *@param QStringList Pointer to the list to be filled.
		 */
		void populateCategoryQList(const int &, QStringList *const);
		/** Populates the category QListView with the names of the chosen category.
		 *
		 *@param int Category type
		 *@param QListView Pointer to QListView
		 */
		void populateCategoryQListView(const int &, QListView *const);
		/** Fills up the information for the titleview.
		 *
		 * Returns error code
		 *@param QString Artist name
		 *@param QListView Pointer to QListView to be populated.
		 *@param enabledDislayedOptions Pointer to struct containing column information.
		 */
		int populateTitle(const QString &, QListView *const, const enabledTitleColumns *const);
		/** Populates a QListView with the track information of a specified CD ID.
		 *
		 * Returns error code
		 *@param QString CD ID
		 *@param QListView Pointer to QListView
		 */
		int populateTrackListView(const QString &, QListView *const);
		/** Restores the db from a PostgreSQL dump file.
		 *
		 * @param string Filename to restore
		 * Returns error code.
		 */
		int restore(const QString &string);
		/** Tests the connection with the db to insure that it is reliable.
		 *
		 * Returns error code.
		 */
		int testConnect( );
		/** Sets the QString to the total number of artists.
		 *
		 * Returns error code
		 *@param QString Pass a string to have its value set.
		 */
		int totalArtist(QString &);
		/** Sets the number of collected titles by the current artist.
		 *
		 * Returns error code
		 *@param QString Artist name
		 *@param QString This will get set to the number.
		 */
		int totalCollectedTitlesByArtist(const QString &, QString &);
		/** Sets the total length of the titles by the current artist.
		 *
		 * Returns error code
		 *@param QString Artist name
		 *@param QString This will get set to the number.
		 */
		int totalLengthByArtist(const QString &, QString &);
		/** Sets the number of titles by the current artist.
		 *
		 * Returns error code
		 *@param QString Artist name
		 *@param QString This will get set to the number.
		 */
		int totalTitlesByArtist(const QString &, QString &);
		/** Moves a track down one number for a specific title.
		 *
		 * Returns error code
		 *@param TrackInfo top track
		 *@param TrackInfo bottom track
		 */
		int trackDown(const TrackInfo *, const TrackInfo *);
		/** Moves a track up one number for a specific title.
		 *
		 * Returns error code
		 *@param TrackInfo top track
		 *@param TrackInfo bottom track
		 */
		int trackUp(const TrackInfo *, const TrackInfo *);
		/** Updates a particular artist.
		 *
		 * Returns error code
		 *@param ArtistInfo Pointer to artist information.
		 */
		int updateArtist(const ArtistInfo *const);
		/** Updates a specific category entry.
		 *
		 *@param int Category type
		 *@param QString new value
		 *@param QString old value
		 */
		int updateCategory(const int &, const QString &, const QString &);
		/** Updates the length (title.cdlen) of the title.
		 *
		 *@param QString CD ID
		 */
		int updateTime(const QString &);
		/** Updates the title.
		 *
		 * Returns error code
		 *@param TitleInfo Pointer to title information.
		 */
		int updateTitle(const TitleInfo *const);
		/** Updates a specific track.
		 *
		 *@param TrackInfo Pointer to TrackInfo struct containing track ID to update as well as the updated information.
		 */
		int updateTrack(const TrackInfo *const );
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 */
		int artistReport(QString &, const reportInfo *const, const colorInfo *const);
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 */
		int bteReport(QString &, const reportInfo *const, const colorInfo *const);
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 * @param int Report Type (Type, Label, Genre, etc)
		 * @param QString Value
		 */
		int categorySearchReport(QString &, const reportInfo *const, const colorInfo *const, const int &, const QString &);
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 */
		int collectionReport(QString &, const reportInfo *const, const colorInfo *const);
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 * @param QString Custom report string
		 */
		int customReport(QString &, const reportInfo *const, const colorInfo *const, const QString &);
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 */
		int dateRangeReport(QString &, const reportInfo *const, const colorInfo *const);
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 * @param QString Value
		 * @param report type (artist, title, track)
		 */
		int nameSearchReport(QString &, const reportInfo *const, const colorInfo *const, const QString &, const int &);
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 */
		int statisticsReport(QString &, const reportInfo *const, const colorInfo *const);
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 * @param QString AID
		 */
		int titlesByArtist(QString &, const reportInfo *const, const colorInfo *const, const QString &);
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 */
		int topCDsReport(QString &, const reportInfo *const, const colorInfo *const);
		/**
		 * Generates an HTML report.
		 * @param QString HTML String
		 * @param reportInfo reportInfo struct containing report variables
		 * @param colorInfo colorInfo struct containing color variables
		 * @param QString CDID
		 */
		int tracksByTitle(QString &, const reportInfo *const, const colorInfo *const, const QString &);
	private:
		/** Checks the result of the query or whatever and returns the error code.
		 *
		 * Returns error code.
		 */
		int execCheck() const;
		/** Fills the QMap with the name,aid information for the kcombobox in the artisttitle class.
		 *
		 * Returns error code.
		 */
		int fillArtistQMap();
		/** Fills in the qmap info. Should be done just after a connect and musicArtistTitle is filled and anytime the boxset table is altered.
		 *
		 * The int value is a const value (see @ref include.h) which tells the function which map to fill. maps are cleared first, not appended.
		 *
		 * Returns error code.
		 *@param int map
		 */
		int fillQMap(const int &);
		/** Fills the boxset qmap the the data swapped around (required in some instances). Fills from cache.
		 * The int value is a const value (see @ref include.h) which tells the function which map to fill. maps are cleared first, not appended.
		 * The corresponding qmap MUST be built first as the reverse qmap's are assembled from them (otherwise they will be empty).
		 *
		 * Returns error code.
		 *@param int map
		 */
		void fillReverseQMap(const int &);
		/**
		 * Support function for backup. Help reduce redundant code.
		 *
		 *@param QString id type
		 *@param QString table name
		 */
		int backupTable(const QString &, const QString &);
	private:
		StringHashQMap *artist, *boxset, *formatmap, *genremap, *labelmap, *methodmap;
		StringHashQMap *rboxset, *typemap, *rformatmap, *rgenremap, *rlabelmap, *rmethodmap, *rtypemap;
		ExecStatusType stat;
		int a;
		QFile *htmlrep;
		QString sqlstring;
		QTextStream *s;
};

/**
 * Generates DOCTYPE and charset information
 * @param QString string to append
 * @param QString Language
 */
void HTMLDocType(QString &, const QString &);
/**
 * Generates HEAD and META information for reports.
 * Also sets charset of report and inserts the CSS information.
 * @param QString string to append
 * @param QString report title
 * @param QString metaname
 * @param QString report version
 * @param reportInfo reportInfo struct
 * @param colorInfo colorInfo struct
 */
void HTMLHeader(QString &, const QString &, const QString &, const QString &, const reportInfo *const, const colorInfo *const);
/**
 * Generates BODY tag.
 * @param QString string to append
 */
void HTMLBody(QString &);
/**
 * Starts a new table
 * @param QString string to append
 */
void HTMLTable(QString &);
/**
 * Generates Full Report Header (including information about Date Range, Collected, etc).
 * @param QString string to append
 * @param QString number of columns
 * @param QString report name
 * @param reportInfo reportInfo struct
 */
void HTMLTableHeader(QString &, const QString &, const QString &, const reportInfo *const);
/**
 * Generates a simple table header (only colspan's and includes a title).
 * @param QString string to append
 * @param QString number of columns
 * @param QString table name
 */
void HTMLSimpleTableHeader(QString &, const QString &, const QString &);
/**
 * Generates column headers
 * @param QString string to append
 * @param QStringList color list
 */
void HTMLColumnHeaders(QString &, const QStringList *const);
/**
 * Populates the QString with table row data from the server.
 * @param QString string to append
 * @param QStringList list of column names
 * @param int > 0 splits makes to columns, other does not
 */
void HTMLPopulateData(dbMusic *, QString &, const QStringList *const, const int &);
/**
 * A function for statistics to reduce code base.
 * Populates with category information.
 * @param QString string to append
 * @param float denominator
 * @param QStringList color list
 * @param reportInfo reportInfo struct
 */
void HTMLPopulateData2(dbMusic *, QString &, const float &, const QStringList *const, const reportInfo *const);
/**
 * Fills in a sum header for the end of a report
 * @param QString string to append
 * @param QString colspan
 * @param QString name
 * @param QString value
 */
void HTMLSumHeader(QString &, const QString &, const QString &, const QString &);
/**
 * Appends the date, user, dbname information
 * @param QString string to append
 */
void HTMLSysInfo(QString &, const QString &);
/**
 * Inserts CSS information into the document. MUST be used or document formatting will not occur.
 * @param QString string to append
 * @param reportInfo reportInfo struct
 * @param colorInfo colorInfo struct
 */
void HTMLCSS(QString &, const reportInfo *const, const colorInfo *const);
/**
 * Used to add a single contraint to a query in the form of x=y.
 * Used to add the collect/follow syntax to a query
 * @param QString string to append
 * @param QString value
 * @param reportInfo reportInfo struct
 */
void queryBoolStatus(QString &, const QString &, const reportInfo *const);
/**
 * Adds matching for two fields (a.name=t.name)
 * @param QString string to append
 * @param QString first value
 * @param QString second value
 */
void queryMatch(QString &, const QString &, const QString &);
/**
 * Adds ORDER BY syntax to a query
 * @param QString string to append
 * @param QStringList order list
 * @param reportInfo reportInfo struct
 */
void queryOrderBy(QString &, const QStringList *const, const reportInfo *const);
/**
 * Adds the date range syntax to a query
 * @param QString string to append
 * @param QString begin
 * @param QString end
 * @param reportInfo reportInfo struct
 */
void queryRange(QString &, const QString &, const QString &, const reportInfo *const);

#endif

