#if   !defined(__ABSTRACTSEMAPHORE_HPP)
#define  __ABSTRACTSEMAPHORE_HPP

/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/

#if !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__SEMAPHOREEXCEPTION_HPP)
#include <SemaphoreException.hpp>
#endif

namespace   corelinux
{

   /**
   Semaphore method return enumeration
   */

   enum SemaphoreOperationStatus
   {
      /// Kernel error, errno set

      KERNELERROR=-1,

      /// Call success

      SUCCESS=0,

      /// Semaphore balked request

      BALKED,

      /// Semaphore request timed out

      TIMEDOUT,

      /// Semaphore unavailable for request

      UNAVAILABLE
   };

   DECLARE_CLASS( SemaphoreGroup );
   DECLARE_CLASS( AbstractSemaphore );

   /**
   A AbstractSemaphore supports the protocol that processes and/or threads 
   agree to follow for the purpose of controlled access to a resource. The
   resource can be anything that the developer considers to need access
   controls on such as memory, hardware, methods, computer instructions,
   and so on.
   
   Callers can elect to avoid being put into a blocked state and
   return immediately without control to the resource. Callers may also
   request that they are put into a blocked state for a specified amount
   of time. If, at the end of the specified time, the request has not
   been satisfied, it is returned with a Timeout indicator.
   
   The owner or creator of the AbstractSemaphore can elect to enforce balking
    behavior on a AbstractSemaphore. When so designated, the 
   AbstractSemaphore can turn back any request until some condition in their 
   solution space is met regardless of the callers blocking options. If a 
   caller access attempt is balked, is it returned with a Balked indicator.
   */

   class AbstractSemaphore : public Synchronized
   {

   public:


      //
      // Constructors and destructors
      //

                           /**
                           Default constructor
                           */

                           AbstractSemaphore
                              ( 
                                 SemaphoreGroupPtr, 
                                 SemaphoreIdentifierRef 
                              )
                              throw ( NullPointerException );

                           /// Virtual Destructor

      virtual              ~AbstractSemaphore( void );

      //
      // Operator overloads
      //
               /// Equality operator returns true if identifiers match

               bool     operator==( AbstractSemaphoreCref aRef ) const;

      //
      // Accessors
      //

               /// Returns true if balking enabled

      virtual  bool  isBalkingEnabled( void ) const = 0;

               /// Returns true if recursion allowed

      virtual  bool  isRecursionEnabled( void ) const = 0;

               /// Return a reference to this AbstractSemaphore identifier

               SemaphoreIdentifierCref    getIdentifier( void ) const ;

               /// Returns a reference to the SemaphoreGroup identifier

               SemaphoreGroupIdentifierCref  getGroupIdentifier( void ) const;

               /// Returns the identifier of who currently owns the semaphore

      virtual  ThreadIdentifierCref getOwningThreadIdentifier( void ) 
                                                      const = 0;

               /// Return the depth of the recursion for the owner

      virtual  CounterCref    getRecursionQueueLength( void ) const = 0;

               /// Returns the current value of the semaphore

               Int   getValue( void );

               /// Retrieves the initial value for a semaphore

               Int   getInitialValue( void );

               /// Ask if AbstractSemaphore instance is locked

      virtual  bool  isLocked( void ) = 0;


      //
      // Mutators
      //

               /// Request the lock, wait for availability

      virtual  SemaphoreOperationStatus   lockWithWait( void ) 
                     throw(SemaphoreException) = 0;

               /// Request the lock without waiting

      virtual  SemaphoreOperationStatus   lockWithNoWait( void )
                     throw(SemaphoreException) = 0;

               /// Request the AbstractSemaphore but timeout if not available

//      virtual  SemaphoreOperationStatus   lockWithTimeOut( Timer ) 
//                   throw(SemaphoreException) = 0;

               /// Release the lock

      virtual  SemaphoreOperationStatus   release( void ) 
                     throw(SemaphoreException) = 0;


   protected:

      //
      // Constructors
      //

                           AbstractSemaphore( AbstractSemaphoreCref ) 
                              throw(Assertion);

      //
      // Operator overloads
      //
               AbstractSemaphoreRef operator=( AbstractSemaphoreCref ) 
                              throw(Assertion);

      //
      // Accessors
      //

               /// Returns a reference to the AbstractSemaphore identifier

      inline   SemaphoreIdentifierRef   getId( void ) 
               {
                  return theSemIdentifier;
               }


               /// Returns a reference to the group identifier

      inline   Int   getGroupId( void ) const
               {
                  return theGroupIdentifier;
               }

               /// Returns a reference to the owning thread

      virtual  ThreadIdentifierRef  getOwnerId( void ) = 0;



      //
      // Mutators
      //
               /// Calls kernel lock mechanism

               SemaphoreOperationStatus   setLock( Int );

               /// Calls kernel unlock mechanism

               SemaphoreOperationStatus   setUnlock( Int );

               /// Calls kernel zero mechanism

               SemaphoreOperationStatus   waitZero( Int );

               /// Sets the value for the AbstractSemaphore

               SemaphoreOperationStatus   setValue( Int );


   private:

               ///   The group of instance origination

               SemaphoreGroupPtr    theGroup;

               /// The group instance identifier

               Int   theGroupIdentifier;

               /// The thread identifier

               SemaphoreIdentifier  theSemIdentifier;

   };
}

#endif // if !defined(__ABSTRACTSEMAPHORE_HPP)

/*
   Common rcs information do not modify
   $Author: dulimart $
   $Revision: 1.5 $
   $Date: 2000/09/09 06:54:53 $
   $Locker:  $
*/




