#line 2 "compress.c"
/*-
 * Copyright (c) 1985, 1986, 1992, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Diomidis Spinellis and James A. Woods, derived from original
 * work by Spencer Thomas and Joseph Orost.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* $Id: compress.c,v 1.21 2002/02/03 11:14:58 dolecek Exp $ */

/* rcsid of imported file:
 *	NetBSD: zopen.c,v 1.6 1997/09/15 10:58:39 lukem Exp
 */

/*
 * following code is almost plain copy of code as found
 * in NetBSD 1.3_BETA's compress(1) - just some hooks to use 
 * CSacek's output routines were added, alias defines removed and quite
 * a bit of code was erased as it won't be used within CSacek. Thanks again
 * for NetBSD for beeing with us.
 *
 * 	Jaromir Dolecek <dolecek@ics.muni.cz>
 */

/*-
 * fcompress.c - File compression ala IEEE Computer, June 1984.
 *
 * Compress authors:
 *		Spencer W. Thomas	(decvax!utah-cs!thomas)
 *		Jim McKie		(decvax!mcvax!jim)
 *		Steve Davies		(decvax!vax135!petsd!peora!srd)
 *		Ken Turkowski		(decvax!decwrl!turtlevax!ken)
 *		James A. Woods		(decvax!ihnp4!ames!jaw)
 *		Joe Orost		(decvax!vax135!petsd!joe)
 *
 * Cleaned up and converted to library returning I/O streams by
 * Diomidis Spinellis <dds@doc.ic.ac.uk>.
 *
 */

#include "csacek.h"

#ifdef CSA_WANT_COMPRESSION

#include <ctype.h>
#include <errno.h>

#define	BITS		16		/* Default bits. */
#define	HSIZE		69001		/* 95% occupancy */

/* A code_int must be able to hold 2**BITS values of type int, and also -1. */
typedef long code_int;
typedef long count_int;

static const unsigned char magic_header[] = {0x1f, 0x9d};	/* 1F 9D */

#define	BIT_MASK	0x1f		/* Defines for third byte of header. */
#define	BLOCK_MASK	0x80

/*
 * Masks 0x40 and 0x20 are free.  I think 0x20 should mean that there is
 * a fourth header byte (for expansion).
 */
#define	INIT_BITS 9			/* Initial number of bits/code. */

#define	MAXCODE(n_bits)	((1 << (n_bits)) - 1)

struct s_zstate {
	FILE *zs_fp;			/* File stream for I/O */
	char zs_mode;			/* r or w */
	enum {
		S_START, S_MIDDLE, S_EOF
	} zs_state;			/* State of computation */
	int zs_n_bits;			/* Number of bits/code. */
	int zs_maxbits;			/* User settable max # bits/code. */
	code_int zs_maxcode;		/* Maximum code, given n_bits. */
	code_int zs_maxmaxcode;		/* Should NEVER generate this code. */
	count_int zs_htab [HSIZE];
	u_int zs_codetab [HSIZE];
	code_int zs_hsize;		/* For dynamic table sizing. */
	code_int zs_free_ent;		/* First unused entry. */
	/*
	 * Block compression parameters -- after all codes are used up,
	 * and compression rate changes, start over.
	 */
	int zs_block_compress;
	int zs_clear_flg;
	long zs_ratio;
	count_int zs_checkpoint;
	int zs_offset;
	long zs_in_count;		/* Length of input. */
	long zs_bytes_out;		/* Length of compressed output. */
	long zs_out_count;		/* # of codes output (for debugging). */
	unsigned char zs_buf[BITS];
	union {
		struct {
			long zs_fcode;
			code_int zs_ent;
			code_int zs_hsize_reg;
			int zs_hshift;
		} w;			/* Write paramenters */
		struct {
			unsigned char *zs_stackp;
			int zs_finchar;
			code_int zs_code, zs_oldcode, zs_incode;
			int zs_roffset, zs_size;
			unsigned char zs_gbuf[BITS];
		} r;			/* Read parameters */
	} u;
};

/*
 * To save much memory, we overlay the table used by compress() with those
 * used by decompress().  The tab_prefix table is the same size and type as
 * the codetab.  The tab_suffix table needs 2**BITS characters.  We get this
 * from the beginning of htab.  The output stack uses the rest of htab, and
 * contains characters.  There is plenty of room for any possible stack
 * (stack used to be 8000 characters).
 */

#define	htabof(i)	(zs->zs_htab[i])
#define	codetabof(i)	(zs->zs_codetab[i])

#define	tab_prefixof(i)	codetabof(i)
#define	tab_suffixof(i)	((unsigned char *)(htab))[i]
#define	de_stack	((unsigned char *)&tab_suffixof(1 << BITS))

#define	CHECK_GAP 10000		/* Ratio check interval. */

/*
 * the next two codes should not be changed lightly, as they must not
 * lie within the contiguous general code space.
 */
#define	FIRST	257		/* First free entry. */
#define	CLEAR	256		/* Table clear output code. */

static int	x_cl_block __P((csa_params_t *));
static void	x_cl_hash __P((struct s_zstate *, count_int));
static int	x_output __P((csa_params_t *, code_int));
static int	x_zwrite __P((csa_params_t *, const char *, int));

/*-
 * Algorithm from "A Technique for High Performance Data Compression",
 * Terry A. Welch, IEEE Computer Vol 17, No 6 (June 1984), pp 8-19.
 *
 * Algorithm:
 * 	Modified Lempel-Ziv method (LZW).  Basically finds common
 * substrings and replaces them with a variable size code.  This is
 * deterministic, and can be done on the fly.  Thus, the decompression
 * procedure needs no input table, but tracks the way the table was built.
 */

/*-
 * compress write
 *
 * Algorithm:  use open addressing double hashing (no chaining) on the
 * prefix code / next character combination.  We do a variant of Knuth's
 * algorithm D (vol. 3, sec. 6.4) along with G. Knott's relatively-prime
 * secondary probe.  Here, the modular division first probe is gives way
 * to a faster exclusive-or manipulation.  Also do block compression with
 * an adaptive reset, whereby the code table is cleared when the compression
 * ratio decreases, but after the table fills.  The variable-length output
 * codes are re-sized at this point, and a special CLEAR code is generated
 * for the decompressor.  Late addition:  construct the table according to
 * file size for noticeable speed improvement on small files.  Please direct
 * questions about this implementation to ames!jaw.
 */
static int
x_zwrite(p, wbp, num)
	csa_params_t *p;
	const char *wbp;
	int num;
{
	void *cookie=p->c_cookie;
	code_int i;
	int c, disp;
	struct s_zstate *zs;
	const unsigned char *bp;
	unsigned char tmp;
	int count;

	if (num == 0)
		return (0);

	zs = cookie;
	count = num;
	bp = (const unsigned char *)wbp;
	if (zs->zs_state == S_MIDDLE)
		goto middle;
	zs->zs_state = S_MIDDLE;

	zs->zs_maxmaxcode = 1L << zs->zs_maxbits;
	if (csa_add_output(p, (const char *)magic_header, sizeof(magic_header),
		CSA_OUT_RAW) != 0)
		return (-1);

	tmp = (unsigned char)(zs->zs_maxbits | zs->zs_block_compress);
	if (csa_add_output(p, (char *)&tmp, sizeof(tmp), CSA_OUT_RAW) != 0)
		return (-1);

	zs->zs_offset = 0;
	zs->zs_bytes_out = 3;		/* Includes 3-byte header mojo. */
	zs->zs_out_count = 0;
	zs->zs_clear_flg = 0;
	zs->zs_ratio = 0;
	zs->zs_in_count = 1;
	zs->zs_checkpoint = CHECK_GAP;
	zs->zs_maxcode = MAXCODE(zs->zs_n_bits = INIT_BITS);
	zs->zs_free_ent = ((zs->zs_block_compress) ? FIRST : 256);

	zs->u.w.zs_ent = *bp++;
	--count;

	zs->u.w.zs_hshift = 0;
	zs->u.w.zs_fcode = (long)zs->zs_hsize;
	for (; zs->u.w.zs_fcode < 65536L; zs->u.w.zs_fcode *= 2L)
		zs->u.w.zs_hshift++;
	zs->u.w.zs_hshift = 8 - zs->u.w.zs_hshift;
		/* Set hash code range bound. */

	zs->u.w.zs_hsize_reg = zs->zs_hsize;
	x_cl_hash(zs, (count_int)zs->u.w.zs_hsize_reg);	/* Clear hash table. */

middle:	for (i = 0; count--;) {
		c = *bp++;
		zs->zs_in_count++;
		zs->u.w.zs_fcode = (long)(((long)c << zs->zs_maxbits)
				+ zs->u.w.zs_ent);
		i = ((c << zs->u.w.zs_hshift) ^ zs->u.w.zs_ent);
				/* Xor hashing. */

		if (htabof(i) == zs->u.w.zs_fcode) {
			zs->u.w.zs_ent = codetabof(i);
			continue;
		} else if ((long)htabof(i) < 0)	/* Empty slot. */
			goto nomatch;
		disp = zs->u.w.zs_hsize_reg - i;
				/* Secondary hash (after G. Knott). */
		if (i == 0)
			disp = 1;
probe:		if ((i -= disp) < 0)
			i += zs->u.w.zs_hsize_reg;

		if (htabof(i) == zs->u.w.zs_fcode) {
			zs->u.w.zs_ent = codetabof(i);
			continue;
		}
		if ((long)htabof(i) >= 0)
			goto probe;
nomatch:	if (x_output(p, (code_int) zs->u.w.zs_ent) == -1)
			return (-1);
		zs->zs_out_count++;
		zs->u.w.zs_ent = c;
		if (zs->zs_free_ent < zs->zs_maxmaxcode) {
			codetabof(i) = zs->zs_free_ent++;/* code -> hashtable */
			htabof(i) = zs->u.w.zs_fcode;
		} else if ((count_int)zs->zs_in_count >=
		    zs->zs_checkpoint && zs->zs_block_compress) {
			if (x_cl_block(p) == -1)
				return (-1);
		}
	}
	return (num);
}

/*-
 * Output the given code.
 * Inputs:
 * 	code:	A n_bits-bit integer.  If == -1, then EOF.  This assumes
 *		that n_bits =< (long)wordsize - 1.
 * Outputs:
 * 	Outputs code to the file.
 * Assumptions:
 *	Chars are 8 bits long.
 * Algorithm:
 * 	Maintain a BITS character long buffer (so that 8 codes will
 * fit in it exactly).  Use the VAX insv instruction to insert each
 * code in turn.  When the buffer fills up empty it and start over.
 */

static unsigned char lmask[9] =
	{0xff, 0xfe, 0xfc, 0xf8, 0xf0, 0xe0, 0xc0, 0x80, 0x00};
static unsigned char rmask[9] =
	{0x00, 0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xff};

static int
x_output(p, ocode)
	csa_params_t *p;
	code_int ocode;
{
	struct s_zstate *zs=(struct s_zstate *)p->c_cookie;
	unsigned int bits, r_off;
	unsigned char *bp;
	unsigned long oocode = (unsigned long) ocode;

	r_off = zs->zs_offset;
	bits = zs->zs_n_bits;
	bp = zs->zs_buf;
	if (ocode >= 0) {
		/* Get to the first byte. */
		bp += (r_off >> 3);
		r_off &= 7;
		/*
		 * Since ocode is always >= 8 bits, only need to mask the first
		 * hunk on the left.
		 */
		*bp = (*bp & rmask[r_off]) | ((ocode << r_off) & lmask[r_off]);
		bp++;
		bits -= (8 - r_off);
		oocode >>= 8 - r_off;
		/* Get any 8 bit parts in the middle (<=1 for up to 16 bits). */
		if (bits >= 8) {
			*bp++ = (unsigned char) oocode;
			oocode >>= 8;
			bits -= 8;
		}
		/* Last bits. */
		if (bits)
			*bp = (unsigned char) oocode;
		zs->zs_offset += zs->zs_n_bits;
		if (zs->zs_offset == (zs->zs_n_bits << 3)) {
			bp = zs->zs_buf;
			bits = zs->zs_n_bits;
			zs->zs_bytes_out += bits;
			if (csa_add_output(p, (char *)bp, bits,
			    CSA_OUT_RAW) != 0)
				return (-1);
			bp += bits;
			bits = 0;
			zs->zs_offset = 0;
		}
		/*
		 * If the next entry is going to be too big for the ocode size,
		 * then increase it, if possible.
		 */
		if (zs->zs_free_ent > zs->zs_maxcode || (zs->zs_clear_flg>0)) {
		       /*
			* Write the whole buffer, because the input side won't
			* discover the size increase until after it has read it.
			*/
			if (zs->zs_offset > 0) {
				if (csa_add_output(p, (char *)zs->zs_buf,
				    (size_t) zs->zs_n_bits, CSA_OUT_RAW) != 0)
					return (-1);

				zs->zs_bytes_out += zs->zs_n_bits;
			}
			zs->zs_offset = 0;

			if (zs->zs_clear_flg) {
				zs->zs_maxcode=MAXCODE(zs->zs_n_bits=INIT_BITS);
				zs->zs_clear_flg = 0;
			} else {
				zs->zs_n_bits++;
				if (zs->zs_n_bits == zs->zs_maxbits)
					zs->zs_maxcode = zs->zs_maxmaxcode;
				else
					zs->zs_maxcode = MAXCODE(zs->zs_n_bits);
			}
		}
	} else {
		/* At EOF, write the rest of the buffer. */
		if (zs->zs_offset > 0) {
			zs->zs_offset = (zs->zs_offset + 7) / 8;
			if (csa_add_output(p, (char *)zs->zs_buf,
				    (size_t) zs->zs_offset, CSA_OUT_RAW) != 0)
				return (-1);

			zs->zs_bytes_out += zs->zs_offset;
		}
		zs->zs_offset = 0;
	}
	return (0);
}

static int
x_cl_block(p)			/* Table clear for block compress. */
	csa_params_t *p;
{
	struct s_zstate *zs=(struct s_zstate *)p->c_cookie;
	long rat;

	zs->zs_checkpoint = zs->zs_in_count + CHECK_GAP;

	if (zs->zs_in_count > 0x007fffff) {	/* Shift will overflow. */
		rat = ((unsigned long) zs->zs_bytes_out) >> 8;
		if (rat == 0)		/* Don't divide by zero. */
			rat = 0x7fffffff;
		else
			rat = zs->zs_in_count / rat;
	} else
		rat = (zs->zs_in_count << 8) / zs->zs_bytes_out;
					/* 8 fractional bits. */
	if (rat > zs->zs_ratio)
		zs->zs_ratio = rat;
	else {
		zs->zs_ratio = 0;
		x_cl_hash(zs, (count_int) zs->zs_hsize);
		zs->zs_free_ent = FIRST;
		zs->zs_clear_flg = 1;
		if (x_output(p, (code_int) CLEAR) == -1)
			return (-1);
	}
	return (0);
}

static void
x_cl_hash(zs, cl_hsize)			/* Reset code table. */
	struct s_zstate *zs;
	count_int cl_hsize;
{
	count_int *htab_p;
	long i, m1;

	m1 = -1;
	htab_p = zs->zs_htab + cl_hsize;
	i = cl_hsize - 16;
	do {			/* Might use Sys V memset(3) here. */
		*(htab_p - 16) = m1;
		*(htab_p - 15) = m1;
		*(htab_p - 14) = m1;
		*(htab_p - 13) = m1;
		*(htab_p - 12) = m1;
		*(htab_p - 11) = m1;
		*(htab_p - 10) = m1;
		*(htab_p - 9) = m1;
		*(htab_p - 8) = m1;
		*(htab_p - 7) = m1;
		*(htab_p - 6) = m1;
		*(htab_p - 5) = m1;
		*(htab_p - 4) = m1;
		*(htab_p - 3) = m1;
		*(htab_p - 2) = m1;
		*(htab_p - 1) = m1;
		htab_p -= 16;
	} while ((i -= 16) >= 0);
	for (i += 16; i > 0; i--)
		*--htab_p = m1;
}

/* 
 * the content of this function has been taken from function zclose()
 */
int 
csa_init_compress(p)
  csa_params_t *p;
{
	struct s_zstate *zs;

	if ((zs = ap_pcalloc(p->pool_req, sizeof(struct s_zstate))) == NULL)
		return (-1);

	zs->zs_maxbits = BITS;		 /* User settable max # bits/code. */
	zs->zs_maxmaxcode = 1 << zs->zs_maxbits;
					/* Should NEVER generate this code. */
	zs->zs_hsize = HSIZE;		/* For dynamic table sizing. */
	zs->zs_free_ent = 0;		/* First unused entry. */
	zs->zs_block_compress = BLOCK_MASK;
	zs->zs_clear_flg = 0;
	zs->zs_ratio = 0;
	zs->zs_checkpoint = CHECK_GAP;
	zs->zs_in_count = 1;		/* Length of input. */
	zs->zs_out_count = 0;		/* # of codes output (for debugging). */
	zs->zs_state = S_START;
	zs->u.r.zs_roffset = 0;
	zs->u.r.zs_size = 0;
	zs->zs_mode = 'w';

	p->c_cookie = (void *) zs;

	return 0;
}

/*
 * just a wrapper
 */
int
csa_compress(p, wbp, num)
  csa_params_t *p;
  const char *wbp;
  int num;
{
	return x_zwrite(p, wbp, num);
}

/*
 * almost identical to zclose(), just won't call fclose()
 */
int
csa_done_compress(p)
  csa_params_t *p;
{
	struct s_zstate *zs;

	zs = (struct s_zstate *) p->c_cookie;
	if (zs->zs_mode == 'w') {	/* Put out the final code. */
		if (x_output(p, (code_int) zs->u.w.zs_ent) == -1)
			return (-1);
		zs->zs_out_count++;
		if (x_output(p, (code_int) - 1) == -1)
			return (-1);
	}
	p->c_cookie = (void *) NULL;
	return (0);
}
#endif /* CSA_WANT_COMPRESSION */
