-- File Name: motif_graphics.adb    Unit Type:  Package body

-- Unit Name: Motif_Graphics

-- By  Jim Bean;  Original version 24-Oct-96

-- Abstract: Low Level Display CRT data for HUD project

-- Environment:  Embedded Intel using Linux kernel.
--               GNU Linux Ada

-- Changes: 18-Sep-97 Hans-Frieder Vogt
--          use Default_Depth instead of 8 bits
--          2 Mar 2002 Hans-Frieder Vogt
--          use Interfaces.C instead of System.Unsigned_Types

with X_Lib;
with X_toolkit;

with Interfaces.C;

with Text_IO;
with Common;

package body Motif_Graphics is

   HPix: constant Interfaces.C.unsigned_short := High;
   WPix: constant Interfaces.C.unsigned_short := Wide;
   H_Pix: constant Integer := High;
   W_Pix: constant Integer := Wide;

   type COLOR_TABLES is array (Common.Color_Indices) of X_Lib.Pixel;

   CRT             : X_Lib.X_Image_Pointer;

   Color_Table     : Color_Tables;

----------------------------------------------------------

   procedure Make_Colors is
      type A_COLOR is record
         R: Interfaces.C.unsigned_short;
         G: Interfaces.C.unsigned_short;
         B: Interfaces.C.unsigned_short;
      end record;

      type COLOR_VALS is array (Common.COLOR_INDICES) of A_COLOR;
      Color_Val: constant COLOR_VALS :=
         (Common.Red   => (R => 16#FFFF#, G => 0,        B => 0),
          Common.Blue  => (R => 0,        G => 0,        B => 16#FFFF#),
          Common.Black => (R => 0,        G => 0,        B => 0),
          Common.White => (R => 16#FFFF#, G => 16#FFFF#, B => 16#FFFF#));

      ColorCell: X_Lib.X_Color;
   begin
      for I in Common.Red..Common.White loop
         ColorCell.red   := Color_Val (I).R;
         ColorCell.green := Color_Val (I).G;
         ColorCell.blue  := Color_Val (I).B;

         X_Lib.X_Alloc_Color (Common.Display, Common.Color_Map, ColorCell);

         Color_Table (I) := ColorCell.Pix;

      end loop;
   end Make_Colors;

----------------------------------------------------------

   procedure Make_CRT is
      PixMap    : X_Lib.Pixmap_Id;
   use X_Lib;
   begin

      Make_Colors;

      -- Make an empty 320x200 PixMap.
      Pixmap := X_Lib.X_Create_Pixmap
               (Common.Display, Common.Window, WPix, HPix,
	       X_Lib.X_Default_Depth_Of_Screen (X_Toolkit.Xt_Screen (Common.Appshell)));

      -- Convert that to an empty Image.
      CRT    := X_Lib.X_Get_Image
               (Common.Display, Pixmap, 0, 0, W_Pix, H_Pix,
                    X_Lib.X_All_Planes, X_Lib.Z_Pixmap);

      -- Fill the Image (CRT) with white pixels.
      for I in 0..H_Pix-1 loop
         for J in 0..W_Pix-1 loop
            X_Lib.X_Put_Pixel (CRT, J, I, Color_Table (Common.White));
         end loop;
      end loop;

   exception
      when others => Text_IO.Put_Line ("Other error in Make_CRT");
   end Make_CRT;

----------------------------------------------------------

   procedure Show_CRT is
   begin
      X_Lib.X_Put_Image (Common.Display, Common.Window, Common.GC, CRT,
                   0, 0, Integer (Common.Origin_X),
                         Integer (Common.Origin_Y), W_Pix, H_Pix);
   exception
      when others => Text_IO.Put_Line ("Error in Show_CRT");
   end Show_CRT;

-----------------------------------------------------------------------

   procedure Put_Pixel (X_Pos: in Integer;
                        Y_Pos: in Integer;
                        Color: in Common.Color_Indices) is
   begin
      X_Lib.X_Put_Pixel (CRT, Integer (X_Pos),
                              Integer (Y_Pos), Color_Table (Color));
   end;

----------------------------------------------------------

   procedure Draw_Line (A: in A_Point;
                        B: in A_Point;
                        C: in Common.Color_Indices) is
   begin
      if    A.Y = B.Y then  -- Horizontal line.
         for X in A.X .. B.X loop
            Put_Pixel (Integer (X), Integer (A.Y), C);
         end loop;

      elsif A.X = B.X then  -- Vertical line.
         for Y in A.Y .. B.Y loop
            Put_Pixel (Integer (A.X), Integer (Y), C);
         end loop;
      end if;
   exception
      when others => Text_IO.Put_Line ("Exception in Draw_Line");
   end Draw_Line;

----------------------------------------------------------

   procedure Draw_Box is
   begin
      Draw_Line ((0,0),      (Wide-1,0),      Common.Red);
      Draw_Line ((Wide-1,0), (Wide-1,High-1), Common.Red);
      Draw_Line ((0,High-1), (Wide-1,High-1), Common.Red);
      Draw_Line ((0,0),      (0,High-1),      Common.Red);
   end Draw_Box;

end Motif_Graphics;
