-- `Topal': GPG/Pine integration
--
-- Copyright (C) 2001  Phillip J. Brooke
--
--     This program is free software; you can redistribute it and/or modify
--     it under the terms of the GNU General Public License as published by
--     the Free Software Foundation; either version 2 of the License, or
--     (at your option) any later version.
--
--     This program is distributed in the hope that it will be useful,
--     but WITHOUT ANY WARRANTY; without even the implied warranty of
--     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--     GNU General Public License for more details.
--
--     You should have received a copy of the GNU General Public License
--     along with this program; if not, write to the Free Software
--     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

with Ada.IO_Exceptions;
with Ada.Strings.Fixed;
with Ada.Text_IO;
with Externals.Simple;
with Keys;
with Menus;              use Menus;
with Misc;
with Readline;

package body Configuration is

   procedure Parse_AKE (KE   : in  UBS;
                        Key   : out UBS;
                        Email  : out UBS) is
      KES : String := ToStr(KE);
      Sep : Natural;
   begin
      Sep := Ada.Strings.Fixed.Index(Source => KES,
                                     Pattern => ",");
      Key := ToUBS(KES(KES'First..Sep-1));
      Email := ToUBS(KES(Sep+1..KES'Last));
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Parse_AKE");
         raise;
   end Parse_AKE;

   procedure Parse_Config_Line (Line   : in  String;
                                Name   : out UBS;
                                Value  : out UBS) is
      Sep : Natural;
   begin
      Sep := Ada.Strings.Fixed.Index(Source => Line,
                                     Pattern => "=");
      Name := ToUBS(Line(Line'First..Sep-1));
      Value := ToUBS(Line(Sep+1..Line'Last));
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Parse_Config_Line");
         raise;
   end Parse_Config_Line;

   function Set_Two_Way (S : String) return Boolean is
   begin
      if S = "on" then
         return True;
      elsif S = "off" then
         return False;
      else
         raise Switch_Parse_Error;
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Set_Two_Way");
         raise;
   end Set_Two_Way;

   procedure Read_Config_File is
      use Ada.Text_IO;
      use Misc;
      CF : File_Type;
   begin
      Open(File => CF,
           Mode => In_File,
           Name => ToStr(Topal_Directory) & "/config");
  Config_Loop:
      loop
         begin
            declare
               The_Line : String := ToStr(Misc.Unbounded_Get_Line(CF));
               Name     : UBS;
               Value    : UBS;
            begin
               Parse_Config_Line(The_Line, Name, Value);
               declare
                  N : String := ToStr(Name);
                  V : UBS renames Value;
               begin
                  if N = "my-key" then
                     Config.my_key := V;
                  -- Binary command options.
                  elsif N = "cat-binary" then
                     null; -- Silently ignored.
                  elsif N = "chmod-binary" then
                     Config.chmod_binary := V;
                  elsif N = "clear-binary" then
                     Config.clear_binary := V;
                  elsif N = "date-binary" then
                     Config.date_binary := V;
                  elsif N = "diff-binary" then
                     Config.diff_binary := V;
                  elsif N = "dos2unix-binary" then
                     Config.Dos2Unix_binary := V;
                  elsif N = "echo-binary" then
                     null; -- Silently ignored.
                  elsif N = "formail-binary" then
                     Config.formail_binary := V;
                  elsif N = "gpg-binary" then
                     Config.gpg_binary := V;
                  elsif N = "grep-binary" then
                     Config.grep_binary := V;
                  elsif N = "less-binary" then
                     Config.less_binary := V;
                  elsif N = "md5sum-binary" then
                     Config.md5sum_binary := V;
                  elsif N = "metamail-binary" then
                     Config.metamail_binary := V;
                  elsif N = "mimeconstruct-binary" then
                     Config.mimeconstruct_binary := V;
                  elsif N = "mkdir-binary" then
                     Config.mkdir_binary := V;
                  elsif N = "mv-binary" then
                     Config.mv_binary := V;
                  elsif N = "rm-binary" then
                     Config.rm_binary := V;
                  elsif N = "sed-binary" then
                     Config.sed_binary := V;
                  elsif N = "stty-binary" then
                     Config.stty_binary := V;
                  elsif N = "tee-binary" then
                     Config.tee_binary := V;
                  elsif N = "test-binary" then
                     Config.test_binary := V;
                  elsif N = "general-options" then
                     Config.general_options := V;
                  elsif N = "gpg-options" then
                     Config.gpg_options := V;
                  elsif N = "receiving-options" then
                     Config.receiving_options := V;
                  elsif N = "sending-options" then
                     Config.sending_options := V;
                  elsif N = "decrypt-not-cached" then
                     Config.decrypt_not_cached :=  String_To_Integer(V);
                  elsif N = "decrypt-not-cached-use-cache" then
                     Config.decrypt_not_cached_use_cache :=  String_To_Integer(V);
                  elsif N = "decrypt-cached" then
                     Config.decrypt_cached := String_To_Integer(V);
                  elsif N = "decrypt-cached-use-cache" then
                     Config.decrypt_cached_use_cache := String_To_Integer(V);
                  elsif N = "verify-not-cached" then
                     Config.verify_not_cached := String_To_Integer(V);
                  elsif N = "verify-not-cached-use-cache" then
                     Config.verify_not_cached_use_cache := String_To_Integer(V);
                  elsif N = "verify-cached" then
                     Config.verify_cached := String_To_Integer(V);
                  elsif N = "verify-cached-use-cache" then
                     Config.verify_cached_use_cache := String_To_Integer(V);
                  elsif N = "decrypt-cached-fast-continue" then
                     Config.Decrypt_Cached_Fast_Continue := Set_Two_Way(ToStr(V));
                  elsif N = "verify-cached-fast-continue" then
                     Config.Verify_Cached_Fast_Continue := Set_Two_Way(ToStr(V));
                  elsif N = "verify-not-cached-fast-continue" then
                    Config.Verify_Not_Cached_Fast_Continue := Set_Two_Way(ToStr(V));
                  elsif N = "inline-separate-output" then
                     Config.Inline_Separate_Output := Set_Two_Way(ToStr(V));
                  elsif N = "fe-simple" then
                     Config.FE_Simple := Set_Two_Way(ToStr(V));
                  elsif N = "debug" then
                     Config.Debug := Set_Two_Way(ToStr(V));
                  elsif N = "ake" then
                     -- Break the up the key,email pair.
                     declare
                        K : UBS;
                        E : UBS;
                     begin
                        Parse_AKE(V, K, E);
                        Config.AKE_Count := Config.AKE_Count + 1;
                        UAP.Set(Config.AKE_Key, Config.AKE_Count, K);
                        UAP.Set(Config.AKE_Email, Config.AKE_Count, E);
                     end;
                  elsif N = "xk" then
                     -- This is a key to be excluded.
                     Config.XK_Count := Config.XK_Count + 1;
                     UAP.Set(Config.XK_Key, Config.XK_Count, V);
                  else
                     Put_Line("Bogus line: " & The_Line);
                     raise Config_Parse_Error;
                  end if;
               end;
            end;
         exception
            when Ada.IO_Exceptions.End_Error =>
               exit Config_Loop;
         end;
      end loop Config_Loop;
      Close(CF);
   exception
      when Ada.IO_Exceptions.Name_Error =>
         -- Silently ignore the lack of a config file.
         Close(CF);
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Read_Config_File");
         raise;
   end Read_Config_File;

   procedure Dump (Overwrite_Config : in Boolean := False) is
      use Ada.Text_IO;
      use Misc;
      F : File_Type;
   begin
      if Overwrite_Config then
         if Externals.Simple.Test_F(ToStr(Topal_Directory) & "/config") then
            Externals.Simple.Mv_F(ToStr(Topal_Directory) & "/config",
                                  ToStr(Topal_Directory) & "/config.bak");
         end if;
         Create(File => F,
                Mode => Out_File,
                Name => ToStr(Topal_Directory) & "/config");
         Set_Output(F);
      end if;
      Put_Line("my-key=" & ToStr(Config.My_Key));
      Put_Line("chmod-binary=" & ToStr(Config.Chmod_Binary));
      Put_Line("clear-binary=" & ToStr(Config.Clear_Binary));
      Put_Line("date-binary=" & ToStr(Config.Date_Binary));
      Put_Line("diff-binary=" & ToStr(Config.Diff_Binary));
      Put_Line("dos2unix-binary=" & ToStr(Config.Dos2Unix_Binary));
      Put_Line("formail-binary=" & ToStr(Config.Formail_Binary));
      Put_Line("gpg-binary=" & ToStr(Config.Gpg_Binary));
      Put_Line("grep-binary=" & ToStr(Config.Grep_Binary));
      Put_Line("less-binary=" & ToStr(Config.Less_Binary));
      Put_Line("md5sum-binary=" & ToStr(Config.Md5sum_Binary));
      Put_Line("metamail-binary=" & ToStr(Config.Metamail_Binary));
      Put_Line("mimeconstruct-binary=" & ToStr(Config.Mimeconstruct_Binary));
      Put_Line("mkdir-binary=" & ToStr(Config.Mkdir_Binary));
      Put_Line("mv-binary=" & ToStr(Config.Mv_Binary));
      Put_Line("rm-binary=" & ToStr(Config.Rm_Binary));
      Put_Line("sed-binary=" & ToStr(Config.Sed_Binary));
      Put_Line("stty-binary=" & ToStr(Config.Stty_Binary));
      Put_Line("tee-binary=" & ToStr(Config.Tee_Binary));
      Put_Line("test-binary=" & ToStr(Config.Test_Binary));
      Put_Line("general-options=" & ToStr(Config.General_Options));
      Put_Line("gpg-options=" & ToStr(Config.Gpg_Options));
      Put_Line("receiving-options=" & ToStr(Config.Receiving_Options));
      Put_Line("sending-options=" & ToStr(Config.Sending_Options));
      Put_Line("decrypt-not-cached="
               & Trim_Leading_Spaces(Integer'Image(Config.Decrypt_Not_Cached)));
      Put_Line("decrypt-not-cached-use-cache="
               & Trim_Leading_Spaces(Integer'Image(Config.Decrypt_Not_Cached_Use_Cache)));
      Put_Line("decrypt-cached="
               & Trim_Leading_Spaces(Integer'Image(Config.Decrypt_Cached)));
      Put_Line("decrypt-cached-use-cache="
               & Trim_Leading_Spaces(Integer'Image(Config.Decrypt_Cached_Use_Cache)));
      Put_Line("verify-not-cached="
               & Trim_Leading_Spaces(Integer'Image(Config.Verify_Not_Cached)));
      Put_Line("verify-not-cached-use-cache="
               & Trim_Leading_Spaces(Integer'Image(Config.Verify_Not_Cached_Use_Cache)));
      Put_Line("verify-cached="
               & Trim_Leading_Spaces(Integer'Image(Config.Verify_Cached)));
      Put_Line("verify-cached-use-cache="
               & Trim_Leading_Spaces(Integer'Image(Config.Verify_Cached_Use_Cache)));
      Put_Line("decrypt-cached-fast-continue="
               & ToStr(Two_Way(Config.Decrypt_Cached_Fast_Continue)));
      Put_Line("verify-cached-fast-continue="
               & ToStr(Two_Way(Config.Verify_Cached_Fast_Continue)));
      Put_Line("verify-not-cached-fast-continue="
               & ToStr(Two_Way(Config.Verify_Not_Cached_Fast_Continue)));
      Put_Line("inline-separate-output=" & ToStr(Two_Way(Config.Inline_Separate_Output)));
      Put_Line("fe-simple=" & ToStr(Two_Way(Config.FE_Simple)));
      Put_Line("debug=" & ToStr(Two_Way(Config.Debug)));
      for I in 1 .. Config.AKE_Count loop
         Put_Line("ake="
                  & ToStr(UAP.Value(Config.AKE_Key, I))
                  & ","
                  & ToStr(UAP.Value(Config.AKE_Email, I)));
      end loop;
      for I in 1 .. Config.XK_Count loop
         Put_Line("xk="
                  & ToStr(UAP.Value(Config.XK_Key, I)));
      end loop;
      if Overwrite_Config then
         Set_Output(Standard_Output);
         Close(F);
      end if;
   exception
      when others =>
         Set_Output(Standard_Output);
         Put_Line("Problem in Configuration.Dump, probably file related");
         Close(F);
         raise;
   end Dump;

   procedure Edit_Configuration is
      use Ada.Text_IO;
      use Misc;
      Default_Config : Config_Record;
      Three_Str : constant array (Integer range 1..3) of UBS
        := (1 => ToUBS("Always"),
            2 => ToUBS("Ask"),
            3 => ToUBS("Never"));
      Five_Str : constant array (Integer range 1..5) of UBS
        := (1 => ToUBS("Use"),
            2 => ToUBS("Replace"),
            3 => ToUBS("Ask-replace"),
            4 => ToUBS("Ignore cache"),
            5 => ToUBS("Offer all"));
      Three_Values : constant array (Three_Way_Index) of Integer
        := (Always => 1,
            Ask    => 2,
            Never  => 3);
      Five_Values : constant array (Five_Way_Index) of Integer
        := (UUse        => 1,
            Replace     => 2,
            AskReplace  => 3,
            IgnoreCache => 4,
            OfferAll    => 5);
      Selection   : Edit_Index;
      Selection2O : Options_Index;
      Selection2S : Settings_Index;
   begin
      Default_Configuration(Default_Config);
  Edit_Loop:
      loop
         Put_Line("Configuration:             Debugging currently "
                  & ToStr(Two_Way(Config.Debug)));
         Put_Line("                           Inline separate output currently "
                  & ToStr(Two_Way(Config.Inline_Separate_Output)));
         Put_Line("Configuration is not saved beyond this session unless explicitly saved");
         Selection := Edit_Menu;
         case Selection is
            when Quit => -- Quit editing.
               New_Line(3);
               exit Edit_Loop;
            when Save => -- Save changes.
               Dump(Overwrite_Config => True);
               Put_Line("Configuration saved");
               New_Line(3);
            when Binary_Paths => -- Binary paths.
               New_Line(3);
               Put_Line("Currently not implemented.  Use topal -dump, then edit the configuration file.");
               New_Line(3);
            when GPG_Options => -- GPG options.
               Options_Loop:
                 loop
                    Put_Line("Options submenu (all relevant options are concatenated together):");
                    Put_Line("  GPG options (all GPG operations): "
                             & NL & "    " & ToStr(Config.GPG_Options));
                    Put_Line("  General options (send & receive): "
                             & NL & "    " & ToStr(Config.General_Options));
                    Put_Line("  Receiving options (receive only): "
                             & NL & "    " & ToStr(Config.Receiving_Options));
                    Put_Line("  Sending options (send only): "
                             & NL & "    " & ToStr(Config.Sending_Options));
                    Selection2O := Options_Menu;
                    case Selection2O is
                       when Quit =>
                          New_Line(3);
                          exit Options_Loop;
                       when Change_General =>
                          Readline.Add_History(ToStr(Config.General_Options));
                          Config.General_Options := ToUBS(Readline.Get_String("General options: "));
                       when Change_GPG =>
                          Readline.Add_History(ToStr(Config.GPG_Options));
                          Config.GPG_Options := ToUBS(Readline.Get_String("GPG options: "));
                       when Change_Receiving =>
                          Readline.Add_History(ToStr(Config.Receiving_Options));
                          Config.Receiving_Options := ToUBS(Readline.Get_String("Receiving options: "));
                       when Change_Sending =>
                          Readline.Add_History(ToStr(Config.Sending_Options));
                          Config.Sending_Options := ToUBS(Readline.Get_String("Sending options: "));
                       when Default_General =>
                          Config.General_Options := Default_Config.General_Options;
                       when Default_GPG =>
                          Config.GPG_Options := Default_Config.GPG_Options;
                       when Default_Receiving =>
                          Config.Receiving_Options := Default_Config.Receiving_Options;
                       when Default_Sending =>
                          Config.Sending_Options := Default_Config.Sending_Options;
                       when Default_All =>
                          Config.General_Options := Default_Config.General_Options;
                          Config.GPG_Options := Default_Config.GPG_Options;
                          Config.Receiving_Options := Default_Config.Receiving_Options;
                          Config.Sending_Options := Default_Config.Sending_Options;
                    end case;
                 end loop Options_Loop;
            when DV_Settings => -- Decrypt/verify settings.
               Settings_Loop:
                 loop
                    Put_Line("Decrypt/Verify settings submenu:");
                    Put_Line("  Decrypt/cached:                  " & ToStr(Three_Str(Config.Decrypt_Cached)));
                    Put_Line("  Decrypt/cached, cache usage:     " & ToStr(Five_Str(Config.Decrypt_Cached_Use_Cache)));
                    Put_Line("  Decrypt/not cached:              " & ToStr(Three_Str(Config.Decrypt_Not_Cached)));
                    Put_Line("  Decrypt/not cached, cache usage: " & ToStr(Three_Str(Config.Decrypt_Not_Cached_Use_Cache)));
                    Put_Line("  Verify/cached:                   " & ToStr(Three_Str(Config.Verify_Cached)));
                    Put_Line("  Verify/cached, cache usage:      " & ToStr(Five_Str(Config.Verify_Cached_Use_Cache)));
                    Put_Line("  Verify/not cached:               " & ToStr(Three_Str(Config.Verify_Not_Cached)));
                    Put_Line("  Verify/not cached, cache usage:  " & ToStr(Three_Str(Config.Verify_Not_Cached_Use_Cache)));
                    Put_Line("  Decrypt/cached fast continue:    " & ToStr(Two_Way(Config.Decrypt_Cached_Fast_Continue)));
                    Put_Line("  Verify/cached fast continue:     " & ToStr(Two_Way(Config.Verify_Cached_Fast_Continue)));
                    Put_Line("  Verify/not cached fast continue: " & ToStr(Two_Way(Config.Verify_Not_Cached_Fast_Continue)));
                    Selection2S := Settings_Menu;
                    case Selection2S is
                       when Decrypt_Cached =>
                          Config.Decrypt_Cached := Three_Values(Three_Way_Menu);
                       when Decrypt_Cached_Use =>
                          Config.Decrypt_Cached_Use_Cache := Five_Values(Five_Way_Menu);
                       when Decrypt_Not_Cached =>
                          Config.Decrypt_Not_Cached := Three_Values(Three_Way_Menu);
                       when Decrypt_Not_Cached_Use =>
                          Config.Decrypt_Not_Cached_Use_Cache := Three_Values(Three_Way_Menu);
                       when Verify_Cached =>
                          Config.Verify_Cached := Three_Values(Three_Way_Menu);
                       when Verify_Cached_Use =>
                          Config.Verify_Cached_Use_Cache := Five_Values(Five_Way_Menu);
                       when Verify_Not_Cached =>
                          Config.Verify_Not_Cached := Three_Values(Three_Way_Menu);
                       when Verify_Not_Cached_Use =>
                          Config.Verify_Not_Cached_Use_Cache := Three_Values(Three_Way_Menu);
                       when Quit =>
                          New_Line(3);
                          exit Settings_Loop;
                       when All_Defaults =>
                          Config.Decrypt_Cached := Default_Config.Decrypt_Cached;
                          Config.Decrypt_Cached_Use_Cache := Default_Config.Decrypt_Cached_Use_Cache;
                          Config.Decrypt_Not_Cached := Default_Config.Decrypt_Not_Cached;
                          Config.Decrypt_Not_Cached_Use_Cache := Default_Config.Decrypt_Not_Cached_Use_Cache;
                          Config.Verify_Cached := Default_Config.Verify_Cached;
                          Config.Verify_Cached_Use_Cache := Default_Config.Verify_Cached_Use_Cache;
                          Config.Verify_Not_Cached := Default_Config.Verify_Not_Cached;
                          Config.Verify_Not_Cached_Use_Cache := Default_Config.Verify_Not_Cached_Use_Cache;
                       when Toggle_Decrypt_Cached_Fast =>
                          Config.Decrypt_Cached_Fast_Continue := not Config.Decrypt_Cached_Fast_Continue;
                       when Toggle_Verify_Cached_Fast =>
                          Config.Verify_Cached_Fast_Continue := not Config.Verify_Cached_Fast_Continue;
                       when Toggle_Verify_Not_Cached_Fast =>
                          Config.Verify_Not_Cached_Fast_Continue := not Config.Verify_Not_Cached_Fast_Continue;
                    end case;
                 end loop Settings_Loop;
            when Own_Key => -- Own key.
               New_Line(2);
               Put_Line("Signing key is currently: "
                        & ToStr(Config.My_Key));
               if YN_Menu = Yes then
                  -- Now, find all secret keys and offer them as a `select 1' list.
                  declare
                     SKL            : Keys.Key_List;
                     Pattern        : UBS;
                     New_Secret_Key : UBS;
                     Use_Old_Key    : Boolean;
                  begin
                     Keys.Empty_Keylist(SKL);
                     Pattern := ToUBS(Readline.Get_String("Type GPG search pattern: "));
                     Keys.Add_Secret_Keys(Pattern, SKL);
                     Keys.Select_Key_From_List(SKL, New_Secret_Key, Use_Old_Key);
                     if not Use_Old_Key then
                        Config.My_Key := New_Secret_Key;
                     end if;
                  end;
               end if;
            when Key_Assoc => -- Key associations.
               New_Line(3);
               Put_Line("Currently not implemented.  Use topal -dump, then edit the configuration file.");
               New_Line(3);
            when Key_Excl => -- Key exclusions.
               New_Line(3);
               Put_Line("Currently not implemented.  Use topal -dump, then edit the configuration file.");
               New_Line(3);
            when Set_Defaults => -- Set all defaults.
               Copy_Configuration(Config, Default_Config);
               Put_Line("All configuration settings are now default");
               New_Line(3);
            when Toggle_Debug => -- Toggle debugging.
               New_Line(3);
               Config.Debug := not Config.Debug;
            when Toggle_ISO => -- Toggle inline separate output.
               New_Line(3);
               Config.Inline_Separate_Output := not Config.Inline_Separate_Output;
            when Reset => -- Restore to current saved file.
               New_Line(3);
               Copy_Configuration(Config, Default_Config);
               Read_Config_File;
               Put_Line("Read saved configuration");
               New_Line(3);
         end case;
      end loop Edit_Loop;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Edit_Configuration");
         raise;
   end Edit_Configuration;

   procedure Default_Configuration (C : in out Config_Record) is
   begin
      C.My_Key := ToUBS("");
      C.Chmod_Binary := ToUBS("/bin/chmod");
      C.Clear_Binary := ToUBS("/usr/bin/clear");
      C.Date_Binary := ToUBS("/bin/date");
      C.Diff_Binary := ToUBS("/usr/bin/diff");
      C.Dos2Unix_Binary := ToUBS("/usr/bin/dos2unix");
      C.Formail_Binary := ToUBS("/usr/bin/formail");
      C.Gpg_Binary := ToUBS("/usr/bin/gpg");
      C.Grep_Binary := ToUBS("/bin/grep");
      C.Less_Binary := ToUBS("/usr/bin/less");
      C.Md5sum_Binary := ToUBS("/usr/bin/md5sum");
      C.Metamail_Binary := ToUBS("/usr/bin/metamail");
      C.Mimeconstruct_Binary := ToUBS("/usr/bin/mime-construct");
      C.Mkdir_Binary := ToUBS("/bin/mkdir");
      C.Mv_Binary := ToUBS("/bin/mv");
      C.Rm_Binary := ToUBS("/bin/rm");
      C.Sed_Binary := ToUBS("/bin/sed");
      C.Stty_Binary := ToUBS("/bin/stty");
      C.Tee_Binary := ToUBS("/usr/bin/tee");
      C.Test_Binary := ToUBS("/usr/bin/test");
      C.General_Options := ToUBS("");
      C.Gpg_Options := ToUBS("--no-options");
      C.Receiving_Options := ToUBS("--keyserver=wwwkeys.uk.pgp.net");
      C.Sending_Options := ToUBS("--force-v3-sigs --compress-algo 1 --cipher-algo cast5");
      C.Decrypt_Not_Cached := 2;
      C.Decrypt_Not_Cached_Use_Cache := 1;
      C.Decrypt_Cached := 2;
      C.Decrypt_Cached_Use_Cache := 1;
      C.Verify_Not_Cached := 1;
      C.Verify_Not_Cached_Use_Cache := 1;
      C.Verify_Cached := 1;
      C.Verify_Cached_Use_Cache := 1;
      C.Decrypt_Cached_Fast_Continue := False;
      C.Verify_Cached_Fast_Continue := True;
      C.Verify_Not_Cached_Fast_Continue := False;
      C.Inline_Separate_Output := False;
      C.FE_Simple := False;
      C.No_Clean := False;
      C.Debug := False;
      UAP.Create(C.AKE_Key, Initial_KL);
      UAP.Create(C.AKE_Email, Initial_KL);
      C.AKE_Count := 0;
      UAP.Create(C.XK_Key, Initial_KL);
      C.XK_Count := 0;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Default_Configuration");
         raise;
   end Default_Configuration;

   procedure Copy_Configuration(Left  : in out Config_Record;
                                Right : in     Config_Record) is
   begin
      Left.My_Key := Right.My_Key;
      Left.Chmod_Binary := Right.Chmod_Binary;
      Left.Clear_Binary := Right.Clear_Binary;
      Left.Date_Binary := Right.Date_Binary;
      Left.Diff_Binary := Right.Diff_Binary;
      Left.Dos2Unix_Binary := Right.Dos2Unix_Binary;
      Left.Formail_Binary := Right.Formail_Binary;
      Left.Gpg_Binary := Right.Gpg_Binary;
      Left.Grep_Binary := Right.Grep_Binary;
      Left.Less_Binary := Right.Less_Binary;
      Left.Md5sum_Binary := Right.Md5sum_Binary;
      Left.Metamail_Binary := Right.Metamail_Binary;
      Left.Mimeconstruct_Binary := Right.Mimeconstruct_Binary;
      Left.Mkdir_Binary := Right.Mkdir_Binary;
      Left.Mv_Binary := Right.Mv_Binary;
      Left.Rm_Binary := Right.Rm_Binary;
      Left.Sed_Binary := Right.Sed_Binary;
      Left.Stty_Binary := Right.Stty_Binary;
      Left.Tee_Binary := Right.Tee_Binary;
      Left.Test_Binary := Right.Test_Binary;
      Left.General_Options := Right.General_Options;
      Left.Gpg_Options := Right.Gpg_Options;
      Left.Receiving_Options := Right.Receiving_Options;
      Left.Sending_Options := Right.Sending_Options;
      Left.Decrypt_Not_Cached := Right.Decrypt_Not_Cached;
      Left.Decrypt_Not_Cached_Use_Cache := Right.Decrypt_Not_Cached_Use_Cache;
      Left.Decrypt_Cached := Right.Decrypt_Cached;
      Left.Decrypt_Cached_Use_Cache := Right.Decrypt_Cached_Use_Cache;
      Left.Verify_Not_Cached := Right.Verify_Not_Cached;
      Left.Verify_Not_Cached_Use_Cache := Right.Verify_Not_Cached_Use_Cache;
      Left.Verify_Cached := Right.Verify_Cached;
      Left.Verify_Cached_Use_Cache := Right.Verify_Cached_Use_Cache;
      Left.Decrypt_Cached_Fast_Continue := Right.Decrypt_Cached_Fast_Continue;
      Left.Verify_Cached_Fast_Continue := Right.Verify_Cached_Fast_Continue;
      Left.Verify_Not_Cached_Fast_Continue := Right.Verify_Not_Cached_Fast_Continue;
      Left.Inline_Separate_Output := Right.Inline_Separate_Output;
      Left.FE_Simple := Right.FE_Simple;
      Left.No_Clean := Right.No_Clean;
      Left.Debug := Right.Debug;
      UAP.Copy(Left.AKE_Key, Right.AKE_Key);
      UAP.Copy(Left.AKE_Email, Right.AKE_Email);
      Left.AKE_Count := Right.AKE_Count;
      UAP.Copy(Left.XK_Key, Right.XK_Key);
      Left.XK_Count := Right.XK_Count;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Configuration.Copy_Configuration");
         raise;
   end Copy_Configuration;

end Configuration;
