/*
 * snes9express
 *  prefs.cpp
 *  (C) 1998-1999, David Nordlund
 *  Licensed under the "Artistic" license
 *  To view the details of the Artistic license, see the file ./ARTISTIC,
 *  or go to http://www.opensource.org/artistic-license.html
 */

#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "interface.h"
#include "misc.h"
#include "pix/bnr.xpm"
#include "pix/ball.xpm"

/** Set up the preferences GUI */
s9x_Prefs::s9x_Prefs(fr_Element*parent):
fr_Window(parent, PROG " Preferences"),
Banner(this, PROG, bnr_xpm),
Notebook(this),
PgAppear(this, "Appearance"),
PgPaths(this, "Paths"),
PgReset(this, "Reset"),
PgPower(this, "Power"),
Appearance(this, "Appearance"),
VideoDriver(this, "Video Driver", false),
GUI(&Appearance, S9X_PLAIN_GUI),
SNES(&Appearance, "SNES"),
UseX11(&VideoDriver, "Runs X11 snes9x"),
UseGlide(&VideoDriver, "Runs Glide snes9x"),
Start2Prof(this, "Default Profile", true),
Start2Last(this, "Last run options", false),
Reset2Prof(this, "Default Profile", true),
Reset2Last(this, "Last run options", false),
Reset2Args(this, "Command-line", false),
Snes9xVersion(this, "", SNES9X_VERSION, 1.00, SNES9X_VERSION, 2),
ROMdir(this, "Default ROM dir", S9X_ROMENV, S9X_ROMDIR, true),
X11Executable(this, "snes9x X11 executable", "SNES9X_EXEC", "snes9x"),
GlideExecutable(this, "snes9x Glide executable", "SNES9X_GEXEC", "gsnes9x"),
BtnApply(this, "Apply"),
BtnSave(this, "Save"),
BtnClose(this, "Close"),
BtnOK(this, "Ok"),
NoFork(this, "Closes " PROG, false),
NoESD(this, "Turns off ESD", false),
PrefsFile("prefs")
{
   fr_ButtonBox BtnBox(this);

   SetIcon("Preferences", ball_xpm);
   SetGridSize(1, 4, false);
   SetPadding(3, 3);
   Pad();
   SetStretch(Fill, Shrink);
   AddListener(this);

   Pack(Banner);
	
   Notebook.SetPadding(3, 3);
   Pack(Notebook);
   
   CreatePgAppear();
   CreatePgPaths();
   CreatePgReset();
   CreatePgPower();   
	
   Notebook.AddPage(PgAppear);
   Notebook.AddPage(PgPaths);
   Notebook.AddPage(PgReset);
   Notebook.AddPage(PgPower);
	
   BtnApply.AddListener(this);
   BtnSave.AddListener(this);
   BtnClose.AddListener(this);
   BtnOK.AddListener(this);
   BtnApply.SetTooltip("Apply these settings (does not Save)");
   BtnSave.SetTooltip("Save these settings (does not Apply)");
   BtnClose.SetTooltip("Close this window (does not Apply or Save)");
   BtnOK.SetTooltip("Apply, Save, & Close");
   BtnBox.AddButton(BtnApply);
   BtnBox.AddButton(BtnSave);
   BtnBox.AddButton(BtnClose);
   BtnBox.AddButton(BtnOK, true);
   SetStretch(Normal, Normal);
   Pack(BtnBox);
   BtnOK.GrabFocus();
   
   SetToDefaults();
   Load();
}

void s9x_Prefs::CreatePgAppear() {
   PgAppear.SetGridSize(1, 2, false);

   SNES.Args << fr_CaseInsensitive << "--gui=snes";
   SNES.SetTooltip("Make the GUI resemble the North American SNES console");
   GUI.Args << fr_CaseInsensitive << "--gui" << "--gui=gtk" << "--gui=gnome";
   GUI.SetTooltip("Use an ordinary appearance");
   PgAppear.Pack(Appearance);   

   fr_Box Box(this);
   Box.SetName("snes9x version");
   Box.SetGridSize(3, 1, false);
   fr_Label lbl(this, "snes9x version:");
   Box.Pack(lbl);

   Snes9xVersion.SetName("snes9x version:");
   Snes9xVersion.Args << fr_CaseInsensitive << "--snes9xversion";
   Snes9xVersion.SetTooltip("Set the version of snes9x you are using. "
			    "The appropraite options will be available "
			    "in the interface.");
   Box.Pack(Snes9xVersion);
   Box.Frame();
   PgAppear.Pack(Box);
}

void s9x_Prefs::CreatePgPaths() {
   char *Home, *DefSnapDir = 0;
   PgPaths.SetGridSize(1, 2, false);
   ROMdir.Args << fr_CaseInsensitive << "--romdir";
   ROMdir.AddLabel("ROMs:");
   ROMdir.SetTooltip("Enter the path of the default directory for ROMs");
   if((Home = fr_HomeDir()))
     DefSnapDir = new char[strlen(Home)+strlen(S9X_SNAPSHOTDIR)+8];
   if(DefSnapDir)
     sprintf(DefSnapDir, "%s/%s", Home, S9X_SNAPSHOTDIR);
   SnapDir = new fr_File(this, "Default Snapshot dir", S9X_SNAPSHOTENV,
			 DefSnapDir, true),
   SnapDir->Args << fr_CaseInsensitive << "--snapshotdir";
   SnapDir->AddLabel("Snapshots:");
   SnapDir->SetTooltip("Enter the path of the default directory for Snapshots");
   if(DefSnapDir) delete DefSnapDir;
   
   fr_Box Box1(this);
   Box1.SetName("Default Directories");
   Box1.SetGridSize(1, 2, true);
   Box1.Frame();
   Box1.Pack(ROMdir);
   Box1.Pack(*SnapDir);
   PgPaths.Pack(Box1);
	
   X11Executable.AddLabel("X11:");
   X11Executable.Args << fr_CaseInsensitive << "--snes9x" << "--exec";
   X11Executable.SetTooltip("Enter the path of the snes9x X11 executable file");

   GlideExecutable.AddLabel("Glide:");
   GlideExecutable.Args << fr_CaseInsensitive << "--gsnes9x";
   GlideExecutable.SetTooltip("Enter the path of the snes9x Glide executable file (for 3dfx graphics cards)");
	
   fr_Box Box2(this);
   Box2.SetName("Executables");
   Box2.SetGridSize(1, 2, true);
   Box2.Frame();
   Box2.Pack(X11Executable);
   Box2.Pack(GlideExecutable);
   PgPaths.Pack(Box2);	
}

void s9x_Prefs::CreatePgReset() {
   PgReset.SetGridSize(1, 2, true);
   
   fr_Box Box1(this);
   Box1.SetName("Reset Button loads ...");
   Box1.SetGridSize(2, 1, true);

   Reset2Prof.Args << fr_CaseInsensitive << "--reset+profile";
   Reset2Prof.NotArgs << "--reset-profile";
   Reset2Prof.SetTooltip("Load the Default Profile");
   Box1.Pack(Reset2Prof);
   
   Reset2Last.Args << fr_CaseInsensitive << "--reset+last";
   Reset2Last.SetTooltip("Set the options to those that were used"
			 " the last time you ran snes9x");
   Box1.Pack(Reset2Last);
   Reset2Args.Args << fr_CaseInsensitive << "--reset+args";
   Reset2Args.SetTooltip("Re-read the " PROG " command line options\n (if any"
			 " were issued.  This is always done at start up)");
   Box1.Pack(Reset2Args);
   Box1.Frame();
   
   fr_Box Box2(this);
   Box2.SetName("Start-Up loads ...");
   Box2.SetGridSize(2, 1, true);

   Start2Prof.Args << fr_CaseInsensitive << "--start+profile";
   Start2Prof.NotArgs << "--start-profile";
   Start2Prof.SetTooltip("Load the Default Profile");
   Box2.Pack(Start2Prof);
   
   Start2Last.Args << fr_CaseInsensitive << "--start+last";
   Start2Last.SetTooltip("Set the options to those that were used"
			 " the last time you ran snes9x");
   Box2.Pack(Start2Last);
   Box2.Frame();
   
   PgReset.Pack(Box1);
   PgReset.Pack(Box2);
}

void s9x_Prefs::CreatePgPower() {
   PgPower.SetGridSize(1, 2, false);
   fr_Box Box1(this);
   Box1.SetName("Power Button ...");
   Box1.SetGridSize(2, 2, false);

   NoFork.Args << fr_CaseInsensitive << "--no-fork-on-exec";
   NoFork.SetTooltip("Exit " PROG " prior to launching snes9x. "
		     " Leaving this unchecked will have " PROG
		     " return after snes9x has finished.");
   NoFork.AddListener(this);

   NoESD.Args << fr_CaseInsensitive << "--turn-off-esd";
   NoESD.SetTooltip("Tell the E Sound Daemon to release the sound"
		     " device, so that snes9x will be able to use it.");
   NoESD.AddListener(this);

   UseX11.Args << fr_CaseInsensitive << "--driver=x11";
   UseX11.SetTooltip("Use the regular X11 snes9x executable");
	
   UseGlide.Args << fr_CaseInsensitive << "--driver=glide";
   UseGlide.SetTooltip("Use the Glide snes9x executable");
	
   Box1.Pack(NoFork);
   Box1.Pack(NoESD);
   Box1.Pack(UseX11);
   Box1.Pack(UseGlide);
   Box1.Frame();
   PgPower.Pack(Box1);
}

s9x_Prefs::~s9x_Prefs() {
   delete SnapDir;
}

void s9x_Prefs::SetToDefaults() {
   Appearance.SetToDefault();
   Reset2Prof.SetToDefault();
   Reset2Args.SetToDefault();
   Reset2Last.SetToDefault();
   Start2Prof.SetToDefault();
   Start2Last.SetToDefault();
   NoFork.SetToDefault();
   ROMdir.SetToDefault();
   SnapDir->SetToDefault();
   X11Executable.SetToDefault();
   GlideExecutable.SetToDefault();
   Snes9xVersion.SetToDefault();
   VideoDriver.SetToDefault();
   NoESD.SetToDefault();
#ifndef S9X_INCLUDE_ESD
   NoESD.EnableIf(0);
#endif
}

char*s9x_Prefs::GetROMdir() {
   return ROMdir.GetFileName();
}

char*s9x_Prefs::GetSnapDir() {
   return SnapDir->GetFileName();
}

void s9x_Prefs::SiftArgs(fr_ArgList& L) {
   Appearance.SiftArgs(L);
   NoFork.SiftArgs(L);
   Reset2Prof.SiftArgs(L);
   Reset2Args.SiftArgs(L);
   Reset2Last.SiftArgs(L);
   Start2Prof.SiftArgs(L);
   Start2Last.SiftArgs(L);
   ROMdir.SiftArgs(L);
   SnapDir->SiftArgs(L);
   X11Executable.SiftArgs(L);
   GlideExecutable.SiftArgs(L);
   Snes9xVersion.SiftArgs(L);
   VideoDriver.SiftArgs(L);
   NoESD.SiftArgs(L);
}

void s9x_Prefs::CompileArgs(fr_ArgList& L) {
   Appearance.CompileArgs(L);
   NoFork.CompileArgs(L);
   Reset2Prof.CompileArgs(L);
   Reset2Args.CompileArgs(L);
   Reset2Last.CompileArgs(L);
   Start2Prof.CompileArgs(L);
   Start2Last.CompileArgs(L);
   ROMdir.CompileArgs(L);
   SnapDir->CompileArgs(L);
   X11Executable.CompileArgs(L);
   GlideExecutable.CompileArgs(L);
   Snes9xVersion.CompileArgs(L);
   VideoDriver.CompileArgs(L);
   NoESD.CompileArgs(L);
}

void s9x_Prefs::Load() {
   try {
      PrefsFile.open(ios::in);
      fr_ArgList ArgList(25);
      char *buf = new char[FR_PATH_MAX];

      ArgList << fr_CaseInsensitive;
      while(PrefsFile >> buf) {
	 buf[FR_PATH_MAX-1] = 0;
	 ArgList << buf;
      };
      PrefsFile.close();
      SiftArgs(ArgList);
      delete[] buf;
   } catch(...) {
      fr_Mesg("Error loading preferences file!  :(");
   };
}

void s9x_Prefs::Save() {
   try {
      fr_ArgList ArgList(25);
      CompileArgs(ArgList);
      int c = ArgList.CountArgs();
      if(c<1) {
	 PrefsFile.erase();
	 return;
      };
      PrefsFile.open(ios::out);
      PrefsFile << "# " << PROG << " preferences" << endl;
      for(int i=0; i<c; i++)
	PrefsFile << ArgList[i] << endl;
      PrefsFile.close();
   } catch(...) {
      fr_Mesg("Error saving preferences file!  :(");
   }
}

void s9x_Prefs::Apply() {
   static char RomDirEnv[FR_PATH_MAX], SnapDirEnv[FR_PATH_MAX];
   char *rd;
   s9x_Interface*I;
	
   I = (s9x_Interface*)Parent;

   if(Appearance.GetSelected() == &SNES)
     I->ViewExpanded(true);
   else
     I->ViewExpanded(false);
       
   rd = ROMdir.GetFileName();
   sprintf(RomDirEnv, "%s=%s", S9X_ROMENV, rd?rd:"");
   putenv(RomDirEnv);
   rd = SnapDir->GetFileName();
   sprintf(SnapDirEnv, "%s=%s", S9X_SNAPSHOTENV, rd?rd:"");
   putenv(SnapDirEnv);
	
   if(VideoDriver.GetSelected() == &UseGlide)
     I->SetExecutable(GlideExecutable.GetFileName());
   else
     I->SetExecutable(X11Executable.GetFileName());
	
   I->Set9xVersion(Snes9xVersion.GetValue());
}

bool s9x_Prefs::Reincarnate() {
   return !(NoFork.GetState());
}

bool s9x_Prefs::ResetToProfile() {
   return Reset2Prof.GetState();
}

bool s9x_Prefs::ResetToArgs() {
   return Reset2Args.GetState();
}

bool s9x_Prefs::ResetToLast() {
   return Reset2Last.GetState();
}

bool s9x_Prefs::StartToProfile() {
   return Start2Prof.GetState();
}

bool s9x_Prefs::StartToLast() {
   return Start2Last.GetState();
}

bool s9x_Prefs::DisableESD() {
   return NoESD.GetState();
}

void s9x_Prefs::EventOccurred(fr_Event*e) {
   if(e->Is(BtnApply)||e->Is(BtnOK))
     Apply();
   if(e->Is(BtnSave)||e->Is(BtnOK))
     Save();
   if(e->Is(BtnClose)||e->Is(BtnOK))
     SetVisibility(false);
#ifdef S9X_INCLUDE_ESD
   if(e->Is(NoFork))
     NoESD.EnableIf(!NoFork.GetState());
   else if(e->Is(NoESD))
     NoFork.EnableIf(!NoESD.GetState());
#endif
   if(e->Is(this, fr_Destroy))
     SetVisibility(false);
   else if(e->Is(this, fr_MenuClick))
     SetVisibility(true);
}
