/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.util.ResourceBundle;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import com.redhat.rhdb.vise.resources.ActionResources;

/*
 * need to set things so that the dialog will take a model, clone it, work on
 * the clone, and set the original values to clone values if and only if the
 * user clicks ok.
 */

/**
 * Modal dialog for editing database models. Only one model can be
 * edited at a time. Currently, the owner of objects of this class
 * must be a Dialog object.
 *
 * @author Liam Stewart
 * @author Maintained by <a href="mailto:fnasser@redhat.com">Fernando Nasser</a>
 * @version 1.2.0
 *
 * @see com.redhat.rhdb.vise.DatabaseModel
 */
public class DatabaseEditorDialog extends JDialog implements ActionListener {
	private ActionFactory afactory;
	private GenericAction testAction;
    private ResourceBundle res;

	private int result;
	private Dialog owner;
	
	private DatabaseModel dbmodel;
	private ConfigurationModel cm;
	
	private JTextField nickfield, dbfield, hostfield, portfield, userfield;
	private JPasswordField passfield;
	private JButton okButton, cancelButton, testButton;
	private	JCheckBox savepasswd_button;

	private TestDatabaseConnectionDialog tester;

	/** User has closed window with intention of keeping changes. */
	public static final int OK_OPTION = 0;

	/** User has closed window with intention of discarding changes. */
	public static final int CANCEL_OPTION = 1;

	/** User has closed window with intention of discarding changes. */
	public static final int CLOSED_OPTION = 2;
	
	/**
	 * Creates a new <code>DatabaseEditorDialog</code> instance.
	 *
	 * @param owner a <code>Dialog</code> value
	 */
	public DatabaseEditorDialog(Dialog owner)
	{
		super(owner, true);
		
		this.owner = owner;
		
		// Listeners
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e)
			{
				close();
			}
		});

		// Actions
		afactory = new ActionFactory();

		testAction   = afactory.getAction(ActionFactory.ACTION_TEST);

        res = ResourceBundle.getBundle("com.redhat.rhdb.vise.resources.ActionResources");

		// initialize components
		initComponents();
	}

	/**
	 * Show the dialog, first setting its location relative to that of
	 * its owner.
	 *
	 */
	public void show()
	{
		pack();
		setLocationRelativeTo(owner);
		nickfield.requestFocus();
		
		super.show();
	}

	/**
	 * How/why was the dialog last closed?
	 *
	 * @return an <code>int</code> value
	 */
	public int getResult()
	{
		return result;
	}

	/**
	 * Set thd DatabaseModel that this dialog is supposed to edit.
	 *
	 * @param d a <code>DatabaseModel</code> value
	 */
	public void setDatabaseModel(DatabaseModel d)
	{
		this.dbmodel = d;
		
		update();
	}

	/**
	 * Get the DatabaseModel that is currently editable.
	 *
	 * @return a <code>DatabaseModel</code> value
	 */
	public DatabaseModel getDatabaseModel()
	{
		updateDatabaseModel();

		return dbmodel;
	}

	/**
	 * Set the ConfigurationModel used when doing validation tests.
	 *
	 * @param cm a <code>ConfigurationModel</code> value
	 */
	public void setConfigurationModel(ConfigurationModel cm)
	{
		this.cm = cm;
	}

	/**
	 * Get the ConfigurationModel used when doing validation tests.
	 *
	 * @return a <code>ConfigurationModel</code> value
	 */
	public ConfigurationModel getConfigurationModel()
	{
		return cm;
	}
	
	//
	// ActionListener
	//

	/**
	 * Listener for ActionEvents.
	 *
	 * @param e an <code>ActionEvent</code> value
	 */
	public void actionPerformed(ActionEvent e)
	{
		String command = e.getActionCommand();

		if (command.equals(res.getString(ActionResources.OK_COMMAND)))
		{
			if (validate_db())
				ok();
		}
		else if (command.equals(res.getString(ActionResources.CANCEL_COMMAND)))
		{
			cancel();
		}
		else if (command.equals(testAction.getActionCommand()))
		{
			if (validate_db())
				test();
		}
		else
		{
			System.err.println("DatabaseEditorDialog: unknown action command" + command);
		}
	}

	//
	// private methods
	//

	private void cancel()
	{
		result = CANCEL_OPTION;
		hide();
	}

	private void close()
	{
		result = CLOSED_OPTION;
		hide();
	}

	private void ok()
	{
		result = OK_OPTION;
		hide();
	}
	
	/*
	 * repopulate text fields
	 */
	private void update()
	{
		nickfield.setText(dbmodel.getName());
		dbfield.setText(dbmodel.getDatabase());
		hostfield.setText(dbmodel.getHost());
		portfield.setText(dbmodel.getPort());
		userfield.setText(dbmodel.getUser());
		passfield.setText(dbmodel.getPassword());
		savepasswd_button.setSelected(dbmodel.isSavePassword());
	}

	/*
	 * create all components
	 */
	private void initComponents()
	{
		JPanel content;
		JPanel buttonPanel;
		JPanel dbPanel;
		JComponent dbeditor;

		content = new JPanel();
		content.setLayout(new BorderLayout());
		content.setBorder(BorderFactory.createEmptyBorder(12, 12, 11, 11));
		setContentPane(content);
		
		dbPanel = new JPanel();
		dbPanel.setLayout(new BoxLayout(dbPanel, BoxLayout.Y_AXIS));

		dbeditor = createDatabaseEditorPanel();
		dbPanel.add(dbeditor);
		dbPanel.add(Box.createRigidArea(new Dimension(0, 17)));

		content.add(dbPanel, BorderLayout.CENTER);

		buttonPanel = new ButtonPanel(this, testAction);
		content.add(buttonPanel, BorderLayout.SOUTH);

		tester = new TestDatabaseConnectionDialog(this);
	}

	/*
	 * create the panel for editing database model properties
	 */
	private JPanel createDatabaseEditorPanel()
	{
		JPanel panel, panel2;
		JLabel label;
		
		GridBagLayout gbl = new GridBagLayout();
		GridBagConstraints gbc = new GridBagConstraints();

		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.anchor = GridBagConstraints.EAST;
		gbc.insets = new Insets(0, 0, 11, 0);
		gbc.weighty = 0.0;

		panel = new JPanel();
		panel.setLayout(gbl);

		nickfield = new JTextField(20);
		dbfield = new JTextField(20);
		hostfield = new JTextField(20);
		portfield = new JTextField(20);
		userfield = new JTextField(20);
		passfield = new JPasswordField(20);

		// labels

		gbc.gridx = 0;
		gbc.weightx = 0;
		gbc.insets = new Insets(0, 0, 5, 11);

		label = new JLabel(ExplainResources.getString(ExplainResources.DBEDIT_LABEL_NICKNAME));
		label.setLabelFor(nickfield);
		label.setDisplayedMnemonic(((Integer)ExplainResources.getObject(ExplainResources.DBEDIT_LABEL_NICKNAME_MNEMONIC)).intValue());
		gbc.gridy = 0;
		gbl.setConstraints(label, gbc);
		panel.add(label);

		label = new JLabel(ExplainResources.getString(ExplainResources.DBEDIT_LABEL_DATABASE));
		label.setLabelFor(dbfield);
		label.setDisplayedMnemonic(((Integer)ExplainResources.getObject(ExplainResources.DBEDIT_LABEL_DATABASE_MNEMONIC)).intValue());
		gbc.gridy = 1;
		gbl.setConstraints(label, gbc);
		panel.add(label);
		
		label = new JLabel(ExplainResources.getString(ExplainResources.DBEDIT_LABEL_HOST));
		label.setLabelFor(hostfield);
		label.setDisplayedMnemonic(((Integer)ExplainResources.getObject(ExplainResources.DBEDIT_LABEL_HOST_MNEMONIC)).intValue());
		gbc.gridy = 2;
		gbl.setConstraints(label, gbc);
		panel.add(label);
		
		label = new JLabel(ExplainResources.getString(ExplainResources.DBEDIT_LABEL_PORT));
		label.setLabelFor(portfield);
		label.setDisplayedMnemonic(((Integer)ExplainResources.getObject(ExplainResources.DBEDIT_LABEL_PORT_MNEMONIC)).intValue());
		gbc.gridy = 3;
		gbl.setConstraints(label, gbc);
		panel.add(label);
		
		label = new JLabel(ExplainResources.getString(ExplainResources.DBEDIT_LABEL_USER));
		label.setLabelFor(userfield);
		label.setDisplayedMnemonic(((Integer)ExplainResources.getObject(ExplainResources.DBEDIT_LABEL_USER_MNEMONIC)).intValue());
		gbc.gridy = 4;
		gbl.setConstraints(label, gbc);
		panel.add(label);
		
		label = new JLabel(ExplainResources.getString(ExplainResources.DBEDIT_LABEL_PASSWORD));
		label.setLabelFor(passfield);
		label.setDisplayedMnemonic(((Integer)ExplainResources.getObject(ExplainResources.DBEDIT_LABEL_PASSWORD_MNEMONIC)).intValue());
		gbc.gridy = 5;
		gbc.insets = new Insets(0, 0, 0, 11);
		gbl.setConstraints(label, gbc);
		panel.add(label);
		
		savepasswd_button = new JCheckBox(ExplainResources.getString(ExplainResources.PREFS_LABEL_SAVEPASSWORDS));
		savepasswd_button.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e)
			{
				int state = e.getStateChange();

				if (state == ItemEvent.SELECTED && !dbmodel.isSavePassword())
				{
					dbmodel.setSavePassword(true);
				}
				else if (state == ItemEvent.DESELECTED && dbmodel.isSavePassword())
				{
					dbmodel.setSavePassword(false);
				}
			}
		});
		savepasswd_button.setMnemonic(((Integer)ExplainResources.getObject(ExplainResources.PREFS_LABEL_SAVEPASSWORDS_MNEMONIC)).intValue());
		savepasswd_button.setToolTipText(ExplainResources.getString(ExplainResources.PREFS_LABEL_SAVEPASSWORDS_TOOLTIP));
		gbc.gridy = 6;
		gbc.gridx = 1;
		gbc.insets = new Insets(11, 0, 0, 0);
		gbl.setConstraints(savepasswd_button, gbc);
		panel.add(savepasswd_button);

		// text fields

		gbc.gridx = 1;
		gbc.weightx = 1;
		gbc.insets = new Insets(0, 0, 5, 0);

		gbc.gridy = 0;
		gbl.setConstraints(nickfield, gbc);
		panel.add(nickfield);

		gbc.gridy = 1;
		gbl.setConstraints(dbfield, gbc);
		panel.add(dbfield);

		gbc.gridy = 2;
		gbl.setConstraints(hostfield, gbc);
		panel.add(hostfield);

		gbc.gridy = 3;
		gbl.setConstraints(portfield, gbc);
		panel.add(portfield);

		gbc.gridy = 4;
		gbl.setConstraints(userfield, gbc);
		panel.add(userfield);

		gbc.gridy = 5;
		gbc.insets = new Insets(0, 0, 0, 0);
		gbl.setConstraints(passfield, gbc);
		panel.add(passfield);

		// container panel

		gbl = new GridBagLayout();
		gbc = new GridBagConstraints();

		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.anchor = GridBagConstraints.NORTH;
		gbc.weighty = 1;
		gbc.weightx = 1;

		panel2 = new JPanel();
		panel2.setLayout(gbl);
		gbl.setConstraints(panel, gbc);
		panel2.add(panel);
		
		return panel2;
	}

	private boolean validate_db()
	{
		// trim all text
		nickfield.setText(nickfield.getText().trim());
		dbfield.setText(dbfield.getText().trim());
		hostfield.setText(hostfield.getText().trim());
		portfield.setText(portfield.getText().trim());
		userfield.setText(userfield.getText().trim());
		passfield.setText((new String(passfield.getPassword())).trim());

		// do a validity check (in this order):
		//  - nickname
		//  - database
		//  - hostname
		//  - port number (it must be a number)
		//  - user

		if (nickfield.getText() == null || nickfield.getText().length() == 0)
		{
			JOptionPane.showMessageDialog(this,
										  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_NICK_MISSING),
										  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_NICK_MISSING_TITLE),
										  JOptionPane.ERROR_MESSAGE);
			nickfield.requestFocus();

			return false;
		}

		// the whole passing in a ConfigurationModel thing is a
		// bit ugly but it works for now...
		if (dbmodel.getName() != null && !dbmodel.getName().equals(nickfield.getText()))
		{
			if (cm != null && cm.getDatabaseModel(nickfield.getText()) != null)
			{
				JOptionPane.showMessageDialog(this,
											  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_NICK_USED),
											  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_NICK_USED_TITLE),
											  JOptionPane.ERROR_MESSAGE);
				nickfield.requestFocus();

				return false;
			}
		}

		if (dbfield.getText() == null || dbfield.getText().trim().length() == 0)
		{
			JOptionPane.showMessageDialog(this,
										  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_DB_MISSING),
										  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_DB_MISSING_TITLE),
										  JOptionPane.ERROR_MESSAGE);
			dbfield.requestFocus();

			return false;
		}

		if (hostfield.getText() == null || hostfield.getText().length() == 0)
		{
			JOptionPane.showMessageDialog(this,
										  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_HOST_MISSING),
										  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_HOST_MISSING_TITLE),
										  JOptionPane.ERROR_MESSAGE);
			hostfield.requestFocus();

			return false;
		}

		if (portfield.getText() == null || portfield.getText().trim().length() == 0)
		{
			JOptionPane.showMessageDialog(this,
										  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_PORT_MISSING),
										  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_PORT_MISSING_TITLE),
										  JOptionPane.ERROR_MESSAGE);
			portfield.requestFocus();

			return false;
		}
		else
		{
			try {
				Integer.parseInt(portfield.getText());
			} catch (NumberFormatException ex) {
				JOptionPane.showMessageDialog(this,
											  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_PORT_INT),
											  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_PORT_INT_TITLE),
											  JOptionPane.ERROR_MESSAGE);
				portfield.requestFocus();

				return false;
			}
		}

		if (userfield.getText() != null && userfield.getText().trim().length() == 0)
		{
			JOptionPane.showMessageDialog(this,
										  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_USER_MISSING),
										  ExplainResources.getString(ExplainResources.DBEDIT_ERROR_USER_MISSING_TITLE),
										  JOptionPane.ERROR_MESSAGE);
			userfield.requestFocus();

			return false;
		}

		return true;
	}

	private void updateDatabaseModel()
	{
		dbmodel.setName(nickfield.getText());
		dbmodel.setDatabase(dbfield.getText());
		if (hostfield.getText().equals(""))
			dbmodel.setHost("localhost");
		else
			dbmodel.setHost(hostfield.getText());
		if (portfield.getText().equals(""))
			dbmodel.setPort("5432");
		else
			dbmodel.setPort(portfield.getText());
		dbmodel.setUser(userfield.getText());
		dbmodel.setPassword(new String(passfield.getPassword()));
		dbmodel.setSavePassword(savepasswd_button.isSelected());
	}

	private boolean test()
	{
		updateDatabaseModel();
		tester.setDatabaseModel(dbmodel);
		tester.show();

		return true;
	}

}// DatabaseEditorDialog
