/* This file is part of q-tools, a collection of performance tools
   Copyright (c) 2003 Hewlett-Packard Development Company, L.P.
   Contributed by David Mosberger-Tang <davidm@hpl.hp.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA  02111-1307  USA  */

/* This module implements a data-structure to sum up counts for
   arbitrary address pairs.  The data-structure is intended to be used
   for collecting procedure call-counts, for example.  */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include "q-syscollect.h"
#include "call-counts.h"

#define HASH_SHIFT	16
#define HASH_TABLE_LEN	(1 << HASH_SHIFT)

struct cc_edge
  {
    uintptr_t from;
    uintptr_t to;
    size_t count;
    struct cc_edge *next;		/* collision-chain */
  };

static struct cc_edge *cc_free_list;

/* It should be OK to use a fixed-size hash-table here because we move
   recently accessed elements to the front of the collision-queue on
   each access.  */
struct call_count
  {
    struct cc_edge *htable[HASH_TABLE_LEN];
  };

static inline unsigned long
hash (uintptr_t from, uintptr_t to)
{
  union {
    unsigned long l;
    long double ld;
  } u;

  u.ld = 65535.0 * (long double) from * (long double) to;
  return u.l & (HASH_TABLE_LEN - 1);
}

static inline void
new_edge (struct call_count *cc, uintptr_t from, uintptr_t to,
	  size_t count, unsigned long index)
{
  static unsigned long alloc_count = 2048;
  unsigned long i;
  struct cc_edge *e;

  if (!cc_free_list)
    {
      alloc_count *= 2;
      e = malloc (alloc_count * sizeof (struct cc_edge));
      if (!e)
	{
	  fprintf (stderr, "call-counts.c.new_edge: out of memory\n");
	  return;
	}
      for (i = 0; i < alloc_count; ++i)
	{
	  e->next = cc_free_list;
	  cc_free_list = e++;
	}
    }
  e = cc_free_list;
  cc_free_list = cc_free_list->next;

  e->from = from;
  e->to = to;
  e->count = count;
  e->next = cc->htable[index];
  cc->htable[index] = e;
}

struct call_count *
call_count_create (void)
{
  struct call_count *cc = malloc (sizeof (*cc));

  if (!cc)
    return NULL;

  memset (cc, 0, sizeof (*cc));
  return cc;
}

void
call_count_destroy (struct call_count *cc)
{
  free (cc);
}

void
call_count_add (struct call_count *cc, uintptr_t from, uintptr_t to,
		size_t count)
{
  struct cc_edge *e, *prev;
  unsigned long index = hash (from, to);

  e = cc->htable[index];
  if (likely (e))
    {
      if (likely (e->from == from && e->to == to))
	{
	  /* fast path... */
	  e->count += count;
	  return;
	}
      else
	/* walk the collision-chain:  */
	for (prev = e, e = e->next; e; prev = e, e = e->next)
	  if (e->from == from && e->to == to)
	    {
	      e->count += count;
	      /* move to front of collision chain: */
	      prev->next = e->next;
	      e->next = cc->htable[index];
	      cc->htable[index] = e;
	      return;
	    }
    }
  new_edge (cc, from, to, count, index);
}

int
call_count_extract (struct call_count *cc,
		    int (*callback) (void *arg, uintptr_t from, uintptr_t to,
				     size_t count),
		    void *arg)
{
  unsigned long i;
  struct cc_edge *e;
  int ret = 0;

  for (i = 0; i < HASH_TABLE_LEN; ++i)
    if (cc->htable[i])
      {
	for (e = cc->htable[i]; e; e = e->next)
	  {
	    ret = (*callback) (arg, e->from, e->to, e->count);
	    if (ret < 0)
	      break;
	  }
      }
  return ret;
}
