// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc_v2/sapphire.cpp,v 1.7 2002/01/09 14:50:13 weldon Exp $
//

#ifdef GC_SAPPHIRE
#include "sapphire.h"
#include "block_store.h"
#include "gc_for_orp.h"
#include "orp_threads.h"
#include "sync_bits.h"
#include "train_generation.h"
#include "los.h"
#include "orp_synch.h"
#include "to_from_table.h"
#include "ssb.h"
#include "orp_synch.h"
#include "gc_perf.h"

// For some reason this must be seperated out. I think it is a VC++ bug.
extern Card_Table *p_global_card_table_manager;


// A couple of forward references.

// Objects in c space are objects in los and objects in copy space.
// Objects in copy space are objects in from space and objects in to space.
// Objects in u space are objects not in c space.
// Color issues.
// Objects in los can be white, or black once they have been scanned.
//   They can be on an ssb queue but that is not useful w.r.t. to color.
// Objects in "to" space can be gray if they lay beyond the cheney repoint pointer.
//   or black if they have been repointed.
// Objects in "from" space with no counterpart in "to" space are white.
//   otherwise they are whatever color their "from" space objects are.
// Question - do we have to repoint "from" objects or is this just an optimization.
//   I think we don't have to repoint "from" objects.
// 
// Objects in u space have no color!
bool is_object_in_c_space (Java_java_lang_Object *p_obj);
bool is_object_in_u_space (Java_java_lang_Object *p_obj);
bool is_object_in_copy_space (Java_java_lang_Object *p_obj);
bool is_object_in_from_space (Java_java_lang_Object *p_obj);
bool is_object_in_to_space (Java_java_lang_Object *p_obj);
bool is_object_in_los (Java_java_lang_Object *p_obj)
{
    return p_global_bs->is_object_in_large_object_space(p_obj);
}

// Object color changing routines.

// Valid in the reserve phase. Takes a slot to assist in locating "to" space
// and the object to turn from white to gray.
void sapphire_gray_white_object(Java_java_lang_Object **pp_slot, Java_java_lang_Object *p_obj);

// Valid in the reserve phase. Takes an object that could be anywhere and scans it
// calling sapphire_gray_white_object on all slots holding c space objects.
void sapphire_scan_graying_white(Java_java_lang_Object *p_obj);

// Using cards and remembered sets calls sapphire_gray_white_object on all objects transitively
// reachable from u space.
bool sapphire_gray_white_objects_reachable_from_u();

void sapphire_repoint_refs_in_u_los_and_to_space ();
void sapphire_make_reserve_phase_consistent (Java_java_lang_Object *p_from_obj);
void sapphire_make_repoint_flip_phase_consistent (Java_java_lang_Object *p_obj, 
                                                  Java_java_lang_Object *p_shadow_obj);
Java_java_lang_Object *get_from_object (Java_java_lang_Object *p_obj);
Java_java_lang_Object *get_to_object (Java_java_lang_Object *p_obj);
bool is_sapphire_link_block (POINTER_SIZE_INT *p_header);

void sapphire_blacken_gray_object (Java_java_lang_Object *p_obj);
void sapphire_blacken_gray_objects();
void sapphire_flip_thread ();
void sapphire_repoint (Java_java_lang_Object *p_obj);
// void sapphire_gray_white_object (Java_java_lang_Object **pp_slot);

void sapphire_maintain_consistency (Java_java_lang_Object *p_obj, 
                                    unsigned offset,
                                    Java_java_lang_Object *value);

sapphire_tri_color sapphire_object_color (Java_java_lang_Object *base_of_object_holding_ref);
void sapphire_repoint_array (Java_java_lang_Object *p_obj);
void sapphire_repoint_refs_in_u();
void sapphire_flip_all_threads();
void sapphire_stabilize_all_threads();
void sapphire_release_old_c_area();
bool sapphire_scan_all_stacks ();
void sapphire_scan_graying_white (Java_java_lang_Object *p_obj);
void sapphire_array_scan_graying_white (Java_java_lang_Object *p_obj);
void sapphire_scan_object_asserting_no_old_pointers (Java_java_lang_Object *base_of_object_holding_ref);
void spin_lock_release (Java_java_lang_Object *obj);
void sapphire_los_mark (Object_Gc_Header *p_gc_header);
bool is_header_locked (Java_java_lang_Object *p_obj);
bool is_object_in_to_space (Java_java_lang_Object *p_obj);

// Set to true is stack scanning finds a white object.
bool global_white_object_found;

//
// This is a hash table that given the "to" version of an object can find
// the corresponding "from" version of the object. It can also be used to
// determine if an object is in "to" space.
// 
To_From_Table *to_from_map = new To_From_Table();

//
// All pointers from U space, to space, and los space are placed into this
// ssb. In fact all slots in the heap that are not in "from" space that 
// point to "from" space will be in this ssb. It is conservative in the 
// sense that the value in the slot may change before it is scanned and 
// therefore not hold a pointer into "from" space.
// Pointers are placed into this ssb during card scanning, remset scanning, los scanning,
// to space scanning of live objects and by the write barriers.
// 
// This is then rewound at the start of the repoint phase and scanned.
// 
// c_space_ssb includes slots in los.
ssb_container *c_space_ssb = NULL;

// The slot discovered by the GC that need to be updated.
ssb_container *c_space_ssb_collector = NULL;

ssb_container *ssb_repoint = NULL;

volatile sapphire_states sapphire_state;

// 
// Sapphire needs to manages two versions of objects, a "from" version
// and a "to" version. 
// Given a reference to either of these versions one must be able to 
// locate the other version as well as be able to atomically lock the 
// object. To support this we use the same 
// mechanism that the locking code uses to escape from thin locks. The "from"
// version points to sapphire link block which in turn points to the "to" version. 
// To get from the "to" object to the "from" object one uses a table designed for
// this purpose. The header of the "to" version is used for locking. This allows
// full speed synchronization once we have the "to" version. It also allows the
// cleanup after a gc to just release the sapphire link blocks.
// This is set up in the mark_copy_phase when space for the new version is reserved.
//
//  |-----------|        |               |               |-----------|
//  | header ----------> |  to ----------|-------------->|lock header|   
//  |-----------|        |  from         |               |-----------|
//  |           |        |  color        |               |           |
//  |           |        |               |               |           | 
//  |-----------|              ...                       |-----------|
//      ...                                                   ...
// 
// Locking is simple. We look at the header and if it points to a sapphire link block
// then we move on the the "to" version of the object and do the locking. 
// If not then we do the locking with the header we have.
//
//
// Write barrier code.
//
// sapphire_write_barrier is called anytime a field in an object is changed. This includes both
// fields holding pointers and fields holding non pointers.
//
// Input - the base of the object that has just been written to.
//
// Cards are marked for the new area only since the old area will be released once the
// GC is done and the write barriers are enough to make sure we don't lose any pointers
// written to the old space.
//

void  gc_sapphire_heap_write_ref(Java_java_lang_Object *p_object_holding_ref, 
                                 unsigned offset,
                                 Java_java_lang_Object *value
                                 )
{
//    gc_trace (p_object_holding_ref, "gc_heap_write_ref this object has a slot that is being updated.");
//    gc_trace (value, "gc_heap_write_ref slot holding this value is noted.");
    // Check to make sure the _p_card_last_object_table has a value for this 
    // object.
    
    // Mark the object passed in unless it is in from space, in which case
    // we mark the "to" object.
    Java_java_lang_Object *p_object_to_mark = p_object_holding_ref;
    // Get address of slot.
    Java_java_lang_Object **p_slot = 
        (Java_java_lang_Object **)(((char *)p_object_holding_ref) + offset);    
    switch (sapphire_state) {
    case normal:
    case stabilize_phase: {
            *p_slot = value;
            break;
                 } // Just do card marking for pointers as below
    case mark_reserve_phase: {
        // The mutators can not see a 'to' object version, so we are writing 
        // to C 'from' space, los, or U.
        // The object with the slot can be in either C (collected) or U (uncollected) 
        // space.
        // Writes to U space object slots.
        //		References to white C objects are placed in the c_space_ssb. 
        //      This prevents a thread from writing a white pointer into the heap.
        // Writes to C space object slots
        //		The object being written into can be white, gray or black.
        //		White - has not been encountered before
        //			light gray it by placing the slot in the ssb, 
        //          gc thread will copy it making it dark grey and cheney will blacken it.
        //          This will require setting up pointers between the two spaces.
        //          Since gray can point to white we can safely not process the 
        //          referent (value).
        //		gray - The object has been encountered before and has a copy in new space
        //			but the cheney scan has not reached it, otherwise it would be black.
        //          Do nothing since when the cheney scan arrives it will do 
        //          the copy of the new value and make the objects consistent.
        //          Since gray can point to white we can safely not process the 
        //          referent (value).
        //		Black - The object has been encountered before and copied to new space
        //          and the chency scan has already reached and processed it. We can not
        //          write a reference to a white object into a black object so we need
        //          to gray the value before we write it. We then need to make both
        //          copies consistent. Since we are in this phase the no slot has been
        //          repointed to the "to" version of the object.
        // At the end of this phase all object in copy space will have both a to and a from
        // version but no slot will point to the "to" versions.
        *p_slot = value;
        if (is_object_in_c_space (value)) {
            if (sapphire_object_color (value) == white) {
                ssb_write_object ((ssb_container *)p_TLS_orpthread->ssb_container, value);
            }
        }
       
        if (is_object_in_u_space (p_object_holding_ref)) {
            // U -> ?
            if (is_object_in_c_space (value)) {
                // U -> C
                // record the slot pointing to c space. note slot can change.
                ssb_write_slot ((ssb_container *)p_TLS_orpthread->ssb_container, p_slot); 
            } else {
                // U -> U
                // This object is not interesting during this collection but might be 
                // later so we mark the corresponding card below but that is all we need
                // to do. Objects in U don't have color since we use remset to track
                // pointers.
            }
            break;
        }
        // p_object_holding_ref is in C space.

        // slot is in C (collected) space, possible LOS
        // C -> ?
        assert (is_object_in_c_space(p_object_holding_ref));

        // It can be marked and in LOS which is to_space.
        // considered to_space.  assert (is_object_in_from_space (p_object_holding_ref));

        if (sapphire_object_color (p_object_holding_ref) == white) {
            // C-white -> C-?
            ssb_write_object ((ssb_container *)p_TLS_orpthread->ssb_container, p_object_holding_ref);
            // We have a slot in From space without a To space slot so we 
            // don't mark it instead we use the ssb to tell the 
            // gc about this object.
        } else  { // c-gray/black -> C-? p_object_holding_ref has "to" version...
            if (is_object_in_copy_space(p_object_holding_ref)) {
                // C-copy-gray/black -> C-?
                // make sure "to" version gets the same value.
                // Get address of to slot, we can only see "from" at this point.
                assert (is_object_in_from_space(p_object_holding_ref));
                assert (!is_object_in_to_space(value));
                Java_java_lang_Object *p_to_version = get_to_object (p_object_holding_ref);
                Java_java_lang_Object **p_to_slot = 
                    (Java_java_lang_Object **)(((char *)p_to_version) + offset);
                *p_to_slot = value; // if this is a from value it will be repointed because
                                    // of the following.
                ssb_write_slot ((ssb_container *)p_TLS_orpthread->ssb_container, p_to_slot); 
                ssb_write_slot ((ssb_container *)p_TLS_orpthread->ssb_container, p_slot); 
            } else {
                // p_object_holding_ref is in black and in LOS
                assert (p_object_holding_ref == p_object_to_mark);
                assert (is_object_in_los(p_object_holding_ref));
            }
        }  
             
        if (sapphire_object_color(p_object_holding_ref) != white) {
            if (is_object_in_copy_space(p_object_holding_ref)) {
                // If we already had a "to" object then mark it for next GC
                p_object_to_mark = get_to_object (p_object_holding_ref);
            }
        }
        break; } // end mark_reserve_copy_phase
    case repoint_phase:
    case flip_phase: {
        //
        // At this point every object in copy space has consistent "from" and "to" 
        // versions so we can let the mutator see "to" objects since they can't
        // "read" ahead of the cheney scan and get an inconsistent object.
        // Regardless of where the object is if the referent (value) is in copy space
        // install the "to" object value.
        //
        // If the object is in U space 
        //      U->?
        //
        //      U -> C-copy
        //          install the "to" object pointer.
        //          if the referent (value) is in U space
        //      U -> C-los
        //      U -> U
        //           install the value
        //
        // If the object is in C space then it is black
        //      C -> ?
        //		For both the "to" and "from" versions of the object
        //          If the referent (value) is into C space, 
        //          C -> C
        //          install the "to" object version pointer in both 
        //          the "to" and "from" versions,
        //          keeping them consistent.
        //
        //          If the referent (value) is in U space
        //          C -> U
        //          Install the U value in both the "to" and "from" versions keeping them
        //          consistent.
        // Note that we never install a "from" value. We don't need to place anything
        // in the c_space_ssb since all slots have been noted and we are not
        // creating any slot with a "from" version pointer in it. The card mark
        // will find the slot during the next GC.
        //
        // We can't combine the repoint phase with the flip phase since once we flip 
        // a thread it still has access to the U area which might hold a pointer 
        // into old space. 
        //
        // Memory barriers - If a memory barrier is encounted this code will need to
        // be completed as part of the memory barrier.
        //
        // sapphire_repoint takes an object that could be in "FROM" or "TO" space 
        // and make sure that all pointers in it point to to space. 
        
        // Get the "to" object if there is one, we never install a "from" value in this
        // phase.
        
        Java_java_lang_Object *new_value = value;
        if (is_object_in_from_space (value)) {
            new_value = get_to_object (value);
        }

        if (is_object_in_u_space (p_object_holding_ref)) { // it is in U space
            // U -> ?
            *p_slot = new_value;
            // entire object is repointed to new space so we are done.
        } else {
            // C -> ?
            // object being written into is in C space and has been repointed;
            assert (is_object_in_c_space (p_object_holding_ref));
            assert (sapphire_object_color (p_object_holding_ref) == black);
            sapphire_maintain_consistency(p_object_holding_ref, offset, new_value);
            p_object_to_mark = get_to_object (p_object_holding_ref);
        }
        break; 
                        }
    } // end switch
    
#ifdef GC_GEN_V3
    if (!is_object_in_from_space(p_object_to_mark)) {
        // GC_FIXED_V1 and GC_SAPPHIRE don't need to do card marking.
        // Mark the card. BUG BUG what about a write barrier in an older generation......
        p_global_card_table_manager->mark_card (p_object_to_mark);
    }
#endif
}

// Treat this write as if the entire object was written to by something like
// array copy. This is only valid for objects that don't contain pointers.
void gc_sapphire_heap_wrote_object (Java_java_lang_Object *p_object_holding_ref)
{
    
    Java_java_lang_Object *p_object_to_mark = p_object_holding_ref;
    switch (sapphire_state) {
    case normal:
    case stabilize_phase: {
        break; } // Just do card marking for pointers as below
    case mark_reserve_phase: {
        // assert(0); // remove and debug.
        // We have just written an old space object (assert this) since nobody can 
        // see new space yet.
        // The object can be in either C (collected) or U (uncollected) space.
        // Writing a U space object.
        //	    If we find references in the U space object we assert. 
        //      pointing to C objects then we gray the C object. 
        //      This prevents a thread from writing a white pointer into the heap.
        // Writes to C space object slots
        //		The object being written into can be white, gray or black.
        //		White - has not been encountered before
        //			gray it by copying it to new space, cheney will blacken it.
        //          This will require setting up pointers between the two spaces.
        //		gray - The object has been encountered before and copied to new space
        //			but the cheney scan has not reached it. Do nothing since when the
        //          cheney scan arrives it will do the copy of the new value.
        //		Black - The object has been encountered before and copied to new space
        //          and the chency scan has already reached and processed it. We can not
        //          write a reference to a white object into a black object so we need
        //          to gray the value before we write it.
        // If the object being written into and the value being written are both in
        // C space then if the object being written into is black then we have to make sure
        // the to and from object are made consistent.
        
        ssb_write_object ((ssb_container *)p_TLS_orpthread->ssb_container, p_object_holding_ref);
        
        break; 
                          } // end mark_reserve_phase

    case repoint_phase: {
        //
        // If the object is in U space 
        //		Scan the object making sure that all the pointers into C space
        //		point into new space. This is needed so that we can show progress.
        //
        // If the object is in C space then it is black
        //		If it is in new space
        //			Scan the object making sure that all the pointers into C space
        //			point into new space. (An old space pointer might have been the one
        //			that just got installed into the object.)
        //          Then copy the object into old space to maintain dynamic consistency.
        //      If it is in old space
        //			Scan the object making sure that all the C space pointers
        //			point into new space. Then copy the object into new space to
        //          maintain dynamic consistency.
        //			
        // We can't combine the repoint phase with the flip phase since once we flip a thread it
        // still has access to the U area which might hold a pointer into old space. 
        //
        // Memory barriers - If a memory barrier is encounted this code will need to
        // be completed as part of the memory barrier.
        //
        // Only install new space pointers.
        //
        // sapphire_repoint takes an object that could be in "FROM" or "TO" space 
        // and make sure that all pointers in it point to to space. 
        // BUG BUG BUG There is a race condition here since a pointer
        //  to "FROM" space could have been stored here by the jitted code and 
        //  not changed until the call to repoint. This means that a FLIPPED thread 
        // could pick up this "FROM" pointer when it should not."
        
        // Get the "to" object if there is one.
        if (is_object_in_u_space (p_object_holding_ref)) { // it is in U space
            // U -> ?
            sapphire_repoint (p_object_holding_ref);
            // entire object is repointed to new space so we are done.
        } else {
            if (is_object_in_copy_space(p_object_holding_ref)) {
                Java_java_lang_Object *shadow;
                if (is_object_in_from_space (p_object_holding_ref)) {
                    shadow = get_to_object (p_object_holding_ref);
                    sapphire_make_repoint_flip_phase_consistent(p_object_holding_ref, shadow);
                    p_object_to_mark = get_to_object(p_object_holding_ref);
                } else {
                    shadow = get_from_object (p_object_holding_ref);
                    sapphire_make_repoint_flip_phase_consistent(p_object_holding_ref, shadow);
                }
            }
        }
        break; }
    case flip_phase: {
        // We have a black object (Assert this). It can be in C or U space, if it is in C space
        // it can be in old or new space. This gives us three different possibilities.
        //
        // If in U space
        //		Do nothing. - U space can hold no pointers into C old space.
        //		Scan the object and assert if we find an old space pointer.
        //
        // If in C space
        //		Use the same logic as in the repoint phase. After that logic is done
        //      Flip the thread so that it will never see an old space pointer again.
        //
        // Memory barriers - If a memory barrier is encounted this code will need to
        // be completed as part of the memory barrier.
        //
        //
        // Only install new space pointers.
        //
        
        ///////////////////////////////////////////////
        //      sapphire_repoint takes an object that could be in "FROM" or "TO" space and make sure that
        //      all pointers in it point to to space. BUG BUG BUG There is a race condition here since a pointer
        //      to "FROM" space could have been stored here by the jitted code and not changed until the call
        //      to repoint. This means that a FLIPPED thread could pick up this "FROM" pointer when it should not."
        

        if (is_object_in_u_space (p_object_holding_ref)) {
#ifdef _DEBUG
            sapphire_scan_object_asserting_no_old_pointers (p_object_holding_ref);
#endif
        } else {
            assert (is_object_in_c_space (p_object_holding_ref));
            assert (sapphire_object_color(p_object_holding_ref) == black);
            if (is_object_in_copy_space(p_object_holding_ref)) {
                Java_java_lang_Object *shadow;
                if (is_object_in_from_space (p_object_holding_ref)) {
                    shadow = get_to_object (p_object_holding_ref);
                    sapphire_make_repoint_flip_phase_consistent(p_object_holding_ref, shadow);
                    p_object_to_mark = get_to_object(p_object_holding_ref);
                } else {
                    shadow = get_from_object (p_object_holding_ref);
                    sapphire_make_repoint_flip_phase_consistent(p_object_holding_ref, shadow);
                }
            }
        }
        break; }
    } // end switch
    // DO WE NEED TO MARK CARDS OF NON_POINTER CONTAINING OBJECTS???????????    
#ifdef GC_GEN_V3
    // GC_FIXED_V1 doesn't need to do card marking.
    // Mark the card. BUG BUG what about a write barrier in an older generation......
    p_global_card_table_manager->mark_card (p_object_holding_ref);
#endif
}

// There is an interesting "flicker" value problem here.
// Thread A and thread B are competing to set this value.
// Assume thread B starts by writing valb but thread A ends up "winning" 
// and setting the value to vala first.
// Thread B follows and redoes the writes of valb.
// A thread C might see valb, then vala, then valb. This can only happen
// with unsynchronized code so the Java spec is silent on the matter.
//
// Non reference writes caller does conversion. The following code uses
// text editor polymorphism. If one needs to edit one of the copies one should 
// change all the copies.
void sapphire_make_int8_slot_consistent (Java_java_lang_Object *p_from_obj, 
                                         unsigned offset, 
                                         int8 value)
{
    assert (is_object_in_from_space (p_from_obj));
    assert (sapphire_object_color (p_from_obj) == black); // It should be black
    Java_java_lang_Object *p_to_obj = get_to_object (p_from_obj);
    volatile int8 *p_from_slot = 
        (volatile int8 *)(((char *)p_from_obj) + offset);
    volatile int8 *p_to_slot = 
        (volatile int8 *)(((char *)p_to_obj) + offset);
    *p_from_slot = value;
    *p_to_slot = value;
    while (*p_from_slot != *p_to_slot) {
        *p_from_slot = value;
        *p_to_slot = value;
    }
    gc_trace (p_from_obj, "Making from object int8 consistent.");
    gc_trace (p_to_obj, "Making to object int8 consistent.");
    return;
}

void gc_sapphire_heap_write_int8 (Java_java_lang_Object *p_object_with_slot,
                                  unsigned offset,
                                  int8 value)
{
#ifdef GC_FIXED_V1
    int8 *p_slot = 
        (int8 *)(((char *)p_object_with_slot) + offset);
    *p_slot = value;
    return; 
#endif
    switch (sapphire_state) {
    case normal:
    case stabilize_phase: {
        break; } 
    case mark_reserve_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) { 
            if (sapphire_object_color (p_object_with_slot) == black) {
                sapphire_make_int8_slot_consistent (get_from_object (p_object_with_slot),
                    offset, 
                    value);
                return;
            }
        }
        
        break; 
                          } // end mark_reserve_phase

    case repoint_phase: 
    case flip_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) {
            assert (sapphire_object_color(p_object_with_slot) == black);
            sapphire_make_int8_slot_consistent (get_from_object (p_object_with_slot), 
                offset, 
                value);
            return;
        }
        break; }
    } // end switch
    // If we are here there is only one copy that has to be updated, so do it.
    int8 *p_slot = 
        (int8 *)(((char *)p_object_with_slot) + offset);    
    *p_slot = value;
}


void sapphire_make_int16_slot_consistent (Java_java_lang_Object *p_from_obj, 
                                          unsigned offset, 
                                          int16 value)
{
    assert (is_object_in_from_space (p_from_obj));
    assert (sapphire_object_color (p_from_obj) == black); // It should be black
    Java_java_lang_Object *p_to_obj = get_to_object (p_from_obj);
    volatile int16 *p_from_slot = 
        (volatile int16 *)(((char *)p_from_obj) + offset);
    volatile int16 *p_to_slot = 
        (volatile int16 *)(((char *)p_to_obj) + offset);
    *p_from_slot = value;
    *p_to_slot = value;
    while (*p_from_slot != *p_to_slot) {
        *p_from_slot = value;
        *p_to_slot = value;
    }
    gc_trace (p_from_obj, "Making from object int16 consistent.");
    gc_trace (p_to_obj, "Making to object int16 consistent.");
    return;
}

void gc_sapphire_heap_write_int16 (Java_java_lang_Object *p_object_with_slot,
                                   unsigned offset,
                                   int16 value)
{
#ifdef GC_FIXED_V1
    int16 *p_slot = 
        (int16 *)(((char *)p_object_with_slot) + offset);
    *p_slot = value;
    return; 
#endif
    switch (sapphire_state) {
    case normal:
    case stabilize_phase: {
        break; } 
    case mark_reserve_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) { 
            if (sapphire_object_color (p_object_with_slot) == black) {
                sapphire_make_int16_slot_consistent (get_from_object (p_object_with_slot),
                    offset, 
                    value);
                return;
            }
        }
        
        break; 
                          } // end mark_reserve_phase


    case repoint_phase: 
    case flip_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) {
            assert (sapphire_object_color(p_object_with_slot) == black);
            sapphire_make_int16_slot_consistent (get_from_object (p_object_with_slot), 
                offset, 
                value);
            return;
        }
        break; }
    } // end switch
    // If we are here there is only one copy that has to be updated, so do it.
    int16 *p_slot = 
        (int16 *)(((char *)p_object_with_slot) + offset);    
    *p_slot = value;
}


void sapphire_make_uint16_slot_consistent (Java_java_lang_Object *p_from_obj, 
                                           unsigned offset, 
                                           uint16 value)
{
    assert (is_object_in_from_space (p_from_obj));
    assert (sapphire_object_color (p_from_obj) == black); // It should be black
    Java_java_lang_Object *p_to_obj = get_to_object (p_from_obj);
    volatile uint16 *p_from_slot = 
        (volatile uint16 *)(((char *)p_from_obj) + offset);
    volatile uint16 *p_to_slot = 
        (volatile uint16 *)(((char *)p_to_obj) + offset);
    *p_from_slot = value;
    *p_to_slot = value;
    while (*p_from_slot != *p_to_slot) {
        *p_from_slot = value;
        *p_to_slot = value;
    }
    gc_trace (p_from_obj, "Making from object uint16 consistent.");
    gc_trace (p_to_obj, "Making to object uint16 consistent.");
    return;
}

void gc_sapphire_heap_write_uint16 (Java_java_lang_Object *p_object_with_slot,
                                    unsigned offset,
                                    uint16 value)
{
#ifdef GC_FIXED_V1
    uint16 *p_slot = 
        (uint16 *)(((char *)p_object_with_slot) + offset);
    *p_slot = value;
    return; 
#endif
    switch (sapphire_state) {
    case normal:
    case stabilize_phase: {
        break; } 
    case mark_reserve_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) { 
            if (sapphire_object_color (p_object_with_slot) == black) {
                sapphire_make_uint16_slot_consistent (get_from_object (p_object_with_slot),
                    offset, 
                    value);
                return;
            }
        }
        
        break; 
                          } // end mark_reserve_phase

    case repoint_phase: 
    case flip_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) {
            assert (sapphire_object_color(p_object_with_slot) == black);
            sapphire_make_uint16_slot_consistent (get_from_object (p_object_with_slot), 
                offset, 
                value);
            return;
        }
        break; }
    } // end switch
    // If we are here there is only one copy that has to be updated, so do it.
    uint16 *p_slot = 
        (uint16 *)(((char *)p_object_with_slot) + offset);    
    *p_slot = value;
}


void sapphire_make_float_slot_consistent (Java_java_lang_Object *p_from_obj, 
                                          unsigned offset, 
                                          float value)
{
    assert (is_object_in_from_space (p_from_obj));
    assert (sapphire_object_color (p_from_obj) == black); // It should be black
    Java_java_lang_Object *p_to_obj = get_to_object (p_from_obj);
    volatile float *p_from_slot = 
        (volatile float *)(((char *)p_from_obj) + offset);
    volatile float *p_to_slot = 
        (volatile float *)(((char *)p_to_obj) + offset);
    *p_from_slot = value;
    *p_to_slot = value;
    while (*p_from_slot != *p_to_slot) {
        *p_from_slot = value;
        *p_to_slot = value;
    }
    gc_trace (p_from_obj, "Making from object float consistent.");
    gc_trace (p_to_obj, "Making to object float consistent.");
    return;
}

void gc_sapphire_heap_write_float (Java_java_lang_Object *p_object_with_slot,
                                   unsigned offset,
                                   float value)
{
#ifdef GC_FIXED_V1
    float *p_slot = 
        (float *)(((char *)p_object_with_slot) + offset);
    *p_slot = value;
    return; 
#endif
    switch (sapphire_state) {
    case normal:
    case stabilize_phase: {
        break; } 
    case mark_reserve_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) { 
            if (sapphire_object_color (p_object_with_slot) == black) {
                sapphire_make_float_slot_consistent (get_from_object (p_object_with_slot),
                    offset, 
                    value);
                return;
            }
        }
        
        break; 
                          } // end mark_reserve_phase

    case repoint_phase: 
    case flip_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) {
            assert (sapphire_object_color(p_object_with_slot) == black);
            sapphire_make_float_slot_consistent (get_from_object (p_object_with_slot), 
                offset, 
                value);
            return;
        }
        break; }
    } // end switch
    // If we are here there is only one copy that has to be updated, so do it.
    float *p_slot = 
        (float *)(((char *)p_object_with_slot) + offset);    
    *p_slot = value;
}


void sapphire_make_double_slot_consistent (Java_java_lang_Object *p_from_obj, 
                                           unsigned offset, 
                                           double value)
{
    assert (is_object_in_from_space (p_from_obj));
    assert (sapphire_object_color (p_from_obj) == black); // It should be black
    Java_java_lang_Object *p_to_obj = get_to_object (p_from_obj);
    volatile double *p_from_slot = 
        (volatile double *)(((char *)p_from_obj) + offset);
    volatile double *p_to_slot = 
        (volatile double *)(((char *)p_to_obj) + offset);
    *p_from_slot = value;
    *p_to_slot = value;
    while (*p_from_slot != *p_to_slot) {
        *p_from_slot = value;
        *p_to_slot = value;
    }
    gc_trace (p_from_obj, "Making from object double consistent.");
    gc_trace (p_to_obj, "Making to object double consistent.");
    return;
}

void gc_sapphire_heap_write_double (Java_java_lang_Object *p_object_with_slot,
                                    unsigned offset,
                                    double value)
{
#ifdef GC_FIXED_V1
    double *p_slot = 
        (double *)(((char *)p_object_with_slot) + offset);
    *p_slot = value;
    return; 
#endif
    switch (sapphire_state) {
    case normal:
    case stabilize_phase: {
        break; } 
    case mark_reserve_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) { 
            if (sapphire_object_color (p_object_with_slot) == black) {
                sapphire_make_double_slot_consistent (get_from_object (p_object_with_slot),
                    offset, 
                    value);
                return;
            }
        }
        
        break; 
                          } // end mark_reserve_phase

    case repoint_phase: 
    case flip_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) {
            assert (sapphire_object_color(p_object_with_slot) == black);
            sapphire_make_double_slot_consistent (get_from_object (p_object_with_slot), 
                offset, 
                value);
            return;
        }
        break; }
    } // end switch
    // If we are here there is only one copy that has to be updated, so do it.
    double *p_slot = 
        (double *)(((char *)p_object_with_slot) + offset);    
    *p_slot = value;
}


void sapphire_make_int32_slot_consistent (Java_java_lang_Object *p_from_obj, 
                                          unsigned offset, 
                                          int32 value)
{
    assert (is_object_in_from_space (p_from_obj));
    assert (sapphire_object_color (p_from_obj) == black); // It should be black
    Java_java_lang_Object *p_to_obj = get_to_object (p_from_obj);
    volatile int32 *p_from_slot = 
        (volatile int32 *)(((char *)p_from_obj) + offset);
    volatile int32 *p_to_slot = 
        (volatile int32 *)(((char *)p_to_obj) + offset);
    *p_from_slot = value;
    *p_to_slot = value;
    while (*p_from_slot != *p_to_slot) {
        *p_from_slot = value;
        *p_to_slot = value;
    }
    gc_trace (p_from_obj, "Making from object int32 consistent.");
    gc_trace (p_to_obj, "Making to object int32 consistent.");
    return;
}

void gc_sapphire_heap_write_int32 (Java_java_lang_Object *p_object_with_slot,
                                   unsigned offset,
                                   int32 value)
{
#ifdef GC_FIXED_V1
    int32 *p_slot = 
        (int32 *)(((char *)p_object_with_slot) + offset);
    *p_slot = value;
    return; 
#endif
    switch (sapphire_state) {
    case normal:
    case stabilize_phase: {
        break; } 
    case mark_reserve_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) { 
            if (sapphire_object_color (p_object_with_slot) == black) {
                sapphire_make_int32_slot_consistent (get_from_object (p_object_with_slot),
                    offset, 
                    value);
                return;
            }
        }
        
        break; 
                          } // end mark_reserve_phase

    case repoint_phase: 
    case flip_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) {
            assert (sapphire_object_color(p_object_with_slot) == black);
            sapphire_make_int32_slot_consistent (get_from_object (p_object_with_slot), 
                offset, 
                value);
            return;
        }
        break; }
    } // end switch
    // If we are here there is only one copy that has to be updated, so do it.
    int32 *p_slot = 
        (int32 *)(((char *)p_object_with_slot) + offset);    
    *p_slot = value;
}


void sapphire_make_int64_slot_consistent (Java_java_lang_Object *p_from_obj, 
                                          unsigned offset, 
                                          int64 value)
{
    assert (is_object_in_from_space (p_from_obj));
    assert (sapphire_object_color (p_from_obj) == black); // It should be black
    Java_java_lang_Object *p_to_obj = get_to_object (p_from_obj);
    volatile int64 *p_from_slot = 
        (volatile int64 *)(((char *)p_from_obj) + offset);
    volatile int64 *p_to_slot = 
        (volatile int64 *)(((char *)p_to_obj) + offset);
    *p_from_slot = value;
    *p_to_slot = value;
    while (*p_from_slot != *p_to_slot) {
        *p_from_slot = value;
        *p_to_slot = value;
    }
    gc_trace (p_from_obj, "Making from object int64 consistent.");
    gc_trace (p_to_obj, "Making to object int64 consistent.");
    return;
}

void gc_sapphire_heap_write_int64 (Java_java_lang_Object *p_object_with_slot,
                                   unsigned offset,
                                   int64 value)
{
#ifdef GC_FIXED_V1
    int64 *p_slot = 
        (int64 *)(((char *)p_object_with_slot) + offset);
    *p_slot = value;
    return; 
#endif
    switch (sapphire_state) {
    case normal:
    case stabilize_phase: {
        break; } 
    case mark_reserve_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) { 
            if (sapphire_object_color (p_object_with_slot) == black) {
                sapphire_make_int64_slot_consistent (get_from_object (p_object_with_slot),
                    offset, 
                    value);
                return;
            }
        }
        
        break; 
                          } // end mark_reserve_phase

    case repoint_phase: 
    case flip_phase: {
        if (is_object_in_copy_space (p_object_with_slot)) {
            assert (sapphire_object_color(p_object_with_slot) == black);
            sapphire_make_int64_slot_consistent (get_from_object (p_object_with_slot), 
                offset, 
                value);
            return;
        }
        break; }
    } // end switch
    // If we are here there is only one copy that has to be updated, so do it.
    int64 *p_slot = 
        (int64 *)(((char *)p_object_with_slot) + offset);    
    *p_slot = value;
}

// There are some global slots that are shared by different threads. Sapphire 
// needs to know about writes to these slots. One example of such slots is in
// the string pools used by the class loader.
void gc_sapphire_heap_write_global_slot(Java_java_lang_Object **p_slot,
                                        Java_java_lang_Object *value)
{
    assert (p_slot != NULL);
    switch (sapphire_state) {
    case stabilize_phase:
    case normal: 
        {	
            *p_slot = value;
            // Do nothing object will be enumerated when the time comes.
            break; 
        }
    case mark_reserve_phase: 
        {
            // Nobody can see new space yet.
            // The slot is global and not in the heap.
            // Writes of values referring to white objects in C space need to be grayed.
            // Writes to values referring to object in U space need to scanned.
            *p_slot = value; 
            
            // The slot is in U space. gray value;
            if (is_object_in_c_space (value)) { 
                // object is in C (collected) space
                switch (sapphire_object_color (value)) {
                case white: 
                    {
                        // reserve some new space and set up pointers.
                        ssb_write_object((ssb_container *)p_TLS_orpthread->ssb_container, value);
                        break;
                    }
                case gray: 
                    {
                        assert (is_object_in_from_space (value));
                        break;
                    }
                case black:
                    { 
                        assert (is_object_in_los (value) || is_object_in_from_space (value));
                        // Object is already block so we don't have to do anything.
                        break;
                    }
                }  // end switch
            } // end if
            break;
        } // end mark_reserve_phase
        
    case repoint_phase: 
    case flip_phase: 
        {
            // If the object (value)is in U space make sure it points to TO space.
            if (value != NULL) {
                if (is_object_in_c_space(value)){
                    // Repoint it to the correct space.
                    *p_slot = get_to_object(value);
                }
            }
            break;
        }
    } // end switch
}

// The following should generate a st.rel followed by a mf to get sequential consistency
// for volatiles. As of June 12, 2000  this is my (RLH) assumption of what the ORP spec 
// will/should be.
void gc_sapphire_volatile_heap_write_ref (Java_java_lang_Object *p_base_of_object_with_slot,
                                          unsigned offset,
                                          Java_java_lang_Object *value)
{
    gc_sapphire_heap_write_ref (p_base_of_object_with_slot, offset, value);
}

// Non reference writes caller does conversion.

void gc_sapphire_volatile_heap_write_int8 (Java_java_lang_Object *p_base_of_object_with_slot,
                                           unsigned offset,
                                           int8 value)
{
    gc_sapphire_heap_write_int8 (p_base_of_object_with_slot, offset, value);
}

void gc_sapphire_volatile_heap_write_uint16 (Java_java_lang_Object *p_base_of_object_with_slot,
                                             unsigned offset,
                                             uint16 value)
{
    gc_sapphire_heap_write_uint16 (p_base_of_object_with_slot, offset, value);
}

void gc_sapphire_volatile_heap_write_int16 (Java_java_lang_Object *p_base_of_object_with_slot,
                                            unsigned offset,
                                            int16 value)
{
    gc_sapphire_heap_write_int16 (p_base_of_object_with_slot, offset, value);
}


void gc_sapphire_volatile_heap_write_int32 (Java_java_lang_Object *p_base_of_object_with_slot,
                                            unsigned offset,
                                            int32 value)
{
    gc_sapphire_heap_write_int32 (p_base_of_object_with_slot, offset, value);
}


void gc_sapphire_volatile_heap_write_float (Java_java_lang_Object *p_base_of_object_with_slot,
                                            unsigned offset,
                                            float value)
{
    gc_sapphire_heap_write_float (p_base_of_object_with_slot, offset, value);
}


void gc_sapphire_volatile_heap_write_double (Java_java_lang_Object *p_base_of_object_with_slot,
                                             unsigned offset,
                                             double value)
{
    gc_sapphire_heap_write_double (p_base_of_object_with_slot, offset, value);
}


void gc_sapphire_volatile_heap_write_int64 (Java_java_lang_Object *p_base_of_object_with_slot,
                                            unsigned offset,
                                            int64 value)
{
    gc_sapphire_heap_write_int64 (p_base_of_object_with_slot, offset, value);
}

// End of new sapphire write barrier interface code.

// Bookkeeping code
//
// change_all_threads_sapphire_status 
// Purpose - iterate through the threads changing the sapphire_status in TLS.
// Args - state - the state to change all the threads to.
//
// Returns - nil
void change_all_threads_sapphire_status (gc_sapphire_state state)
{
    uint32 n_threads;
    unsigned n;
    n_threads = thread_gc_number_of_threads(); // init iterator
    for(n = 0; n < n_threads; n++) {
        ORP_thread *thread = (ORP_thread *)thread_gc_enumerate_one();
        thread->gc_sapphire_status = state;
    }
    p_TLS_orpthread->gc_sapphire_status = state;
}

//
// sapphire_object_color
// Input  Any object in C space.
//
// Output white if the object is in U space
//        white if the object is in C "from" space and no "to" space has been reserved
//        gray if space has been reserved but not yet scanned
//        black if space has been reserved and the contents have been copied and scanned 
//
sapphire_tri_color sapphire_object_color (Java_java_lang_Object *p_obj)
{
    assert (is_object_in_c_space(p_obj));

    if (p_global_bs->is_address_in_large_object_space((void *)p_obj)) {
        // it is only in from space if it isn't marked, marked objects are considered in to space.
        if (is_object_marked (p_obj)) {
            return black;
        } else {
            return white; // object in LOS are immediately scanned and turned black. 
        }
    }
    
    if (is_object_in_to_space(p_obj)) {
        // If the object is in to_space we need to get the from object
        // to determine the color from the sapphire_link_block
        p_obj = get_from_object(p_obj);
    }
    
    // Object is not in LOS.
    volatile POINTER_SIZE_INT *p_header = P_OBJ_INFO(p_obj);
    
    // Descriminate between Lock_Block and Sapphire_Link_Block
    if (is_sapphire_link_block((POINTER_SIZE_INT *)p_header)) {
        Sapphire_Link_Block *p_link_block;
        p_link_block = (Sapphire_Link_Block *)(*p_header & LOCK_BLOCK_POINTER_MASK);
        return p_link_block->sapphire_color;
    } else {
        // it is a Lock_Block so the color is white.
        return white;
    }
}

//
// sapphire_release_nursery
//
// Comment - An active thread needs to release its nursery into from space. This
//			 routine will release the nursery and then the nursery will be moved into
//			 the "from" c area.
//
void sapphire_release_nursery(ORP_thread *thread)
{
    Nursery *a_nursery = (Nursery *)thread->p_nursery;
    
    if (a_nursery == NULL)  { 
        // This thread doesn't have a nursery associated with it, perhaps
        // it is recycling the nursery.
        return;
    }
    assert (a_nursery != NULL);
    assert (a_nursery->active);
    a_nursery->active = true);
    if (!a_nursery->_is_nursery_clear) {
//        orp_cout << "Nursery is not clear" << endl;
        assert (thread->p_nursery == a_nursery);
        // This nursery has no objects in it yet so we don't need to migrate it to the focus car.
        return;
    }

    // Orphan a_nursery
    thread->p_nursery = NULL;

    thread->p_nursery = p_cycle_nursery(a_nursery, true);
    // a_nursery is on spent list even if a new nursery is not found.
    if (thread->p_nursery == NULL) {
        // We have a race condition, if the other threads use up all the
        // nurseries before we get there and we are trying to swap in a nursery
        // we are screwed and the thread we are trying to swap with will have to 
        // be stopped. (even though it might not be allocating anything.)
        // No nurseries are available ... do something radical ...
        // Try releasing the spent nurseries into the focus car.
        
        // a_nursery is on spent list even if a new nursery is not found.
        Car *p_focus_car = p_global_bs->get_mature_generation()->p_nominate_focus_car();
        p_global_bs->get_young_generation()->sapphire_evict_spent_nurseries_into_focus_car(p_focus_car);
        thread->p_nursery = p_cycle_nursery(NULL, true);
        if (! (thread->p_nursery)) {
            assert (0);   
        }
    }
    // Make sure the new one is empty.
    ((Nursery *)thread->p_nursery)->verify_empty_nursery();
    assert (thread->p_nursery);
}

//
// sapphire_focus
//
void sapphire_focus ()
{
    
    // Each thread is allocating objects in a nursery, scanning
    // the nursery for pointers will be more expensive than collecting them so 
    // we will move all the nurseries into the "from" space 
    // focus car. Any objects that are allocated from this point on will be in fresh nurseries
    // that are not part of "from" space. The sapphire write barrier will track all pointers 
    // into "from" u space during the time sapphire is running.
    
    // Another reason this approach is taken is that it means that no allocation is done in
    // MOS. IE only the GC moves objects into MOS "train" space. This is important
    // since it preserves the speed of allocation and sapphire is responsible for such things
    // as the buffer holding the start of the last object allocated into a card.
    //
    // This approach also means that every object in MOS is born gray and cheney scanned 
    // once as we blacken the gray objects.
    
    // POSSIBLE FUTURE OPTIMIZATION OPPORTUNITY
    // While not a concern here I (Rick) leave this note: Since all objects are cheney scanned
    // that are in MOS the last object in card logic could be changed to a first object in card
    // by having the Cheney Scan create a first object in card table. This would greatly simplify
    // the card scanning logic.
    // POSSIBLE FUTURE OPTIMIZATION OPPORTUNITY
    
    // Clear map to go from to space to from space object.
    
    to_from_map->empty_all();
    
    // Clear the mark bits used by the LOS.
#ifdef SAPPHIRE_TRACE
    orp_cout << "Clearing marks in sapphire_initialize_and_focus" << endl;
#endif
    p_global_bs->get_los_container()->clear_mark_bits();
}

void sapphire_initialize ()
{
    Car *p_focus_car = p_global_bs->get_mature_generation()->p_nominate_focus_car();

    // initialize c_space_ssb to empty
    if (!c_space_ssb) {
        c_space_ssb = (ssb_container *)malloc(sizeof(ssb_container));
        memset(c_space_ssb, 0, sizeof(ssb_container));
    }
    ssb_release(c_space_ssb);

    if (!c_space_ssb_collector) {
        c_space_ssb_collector = (ssb_container *)malloc(sizeof(ssb_container));
        memset (c_space_ssb_collector, 0, sizeof(ssb_container));
    }

    
    if (!ssb_repoint) {
        ssb_repoint = (ssb_container *)malloc(sizeof(ssb_container));
        memset (ssb_repoint, 0, sizeof(ssb_container));
    }
    ssb_release(ssb_repoint);
    ssb_release(c_space_ssb_collector);
    // The previous GC should have left the ssb with all NULLs.
    
    assert (c_space_ssb->ssb_read_ptr == NULL);
    assert (c_space_ssb->ssb_write_ptr == NULL);
    assert (c_space_ssb->current_ssb == NULL);
    assert (c_space_ssb->ssb_limit == NULL);
    assert (c_space_ssb->first_ssb == NULL);
    assert (c_space_ssb->last_ssb == NULL);

    
    assert (c_space_ssb_collector->ssb_read_ptr == NULL);
    assert (c_space_ssb_collector->ssb_write_ptr == NULL);
    assert (c_space_ssb_collector->current_ssb == NULL);
    assert (c_space_ssb_collector->ssb_limit == NULL);
    assert (c_space_ssb_collector->first_ssb == NULL);
    assert (c_space_ssb_collector->last_ssb == NULL);

#ifndef GC_FIXED_V1
    p_focus_car->sapphire_absorb_nurseries();
#else
    assert (global_safepoint_status == enumerate_the_universe);
    orp_cout << "No need to release nurseries in fixed collector." << endl;
#endif;
}

//
// sapphire_move_ssb
// Given a thread that is stopped move its ssb from the TLS to the
// c_space_ssb. This is called from orp_enum.
//
void sapphire_move_ssb(ORP_thread *p_orpthread)
{
    ssb_move_writer_to_reader((ssb_container *)p_orpthread->ssb_container, c_space_ssb);
}

//
// is_object_in_copy_space
// INPUT - any object
// OUTPUT - true if the object is in, either the "from" or "to" space.
//          false if it is in large object space (part of C space)
//                or in U space.
//
// This routine depends on all objects in "to" space not being white.
// 
bool is_object_in_copy_space (Java_java_lang_Object *p_obj)
{
    if (p_obj == NULL) {
        return false;
    }
    
    if (is_object_in_los(p_obj)) {
        return false;
    }

    // Is the object in C "from" space?
    if (is_object_in_from_space(p_obj)) {
        return true;
    }
    
    // If the object is white it is not in C "to" space.
    if (is_object_in_to_space(p_obj)) {
        return true;
    }
    // It's not in from space, to space or LOS so it is in U space.
    return false;
}

//
// is_object_in_c_space
// INPUT - any object
// OUTPUT - true if the object is in the focus area, either the "from" or "to" space.
//
// This routine depends on all objects in "to" space not being white.
// 
bool is_object_in_c_space (Java_java_lang_Object *p_obj)
{
    if (p_obj == NULL) {
        return false;
    }
    
    if (is_object_in_los(p_obj)) {
        return true;
    }

    // Is the object in C "from" space?
    if (is_object_in_copy_space(p_obj)) {
        return true;
    }
    
    // It's not in copy (from or to) space, or LOS so it is in U space.
    return false;
}

//
// is_object_in_u_space
// INPUT - any object
// OUTPUT - true if the object is not in c space (including not being in LOS).
//
// This routine depends on all objects in "to" space not being white.
// 
bool 
is_object_in_u_space (Java_java_lang_Object *p_obj)
{
    return !(is_object_in_c_space(p_obj));
}

//
//                                                                                               
// Input - the object to check
// Output - true if the object is in the focus area and in the "from" area
//        - false otherwise
//
bool 
is_object_in_from_space (Java_java_lang_Object *p_obj)
{
    // The from area of the focus_area is the focus car. 
    
    if (p_obj == NULL) {
        return false;
    }
    
    if (p_global_bs->get_mature_generation()->is_object_in_focus_car(p_obj))
    {
        return true;
    }
    
    return false;
}

//
// is_object_in_to_space
// Input - the object to check
// Output - true if the object is in the focus area and in the "to" area
//        - false otherwise
//
bool is_object_in_to_space (Java_java_lang_Object *p_obj)
{
    // only (and all) objects in to space are in the to_from_map
    return to_from_map->is_present(p_obj);
}

//
// is_object_white
// Input - the object to check
// Returns - true if the object is white.
// Comments - all objects in U space are white, all object in C "to" space are not white.
//            only objects in C "from" space that do not have "to" clones are white.
bool is_object_white (Java_java_lang_Object *obj)
{
    return (sapphire_object_color (obj) == white);
}

void turn_color_black (Java_java_lang_Object *p_obj)
{
    
    volatile POINTER_SIZE_INT *p_header = P_OBJ_INFO(p_obj);
    
    Sapphire_Link_Block *p_link_block;
    p_link_block = (Sapphire_Link_Block *)(*p_header & LOCK_BLOCK_POINTER_MASK);
    assert (p_link_block->sapphire_color != white);  // black can be turned black.
    p_link_block->sapphire_color = black;
}

// gc_sapphire_get_lock_object
// Monitor enter/exit needs to get the "to" object where the locking
// will take place. This routine takes any object and if the object is a 
// "from" space object returns the corresponding "to" space object, otherwise
// it just returns the object itself.
// 
// The caller should hold the busy_bit if he wants to avoid race conditions.
//
// For objects in from space sapphire_object_color will be white if there
// is not a legal Sapphire_Link_Block and it will be gray or black if there is one.
//   

Java_java_lang_Object *gc_sapphire_get_lock_object (Java_java_lang_Object *p_obj)
{
    if (sapphire_state == normal) {
        return p_obj;
    }
    
    if (sapphire_state == stabilize_phase) {
        return p_obj;
    }
    if (is_object_in_copy_space (p_obj)) {
        if (sapphire_object_color(p_obj) != white) {
            return get_to_object(p_obj);
        }
    }
    return p_obj;
}

//
// get_to_object
// Input  - an object in "from" space that has an object reserved in "to" space.
// Output - the "to" space object
// Comments - we don't need the busy bit since the lock block fields we
// are interested in are immutable.
// An object in from space that is not white will have a sapphire link block.
Java_java_lang_Object *get_to_object (Java_java_lang_Object *p_obj)
{
    assert (is_object_in_copy_space(p_obj));
    
    if (is_object_in_to_space(p_obj)){
#ifdef _DEBUG
        Car *p_focus_car = p_global_bs->get_mature_generation()->p_nominate_focus_car();;

        assert (!p_focus_car->is_address_in_this_car(p_obj));
#endif // _DEBUG
        return p_obj;
    }
    assert (is_object_in_from_space(p_obj));
    assert (sapphire_object_color(p_obj) != white);
    
    volatile POINTER_SIZE_INT *p_header = P_OBJ_INFO(p_obj);
    assert (is_sapphire_link_block((POINTER_SIZE_INT *)p_header));
    assert (*p_header != NULL);
    Sapphire_Link_Block *p_link_block;
    p_link_block = (Sapphire_Link_Block *)(*p_header & LOCK_BLOCK_POINTER_MASK);
    assert (p_link_block->sapphire_to_object != NULL);
    assert (!is_object_in_from_space(p_link_block->sapphire_to_object));
    return p_link_block->sapphire_to_object;
}

//
// get_from_object
// Input  - an object in either "from" space or "to" space.
// Output - the "from" space object
// Comments - we don't need the busy bit since the lock block fields we
// are interested in are immutable.
Java_java_lang_Object *get_from_object (Java_java_lang_Object *p_obj)
{
    if (is_object_in_to_space(p_obj)){
        Java_java_lang_Object *result = (Java_java_lang_Object *)to_from_map->get_value(p_obj);
        assert (get_to_object(result) == p_obj);
        return result;
    }
    assert (is_object_in_from_space(p_obj));
    return p_obj;
}


// 
// grab_busy_bit
// Input - the object
// Comments - Spin until the bit becomes available. If the object is in C "from" space
// and not white, go to the C "to" space object and grab its busy bit.
// During the mark_copy phase threads will only have "from" objects so we won't have
// deadlock problems since anyone trying to get the "to" busy bit will already have the
// "from" busy bit.
// During the repoint and flip phase, the "to" busy bit is the one that counts. 
// If a thread holds the "from" busy bit and not the "to" busy bit it must wait.
// If a thread does not hold the "from" busy bit it can still hold the "to" busy bit and not wait.
//
void grab_busy_bit (Java_java_lang_Object *p_obj)
{
    //	spin_lock_get (p_obj);
    Object_Gc_Header *p_header = P_OBJ_INFO(p_obj);
    ACQUIRE_BUSY_BIT(p_header);
    if (is_object_in_from_space(p_obj)) {
        if (is_object_white(p_obj)) {
            // Race condition - sapphire_state can be changed between these statements.
            //			assert (sapphire_state == mark_copy_phase);
            return; // There is no "to" object so we have the only busy_bit.
        }
        Java_java_lang_Object *to_object = get_to_object(p_obj);
        // spin_lock_get (to_object);
        p_header = P_OBJ_INFO(to_object);
        ACQUIRE_BUSY_BIT(p_header);
        return;	
    }
    return; // The object is either in new space or in U space.
}

// release_busy_bit
// Input the object that was used to grab_busy_bit
// Comment - If this is a from object release both the from bit and the to bit.
void release_busy_bit (Java_java_lang_Object *p_obj)
{
    volatile POINTER_SIZE_INT *p_header = P_OBJ_INFO(p_obj);
    volatile POINTER_SIZE_INT *p_to_header;
    assert (*p_header & BUSY_FORWARDING_BIT);
    
    if (is_object_in_from_space(p_obj)) {
        
        if (is_object_white(p_obj)) {
            assert (sapphire_state == mark_reserve_phase);
            RELEASE_BUSY_BIT(p_header);
            return; // There is no "to" object so we have the only busy_bit.
        }
        // The order we release does not make a difference since another thread
        // will need to get the to_object lock before it can do anything.
        p_header = P_OBJ_INFO (p_obj);
        
        Java_java_lang_Object *to_object = get_to_object(p_obj);
        p_to_header = P_OBJ_INFO (to_object);
        RELEASE_BUSY_BIT(p_header); // Release "from object"
        RELEASE_BUSY_BIT(p_to_header); // Release "to" object
        return;	
    }
    
    p_header = P_OBJ_INFO (p_obj);
    RELEASE_BUSY_BIT(p_header);
    
    return; // The object is either in new space or in U space.
}


// Forward reference - use of Sapphire_Link_Block described below.
Sapphire_Link_Block *get_a_sapphire_link_block();
void fixup_lock_block_object_reference(Java_java_lang_Object *p_to_obj);

//
// sapphire_reserve_to_space
//
// Input  A white object in "from" space that the caller holds the busy bit on. 
//		  The train to reserve the space in.
//
// Output A pointer to an object the "from" objects replica in "to" space.
//        The "to" object has not been initialized.
//
// Comment The busy bit is held on *pp_slot, the "from" object so others can't interfere with this 
//         operation. The caller will need to grab the busy bit on the "to" object returned.
//		   The "to" object is complete in the sense that it has a header and a vtable pointer.
//         The fields are not filled in the object here.
//		   The to_train is passed in and if the slot is not in the focus area it is the same
//         train that slot is in. If slot is in the focus area and in from space then the space
//         is reserved in the "to" space train. To avoid race condition problems
//         we pass the p_obj in "from" space that needs to have space reserved for it.
//         Since the mutator could do an asynchronize modification of p_slot we need
//         to make sure that is not a problem.
//
// Why pp_slot and NOT p_obj? answer - 
// Since the selection of where to move the object is dependent on where the
// slot holding the reference to the object is we need to have the slot of interest.
// In pragmatic terms sapphire_select_to_train needs to get the pp_slot.
//

Java_java_lang_Object *sapphire_reserve_to_space (Java_java_lang_Object **pp_slot, 
                                                  Java_java_lang_Object *p_obj)
{
    assert (p_obj->vt);
    //	orp_cout << "Reserving TO space." << endl;
    gc_trace_slot ((void **)pp_slot, (void *)*pp_slot, " Slot refent has to space reserved.");
    gc_trace (*pp_slot, "Sapphire reserving to space for this from object");
    // First make sure everything is as we expect.
    
    volatile POINTER_SIZE_INT *p_header = P_OBJ_INFO(p_obj);
    
    volatile POINTER_SIZE_INT old_header_value = *p_header;
    
    assert (is_object_in_c_space(p_obj));
    assert (is_object_in_from_space(p_obj));
    assert (is_object_white(p_obj));
    assert ((*p_header & BUSY_FORWARDING_BIT) == BUSY_FORWARDING_BIT); 
    // By convention the caller 
    // has acquired the lock on the from object and will release the lock.
    
    // The slot could be in the the heap or on a stack or in some global.
    // Figure out which train the "to" space is in reserve space in that train.
    Train *to_train = Train::sapphire_select_to_train (pp_slot);
    Java_java_lang_Object *p_to_obj = to_train->p_reserve_space(*pp_slot);
    assert (!is_object_in_from_space(p_to_obj));
    
    Car *p_focus_car = p_global_bs->get_mature_generation()->p_nominate_focus_car();
    assert (!p_focus_car->is_address_in_this_car(p_to_obj));
    // move vtable over
    p_to_obj->vt = p_obj->vt;
    // move header value over with BUSY_FORWARDING_BIT set.
    
    volatile POINTER_SIZE_INT *p_to_header = P_OBJ_INFO(p_to_obj);
    
    *p_to_header = old_header_value;
    orp_fixup_monitor_lock ((void *)P_OBJ_INFO_FROM_START(p_header), (void *)P_OBJ_INFO_FROM_START(p_to_header));
    
    // At this point the "to" object header is ready for business as soon
    // as it releases the BUSY_FORWARDING_BIT
    to_from_map->add_key_value(p_to_obj, p_obj); // p_obj is the from object.
    // The "to" object can now find the corresponding "from" object
    
    // Get a Sapphire_Link_Block which will hold the "to" pointer, the "from" pointer and
    // the lock header shared by the from and to objects.
    Sapphire_Link_Block *linkage = get_a_sapphire_link_block();
    
    linkage->sapphire_to_object = p_to_obj;
    linkage->sapphire_from_object = p_obj;
    linkage->sapphire_color = gray;
    
    // Set up the header value by oring in the SLOW_LOCKING bit sequence as well as
    // the lock bit.
    linkage = (Sapphire_Link_Block *)(((POINTER_SIZE_INT)linkage) | SLOW_LOCKING);
    linkage = (Sapphire_Link_Block *)(((POINTER_SIZE_INT)linkage) | BUSY_FORWARDING_BIT);
    assert (linkage != NULL);
    
    *p_header = (POINTER_SIZE_INT)linkage;
    assert ((*p_header & BUSY_FORWARDING_BIT) == BUSY_FORWARDING_BIT);
    assert ((*p_to_header & BUSY_FORWARDING_BIT) == BUSY_FORWARDING_BIT); 
    gc_trace (p_to_obj, "To space reserved.");
    return p_to_obj;
}
//
// sapphire_using_this_block
// 
// Comment - This routine is called by the thread codes normal free_this_block routine.
//           If the block is being used by sapphire return true. Notice that sapphire only
//			 uses the first block on the list so it returns false if the block isn't the
//			 first one on the list.
bool sapphire_using_this_block (ORP_thread *p_orpthread, Sapphire_Link_Block *p_block)
{
    if (p_block->sapphire_color == white) {
        return false;
    }
    return true;
}

// Sapphire_Link_Blocks are provided so that if we are holding a "from" object
// then we can locate the "to" object through the Sapphire_Link_Block. The pointer to
// the Sapphire_Link_Block is held in the header of the object usually reserved for
// hash and lock bits. monitor_enter and monitor_exit use the link block to locate the
// "to" object and manipulate the header in the "to" object.
//
// Since the header in the "to" object is identical to the headers used in normal case
// it can be optimized. However, to maintain consistency when one writes in a "to" object
// one needs to locate the related "from" object. This is done using a external data
// structure that is indexed using the address of the "to" object. It is not important
// how this mapping is done, one can imagine a hash table, the important thing is that 
// it does not require any space in the "to" object so that "to" objects can run at
// full speed for synchronization and so forth.

// Manipulation of this block needs to acquire the p_sapphire_link_block_lock before
// adding or removing any sapphire link blocks.
volatile Sapphire_Link_Block *p_free_sapphire_link_blocks = NULL;
volatile Sapphire_Link_Block *p_active_sapphire_link_blocks = NULL;
volatile Sapphire_Link_Block *p_last_active_sapphire_link_block = NULL;

//
// Given an object header does it point to a sapphire_link_block.
// This is pretty hacky code and a better but equally efficient way to do
// this would be nice if anyone has a suggestion.
//
// First if the SLOW_LOCKING isn't set then we know we don't have a pointer
// and we can't be pointing to a sapphire_link_block.
// If SLOW_LOCKING is set then we could be pointing to either a sapphire_link_block or
// we could be pointing to a lock block. The first field in a lock block points back to
// the object being locked. The first field in a sapphire_link_block points to the
// "to" version of the object. We use this fact to determine if we have a 
// sapphire_link_block.
//

// This is a best guess given that if the busy bit is not held when this 
// is called there is a race condition where lock block could be about to
// be installed.
bool is_sapphire_link_block (POINTER_SIZE_INT *p_header)
{
    if ((*p_header & SLOW_LOCKING) == SLOW_LOCKING) {
        // The color is in the lock block.
        Sapphire_Link_Block *p_link_block;
        p_link_block = (Sapphire_Link_Block *)(*p_header & LOCK_BLOCK_POINTER_MASK);
        if (p_link_block->discriminator == sapphire_link_block) {
            return true; // we have a link block
        }
    }
    // else we do not have either a sapphire link block or a lock block.
    return false;
}

// 
// Sapphire_get_link_block
// 
// Comment - Provides a link block from a central repository.
// This requires some sort of locking or each thread having it's own
// lock blocks.
// 

Sapphire_Link_Block *get_a_sapphire_link_block()
{
    volatile Sapphire_Link_Block *p_block;
    // Enter a critical section to protect the free list.
    p_sapphire_link_block_lock->_lock();
    if (p_free_sapphire_link_blocks)  {
        p_block = p_free_sapphire_link_blocks;
        assert (!p_block->sapphire_using);
        p_free_sapphire_link_blocks = 
            p_free_sapphire_link_blocks->p_next_sapphire_link_block;
        //        orp_cout << "reusing block " << (void *)p_block << endl;
    }
    else {
        // Malloc up a new link block and use it.
        //#ifdef _DEBUG
        //        p_orpthread->number_of_lock_blocks_allocated++;
        //#endif
        // WHY DO WE ADD 64 BYTES TO THE END OF THE Sapphire_Link_Block so that
        // the the address will always have lower bits of zero since the header
        // pointing to it needs these bits for the recursion count.
        POINTER_SIZE_INT xx;
        xx = (POINTER_SIZE_INT)malloc( sizeof(Sapphire_Link_Block) + 64); 
        memset( (void *)xx, 0, ( sizeof(Sapphire_Link_Block) + 64) );
        // next, align so that the 5 address LSB's are zero
        // ASSUMPTION: we never call free() on this malloc'ed memory
        xx = xx + 31;
        xx = xx & ~31;
        assert((xx & 31) == 0);
        p_block = (Sapphire_Link_Block *)xx;
        //         orp_cout << "allocated block " << (void *)p_block << endl;
        
        //printf("get_a_block:: p_orpthread = 0x%x  p_block = 0x%x\n", p_orpthread, p_block);
    }
    p_block->discriminator = sapphire_link_block;
    p_block->sapphire_to_object = NULL;
    p_block->sapphire_from_object = NULL;
    p_block->p_next_sapphire_link_block = (Sapphire_Link_Block *)p_active_sapphire_link_blocks;
    if (p_active_sapphire_link_blocks == NULL) {
        // This is the first block allocated and since we add objects to the front
        // of the list it will always be the last block. Save it so that it can be 
        // relinked onto the free list without having to traverse the list.
        p_last_active_sapphire_link_block = p_block;
    }
    p_active_sapphire_link_blocks = p_block;
    
    // Fields of interest to sapphire.
    p_block->sapphire_color = gray;
    
    //    orp_cout << p_block << " link block allocated." << endl;
#ifdef _DEBUG
    p_block->sapphire_using = true;
#endif
    p_sapphire_link_block_lock->_unlock();
    return (Sapphire_Link_Block *)p_block;
}

//
// sapphire_free_all_link_blocks
//
// Comment - Since the "to" objects do not use the p_active_sapphire_link_blocks
//           and the "from" objects are about to be released we simple
//           link this list of objects onto the free list and we are done with it.
//
void sapphire_free_all_link_blocks ()
{
    p_sapphire_link_block_lock->_lock();
#ifdef _DEBUG
    int count = 0;
    volatile Sapphire_Link_Block *p_active_list = (Sapphire_Link_Block *)p_active_sapphire_link_blocks;
    while (p_active_list != p_last_active_sapphire_link_block) {
        assert (p_active_list->sapphire_color == black);
        assert (is_object_in_c_space(p_active_list->sapphire_from_object));
        assert (is_object_in_from_space(p_active_list->sapphire_from_object));
        assert (!is_object_in_from_space(p_active_list->sapphire_to_object));
        assert (p_active_list->sapphire_using = true);
        p_active_list->sapphire_using = false;
        p_active_list->sapphire_to_object = NULL;
        p_active_list->sapphire_from_object = NULL;
        p_active_list->sapphire_color = white;
        assert (p_active_list->p_next_sapphire_link_block != NULL);
        p_active_list = p_active_list->p_next_sapphire_link_block;
        count++;
    }
    if (p_active_list != NULL) {
        // This is probable not possible but if there are not live objects in old space then
        // and empty p_active_list can happen.
        assert (p_active_list->sapphire_color == black);
        assert (is_object_in_c_space(p_active_list->sapphire_from_object));
        assert (is_object_in_from_space(p_active_list->sapphire_from_object));
        assert (!is_object_in_from_space(p_active_list->sapphire_to_object));
        assert (p_active_list->sapphire_using = true);
        p_active_list->sapphire_using = false;
        p_active_list->sapphire_to_object = NULL;
        p_active_list->sapphire_from_object = NULL;
        p_active_list->sapphire_color = white;
        assert (p_active_list->p_next_sapphire_link_block == NULL);
    }
    count++;
//    orp_cout << "Released " << count << " lock blocks." << endl;
#endif // _DEBUG
    /**
    orp_cout << "free link blocks p_free_sapphire_link_blocks is " 
    << (void *)p_free_sapphire_link_blocks << endl;
    
      orp_cout << "free link blocks p_last_active_sapphire_link_block is " 
      << (void *)p_last_active_sapphire_link_block << endl;
      orp_cout << "free link blocks p_active_sapphire_link_blocks is " 
      << (void *)p_active_sapphire_link_blocks << endl;
    **/
    assert (p_last_active_sapphire_link_block->p_next_sapphire_link_block == NULL);
    p_last_active_sapphire_link_block->p_next_sapphire_link_block = p_free_sapphire_link_blocks;
    p_free_sapphire_link_blocks = p_active_sapphire_link_blocks;
    p_last_active_sapphire_link_block = NULL; // We have no active blocks
    p_active_sapphire_link_blocks = NULL;  
    p_sapphire_link_block_lock->_unlock();  
}

void sapphire_release_old_c_area() 
{
//    orp_cout << "Releasing all the lock blocks." << endl;
    sapphire_free_all_link_blocks ();
    Train *p_focus_train = p_global_bs->get_mature_generation()->p_nominate_focus_train();
    assert (p_focus_train);
    p_focus_train->_free_oldest_car();
    // Relink LOS free list(s).
    p_global_bs->get_los_container()->cleanup();
    return; 
}

//
// sapphire_gray_white_object 
// Input a slot holding a white object in the C "from" area. 
//
// Comments - We need to grab the busy bit in the header since we might have a race condition
// between two different threads trying to gray the same object.
//
void sapphire_gray_white_object (Java_java_lang_Object **pp_slot, Java_java_lang_Object *p_obj)
{
    //    assert(is_object_white(*pp_slot)); // What about LOS rescans?
   	gc_trace ((void *)p_obj, "in sapphire_gray_white_object with slot and this obj."); 
    gc_trace_slot ((void **)pp_slot, (void *)p_obj, "graying white object referred to in slot.");

    if (!is_object_in_c_space(p_obj)) {
        //        orp_cout << "g";
        return;
    }
    assert (p_obj->vt);
    grab_busy_bit(p_obj);
    assert (is_header_locked (p_obj));
    if (p_global_bs->is_object_in_large_object_space(p_obj)) {
        // If an object is in LOS and not marked at the top of the routine then
        // it could be marked by the time we get here by another thread. If that
        // is the case then we still need to scan it.
        set_object_marked (p_obj);
        // Mark the object black and scan it immediately
        // Objects in LOS are never marked gray.
        release_busy_bit(p_obj);
        sapphire_scan_graying_white(p_obj);
        return;
    }
    
    // The object could already be gray due to race conditions. If so do not
    // reserve more space for it.
    
    if (is_object_white(p_obj)) {

        // We have a race if another thread
        // grays this object before we have the
        // busy bit. Inspect and remove the assert 
        // release the busy bit and just return
        // If this assert is hit.
    
        Java_java_lang_Object *to_object = sapphire_reserve_to_space (pp_slot, p_obj); 
        sapphire_make_reserve_phase_consistent(p_obj);
        assert (!is_object_in_from_space(to_object));
        assert (is_header_locked(to_object));
        release_busy_bit(p_obj);
        // Scan the "to" version so that slots to "from" versions 
        // will be remembered for repointing.
        sapphire_scan_graying_white(to_object);
        turn_color_black(p_obj);
        return;
    }

    // The rules are that if there is a to object, that is where the locking information
    // is kept.
    release_busy_bit(p_obj);
}

//
// sapphire_array_repoint
//
// Input - an object of type array.
//
// Comment - if it is a primitive array (one without pointers) then
//           init_array_scanner and p_get_array_ref will not produce any
//           descendent (and do it quickly)
//
void sapphire_array_repoint (Java_java_lang_Object *p_array_obj)
{
    Partial_Reveal_Class *p_class = (*((Partial_Reveal_VTable **)p_array_obj))->clss;
    // delete    Partial_Reveal_Class *p_class = get_object_class(p_array_obj);
    // There is no point in scanning an array of primitives for references
    // since there won't be any, (and init_array_scanner asserts on array of primitives!)
    if (is_array_of_primitives(p_array_obj)) {
        return;
    }
    
    // Initialize the array scanner which will scan the array from the
    // top to the bottom. IE from the last element to the first element.
    // The scanner only deals with pointer elements.
    unsigned int offset = init_array_scanner (p_array_obj);
    // Cycle through all the descendents.
    Java_java_lang_Object **pp_child_ref;
    int length = ((JavaArray *)p_array_obj)->length;
    int count = 0;
    while ((pp_child_ref = p_get_array_ref(p_array_obj, offset)) != NULL) {
        count++;
        offset = next_array_ref (offset);
        Java_java_lang_Object *p_target = *pp_child_ref;
        if (p_target != NULL) {
            // The object is somewhat interesting.
            if (is_object_in_copy_space(p_target)) {
                Java_java_lang_Object *to_obj = get_to_object(p_target);
                // If a mutator thread has not changed this slot replace the
                // from slot with the to slot. If a mutator has changed this
                // slot just move on.
                InterlockedCompareExchangePointer ((void **)pp_child_ref, (void *)to_obj, (void *)p_target);
            }
        }
    }
    assert (count == length);
}

//
// sapphire_repoint
//
// Input  an object
// Comments - scan the object looking for from space versions and repoint them
// to the "to" space version.
//
void sapphire_repoint (Java_java_lang_Object *p_obj)
{
#ifdef _DEBUG
    if (is_object_in_los(p_obj)) {
        assert (is_object_marked(p_obj));
    }
#endif // _DEBUG

    // If we are asked to repoint a from object, also repoint the to object

    Java_java_lang_Object *p_to_obj;
    if (is_object_in_from_space(p_obj)) {
        assert (!is_object_in_los(p_obj));
        p_to_obj = get_to_object(p_obj);
        sapphire_repoint(p_to_obj);
    }

    if (is_array(p_obj)) {
        sapphire_array_repoint (p_obj);
        return;
    }

    // Cycle through all the descendents atomically changed "from" pointers
    // to "to" pointers.
    unsigned int *offset_scanner = init_object_scanner (p_obj);
    Java_java_lang_Object **pp_child_ref;
    while ((pp_child_ref = p_get_ref(offset_scanner, p_obj)) != NULL) {
        // Move the scanner to the next reference.
        offset_scanner = p_next_ref (offset_scanner);
        Java_java_lang_Object *p_target = *pp_child_ref;
        if (p_target != NULL) {
            if (is_object_in_copy_space(p_target)) {
                Java_java_lang_Object *to_obj = get_to_object(p_target);
                // If a mutator thread has not changed this slot replace the
                // from slot with the to slot. If a mutator has changed this
                // slot just move on.
                InterlockedCompareExchangePointer ((void **)pp_child_ref, (void *)to_obj, (void *)p_target);
            }
        }
    }
}

//
// sapphire_array_scan_graying_white
//
// Input - an object of type array.
//
// Comment - if it is a primitive array (one without pointers) then
//           init_array_scanner and p_get_array_ref will not produce any
//           descendent (and do it quickly)
//
void sapphire_array_scan_graying_white (Java_java_lang_Object *p_array_obj)
{
    Partial_Reveal_Class *p_class = get_object_class(p_array_obj);
    // There is no point in scanning an array of primitives for references
    // since there won't be any, (and init_array_scanner asserts on array of primitives!)
    if (is_array_of_primitives(p_array_obj)) {
        return;
    }
    
    // Initialize the array scanner which will scan the array from the
    // top to the bottom. IE from the last element to the first element.
    // The scanner only deals with pointer elements.
    unsigned int offset = init_array_scanner (p_array_obj);
    // Cycle through all the descendents.
    Java_java_lang_Object **pp_child_ref;
    while ((pp_child_ref = p_get_array_ref(p_array_obj, offset)) != NULL) {
        offset = next_array_ref (offset);
        Java_java_lang_Object *p_child_obj = *pp_child_ref;
        // Skip null entries.
        if (p_child_obj != NULL) {
            if (is_object_in_c_space(p_child_obj)) {
                if (is_object_white(p_child_obj)){
                    // This is a pointer to a white object, gray it.
                    sapphire_gray_white_object(pp_child_ref, p_child_obj);
                }
                ssb_write_slot(c_space_ssb_collector, pp_child_ref);
                
                ssb_write_slot(ssb_repoint, pp_child_ref);
            }
        }
    }
}

//
// sapphire_scan_graying_white
//
// Input  an object
// Comments - scan the object looking for white referents into C.
// If the referent object is in LOS and unmarked - mark and scan graying white c objects.
// If the referent object is in from space and does not have a "to" version
//     reserve to space, copy the bits and scan_graying_white_c_object.
//     The "to" version of the object will not point into "to" space.
//     
// Assert that the referent object is not in "to" space.
//
// The object being scanned can be any color or in any space.
//
void sapphire_scan_graying_white (Java_java_lang_Object *p_obj)
{    
    if (is_array(p_obj)) {
        sapphire_array_scan_graying_white (p_obj);
        return;
    }
    // Cycle through all the descendents.
    unsigned int *offset_scanner = init_object_scanner (p_obj);
    Java_java_lang_Object **pp_child_ref;
    while ((pp_child_ref = p_get_ref(offset_scanner, p_obj)) != NULL) {
        // Move the scanner to the next reference.
        offset_scanner = p_next_ref (offset_scanner);
        Java_java_lang_Object *p_child_obj = *pp_child_ref;
        assert (p_obj->vt);
        if (p_obj != NULL) {
            // The object is somewhat interesting.
            if (is_object_in_c_space(p_child_obj)) {
                if (is_object_white(p_child_obj)){
                    // This is a pointer to a white object, gray it.
                    sapphire_gray_white_object(pp_child_ref, p_child_obj);
                }
                ssb_write_slot(c_space_ssb_collector, pp_child_ref);
                ssb_write_slot(ssb_repoint, pp_child_ref);
            }
        }
    }
}

//
// sapphire_blacken_gray_object
//
// We have an object that needs to be turned from gray to black. Since
// This object was found while cheney scanning "to" space make sure we
// are scanning such an object.
//
void sapphire_blacken_gray_object (Java_java_lang_Object *p_the_obj)
{
   	if (p_global_bs->is_object_in_large_object_space(p_the_obj)) {
        assert (is_object_marked(p_the_obj));
        return; // LOS objects are scanned and blackened as the are turned gray.
    }
    
    // We can end up here if when we are cheney scanning "to" space objects
    // so we need to figure out where the "from" objects are as well as
    // the to objects.
    assert (sapphire_object_color (p_the_obj) != white);
    Java_java_lang_Object *p_to_obj = p_the_obj;
    if (is_object_in_from_space(p_the_obj)) {
        p_to_obj = get_to_object(p_the_obj);
    }
    assert (sapphire_object_color (p_to_obj) != white);
    Java_java_lang_Object *p_from_obj = get_from_object(p_the_obj);
    assert (sapphire_object_color (p_from_obj) != white);
    
    gc_trace (p_to_obj, "blackening gray object.");
    assert (!is_object_in_from_space (p_to_obj));
    assert (is_object_in_from_space (p_from_obj));
    assert (p_to_obj != p_from_obj);
    //	assert (sapphire_object_color (p_from_obj) == gray); write barrier can write to black
    turn_color_black(p_from_obj);
    sapphire_make_reserve_phase_consistent (p_from_obj);
}

// 
// sapphire_if_gray_black_object
// input -   the object to blacken if it is already gray.
//           a remembered set that is ignored.
// Comment - Passed to the walk routine of the focus "from" car.
//
bool sapphire_if_gray_blacken_object (Object_Gc_Header *p_hdr, Remembered_Set *ignore)
{
    Java_java_lang_Object *p_obj = P_OBJ_FROM_START(p_hdr);
    if (sapphire_object_color (p_obj) == gray) {
        sapphire_blacken_gray_object (p_obj);
    }
    return true;
}

//
// When we execute a cheney scan all objects being scanned end up here.
//
void sapphire_cheney_scan_execute_policy(Java_java_lang_Object *p_obj, bool doing_mos_collection)
{	
    volatile POINTER_SIZE_INT *p_header = P_OBJ_INFO(p_obj);
    Remembered_Set *ignore = NULL;
    sapphire_if_gray_blacken_object ((Object_Gc_Header *)p_header, ignore);
}

//
// sapphire_blacken_gray_objects ()
//
// Comments - Cheney scans "to" space locating gray objects, as we find them
// call sapphire_blacken_gray_object. Since "to" space is distributed throughout
// various cars this routine might have to scan the cars multiple times.
//
void sapphire_blacken_gray_objects ()
{
    //	Generation::rewind_cheney_scan();
    p_global_bs->get_mature_generation()->execute_pending_cheney_scans(true);
    Car *p_focus_car = p_global_bs->get_mature_generation()->p_nominate_focus_car();
    bool ignore = p_focus_car->walk_car (sapphire_if_gray_blacken_object);
}

//
// sapphire_make_repoint_flip_phase_consistent
// 
//
// Comment - Copy the object into the new space updating any pointers to point to new space.
// This routine is called with object in old space and it already has space reserved 
// for it in new space. 
//

void sapphire_make_repoint_flip_phase_consistent (Java_java_lang_Object *p_obj, 
                                                  Java_java_lang_Object *p_shadow_obj)
{    
    if (is_object_in_los(p_obj)) {
        return;
    }
    // It should have just been set to black 
    assert (sapphire_object_color (p_obj) == black); 
    
    // First we repoint p_obj to point to "to" objects. Then we copy the bits over.
    sapphire_repoint(p_obj);

    // Copy code is good for both arrays and objects.
    char *end_of_object = (char *)p_obj + 
        get_real_object_size_bytes (p_obj) - OBJECT_HEADER_SIZE;
    char *obj_scan_point = (char *)p_obj;    // Scan pointer in obj object
    char *shadow_scan_point = (char *)p_shadow_obj;		 // Scan pointer in shadow object
    assert (p_obj->vt == p_shadow_obj->vt); // Should have been set when "to" replica made.
    // Someday use memcopy and skip vt copy for speed.
    while (obj_scan_point < end_of_object) {
        *shadow_scan_point = *obj_scan_point;
        obj_scan_point++;
        shadow_scan_point++;
    }
}

//
// The object passed to this routine will be in "from" space and have
//    a "to" version already reserved with properly installed headers in
//    both the "to" version and the "from" version.
// Reserve phase consistent means that after this has been run then
// If the object is in LOS it has been marked, scanned and does not point to 
//    any white "c" objects.
// If the object is in copy space then there exists both a to and a from version of 
//    the objects and the vtable and all fields are bit equal. This implies that
//    references in both the to and from versions point to the from versions.
//    The header of the from version points to a sapphire link block and the
//    header of the to version holds the lock header.
// 
void sapphire_make_reserve_phase_consistent (Java_java_lang_Object *p_from_obj)
{
    // First copy the raw bits from "from" space into "to" space. This could include
    // pointers into "from" space that now reside in "to" space. This violates the no
    // "to" space to "from" space pointer invariant. Fortunately, this routine is only called
    // before any thread engine can see "to" space and the pointers are updated immediately.
    // So why don't we just intercept the pointers before we write them into "to" space. The reason
    // is simple the mechanisms that decide where to move objects are based on the location of
    // the slot holding a pointer to the object. If we used the "from" space pointer to evict an
    // object with we would move it into the "from" space. That would be broken.
    
    assert (is_object_in_from_space (p_from_obj));
        
    Java_java_lang_Object *p_to_obj = get_to_object (p_from_obj);
   
    gc_trace (p_from_obj, "Making from object consistent.");
    gc_trace (p_to_obj, "Making to object consistent.");
    
    assert (p_from_obj != p_to_obj);
    assert (!is_object_in_from_space (p_to_obj));
    assert (!(p_global_bs->get_mature_generation()->p_nominate_focus_car())->is_address_in_this_car(p_to_obj));
    // This works for both arrays and objects.    
    char *end_of_from_object = (char *)p_from_obj + 
        get_real_object_size_bytes (p_from_obj) - OBJECT_HEADER_SIZE;
    char *object_from_scan_point = (char *)p_from_obj;    // Scan pointer in "from" object
    char *object_to_scan_point = (char *)p_to_obj;		 // Scan pointer in "to" object
    assert (p_to_obj->vt == p_from_obj->vt); // Should have been set when "to" replica made.
    // Someday use memcopy and skip vt copy for speed. This does a byte by byte copy.
    while (object_from_scan_point < end_of_from_object) {
        *object_to_scan_point = *object_from_scan_point;
        while (*object_to_scan_point != *object_from_scan_point) {
            // keep trying until mutator show no conflict.
            *object_to_scan_point = *object_from_scan_point;
        }
        object_from_scan_point++;
        object_to_scan_point++;
    }
}
//
// sapphire_maintain_consistency
// 
// Input -  the object that needs to updated. 
//			the slot in the object being updated.
//          the pointer value to update the object with.
// Comment - This routine is where dynamic consistency is maintained.
//           The invariants are that no "from" space pointers can be placed in "to" space.
//           Since "to" space pointers can be placed in "from" space the algorithm is 
//           to convert any "from" space value to a "to" space value and store it it
//           both "to" and "from" space.
// 
// The value will not be into from space, it will either be into to space or U space.
void sapphire_maintain_consistency (Java_java_lang_Object *p_obj, 
                                    unsigned offset,
                                    Java_java_lang_Object *value)
{ 
    if (p_global_bs->is_object_in_large_object_space(p_obj)) {
        // There is only on LOS copy of each object so it will always be
        // consistent with itself so our job is done here.
        Java_java_lang_Object **p_slot = 
            (Java_java_lang_Object **)(((char *)p_obj) + offset);
        *p_slot = value;
        return;
    }
    assert(!is_object_in_from_space(value));
    Java_java_lang_Object *p_to_obj = get_to_object (p_obj);
    Java_java_lang_Object *p_from_obj = get_from_object (p_obj);
    Java_java_lang_Object **p_to_slot = 
        (Java_java_lang_Object **)(((char *)p_to_obj) + offset); 
    Java_java_lang_Object **p_from_slot = 
        (Java_java_lang_Object **)(((char *)p_from_obj) + offset); 
    // I am not sure we need this loop?????????????    
    do {
        *p_to_slot = value;
        *p_from_slot = value;
    } while (*p_to_slot != *p_from_slot);
}

//
// sapphire_note_c_ref
//
// Comments - This is called by the routines that scan cards to note
// if we have a slot containing a los object.
// 
//

void sapphire_note_slot_with_c_ref(Java_java_lang_Object **p_slot)
{
    Java_java_lang_Object *value = *p_slot;
    // Stop and debug....
    assert (0); // For now there should be no cards marked since there is no older gen.
    if (!is_object_in_c_space(value)) {
        return; // If object isn't in c space ignore here.
    }
    switch (sapphire_state) {
    case normal: {
        assert(0); // why are we scanning cards now?
        break; 
                 } // Just do card marking for pointers as below
    case mark_reserve_phase: {
        ssb_write_object(c_space_ssb_collector, value);
        ssb_write_slot(c_space_ssb_collector, p_slot);
        break;
                             } // end mark_reserve_phase
    case repoint_phase: {
        assert(0);
        break; 
                        }
    case flip_phase:
    case stabilize_phase: {
        assert(0);
        break; }
    } // end switch
}

//
// sapphire_gray_white_object_reachable_from_u
//
// Comments- loop through all the objects in u that hold pointers to focus area (C) objects
// calling sapphire_scan_graying_white.
bool sapphire_gray_white_objects_reachable_from_u ()
{
    bool result = false;
    // Scan all the cards moving all the interesting slots into the appropriate remembered set,
    // including the remset returned by _p_focus_car->get_car_remembered_set();

    // This card scan deals with all u pointers to los also.
    p_global_card_table_manager->scan_cards (p_global_bs->get_mature_generation(),
        p_global_bs->get_young_generation());

    // Process the slots we discovered.

    ssb_move_writer_to_reader(c_space_ssb_collector, c_space_ssb);
    ssb_rewind(c_space_ssb);
    Java_java_lang_Object **pp_obj_ref;
    // Loop through all the pointers in u space that might point into the focus area.
    while ((pp_obj_ref = ssb_read(c_space_ssb)) != NULL) {
        if (ssb_is_object(pp_obj_ref)) {
            Java_java_lang_Object *p_the_obj = ssb_get_object(pp_obj_ref);
            gc_trace (ssb_get_object(pp_obj_ref), "scanning - this u object.");
            sapphire_scan_graying_white(p_the_obj);
            if (is_object_in_c_space(p_the_obj)) {
                if (is_object_white(p_the_obj)){
                    sapphire_gray_white_object(&p_the_obj, p_the_obj);
                    sapphire_blacken_gray_object (p_the_obj);
                }
            }
        } else {
            Java_java_lang_Object *p_obj = *pp_obj_ref;
            if (is_object_in_c_space(p_obj)) {
                if (is_object_white(p_obj)){
                    sapphire_gray_white_object(pp_obj_ref, p_obj);
                    sapphire_blacken_gray_object (p_obj);
                    result = true;
                }
            }
        }
    }

    // Process the slots from previous GCs.
    
    // 	why do I need the young and mature generations???);
    
    // get the cars write barrier
    Car *p_focus_car = p_global_bs->get_mature_generation()->p_nominate_focus_car();
    assert(p_focus_car);
    Remembered_Set *focus_area_remset = p_focus_car->get_car_remembered_set ();
    
    focus_area_remset->rewind();
    // Loop through all the pointers in u space that might point into the focus area.
    while ((pp_obj_ref = focus_area_remset->next()) != NULL) {
        Java_java_lang_Object *p_obj = *pp_obj_ref;
        gc_trace(*pp_obj_ref, "sapphire_reserve_to_space_and_los_mark_objects_with_refs_in_u looking at object.");
        if (is_object_in_c_space(p_obj)) {
            if (is_object_white(p_obj)){
                sapphire_gray_white_object(pp_obj_ref, p_obj);
                result = true;
            }
            if (is_object_in_copy_space(p_obj)) {
                ssb_write_slot (c_space_ssb_collector, pp_obj_ref);
                ssb_write_slot(ssb_repoint, pp_obj_ref);
            }
        }
    }
    return result;
}


//
// sapphire_scan_object_asserting_no_old_pointers
//
// The object should have no pointers into old space, just loop through and make sure.
// This is for debugging purposes only.
//
void sapphire_scan_object_asserting_no_old_pointers (Java_java_lang_Object *base_of_object_holding_ref)
{
    // Just grab the code up above once it is debugged and modify this
    //         *pp_child_ref = get_to_object(*pp_child_ref); to be 
    // assert (*pp_child_ref == get_to_object(*pp_child_ref));
}

//
// sapphire_driver - 
//
// This routine is the top level sapphire routine. As such it is responsible for managing
// the communication with the application threads. This includes blocking them, scanning their
// stacks to get pointers into the head, and flipping them. This routine also controls the
// which phase the write barrier is in.
// 

void sapphire_driver ()
{
    // p_sapphire_lock should be held by caller.so a thread can't destroy the GC
    // system inside an exit.
    unsigned long sapphire_number_of_spent_nurseries = p_global_bs->get_young_generation()->get_number_of_spent_nurseries();
    unsigned long sapphire_number_of_nurseries =  p_global_bs->get_young_generation()->get_number_of_nurseries();
    if ((sapphire_number_of_spent_nurseries * 2) < sapphire_number_of_nurseries ) {
//        orp_cout << " No gc yet, number of spent_nurserys is " << sapphire_number_of_spent_nurseries << endl;
        orp_cout << endl << "<Skip Sapphire GC>" << endl;
        return;
    }
    orp_cout << endl << "<Sapphire GC>" << endl;
    int nthreads = thread_gc_number_of_threads();
//    orp_cout << " top of sapphire_driver .. nthreads is " << nthreads;
    if (nthreads > 1) {
        nthreads = nthreads;
    }
    
    // Turn on the mark_copy write barrier, thus preventing any new pointers to white
    // objects from being placed into the entire heap.
    // DO I need an initialization phase? I think I do or the write barriers will 
    // start doing stuff before things are initialized.
    assert (sapphire_state == normal);
    
    
    // Concerning a race condition as we are moving nurseries into the focus car. There
    // is a time when an object in a nursery can be in U space and then moved into 
    // C space. This can happen during a write barrier. This is probable OK since
    // we have not started the mark phase yet and any such writes can be lost. However,
    // writes to nurseries that are used after a threads nursery is moved into the 
    // focus car need to be tracked. For this reason we assume that the mark_copy_phase
    // starts as soon as we start to evacuate nurseries in to the focus car. 
    
    assert(p_the_safepoint_control_thread == 0);
    p_the_safepoint_control_thread = p_TLS_orpthread;
    assert (global_safepoint_status == nill);
    global_safepoint_status = enumerate_the_universe;            //   vv
    
    // First determine what the focus (or C) area is - this will be the 
    // oldest car in the oldest train, move all the nurseries into this focus car.
    // Do we have conflicts with train algorithm. The check for pointers into the
    // train must be done at the end of the collection. This means that the write barrier
    // for sapphire must note any pointers into the focus car that are not into the focus train.
    
    sapphire_focus();
    assert (sapphire_state == normal);
    sapphire_state = mark_reserve_phase;
    sapphire_initialize();
    // Process the remembered set of the C area. Once this is done the U area will hold
    // no pointers to white objects in the C area. LOS and U objects will hold no
    // pointers to "to" versions of objects. The write barrier maintains this invariant.
    // The current implementation does not do a cheney scan but uses a mark stack to
    // scan the live C objects. At this point the "to" objects all point to "from" objects.
    sapphire_gray_white_objects_reachable_from_u();
    bool local_white_object_found = true;
    bool local_white_object_written = true;
    int check = 0;
    while (local_white_object_found || local_white_object_written) {
        // Call a routine to loop through all the active threads, blocking them and
        // then scanning their stacks for pointers into the C space. This routine will
        // return true if it encounters a white object.
        local_white_object_found = sapphire_scan_all_stacks();
        local_white_object_written = sapphire_gray_white_objects_reachable_from_u();
        check++; // No thread can write white so after 1 pass 
        // They won't be able to read white either.
        // If a white object was encountered repeat the previous two steps.
    }

    local_white_object_found = sapphire_scan_all_stacks();
    assert (!local_white_object_found);
    local_white_object_written = sapphire_gray_white_objects_reachable_from_u();
    assert (!local_white_object_written);
    // At this point all live objects in C space are either marked if they are in LOS
    // or have both "to" and "from" versions if they are in copy space.

//    orp_cout << "keep counts of object in different states so white/black to "
//        << "gray becomes the same as gray to black " << endl;
    
    // At this point U space los and "to" space can be treated uniformly and just repointed.
    // allocation space is in U space and all slots to "from" space should now be organized 
    // into a single SSB container.
    

    // Change the write barrier and the equal routine to the repoint phase. From here on a
    // thread can have pointers to both from space and to to space.
    assert (sapphire_state == mark_reserve_phase);
    sapphire_state = repoint_phase;
    
    // Call a routine to repoint the heap. When we return the repointing phase will be
    // complete and there will be no pointers from the U area into the old C area.
    // The write barrier will maintain this invariant by preventing C pointers from
    // being written into the heap.
    sapphire_repoint_refs_in_u_los_and_to_space();
    
    assert (c_space_ssb_collector->current_ssb == NULL);
    // Change the write barrier to the flip phase, the equal routine is the same as in the
    // repoint phase.
    sapphire_state = flip_phase;
    
    // Call a routine to loop through all the active threads one at a time, flipping them.
    // Once a thread is flipped it can no longer see any pointers into the old C (focus) area.
    // When this routine returns all the stack pointers into the C area are repointed 
    // into the new C area.
    sapphire_flip_all_threads();
    
    sapphire_state = stabilize_phase;

    sapphire_stabilize_all_threads();

    sapphire_state = normal;
    // Call a routine to release the old C area. This also releases the lock blocks that
    // sapphire has been using.
    sapphire_release_old_c_area();
    
//    orp_cout << "??? Calling orp_resume_a_thread_after from sapphire_driver. - cleanup." << endl;
    void orp_resume_a_thread_after (ORP_thread *p_thr, bool debug_breakpoint);
    orp_resume_a_thread_after (p_TLS_orpthread, false);
    
    ssb_release (c_space_ssb);

    global_safepoint_status = nill;                                   //^^^^
    p_the_safepoint_control_thread = 0;
    //	orp_cout << "Rinse and repeat - the sapphire driver code is at the end." << endl;

    
    nthreads = thread_gc_number_of_threads();
//    orp_cout << " bottom of sapphire_driver .. nthreads is " << nthreads;

    if (nthreads > 1) {
        nthreads = nthreads;
    }
    // Rinse and repeat...
}
//
// sapphire_scan_thread
//	INPUT - a thread
//	OUTPUT bool -	true if a pointer to a white object in the focus area is found, 
//					false otherwise
//
// This routine blocks the thread passed to it, moves it to a GC safe point, and then scans the
// stack looking for pointers into the focus area. 
// For each object it finds
// If the object is white then 
//		call sapphire_reserve_new_object to turn the white object gray.
//		set result to true.
// else
//		do nothing.
//
// Once the stack has been scanned the thread is unblocked and result is returned.
//
#if 0
bool sapphire_scan_this_stack (ORE_Thread *a_thread)
{
    move_thread_to_gc_safe_point (a_thread);
    bool xxwhite_object_found = false;
    if (sapphire_scan_stack (a_thread)) {
        xxwhite_object_found = true;
    }
}
#endif // 0
//
// sapphire_scan_all_stacks
//
//	OUTPUT bool -	true if a pointer to a white object in the focus area is found, 
//					false otherwise
//
// This routine loops through all the threads blocking them one at a time and scanning
// their stacks for pointers to white objects in the focus area. If a white object is found
// space is reserved in new space and the object is grayed. 
//
// The caller is responsible turning the object black by copying the contents over and 
// scanning the contents.
// 
// RACE CONDITION IF A NEW THREAD IS CREATED and place on front of thread list!!!!????
//
// Optimization opportunity - 
//	1. don't rescan stacks that have done no work since last scan.
//  2. scan threads that are halted anyway. This gives running threads a chance to 
//     halt on their own.
// 

bool sapphire_scan_all_stacks ()
{
    // sapphire_add_root_set_entry will be called with the slots.
    global_white_object_found = false;
//    orp_cout << "Add timing to sapphire_scan_all_stacke " << endl;    
    change_all_threads_sapphire_status(sapphire_enumerate);
    orp_enumerate_root_set_all_threads();
    // As we enumerate the threads we let them continue to run....
    //	orp_resume_threads_after(false);
    
    return global_white_object_found;
}

// 
// Flip the active thread.
//

void sapphire_flip_thread ()
{
    //	orp_cout << "(not) Flipping a thread" << endl;	
}

bool is_header_locked (Java_java_lang_Object *p_obj)
{
    volatile POINTER_SIZE_INT *p_header = P_OBJ_INFO(p_obj);
    if ((*p_header & BUSY_FORWARDING_BIT) == BUSY_FORWARDING_BIT) {
        return true;
    } else {
        return false;
    }
}
//
// sapphire_repoint_refs_in_u_los_and_to_space
//
// Comment - use the ssb created during the reserve phase to locate all references
//           to "from" space, including those in U, "to" and los spaces. 
// 
// The write barrier will prevent any new pointers to "from" space from being added into U.
//
void sapphire_repoint_refs_in_u_los_and_to_space ()
{
    ssb_move_writer_to_reader(c_space_ssb_collector, c_space_ssb);
    ssb_rewind(c_space_ssb);
    Java_java_lang_Object **pp_obj_ref;
    int count = 0;
    int change_count = 0;
    // Loop through all the pointers in u space that might point into the focus area.
    while ((pp_obj_ref = ssb_read(c_space_ssb)) != NULL) {
        count++;
        if (ssb_is_object(pp_obj_ref)) {
            sapphire_repoint(ssb_get_object(pp_obj_ref));
        } else {
            Java_java_lang_Object *p_target_obj = *pp_obj_ref;
            if (is_object_in_copy_space(p_target_obj)) {
                gc_trace (p_target_obj, "Getting this objects to object and installing.");
                // This is a pointer to a from object, repoint to the "to" copy.
                void *result = InterlockedCompareExchangePointer((void **)pp_obj_ref, 
                    (void *)get_to_object(p_target_obj), 
                    (void *)p_target_obj);
                change_count++;
                assert(!(is_object_in_from_space(*pp_obj_ref)));
                // If it works great, if not the write barrier will take care of it.
            }
        }
    }
//    orp_cout << "Total Count in repoint. " << count << endl;
//    orp_cout << "Change Count in repoint. " << change_count << endl;

}

// sapphire_flip_all_threads
//
// Comment - At this point all the "from" pointers in the heap have been repointed to
// point into "to" space. The write barrier is set up so that no thread can write a "from"
// pointer into the heap. This means that the only "from" pointers are in the thread stacks
// and the thread engine. We now have to "flip" the threads so that they have no pointers into
// "from" space. Once flipped we can release "from" space.
//
// New threads are not an issue since they can never see into "from" space.
//
// HACK HACK - hidden interface.. Rick is at fault.... 
void change_all_threads_sapphire_status (gc_sapphire_state state);
void sapphire_flip_all_threads()
{
#ifdef GC_FIXED_V1
    return; // No need to flip threads where objects don't move.
#else

#ifdef _DEBUG
    ssb *last_ssb_in_c_space_ssb = c_space_ssb->last_ssb;
    // At this point there should be no "from" pointers in the heap.
    assert(c_space_ssb_collector->current_ssb == NULL);
    ssb_move_writer_to_reader(c_space_ssb_collector, c_space_ssb);
    ssb_rewind (c_space_ssb);
    Java_java_lang_Object **pp_obj_ref;
    while ((pp_obj_ref = ssb_read(c_space_ssb)) != NULL) {
        if (ssb_is_object(pp_obj_ref)) {
            // This object can be in to space.
//            assert(!is_object_in_from_space(ssb_get_object(pp_obj_ref)));
        } else {
            Java_java_lang_Object *p_target_obj = *pp_obj_ref;
            if(is_object_in_from_space(p_target_obj)) {
//                orp_cout << "Fixup of bad pointer " << p_target_obj << endl;
                Java_java_lang_Object *p_to_obj = get_to_object(p_target_obj);
//                orp_cout << "                                        pointer at " << pp_obj_ref << endl;
//                orp_cout << "                                                    good pointer " << p_to_obj << endl;
                *pp_obj_ref = p_to_obj;
                assert (is_object_in_to_space(p_to_obj));
            }
        }
    }
    
#endif
    
    change_all_threads_sapphire_status(sapphire_flip);
    
    // First set all the threads to indicate that they are available to be flipped.
    // This will help the write barriers know if they can flip the thread and
    // will let everyone know when a thread is flipped. Flipping a thread multiple
    // times is not a bug, it just wastes time. 
    // BUG BUG BUG
    // Writes of volatiles need to be at GC safe points so we can flip them.
    orp_enumerate_root_set_all_threads();
    
#endif
    
#ifdef _DEBUG
    assert(c_space_ssb_collector->current_ssb == NULL);
    ssb_move_writer_to_reader(c_space_ssb_collector, c_space_ssb);
    ssb_rewind (c_space_ssb);
    while ((pp_obj_ref = ssb_read(c_space_ssb)) != NULL) {
        if (ssb_is_object(pp_obj_ref)) {
            // This object can be in to space.
//            assert(!is_object_in_from_space(ssb_get_object(pp_obj_ref)));
        } else {
            Java_java_lang_Object *p_target_obj = *pp_obj_ref;
            if(is_object_in_from_space(p_target_obj)) {
                orp_cout << "Fixup of bad pointer " << p_target_obj << endl;
                Java_java_lang_Object *p_to_obj = get_to_object(p_target_obj);
                orp_cout << "                                        pointer at " << pp_obj_ref << endl;
                orp_cout << "                                                    good pointer " << p_to_obj << endl;
                *pp_obj_ref = p_to_obj;
                assert (is_object_in_to_space(p_to_obj));
                // We need to flip the threads again since we found a "from" object
                // that could have been read by a thread.
                sapphire_flip_all_threads();
            }
        }
    }
    
#endif
    // Threads are restarted after they are enumerated
    //	orp_resume_threads_after(false);
}
// sapphire_stabilize_all_threads is designed to push all of the code out of
// monitor enter/exit critical sections so that we can move on to a normal state.
// Monitor enter and exit treat normal and stabilize as the same but when we are in 
// the stabilize state the "from" objects are still available though they aren't used.
// This was needed because of the integrity checks done in monitor enter/exit.

void sapphire_stabilize_all_threads()
{
#ifdef GC_FIXED_V1
    return; // No need to flip threads where objects don't move.
#else
    change_all_threads_sapphire_status(sapphire_stabilize);
    
    // First set all the threads to indicate that they are available to be flipped.
    // This will help the write barriers know if they can flip the thread and
    // will let everyone know when a thread is flipped. Flipping a thread multiple
    // times is not a bug, it just wastes time. 
    // BUG BUG BUG
    // Writes of volatiles need to be at GC safe points so we can flip them.
    orp_enumerate_root_set_all_threads();
#endif
#ifdef _DEBUG
    assert(c_space_ssb_collector->current_ssb == NULL);
    ssb_move_writer_to_reader(c_space_ssb_collector, c_space_ssb);
    ssb_rewind (c_space_ssb);
    Java_java_lang_Object **pp_obj_ref;
    while ((pp_obj_ref = ssb_read(c_space_ssb)) != NULL) {
        if (ssb_is_object(pp_obj_ref)) {
            // This object can be in to space.
//            assert(!is_object_in_from_space(ssb_get_object(pp_obj_ref)));
        } else {
            Java_java_lang_Object *p_target_obj = *pp_obj_ref;
            assert(!is_object_in_from_space(p_target_obj));
        }
    }

    
#endif

}
// sapphire_add_root_set_entry
//
// Input - a pointer to a slot holding a reference. The slot is in the thread engine, either
//		   on a stack or in some local place. It is not in the heap.
//
// Comment - This is the call back from the code that enumerates the stacks and the globals.
//
void sapphire_add_root_set_entry(Java_java_lang_Object **pp_obj) 
{
    if (!p_global_bs->is_reference_in_heap (pp_obj)) {
        pp_obj = pp_obj; // For some reason we are enumerating classes that are in the heap?????
    }
    
    Java_java_lang_Object *p_obj = *pp_obj;
    // Is it an interesting?
    if (p_obj == NULL) {
        // it isn't interesting
        return;
    }
    gc_trace (*pp_obj, "Slot holding this is passed to sapphire_add_root_set_entry.");
    if (is_object_in_c_space (p_obj)) {
        // It is interesting. What state is sapphire in.
        switch (sapphire_state) {
        case normal:
            { 
                assert (0);
                break;
            } // Can't happen why is someone scanning a stack?
        case mark_reserve_phase: 
            {
                assert (!is_object_in_to_space(p_obj));
                // At this point we should not be able to see "to" version of objects.
                switch (sapphire_object_color (p_obj)) {
                case white: {
                    ssb_write_object(c_space_ssb_collector, p_obj);
                    break;
#if 0
                    // If an object is in LOS and it was grayed then it is considered
                    // to not be in "from" space any longer.
                    bool in_los = p_global_bs->is_object_in_large_object_space(p_obj);
                    // reserve some new space and set up pointers.
                    gc_trace (p_obj, "graying and then blackening this object.");
                    sapphire_gray_white_object (pp_obj, p_obj);
                    if (!in_los) {
                        assert (is_object_in_from_space (p_obj));
                        assert (sapphire_object_color(*pp_obj) != white);
                        Java_java_lang_Object *to_obj = get_to_object (*pp_obj);
                        assert (sapphire_object_color(to_obj) != white);
                    } else {
                        // No gray in LOS.
                        assert (is_object_marked(*pp_obj));
                        assert (sapphire_object_color(*pp_obj) == black); // race condition?
                    }
                    sapphire_blacken_gray_object (*pp_obj);
                    global_white_object_found = true;
                    break;
#endif
                            }
                case gray: {
                    break;
                    // do nothing, cheney will do the copy and updating. LOS is done.
                           }
                case black:{
                    break;
                    // do nothing, cheney will do the copy and updating. LOS is done.
                           }
                }  // end switch (sapphire_object_color (*ref))
                break;
            } // end mark_reserve_phase
            
        case repoint_phase: 
            {
                assert (0); // All scanning should be completed in the mark_copy_phase
                break;
            }
        case flip_phase: 
            {
                if (is_object_in_from_space(p_obj)) {
                    // Do we have to repoint this object on a stack.
                    Java_java_lang_Object *to_object = get_to_object(p_obj);
                    // There can be to objects on the stack at this time.
                    assert (p_obj != to_object); // LOS objects do not move
                    // So this isn't true I don't think.
                    // halt and debug. 
                    // Wrong -> a marked
                    // LOS object is considered to be
                    // in to space so won't be "flipped"
                    assert (sapphire_object_color (*pp_obj) == black);
                    gc_trace (*pp_obj, "Flip phase updating from object to ref to object");
                    gc_trace (to_object, "slot updated to point to this to object");
                    // We are flipping the stacks so all the work should be done.
                    *pp_obj = to_object;
                }
                break;
            }
        case stabilize_phase:
            {    
                if (is_object_in_from_space(p_obj)) {
                    assert (0);
                    // At this point we should have no "from" objects reachable
                    // from the stack....
                }
                // Do nothing we are just pushing the threads out of the
                // monitor enter/exit routines.
                break;
            }
        default:
            {
                assert (0);
                break;
            }
        } // end switch (sapphire_state)
    } // 	endif (is_object_in_c_space (*pp_obj)) {
    if ((sapphire_state == flip_phase) 
        && is_object_in_c_space (*pp_obj)
        && (sapphire_object_color(*pp_obj) != black)) {
        // If we are in the enumeration phase it could have been scanned
        // Of more importance is that if we are in the flip phase then
        // this had better be black.
        orp_cout << "BUG BUG object is not black and we are flipping." << endl;
        //        assert (sapphire_object_color(*pp_obj) == black);    
    }
    return;
}
// 
// sapphire_gc_malloc_or_null
//
// Comment - Allocate an object in the youngest car of the youngest train. This routine replaces
// the normal gc_malloc_or_null that allocates objects in nurseries. Since sapphire is concurrent
// allocation must preceed in U space while collection is happening.
Java_java_lang_Object *sapphire_gc_malloc_or_null (unsigned object_size, VTable *p_vtable)
{ 
    // The slot could be in the the heap or on a stack or in some global.
    // Figure out which train the "to" space is in reserve space in that train.
    Train *to_train = p_global_bs->get_mature_generation()->p_get_youngest_train();
    Java_java_lang_Object *p_obj = to_train->p_reserve_bytes(object_size);
    if (p_obj != NULL) {
        p_obj->vt = (VTable *)p_vtable;
    }
    return p_obj;
}

Java_java_lang_Object *sapphire_gc_malloc (unsigned object_size, Partial_Reveal_VTable *p_vtable)
{
    return NULL;
    // The slot could be in the the heap or on a stack or in some global.
    // Figure out which train the "to" space is in reserve space in that train.
    Train *to_train = p_global_bs->get_mature_generation()->p_get_youngest_train();
    Java_java_lang_Object *p_obj = to_train->p_reserve_bytes(object_size);
    if (p_obj == NULL) {
        // We need the sapphire_lock......
        assert (0);
        sapphire_driver();
        Train *to_train = p_global_bs->get_mature_generation()->p_get_youngest_train();
        p_obj = to_train->p_reserve_bytes(object_size);
    }
    assert (p_obj);
    gc_trace (p_obj, "sapphire just allocated the object.");
    volatile POINTER_SIZE_INT *p_header = P_OBJ_INFO(p_obj);
    memset ((void *)p_header, 0, object_size);
    p_obj->vt = (VTable *)p_vtable;
    return p_obj;
}

//
// gc_sapphire_equal - do a pointer compare.
// Since there can be two version of an object during certain Sapphire phases
// equal must be smarter than a normal equal. 
// If we are in phases where the two version are visible to application threadst then
// if either of the objects have a "to" version then we will use the "to" version to do
// the pointer compare.
//
int gc_sapphire_equal (Java_java_lang_Object *p_obj1, Java_java_lang_Object *p_obj2)
{
    if (p_obj1 == p_obj2) {
        return true;
    }
    switch (sapphire_state) {
    case normal:
    case initialize_phase:
    case mark_reserve_phase:
    case stabilize_phase:
        {
            return false;
        } // end normal mark_reserve_phase
    case repoint_phase: 
    case flip_phase: 
        {
            Java_java_lang_Object *p_to_obj1 = p_obj1;
            Java_java_lang_Object *p_to_obj2 = p_obj2;
            if (is_object_in_from_space(p_obj1)) {
                p_to_obj1 = get_to_object(p_obj1);
            }
            if (is_object_in_from_space(p_obj2)) {
                p_to_obj2 = get_to_object(p_obj2);
            }
            bool result = (p_to_obj1 == p_to_obj2);
            return result;
        }
    default:
        {
            assert (0);
            break;
        }
    } // end switch (sapphire_state)
    assert (0);
    return false; // Keep the compiler happy.
}

#endif // GC_SAPPHIRE
