// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc_v2/include/gc_space.h,v 1.6 2001/11/05 13:36:35 rlhudson Exp $
//

#ifndef _gc_space_H_
#define _gc_space_H_


//
// A Gc_Space is an abstract class representing a container. A container
// is an entity that can hold Java objects. An example of a container is
// a Car. All containers inherit (directly or indirectly) from Gc_Space.
//
//                          Gc_Space (abstract)
//                              |
//             ------------------------------------
//             |                                  |
//             |                                  |
//        Block_List (abstract)                   |
//      ---------------------                     |
//      |                    |                 Nursery
//     Car                  Step
//
//
//


#include "platform.h"
 
#include "object_layout.h"
#include "gc_header.h"
#include "gc_component.h"

class Block_Store;
class Card_Table;
class Remembered_Set;
class Generation;
class Gc_Fast_Hooks;
class Gc_Plan;

class Gc_Space : public Gc_Component {

public:

	Gc_Space(unsigned      id,
             Gc_Fast_Hooks *p_gc_hooks,
             Gc_Plan       *p_gc_plan,
		     Generation    *p_container_gen,
			 Gc_Space      *p_superior,
             Card_Table    *p_card_table,
			 Block_Store   *p_block_store) 

             : Gc_Component(p_gc_hooks, 
                            p_gc_plan,
                            p_block_store)
    
    {

        //
        // The following uniquely identifies all spaces. Actually it uniquely
        // identifies a step from other steps, a car from other cars, etc...
        //
		identity       = id;
       
        //
        // The GC heap is also tiled into fixed size cards (smaller than blocks),
        // and the card table object is used for card marking write barriers.
        //
		_p_card_table  = p_card_table;
        //
        // Each GC Space has a container, which is the generation it belongs to.
        //
		p_container    = p_container_gen;
        //
        // Every GC Space has a "superior" GC Space which is the promotion target
        // of objects in this space.
        //
		_p_superior    = p_superior;
        //
        // This really needs to be in car.h, since only cars need container write
        // barriers.
        //
		_p_container_write_barrier = NULL;
        //
        // This keeps a record of the last object that was added into this GC Space,
        // to facilitate subsequent scanning of this GC Space.
        //
		_p_last_object_gc_header = NULL;
        //
        // A count of the number of objects allocated into this space so far.
        //
		_resident_object_count   = 0;
        //
        // A count of the number of bytes allocated so far.
        //
		_resident_occupied_bytes = 0;
	}
	//
	// GC Space deletion. Cars get deleted frequently, but steps and
    // Nurseries usually last for the duration of the ORP's life.
	//
	virtual ~Gc_Space() {
	}

	//
	// Do any cleanup at the end of the reclamation. Need to make this
    // a pure virtual function.
	//
	virtual void cleanup(){
		//
		// This had better be specialized.
		//
		orp_exit(1); 
	}
	//
	// The GC space enclosing this space. (For ex: a train encloses cars)
    // I made this public for performance reasons rds 5/28.)
	//
	Generation       *p_container;


    //
    // Return the ID (car ID, Step ID, Nursery ID, etc..)
    //
	inline virtual unsigned long get_id() {
		return identity;
	}
	//
	// Return the current number of objects this space holds.
	//
	int get_resident_object_count() {
		return _resident_object_count;
	}
	//
	// Return the current number of bytes occupied in this space.
	//
	unsigned long get_resident_occupied_bytes() {
		return _resident_occupied_bytes;
	}

	//
	// Membership checking routines
	//
	virtual bool is_my_object(Java_java_lang_Object *p_obj);

	virtual bool is_my_reference(Java_java_lang_Object **pp_obj_ref);

	virtual bool is_my_address(void *p_addr);

    //
    // Is specified reference in a space that this generation contains?
    //
    bool is_reference_in_my_generation(Java_java_lang_Object **pp_obj_ref);
	//
	// Is specified object in a space that this generation contains.
	//
    bool is_address_in_my_generation(void *p_addr);

	unsigned long resident_count() {
		return _resident_object_count;
	}

	unsigned long resident_occupied_bytes() {
		return _resident_occupied_bytes;
	}

  
	//
	// Pointer to beginning of region.
	//

#ifdef OBJECT_SPLITTING
	void *_p_cold_free;
    //
	// Pointer to beginning of cold field region ..
	//
#endif // OBJECT_SPLITTING

protected:

#ifdef OBJECT_SPLITTING

	// To allow finding the size of the nursery or step block...
	friend extern void fast_copy_forward_and_update_locks(Object_Gc_Header *p_old_hdr,
										Object_Gc_Header *p_new_hdr,
										unsigned real_object_size_bytes);
#endif // OBJECT_SPLITTING

	//
	// Restart a cheney scan.
	//
	virtual void _rewind_scan_pointer () 
	{	
		_p_scan = _p_scan_start;
	}
    //
    //  The card table for supporting write barriers
    //
    Card_Table       *_p_card_table;
	//
	// Generation, train, car, nursery or step number:
	//
	unsigned long        identity;
	//
	// The last allocated object in this "segment".
	//
	Object_Gc_Header *_p_last_object_gc_header;

	//
	// Number of objects currently contained in this space
	//
	int _resident_object_count;
	//
	// Space used up by these resident objects
	//
	unsigned long _resident_occupied_bytes;

	//
	// The (pointers to) references of incoming pointers for this space.
	//
	Remembered_Set   *_p_container_write_barrier;

	//
	// The subordinate space that tenures/promotes into this space.
	//
	Gc_Space         *_p_subordinate;
	//
	// The target space to tenure/promote into.
	//
	Gc_Space         *_p_superior;

#ifdef OBJECT_SPLITTING
	void *_p_cold_free;
    //
	// Pointer to beginning of cold field region ..
	//
#endif // OBJECT_SPLITTING

    //
    // Pointer to next object to cheney scan.
    //
    void *_p_scan;
	//
	// Pointer to where we start a cheney scan. This is used by Sapphire since
	// we need to do cheney scans for each phase of sapphire.
	//
	void *_p_scan_start;

private:

    bool _is_object_in_my_generation(Java_java_lang_Object *p_obj);

};

#endif // _gc_space_H_
